(defvar latelf-documentstyle
  "\documentstyle[deduction,fullpage,12pt]{article}")

(defvar latelf-unit-prefix
  "\n\\[ \\begin{array}{c}")

(defvar latelf-unit-postfix
  "\n\\end{array} \\]\n\n\\rule{4in}{.4pt}\n")

(defvar latelf-buffer
  "*latelf*")

(defun fix1s (n)
  (interactive "nMaximal number of 1s: \nr")
  (while (> n 0)
    (goto-char (point-min))
    (let ((old (make-string n (string-to-char "1")))
	  (new (concat "_" (int-to-string n))))
      (while (search-forward old (point-max) t)
	(replace-match new nil t)))
    (setq n (1- n)))
  (goto-char (point-min))
  (while (search-forward "__" (point-max) t)
    (replace-match "_" nil t)))

(defun latelf-before-point (&optional bound)
  (interactive)
  (let ((old-point (point)))
    (save-excursion
      (search-backward "@<" bound)
      (latelf-stream-to-buffer (point) old-point
			       (get-buffer-create latelf-buffer)))))

(defun latelf-region (beg end)
  (interactive "r")
  (goto-char beg)
  (while (search-forward "@<" end t)
    (backward-char 2)
    (latelf-stream-to-buffer (point) end
			     (get-buffer-create latelf-buffer)))
  )

(defun latelf-queries (&optional displayp)
  "Execute all embedded queries in current Elf server state.
Shows progress if prefix argument is given."
  (interactive "P")
  (goto-char (point-min))
  (while (latelf-next-query displayp))
  (message "Done"))

(defun latelf-query (&optional displayp)
  "Execute next embedded query in current Elf server state.
Show progress if prefix argument is given.
Signals error if no query is found."
  (interactive "P")
  (if (not (latelf-next-query displayp))
      (error "Could not find a query")))

(defun latelf-next-query (displayp)
  "Execute next embedded query in current Elf server state."
  (let ((query-begin nil)
	(query-end nil)
	(elf-server-buffer (get-elf-server-buffer))
	;; DYNAMIC's follow
	(latelf-buffer (current-buffer))
	(latelf-unit-prefix "\\[ \\begin{array}{c}")
	(latelf-unit-postfix "\\end{array} \\]")
	)
    (cond ((search-forward "\\query{" (point-max) t)
	   (setq query-begin (point))
	   (backward-char 1)
	   (forward-sexp 1)
	   (setq query-end (1- (point)))
	   (kill-sexp 1)
	   (insert "{}")
	   (backward-char 1)
	   (elf-server-send-command
	    (concat "solve* 1\n"
		    (elf-buffer-substring-dot query-begin query-end)))
	   (elf-server-wait displayp)
	   (save-excursion
	     (set-buffer elf-server-buffer)
	     (latelf-before-point *elf-server-last-process-mark*))
	   t)
	  (t nil))))

(defun latelf-stream-to-buffer (beg end buffer)
  (let ((tokenlist (latelf-stream-to-list beg end))
	(new-begin nil))
    (save-excursion
      (set-buffer buffer)
      (setq new-begin (point))
      (apply 'insert (cons latelf-unit-prefix tokenlist))
      (insert latelf-unit-postfix)
      (if (looking-at "\n") (forward-char 1))
      (save-restriction
	(narrow-to-region new-begin (point))
	(fix1s 9)
	(goto-char (point-min))
	(fill-paragraph nil)))))

(defun latelf-stream-to-list (beg end)
  "Convert token stream in current region to LaTeX source."
  (let ((stream-beg)
	(stream-end)
	(token-start)
	(result nil)
	(spacep nil))
    (goto-char beg)
    (if (not (search-forward "@<" end t))
	(error "No begin stream (@<) found")
      (setq stream-beg (point)))
    (if (not (search-forward ">@" end t))
	(error "No end stream (>@) found")
      (setq stream-end (point)))
    (goto-char stream-beg)
    (while (re-search-forward "[$#]" stream-end 'limit)
      (backward-char 1)
      (setq token-start (point))
      (cond
       ((looking-at "\\$<\\$")
	(setq result (cons "{" result))
	(forward-char 3))
       ((looking-at "\\$>\\$")
	(setq result (cons "}" result))
	(forward-char 3))
       ((looking-at "\\$<<\\$")
	(setq result (cons "(" result))
	(forward-char 4))
       ((looking-at "\\$>>\\$")
	(setq result (cons ")" result))
	(forward-char 4))
       ((looking-at "\\$\\\\\\\\\\$")
	(setq result (cons "\\\\[2em]" result))
	(forward-char 4))
       ((looking-at "\\$\\$")
	(setq result (cons " " result))
	(forward-char 2))
       ((looking-at "##")
	(if (not (search-forward "[" stream-end t))
	    (error "No variable binder ([x:A]) found")
	  (setq token-start (point))
	  (if (not (re-search-forward "[]:]" stream-end t))
	      (error "No variable name delimiter : or ] found")
	    (setq result (cons (buffer-substring token-start (1- (point)))
			       result)))))
       ((looking-at "#<")
	(if (not (search-forward ">#" stream-end t))
	    (error "No end quote (>#) found")
	  (setq result (cons (buffer-substring (+ token-start 2)
					       (- (point) 2))
			     result))))
       ((looking-at "\\$")
	(setq spacep (looking-at "\\$\\\\"))
	(forward-char 1)
	(if (not (search-forward "$" stream-end t))
	    (error "No end token ($) found")
	  (setq result (cons (buffer-substring (+ token-start 1)
					       (- (point) 1))
			     result))
	  (if spacep (setq result (cons " " result)))))
       (t (error "Unrecognized token"))))
    (nreverse result)))
