/*******************************************************************\

Module:

Author: Daniel Kroening, kroening@cs.cmu.edu

\*******************************************************************/

#include "irep.h"
#include "trfalse.h"
#include "i2string.h"

using namespace std;

const irept nil_rep("nil");

/*******************************************************************\

Function:

  Inputs:

 Outputs:

 Purpose:

\*******************************************************************/

void irept::move_to_named_sub(const std::string &name, irept &irep)
 {
  add(name).swap(irep);
  irep.make_nil();
 }

/*******************************************************************\

Function:

  Inputs:

 Outputs:

 Purpose:

\*******************************************************************/

void irept::move_to_sub(irept &irep)
 {
  sub.push_back(nil_rep);
  sub.back().swap(irep);
 }

/*******************************************************************\

Function:

  Inputs:

 Outputs:

 Purpose:

\*******************************************************************/

const string &irept::get(const string &name) const
 {
  const named_subt &s=
    is_comment(name)?comments:named_sub;

  named_subt::const_iterator it=s.find(name);

  if(it==s.end())
   {
    const static std::string empty;
    return empty;
   }

  return it->second.id;
 }

/*******************************************************************\

Function: irept::get_bool

  Inputs:

 Outputs:

 Purpose:

\*******************************************************************/

bool irept::get_bool(const string &name) const
 {
  return atoi(get(name).c_str());
 }

/*******************************************************************\

Function:

  Inputs:

 Outputs:

 Purpose:

\*******************************************************************/

void irept::swap(irept &irep)
 {
  id.swap(irep.id);
  sub.swap(irep.sub);
  named_sub.swap(irep.named_sub);
  comments.swap(irep.comments);
 }

/*******************************************************************\

Function:

  Inputs:

 Outputs:

 Purpose:

\*******************************************************************/

void irept::set(const string &name, const long value)
 {
  add(name).id=i2string((int)value);
 }  

/*******************************************************************\

Function:

  Inputs:

 Outputs:

 Purpose:

\*******************************************************************/

void irept::remove(const string &name)
 {
  named_subt &s=
    is_comment(name)?comments:named_sub;

  named_subt::iterator it=s.find(name);

  if(it!=s.end()) s.erase(it);
 }  

/*******************************************************************\

Function:

  Inputs:

 Outputs:

 Purpose:

\*******************************************************************/

void irept::set(const string &name, const irept &irep)
 {
  irept &added=add(name);
  added.id=irep.id;
  added.sub=irep.sub;
  added.named_sub=irep.named_sub;
  added.comments=irep.comments;
 }  

/*******************************************************************\

Function:

  Inputs:

 Outputs:

 Purpose:

\*******************************************************************/

irept &irept::find(const string &name)
 {
  named_subt &s=
    is_comment(name)?comments:named_sub;

  named_subt::iterator it=s.find(name);

  static irept local_nil_rep;
  local_nil_rep.make_nil();

  if(it==s.end())
    return local_nil_rep;

  return it->second;
 }

/*******************************************************************\

Function:

  Inputs:

 Outputs:

 Purpose:

\*******************************************************************/

const irept &irept::find(const string &name) const
 {
  const named_subt &s=
    is_comment(name)?comments:named_sub;

  named_subt::const_iterator it=s.find(name);

  if(it==s.end())
    return nil_rep;

  return it->second;
 }

/*******************************************************************\

Function:

  Inputs:

 Outputs:

 Purpose:

\*******************************************************************/

irept &irept::add(const string &name)
 {
  named_subt &s=
    is_comment(name)?comments:named_sub;

  pair<named_subt::iterator, bool> result=
    s.insert(pair<std::string, irept>(name, irept("")));

  return result.first->second;
 }

/*******************************************************************\

Function: operator==

  Inputs:

 Outputs:

 Purpose:

\*******************************************************************/

bool operator==(const irept &i1, const irept &i2)
 {
  if(i1.id!=i2.id) return FALSE;

  if(i1.sub!=i2.sub) return FALSE; // recursive call

  if(i1.named_sub!=i2.named_sub) return FALSE; // recursive call

  // comments are NOT checked

  return TRUE;
 }

/*******************************************************************\

Function: irept::to_string

  Inputs:

 Outputs:

 Purpose:

\*******************************************************************/

#include "lispirep.h"

std::string irept::to_string() const
 {
  lispexprt lispexpr;
  irep2lisp(*this, lispexpr);
  return lispexpr.expr2string();
 }

/*******************************************************************\

Function: operator<<

  Inputs:

 Outputs:

 Purpose:

\*******************************************************************/

std::ostream& operator<< (std::ostream& out, const irept &irep)
 {
  out << irep.to_string();
  return out;
 }

/*******************************************************************\

Function: ordering

  Inputs:

 Outputs:

 Purpose: defines ordering on the internal represenation

\*******************************************************************/

bool ordering(const irept &X, const irept &Y)
 {
  if(X.id<Y.id) return TRUE;
  if(Y.id<X.id) return FALSE;

  #if 0
  if(X.sub<Y.sub) return TRUE;  // recursive call
  if(Y.sub<X.sub) return FALSE; // recursive call
  #else

  for(irept::subt::const_iterator
        it1=X.sub.begin(),
        it2=Y.sub.begin();;
        it1++,
        it2++)
   {
    if(it1==X.sub.end())
     {
      if(it2==Y.sub.end())
        break;

      return TRUE;
     }
    else if(it2==Y.sub.end())
      return FALSE;
    else
     {
      if(ordering(*it1, *it2)) return TRUE;
      if(ordering(*it2, *it1)) return FALSE;
     }
   }

  #endif

  #if 0
  if(X.named_sub<Y.named_sub) return TRUE;  // recursive call
  if(Y.named_sub<X.named_sub) return FALSE; // recursive call
  #else

  for(irept::named_subt::const_iterator
        it1=X.named_sub.begin(),
        it2=Y.named_sub.begin();;
        it1++,
        it2++)
   {
    if(it1==X.named_sub.end())
     {
      if(it2==Y.named_sub.end())
        break;

      return TRUE;
     }
    else if(it2==Y.named_sub.end())
      return FALSE;
    else
     {
      if(it1->first<it2->first) return TRUE;
      if(it2->first<it1->first) return FALSE;

      if(ordering(it1->second, it2->second)) return TRUE;
      if(ordering(it2->second, it1->second)) return FALSE;
     }
   }
  #endif

  return FALSE;
 }

/*******************************************************************\

Function: operator<

  Inputs:

 Outputs:

 Purpose: defines ordering on the internal represenation

\*******************************************************************/

bool operator<(const irept &X, const irept &Y)
 {
  return ordering(X, Y);
 }

/*******************************************************************\

Function:

  Inputs:

 Outputs:

 Purpose:

\*******************************************************************/

size_t hash_irep_string(const irep_string &s)
 {
  size_t result=0;

  for(unsigned i=0; i<s.size(); i++)
   {
    result=result<<1;
    result+=s[i];
   }

  return result;
 }

/*******************************************************************\

Function:

  Inputs:

 Outputs:

 Purpose:

\*******************************************************************/

size_t irept::hash() const
 {
  size_t result=hash_irep_string(id);

  forall_irep(it, sub) result=result^it->hash();

  forall_named_irep(it, named_sub)
   {
    result=result^hash_irep_string(it->first);
    result=result^it->second.hash();
   }

  return result;
 }

/*******************************************************************\

Function: indent

  Inputs:

 Outputs:

 Purpose:

\*******************************************************************/

static void indent_str(std::string &s, unsigned indent)
 {
  for(unsigned i=0; i<indent; i++)
    s+=' ';
 }

/*******************************************************************\

Function:

  Inputs:

 Outputs:

 Purpose:

\*******************************************************************/

std::string irept::pretty(unsigned indent) const
 {
  std::string result;

  if(id!="")
   {
    result+=id;
    indent+=2;
   }

  forall_named_irep(it, named_sub)
   {
    result+="\n";
    indent_str(result, indent);

    result+="* ";
    result+=it->first;
    result+=": ";

    result+=it->second.pretty(indent+2);
   }

  forall_named_irep(it, comments)
   {
    result+="\n";
    indent_str(result, indent);

    result+="* ";
    result+=it->first;
    result+=": ";

    result+=it->second.pretty(indent+2);
   }

  unsigned count=0;

  forall_irep(it, sub)
   {
    result+="\n";
    indent_str(result, indent);

    result+=i2string(count++);
    result+=": ";

    result+=it->pretty(indent+2);
   }

  return result;
 }
