/*******************************************************************\

Module:

Author: Daniel Kroening, kroening@cs.cmu.edu

\*******************************************************************/

#include "mp_arith.h"
#include "trfalse.h"
#include "expr.h"
#include "bitvector.h"

/*******************************************************************\

Function:

  Inputs:

 Outputs:

 Purpose:

\*******************************************************************/

void exprt::move_to_operands(exprt &expr)
 {
  operandst &op=operands();
  op.push_back((exprt &)nil_rep);
  op.back().swap(expr);
 }

/*******************************************************************\

Function:

  Inputs:

 Outputs:

 Purpose:

\*******************************************************************/

void exprt::copy_to_operands(const exprt &expr)
 {
  operands().push_back(expr);
 }

/*******************************************************************\

Function: exprt::make_typecast

  Inputs:

 Outputs:

 Purpose:

\*******************************************************************/

void exprt::make_typecast(const typet &_type)
 {
  exprt new_expr;

  new_expr.move_to_operands(*this);
  new_expr.set("type", _type);
  new_expr.id="typecast";

  swap(new_expr);
 }

/*******************************************************************\

Function: exprt::make_not

  Inputs:

 Outputs:

 Purpose:

\*******************************************************************/

void exprt::make_not()
 {
  exprt new_expr;

  new_expr.type()=type();
  new_expr.move_to_operands(*this);
  new_expr.id="not";

  swap(new_expr);
 }

/*******************************************************************\

Function:

  Inputs:

 Outputs:

 Purpose:

\*******************************************************************/

bool exprt::is_constant() const
 {
  return id=="constant";
 }

/*******************************************************************\

Function:

  Inputs:

 Outputs:

 Purpose:

\*******************************************************************/

bool exprt::is_true() const
 {
  return is_constant() &&
         get("value")!="false";
 }

/*******************************************************************\

Function:

  Inputs:

 Outputs:

 Purpose:

\*******************************************************************/

bool exprt::is_false() const
 {
  return is_constant() &&
         get("value")=="false";
 }

/*******************************************************************\

Function:

  Inputs:

 Outputs:

 Purpose:

\*******************************************************************/

void exprt::make_bool(bool value)
 {
  clear();
  id="constant";
  type().id="bool";
  set("value", value?"true":"false");
 }

/*******************************************************************\

Function:

  Inputs:

 Outputs:

 Purpose:

\*******************************************************************/

void exprt::make_true()
 {
  clear();
  id="constant";
  type().id="bool";
  set("value", "true");
 }

/*******************************************************************\

Function:

  Inputs:

 Outputs:

 Purpose:

\*******************************************************************/

void exprt::make_false()
 {
  clear();
  id="constant";
  type().id="bool";
  set("value", "false");
 }

/*******************************************************************\

Function: operator<

  Inputs:

 Outputs:

 Purpose: defines ordering on expressions for canonicalization

\*******************************************************************/

bool operator<(const exprt &X, const exprt &Y)
 {
  return (irept &)X < (irept &)Y;
 }

/*******************************************************************\

Function: exprt::make_zero

  Inputs:

 Outputs:

 Purpose:

\*******************************************************************/

void exprt::make_zero()
 {
  const irep_string type_id=type().id;
  typet old_type=type();

  clear();
  type()=old_type;

  id="constant";

  if(type_id=="bool" ||
     type_id=="rational" ||
     type_id=="real" ||
     type_id=="integer" ||
     type_id=="natural" ||
     type_id=="complex")
   {
    set("value", "0");
   }
  else if(type_id=="unsignedbv" ||
          type_id=="signedbv" ||
          type_id=="floatbv")
   {
    irep_string value;
    unsigned width=bv_width(type());

    for(unsigned i=0; i<width; i++)
      value+='0';

    set("value", value);
   }
  else if(type_id=="pointer")
   {
    set("value", "NULL");
   }
  else
    make_nil();
 }

/*******************************************************************\

Function: exprt::make_one

  Inputs:

 Outputs:

 Purpose:

\*******************************************************************/

void exprt::make_one()
 {
  typet old_type=type();
  const irep_string type_id=old_type.id;

  clear();
  type()=old_type;
  id="constant";

  if(type_id=="bool" ||
     type_id=="rational" ||
     type_id=="real" ||
     type_id=="integer" ||
     type_id=="natural" ||
     type_id=="complex")
   {
    set("value", "1");
   }
  else if(type_id=="unsignedbv" ||
          type_id=="signedbv" ||
          type_id=="floatbv")
   {
    irep_string value;
    for(int i=0; i<atoi(type().get("width").c_str())-1; i++)
      value+='0';
    value+='1';
    set("value", value);
   }
  else
    make_nil();
 }

/*******************************************************************\

Function: exprt::negate

  Inputs:

 Outputs:

 Purpose:

\*******************************************************************/

void exprt::negate()
 {
  const irep_string &type_id=type().id;

  if(type_id=="bool")
    set("value", !get_bool("value"));
  else if(type_id=="integer")
   {
    set("value", integer2string(-string2integer(get("value"))));
   }
  else
    make_nil();
 }

/*******************************************************************\

Function: exprt::is_boolean

  Inputs:

 Outputs:

 Purpose:

\*******************************************************************/

bool exprt::is_boolean() const
 {
  return type().id=="bool";
 }

/*******************************************************************\

Function: exprt::is_zero

  Inputs:

 Outputs:

 Purpose:

\*******************************************************************/

bool exprt::is_zero() const
 {
  if(is_constant())
   {
    const irep_string &value=get("value");
    const irep_string &type_id=type().id;

    if(type_id=="integer" || type_id=="natural")
     {
      mp_integer int_value=string2integer(value);
      if(int_value==0) return TRUE;
     }
    else if(type_id=="unsignedbv" || type_id=="signedbv")
     {
      mp_integer int_value=binary2integer(value, FALSE);
      if(int_value==0) return TRUE;
     }
   }

  return FALSE;
 }

/*******************************************************************\

Function: exprt::is_one

  Inputs:

 Outputs:

 Purpose:

\*******************************************************************/

bool exprt::is_one() const
 {
  if(is_constant())
   {
    const irep_string &value=get("value");
    const irep_string &type_id=type().id;

    if(type_id=="integer" || type_id=="natural")
     {
      mp_integer int_value=string2integer(value);
      if(int_value==1) return TRUE;
     }
    else if(type_id=="unsignedbv" || type_id=="signedbv")
     {
      mp_integer int_value=binary2integer(value, FALSE);
      if(int_value==1) return TRUE;
     }
   }

  return FALSE;
 }

/*******************************************************************\

Function: exprt::sum

  Inputs:

 Outputs:

 Purpose:

\*******************************************************************/

bool exprt::sum(const exprt &expr)
 {
  if(!is_constant() || !expr.is_constant()) return TRUE;
  if(type()!=expr.type()) return TRUE;

  const irep_string &type_id=type().id;

  if(type_id=="integer" || type_id=="natural")
   {
    set("value", integer2string(
      string2integer(get("value"))+
      string2integer(expr.get("value"))));
    return FALSE;
   }
  else if(type_id=="unsignedbv" || type_id=="signedbv")
   {
    set("value", integer2binary(
      binary2integer(get("value"), FALSE)+
      binary2integer(expr.get("value"), FALSE),
      atoi(type().get("width").c_str())));
    return FALSE;
   }

  return TRUE;
 }

/*******************************************************************\

Function: exprt::mul

  Inputs:

 Outputs:

 Purpose:

\*******************************************************************/

bool exprt::mul(const exprt &expr)
 {
  if(!is_constant() || !expr.is_constant()) return TRUE;
  if(type()!=expr.type()) return TRUE;

  const irep_string &type_id=type().id;

  if(type_id=="integer" || type_id=="natural")
   {
    set("value", integer2string(
      string2integer(get("value"))*
      string2integer(expr.get("value"))));
    return FALSE;
   }
  else if(type_id=="unsignedbv" || type_id=="signedbv")
   {
    set("value", integer2binary(
      binary2integer(get("value"), FALSE)*
      binary2integer(expr.get("value"), FALSE),
      atoi(type().get("width").c_str())));
    return FALSE;
   }

  return TRUE;
 }

/*******************************************************************\

Function: exprt::subtract

  Inputs:

 Outputs:

 Purpose:

\*******************************************************************/

bool exprt::subtract(const exprt &expr)
 {
  if(!is_constant() || !expr.is_constant()) return TRUE;

  if(type()!=expr.type()) return TRUE;

  const irep_string &type_id=type().id;

  if(type_id=="integer" || type_id=="natural")
   {
    set("value", integer2string(
      string2integer(get("value"))-
      string2integer(expr.get("value"))));
    return FALSE;
   }
  else if(type_id=="unsignedbv" || type_id=="signedbv")
   {
    set("value", integer2binary(
      binary2integer(get("value"), FALSE)-
      binary2integer(expr.get("value"), FALSE),
      atoi(type().get("width").c_str())));
    return FALSE;
   }

  return TRUE;
 }

