/*******************************************************************\

Module:

Author: Daniel Kroening, kroening@cs.cmu.edu

\*******************************************************************/

#include "trfalse.h"
#include "expr2smv.h"
#include "lispexpr.h"
#include "lispirep.h"

class expr2smvt
 {
 public:
  bool convert_binary(const exprt &src, std::string &dest, const std::string &symbol,
                      unsigned precedence);

  bool convert_unary(const exprt &src, std::string &dest, const std::string &symbol,
                     unsigned precedence);

  bool convert_index(const exprt &src, std::string &dest, unsigned precedence);

  bool convert(const exprt &src, std::string &dest, unsigned &precedence);

  bool convert(const exprt &src, std::string &dest);

  bool convert_symbol(const exprt &src, std::string &dest, unsigned &precedence);

  bool convert_constant(const exprt &src, std::string &dest, unsigned &precedence);

  bool convert_norep(const exprt &src, std::string &dest, unsigned &precedence);

  bool convert(const typet &src, std::string &dest);
 };

/*******************************************************************\

Function: expr2smvt::convert_binary

  Inputs:

 Outputs:

 Purpose:

\*******************************************************************/

bool expr2smvt::convert_binary(const exprt &src, std::string &dest,
                               const std::string &symbol,
                               unsigned precedence)
 {
  if(src.operands().size()<2)
    return convert_norep(src, dest, precedence);

  bool first=TRUE;

  forall_operands(it, src)
   {
    if(first)
      first=FALSE;
    else
     {
      dest+=' ';
      dest+=symbol;
      dest+=' ';
     }

    std::string op;
    unsigned p;

    if(convert(*it, op, p)) return TRUE;

    if(precedence>p) dest+='(';
    dest+=op;
    if(precedence>p) dest+=')';
   }

  return FALSE;
 }

/*******************************************************************\

Function: expr2smvt::convert_unary

  Inputs:

 Outputs:

 Purpose:

\*******************************************************************/

bool expr2smvt::convert_unary(const exprt &src, std::string &dest,
                              const std::string &symbol,
                              unsigned precedence)
 {
  if(src.operands().size()!=1)
    return convert_norep(src, dest, precedence);
    
  std::string op;
  unsigned p;

  if(convert(src.operands()[0], op, p)) return TRUE;

  dest+=symbol;
  if(precedence>p) dest+='(';
  dest+=op;
  if(precedence>p) dest+=')';

  return FALSE;
 }

/*******************************************************************\

Function: expr2smvt::convert_index

  Inputs:

 Outputs:

 Purpose:

\*******************************************************************/

bool expr2smvt::convert_index(const exprt &src, std::string &dest,
                              unsigned precedence)
 {
  if(src.operands().size()!=2)
    return convert_norep(src, dest, precedence);

  std::string op;
  unsigned p;

  if(convert(src.operands()[0], op, p)) return TRUE;

  if(precedence>p) dest+='(';
  dest+=op;
  if(precedence>p) dest+=')';

  if(convert(src.operands()[1], op, p)) return TRUE;

  dest+='[';
  dest+=op;
  dest+=']';

  return FALSE;
 }

/*******************************************************************\

Function: expr2smvt::convert_norep

  Inputs:

 Outputs:

 Purpose:

\*******************************************************************/

bool expr2smvt::convert_norep(const exprt &src, std::string &dest,
                              unsigned &precedence)
 {
  precedence=22;
  dest=src.to_string();
  return FALSE;
 }

/*******************************************************************\

Function: expr2smvt::convert_symbol

  Inputs:

 Outputs:

 Purpose:

\*******************************************************************/

bool expr2smvt::convert_symbol(const exprt &src, std::string &dest,
                               unsigned &precedence)
 {
  precedence=22;
  dest=src.get("identifier");
 
  if(strncmp(dest.c_str(), "smv::", 5)==0)
    dest.erase(0, 5);

  return FALSE;
 }

/*******************************************************************\

Function: expr2smvt::convert_constant

  Inputs:

 Outputs:

 Purpose:

\*******************************************************************/

bool expr2smvt::convert_constant(const exprt &src, std::string &dest,
                                 unsigned &precedence)
 {
  precedence=22;

  const typet &type=src.type();
  const std::string &value=src.get("value");

  if(type.id=="bool")
   {
    if(src.is_true())
      dest+="1";
    else
      dest+="0";
   }
  else if(type.id=="integer" || type.id=="natural")
    dest=value;
  else
    return convert_norep(src, dest, precedence);

  return FALSE;
 }

/*******************************************************************\

Function: expr2smvt::convert

  Inputs:

 Outputs:

 Purpose:

\*******************************************************************/

bool expr2smvt::convert(const exprt &src, std::string &dest, unsigned &precedence)
 {
  precedence=22;

  if(src.id=="+")
    return convert_binary(src, dest, "+", precedence=14);

  else if(src.id=="-")
   {
    if(src.operands().size()<2)
      return convert_norep(src, dest, precedence);
    else     
      return convert_binary(src, dest, "-", precedence=14);
   }

  else if(src.id=="unary-")
   {
    if(src.operands().size()!=1)
      return convert_norep(src, dest, precedence);
    else     
      return convert_unary(src, dest, "-", precedence=16);
   }

  else if(src.id=="index")
    return convert_index(src, dest, precedence=22);

  else if(src.id=="*" || src.id=="/")
    return convert_binary(src, dest, src.id, precedence=14);

  else if(src.id=="<" || src.id==">" ||
          src.id=="<=" || src.id==">=")
    return convert_binary(src, dest, src.id, precedence=9);

  else if(src.id=="=")
    return convert_binary(src, dest, "=", precedence=9);

  else if(src.id=="not")
    return convert_unary(src, dest, "!", precedence=16);

  else if(src.id=="and")
    return convert_binary(src, dest, "&", precedence=7);

  else if(src.id=="or")
    return convert_binary(src, dest, "|", precedence=6);

  else if(src.id=="=>")
    return convert_binary(src, dest, "->", precedence=5);

  else if(src.id=="<=>")
    return convert_binary(src, dest, "<->", precedence=4);

  else if(src.id=="symbol")
    return convert_symbol(src, dest, precedence);

  else if(src.id=="constant")
    return convert_constant(src, dest, precedence);

  else // no SMV language expression for internal representation 
    return convert_norep(src, dest, precedence);

  return FALSE;
 }

/*******************************************************************\

Function: expr2smvt::convert

  Inputs:

 Outputs:

 Purpose:

\*******************************************************************/

bool expr2smvt::convert(const exprt &src, std::string &dest)
 {
  unsigned precedence;
  return convert(src, dest, precedence);
 }

/*******************************************************************\

Function:

  Inputs:

 Outputs:

 Purpose:

\*******************************************************************/

bool expr2smv(const exprt &expr, std::string &code)
 {
  expr2smvt expr2smv;
  return expr2smv.convert(expr, code);
 }

