package comms.dir;

import java.awt.event.ActionListener;
import java.awt.event.WindowListener;
import java.awt.event.WindowEvent;
import java.awt.event.ActionEvent;
import java.awt.event.KeyEvent;
import java.awt.event.ItemListener;
import java.awt.event.ItemEvent;
import java.awt.Frame;
import java.awt.List;
import java.awt.ScrollPane;
import java.awt.Panel;
import java.awt.Button;
import java.awt.Menu;
import java.awt.MenuItem;
import java.awt.MenuBar;
import java.awt.MenuShortcut;
import java.awt.Toolkit;
import java.awt.Dimension;
import java.awt.BorderLayout;
import java.awt.GridBagLayout;
import java.awt.GridBagConstraints;
import java.awt.TextField;
import java.awt.Label;
import java.awt.Dialog;
import java.awt.Choice;
import comms.core.EntryTypes;
import comms.core.PhoneEntry;
import comms.core.PhoneNumber;

/*****************************************************************************
 * <! Copyright 2000, Institute for Complex Engineered Systems,
 *                    Carnegie Mellon University
 *
 * PROJECT: Adaptable 
 *
 * FILE: DirectoryGUI.java 
 * >
 *
 * A GUI interface for the Directory Server.
 * 
 * @author Theodore Q Pham <A HREF="mailto:telamon@CMU.EDU">telamon@CMU.EDU</A>
 *         <br> </br>
 *
 * @version  $Revision: 1.1.2.8 $ $Date: 2000/06/19 19:29:15 $
 *
 * <!
 * REVISION HISTORY:
 *
 * $Log: DirectoryGUI.java,v $
 * Revision 1.1.2.8  2000/06/19 19:29:15  telamon
 * wrapped a print statement in the debug flag
 *
 * Revision 1.1.2.7  2000/06/19 19:25:29  telamon
 * fixed gui exception thrown because of double entry list removal
 *
 * Revision 1.1.2.6  2000/06/15 20:32:42  telamon
 * fixed not implemented AddEntry
 *
 * Revision 1.1.2.5  2000/06/15 20:28:16  telamon
 * fixed not implemented AddEntry
 *
 * Revision 1.1.2.4  2000/06/15 19:58:42  telamon
 * completed implementation of the Add Entry GUI for Directory
 *
 * Revision 1.1.2.3  2000/05/22 23:44:10  telamon
 * changed GUI layout
 *
 * Revision 1.1.2.2  2000/05/22 23:34:58  telamon
 * fixed directory GUI
 *
 * Revision 1.1.2.1  2000/05/10 09:58:42  telamon
 * new directory server
 *
 *
 * >
 ****************************************************************************/
class DirectoryGUI extends Frame implements DirectoryViewCon,
  ActionListener, WindowListener, ItemListener {

  static final boolean debug = false;
  
  static final int kPREFERX = 460;
  static final int kPREFERY = 180;
  
  Directory dir;

  List entryList;
  TextField hostField;
  TextField portField;
  TextField typeField;
  TextField optionalField;
  AddEntryDialog aed;
  Dimension preferD;

  final static String RESET = "1";
  final static int RESET_INT = 1;
  final static String ADD = "2";
  final static int ADD_INT = 2;
  final static String REMOVE ="3";
  final static int REMOVE_INT = 3;
  final static String QUIT = "4";
  final static int QUIT_INT = 4;
  
  DirectoryGUI(Directory dir) {
    super("Directory - "+dir.getHostname()+" "+dir.getPort());
    this.dir = dir;

    MenuBar menuBar = new MenuBar();
    Menu mainMenu = new Menu("Commands");
    MenuItem resetI = new MenuItem("Reset",
				   new MenuShortcut(KeyEvent.VK_R));
    resetI.setActionCommand(RESET);
    resetI.addActionListener(this);
    mainMenu.add(resetI);
    
    MenuItem removeI = new MenuItem("Remove",
				    new MenuShortcut(KeyEvent.VK_E));
    removeI.setActionCommand(REMOVE);
    removeI.addActionListener(this);
    mainMenu.add(removeI);


    MenuItem addI = new MenuItem("Add",
				 new MenuShortcut(KeyEvent.VK_A));
    addI.setActionCommand(ADD);
    addI.addActionListener(this);
    mainMenu.add(addI);
    
    mainMenu.addSeparator();
    MenuItem quitI = new MenuItem("Quit",
				  new MenuShortcut(KeyEvent.VK_Q));
    quitI.setActionCommand(QUIT);
    quitI.addActionListener(this);
    mainMenu.add(quitI);
    
    menuBar.add(mainMenu);

    this.setMenuBar(menuBar);
    this.addWindowListener(this);

    ScrollPane sp = new ScrollPane();
    this.add(sp);
    
    Panel displayPanel = new Panel();    
    sp.add(displayPanel);

    Panel p;
    
    displayPanel.setLayout(new StackLayout(StackLayout.HORIZONTAL));

    entryList = new List(3, false);
    entryList.addItemListener(this);
    displayPanel.add("TALL WIDE*2 FLUSH", entryList);

    p = StackLayout.addVPanel("TOP WIDE*3 FLUSH", displayPanel);
    Panel subp;
    subp = StackLayout.addHWideFlushPanel(p);
    subp.add("FLUSH WIDTH=50", new Label("Host"));
    hostField = new TextField("",24);
    hostField.setEditable(false);
    subp.add("WIDE FLUSH",hostField);

    subp = StackLayout.addHWideFlushPanel(p);
    subp.add("FLUSH WIDTH=50", new Label("Port"));    

    portField = new TextField("",5);
    portField.setEditable(false);
    subp.add("WIDE FLUSH", portField);

    subp = StackLayout.addHWideFlushPanel(p);
    subp.add("FLUSH WIDTH=50", new Label("Type"));
    typeField = new TextField("",10);
    typeField.setEditable(false);
    subp.add("WIDE FLUSH", typeField);

    subp = StackLayout.addHWideFlushPanel(p);    
    subp.add("FLUSH WIDTH=50", new Label("Option"));
    optionalField = new TextField("");
    optionalField.setEditable(false);
    subp.add("WIDE FLUSH", optionalField);
    
    preferD = Toolkit.getDefaultToolkit().getScreenSize();
    if( preferD.width > kPREFERX ) {
      preferD.width = kPREFERX;
    }
    if( preferD.height > kPREFERY ) {
      preferD.height = kPREFERY;
    }
    
    this.validate();
    this.pack();

    aed = new AddEntryDialog();
  }

  public Dimension getPreferredSize() {
    return preferD;
  }
  
  public void actionPerformed(ActionEvent e) {
    if(debug) {
      System.out.println("Command: "+e.getActionCommand());
    }

    int commInt = Integer.parseInt(e.getActionCommand());

    switch(commInt) {
      case RESET_INT:
	dir.resetDirectory();
	break;
      case REMOVE_INT:
	removeNameFromDirectory();	
	break;
      case ADD_INT:
	addEntryToList();	
	break;
      case QUIT_INT:
	this.dispose();	
	break;
    }
  }

  public void windowOpened(WindowEvent e) {}
  
  public void windowClosing(WindowEvent e) {
    this.dispose();
  }

  public void windowClosed(WindowEvent e) {
    System.exit(0);
  }
  
  public void windowIconified(WindowEvent e) {}

  public void windowDeiconified(WindowEvent e) {}

  public void windowActivated(WindowEvent e) {}

  public void windowDeactivated(WindowEvent e) {}
  
  public void itemStateChanged(ItemEvent e) {
    if(debug) System.out.println("state: "+e.getStateChange());
    if(e.getStateChange() == ItemEvent.SELECTED) {
      String name =
	entryList.getItem( ((Integer) e.getItem()).intValue());
      PhoneEntry pe = dir.getPhoneEntry(name);
      if(pe != null) {

	hostField.setText(pe.getPhoneNumber().getHostname());
	portField.setText(Integer.toString(pe.getPhoneNumber().getPortNumber()));
	typeField.setText(EntryTypes.getTypeString(pe.getType()));
	optionalField.setText(pe.getOptional());
      }
    }
  }

  public void initialized() {
    if (debug) {
      System.out.println("Directory initialized"); 
    }
    this.show();
  }

  public void entryAdded(PhoneEntry pe) {
    if(debug) System.out.println("Entry Added: "+pe.toString());
    entryList.add(pe.name);
  }

  public void entryRemoved(PhoneEntry pe) {
    if(debug) System.out.println("Entry Removed: "+pe.toString());
    entryList.remove(pe.name);
  }

  void removeNameFromDirectory() {
    String name;
    //find out what the currently selected name is in the list and remove
    //it
    name = entryList.getSelectedItem();

    if(debug) System.out.println("RemoveName: "+name);  
    
    if(name != null) {
      //take the entry out of the Directory
      dir.removeFromDirectory(name);
    }
  }

  private class AddEntryDialog extends Dialog implements WindowListener,
    ActionListener {

    TextField feedbackField;
    TextField nameField;
    TextField hostField;
    TextField portField;
    Choice typeChoice;
    TextField optionalField;

    final static String ADD = "1";
    final static int ADD_INT = 1;
    final static String CLEAR = "2";
    final static int CLEAR_INT = 2;
    final static String CANCEL = "3";
    final static int CANCEL_INT = 3;

    Button addButton;
    Button clearButton;
    Button cancelButton;
    
    AddEntryDialog() {
      super(DirectoryGUI.this, "Add Directory Entry", true);

      this.addWindowListener(this);

      this.setResizable(false);
      
      Panel p;
      
      this.setLayout(new StackLayout(StackLayout.HORIZONTAL));
      p = StackLayout.addVPanel("TOP WIDE FLUSH", this);
      
      Panel subp;
      subp = StackLayout.addHWideFlushPanel(p);
      subp.add("WIDTH=50 FLUSH", new Label("Name"));

      nameField = new TextField("",15);
      subp.add("WIDE FLUSH",nameField);

      subp = StackLayout.addHWideFlushPanel(p);
      subp.add("WIDTH=50 FLUSH", new Label("Host"));
      hostField = new TextField("",24);
      subp.add("WIDE FLUSH",hostField);

      subp = StackLayout.addHWideFlushPanel(p);
      subp.add("WIDTH=50 FLUSH", new Label("Port"));
      portField = new TextField("",5);
      subp.add("WIDE FLUSH", portField);
      
      subp = StackLayout.addHWideFlushPanel(p);
      subp.add("WIDTH=50 FLUSH", new Label("Type"));

      typeChoice = new Choice();
      for(int i = 0; i < EntryTypes.MAX_TYPE;i++) {
	typeChoice.add(EntryTypes.getTypeString(i));
      }
      subp.add("WIDE FLUSH", typeChoice);

      subp = StackLayout.addHWideFlushPanel(p);      
      subp.add("WIDTH=50 FLUSH", new Label("Option"));
      optionalField = new TextField("");
      subp.add("WIDE FLUSH", optionalField);

      subp = StackLayout.addHWideFlushPanel(p);
      feedbackField = new TextField("");
      feedbackField.setEditable(false);
      subp.add("WIDE FLUSH", feedbackField);
      
      subp = StackLayout.addHWideFlushPanel(p);
      addButton = new Button("Add");
      addButton.setActionCommand(ADD);
      addButton.addActionListener(this);
      subp.add("FLUSH", addButton);

      clearButton = new Button("Clear");
      clearButton.setActionCommand(CLEAR);
      clearButton.addActionListener(this);
      subp.add("FLUSH", clearButton);
      
      cancelButton = new Button("Cancel");
      cancelButton.setActionCommand(CANCEL);
      cancelButton.addActionListener(this);
      subp.add("FLUSH",cancelButton);
      
      this.validate();
      this.pack();
    }

    public void actionPerformed(ActionEvent e) {

      int commInt = Integer.parseInt(e.getActionCommand());

      switch(commInt) {
	case ADD_INT:
	  checkInputAndAdd();
	  break;
	case CLEAR_INT:
	  clear();
	  break;
	case CANCEL_INT:
	  this.hide();
	  break;
      }
    }

    void checkInputAndAdd() {

      clearButton.setEnabled(false);
      cancelButton.setEnabled(false);
      addButton.setEnabled(false);
      
      nameField.setEditable(false);
      hostField.setEditable(false);
      portField.setEditable(false);
      typeChoice.setEnabled(false);
      optionalField.setEditable(false);

      //grab the strings and try to construct a PhoneEntry
      PhoneEntry newPE;
      try {
	newPE = new PhoneEntry(nameField.getText(),
			       new PhoneNumber(hostField.getText(),
					       Integer.parseInt(portField.getText())),
			       typeChoice.getSelectedIndex(),
			       optionalField.getText());
      } catch (NumberFormatException nfe) {
	newPE = null;
      }

      if(newPE != null) {
	if(!newPE.valid()) {
	  newPE = null;
	}
      }

      if(newPE != null) {
	//add the entry to the directory
	DirectoryGUI.this.dir.addToDirectory(newPE);
	this.hide();
      }
      else {
	//throw up an error message and let them try again
	feedbackField.setText("Invalid Entry.  Try again.");
	
	nameField.setEditable(true);
	hostField.setEditable(true);
	portField.setEditable(true);
	typeChoice.setEnabled(true);
	optionalField.setEditable(true);

	clearButton.setEnabled(true);
	cancelButton.setEnabled(true);
	addButton.setEnabled(true);
      }
    }
    
    public void windowOpened(WindowEvent e) {}
  
    public void windowClosing(WindowEvent e) {
      if(debug) System.out.println("AddEntryclosing");
      this.hide();
    }
    
    public void windowClosed(WindowEvent e) {
      if(debug) System.out.println("AddEntryclosed");
    }

    public void windowIconified(WindowEvent e) {}
    
    public void windowDeiconified(WindowEvent e) {}
    
    public void windowActivated(WindowEvent e) {}
    
    public void windowDeactivated(WindowEvent e) {}
    
    void clear() { 
      nameField.setText("");
      hostField.setText("");
      portField.setText("");
      typeChoice.removeAll();
      for(int i = 0; i < EntryTypes.MAX_TYPE;i++) {
	typeChoice.add(EntryTypes.getTypeString(i));
      }
      optionalField.setText("");
      feedbackField.setText("Enter Directory Entry data");
      
      nameField.setEditable(true);
      hostField.setEditable(true);
      portField.setEditable(true);
      typeChoice.setEnabled(true);
      optionalField.setEditable(true);

      clearButton.setEnabled(true);
      cancelButton.setEnabled(true);
      addButton.setEnabled(true);

    }
  }
  
  void addEntryToList() {
    //pop up a dialog box that will allow you to add an entry to the
    //directory

    aed.clear();
    aed.show();
  }
  
  public void reset() {
    entryList.removeAll();
    if(debug) System.out.println("Directory reset.");
  }
}

