package comms.dir;

import comms.core.CommsInputStream;
import comms.core.CommsOutputStream;
import comms.core.EntryTypes;
import comms.core.PhoneEntry;
import comms.core.PhoneNumber;
import java.util.Hashtable;
import java.util.Vector;
import java.util.StringTokenizer;
import java.util.Enumeration;
import java.net.ServerSocket;
import java.net.Socket;
import java.io.IOException;
import java.net.UnknownHostException;

/*****************************************************************************
 * <! Copyright 2000, Institute for Complex Engineered Systems,
 *                    Carnegie Mellon University
 *
 * PROJECT: Adaptable 
 *
 * FILE: Directory.java 
 * >
 *
 * A server that houses the phone book.  This phone book associates
 * names to hostnames and ports.  When an entity first starts
 * communicating via this comms library, it connects to the Directory
 * Server and downloads the phone list.  If an entity ever needs to
 * contact another entity which it doesn't have an entry for, it will
 * requery the Directory Server.
 *
 * The Directory Server may be run both in graphical and non-graphical mode.
 *
 * TODO TED 4/9/00: Add ability to log in and manipulate the Directory
 * Server via a network utility program.
 * 
 * @author Theodore Q Pham <A HREF="mailto:telamon@CMU.EDU">telamon@CMU.EDU</A>
 *         <br> </br>
 *
 * @version  $Revision: 1.1.2.8 $ $Date: 2000/06/15 20:24:47 $
 *
 * <!
 * REVISION HISTORY:
 *
 * $Log: Directory.java,v $
 * Revision 1.1.2.8  2000/06/15 20:24:47  telamon
 * got rid of the directory text interface
 *
 * Revision 1.1.2.7  2000/05/22 23:34:57  telamon
 * fixed directory GUI
 *
 * Revision 1.1.2.6  2000/05/10 09:58:40  telamon
 * new directory server
 *
 *
 * >
 ****************************************************************************/
public class Directory extends Thread {
  /**
   * A Hashtable of all the PhoneEntries in this Directory Server
   * This table should only be read and modified while holding the Directory
   * lock
   */
  private Hashtable phoneTable;
  
  /**
   * A Vector containing the Hashtables for the individual types
   * This table should only be read and modified while holding the Directory
   * lock
   */
  private Vector typeVector;

  /**
   * When PhoneEntries are added or removed, the operation must be done
   * in the main PhoneTable and in the individual type Hashtable.
   */
  
  private ServerSocket serverSocket;
  private DirectoryViewCon dirViewCon;
  private String hostname;

  public final static String VERSION = "$Revision: 1.1.2.8 $";
  
  /** The default port which the name server sets up on. */
  private static final int DEFAULT_PORT = 6500;

  /** Message string tokens for Directory Server Protocol */
  
  /** The string sent which indicates that the router list has been
      sent. */
  private static final String LIST_END_MESSAGE = "done";
  /** The string which informs the directory that the entity wants
      to register. */
  private static final String REGISTER_MESSAGE = "register";
  /** The string which informs the directory that the entity wants
      to register with extended information. */
  private static final String REGISTER_MESSAGE_EXTENDED = "2register";
  /** The string which queries the directory for its contents. */
  private static final String RETRANSMIT_MESSAGE = "retransmit";
  /** The string which queries the directory for its contents with
      extended info. */
  private static final String RETRANSMIT_MESSAGE_EXTENDED = "2retransmit";
  /** The string that tells the directory that the entity should be
      removed (used for graceful entity shutdown). */
  private static final String REMOVE_MESSAGE = "remove";
  /** Match an entity and return the information for that entity. */
  private static final String MATCH_MESSAGE = "match";
  /** Match an entity and return the information for that entity
      with extended info. */
  private static final String MATCH_MESSAGE_EXTENDED = "2match";
  /** Return all entries that match a particular type */
  private static final String MATCH_TYPE_MESSAGE = "TYMA";
  /** Return all extended entries that match a particular type */
  private static final String MATCH_TYPE_MESSAGE_EXTENDED = "2TYMA";
  /** Fail message on match **/
  private  static final String FAIL_MESSAGE = "fail";
  /** Message meaning it's time for the directory to shutdown */
  private static final String EXIT_MESSAGE = "EXIT";
  /** Reset the directory listing message **/
  private static final String RESET_MESSAGE = "RESET";
  
  /** Constructs a new Directory  server at the specified port */
  private Directory(int servicePort) throws IOException,
    UnknownHostException {
    hostname = comms.core.LocalHost.getFullLocalHostName();
    phoneTable = new Hashtable();
    typeVector = new Vector(EntryTypes.MAX_TYPE);
    typeVector.setSize(EntryTypes.MAX_TYPE);
    serverSocket = new ServerSocket(servicePort);
  }

  private void setDirectoryViewCon(DirectoryViewCon dvc) {
    dirViewCon = dvc;
  }

  private void startViewCon() {
    dirViewCon.initialized();
  }

  String getHostname() {
    return hostname;
  }

  int getPort() {
    return serverSocket.getLocalPort();
  }
  
  public void run() {
    Socket socket = null;
    while(true) {
      try {
	socket = serverSocket.accept();
        SocketHandler socketHandler = new SocketHandler(this, socket);
	socketHandler.setDaemon(true);
        socketHandler.start();
	
	/*      } catch (SocketException se) {
	se.printStackTrace();
        System.out.println("Socket timed out ("+se+")... continuing.");
        try {
          if(socket != null)
          {
            socket.close();
          }
        }
        catch(IOException ioe){}
        continue;
        */
	
      } catch (IOException ioe) {
	ioe.printStackTrace();
	System.out.println("An I/O Exception occurred ("+ioe+")... continuing.");
        try {
          if(socket != null) {
            socket.close();
          }
        } catch(IOException ioe2){}
      }
    }
  }

  void handleRequest(String entity, String request, 
		     CommsInputStream inStream,
		     CommsOutputStream outStream) 
    throws IOException, IllegalArgumentException {    
    
    if(request.startsWith(EXIT_MESSAGE)) {
      System.exit(0);
    }
    
    if(!request.startsWith(RETRANSMIT_MESSAGE) &&
       !request.startsWith(RETRANSMIT_MESSAGE_EXTENDED) &&
       !request.startsWith(REGISTER_MESSAGE) &&
       !request.startsWith(REGISTER_MESSAGE_EXTENDED) && 
       !request.startsWith(REMOVE_MESSAGE) &&
       !request.startsWith(MATCH_MESSAGE) &&
       !request.startsWith(MATCH_MESSAGE_EXTENDED) &&
       !request.startsWith(MATCH_TYPE_MESSAGE) &&
       !request.startsWith(MATCH_TYPE_MESSAGE_EXTENDED) &&
       !request.startsWith(RESET_MESSAGE)) {
      System.out.println("Did not receive a proper request.\n");
      throw new IOException("Improper directory request");
    }
    
    if(request.startsWith(REMOVE_MESSAGE)) {
      removeFromDirectory(entity);
      return;
    }
    
    if(request.startsWith(RETRANSMIT_MESSAGE)) {
      sendDirectory(outStream);
      return;
    }
    
    if(request.startsWith(RETRANSMIT_MESSAGE_EXTENDED)) {
      sendDirectoryExtended(outStream);
      return;
    }
    
    if(request.startsWith(MATCH_MESSAGE)) {
      sendPhoneEntry(outStream, entity);
      return;
    }
    
    if(request.startsWith(MATCH_MESSAGE_EXTENDED)) {
      sendPhoneEntryExtended(outStream, entity);
      return;
    }
    
    if(request.startsWith(MATCH_TYPE_MESSAGE)) {
      sendPhoneType(outStream, entity);
      return;
    }
    
    if(request.startsWith(MATCH_TYPE_MESSAGE_EXTENDED)) {
      sendPhoneTypeExtended(outStream, entity);
      return;
    }

    if(request.startsWith(RESET_MESSAGE)) {
      resetDirectory();
      return;
    }
    
    String[] vals;
    vals = inStream.readMessage();
    String phoneNumberString = vals[0];
    PhoneEntry entry =
      ( (request.startsWith(REGISTER_MESSAGE)) ?
        new PhoneEntry(entity, phoneNumberString, false) :
        new PhoneEntry(entity, phoneNumberString, true));
    
    addToDirectory(entry);
    sendDirectory(outStream);
  }

  synchronized void resetDirectory() {
    phoneTable.clear();
    Hashtable typeTable;
    for(int i =0; i < EntryTypes.MAX_TYPE;i++) {
      typeTable = (Hashtable) typeVector.elementAt(i);
      if(typeTable != null) {
	typeTable.clear();
	typeVector.setElementAt(null,i);
      }
    }
    dirViewCon.reset();
  }
  
  /**
   * Sends the list of all phone names, hosts, and port numbers to
   * the specified socket.
   */
  synchronized void sendDirectory(CommsOutputStream outStream)
    throws IOException {

    Enumeration entries;
    entries = phoneTable.elements();

    while(entries.hasMoreElements()) {
      String entryST =
	((PhoneEntry) entries.nextElement()).toString();
      outStream.writeMessage(entryST,"","");
    }
    outStream.writeMessage(LIST_END_MESSAGE,"","");
    outStream.flush();
  }

  /**
   * Sends the list of all phone names, hosts, and port numbers to the
   * specified socket.
   */
  synchronized void sendDirectoryExtended(CommsOutputStream outStream)
    throws IOException {

    Enumeration entries;
    entries = phoneTable.elements();

    while(entries.hasMoreElements()) {
      String entryST =
	((PhoneEntry) entries.nextElement()).toStringExtended();
      outStream.writeMessage(entryST,"","");
    }
    outStream.writeMessage(LIST_END_MESSAGE,"","");
    outStream.flush();
  }
  
  synchronized void sendPhoneEntry(CommsOutputStream outStream,
				   String entry)
    throws IOException {

    PhoneEntry matchedEntry = (PhoneEntry) phoneTable.get(entry);

    if(matchedEntry != null) {  
      outStream.writeMessage(matchedEntry.toString(),"","");
      outStream.flush();
    } else {
      outStream.writeMessage(FAIL_MESSAGE,"","");
      outStream.flush();
    }
  }
  
  synchronized void sendPhoneEntryExtended(CommsOutputStream outStream,
					   String entry)
    throws IOException {
    PhoneEntry matchedEntry = (PhoneEntry) phoneTable.get(entry);

    if(matchedEntry != null) {  
      outStream.writeMessage(matchedEntry.toStringExtended(),"","");
      outStream.flush();
    } else {
      outStream.writeMessage(FAIL_MESSAGE,"","");
      outStream.flush();
    }
  }

  /**
   *  Sends the list of all phone names, hosts, and port numbers that
   *  match a specified type to the specified socket.
   */
  synchronized void sendPhoneType(CommsOutputStream outStream,
				  String arg) 
    throws IOException {
    int type;
    
    try {
      type = Integer.parseInt(arg);
    }
    catch(NumberFormatException nfe) {
      outStream.writeMessage(FAIL_MESSAGE,"","");
      outStream.flush();
      return;
    }
    
    Hashtable typeTable = (Hashtable) typeVector.elementAt(type);
    Enumeration typeTableEnum;
    PhoneEntry currPhoneEntry;
    String entrySt;
    
    if(typeTable != null) {
      typeTableEnum = typeTable.elements();
      
      while(typeTableEnum.hasMoreElements()) {
	currPhoneEntry = (PhoneEntry) typeTableEnum.nextElement();
	entrySt = currPhoneEntry.toString();
	outStream.writeMessage(entrySt,"","");
      }
    }
    
    outStream.writeMessage(LIST_END_MESSAGE,"","");
    outStream.flush();
  }

  /**
   * Sends the list of all extended entries that match a specified
   * type to the specified CommsOutputStream.
   */
  synchronized void sendPhoneTypeExtended(CommsOutputStream outStream, 
					  String arg) 
    throws IOException {
    int type;
    
    try {
      type = Integer.parseInt(arg);
    }
    catch(NumberFormatException nfe) {
      outStream.writeMessage(FAIL_MESSAGE,"","");
      outStream.flush();
      return;
    }

    Hashtable typeTable = (Hashtable) typeVector.elementAt(type);
    Enumeration typeTableEnum;
    PhoneEntry currPhoneEntry;
    String entrySt;
    
    if(typeTable != null) {
      typeTableEnum = typeTable.elements();

      while(typeTableEnum.hasMoreElements()) {
	currPhoneEntry = (PhoneEntry) typeTableEnum.nextElement();
	entrySt = currPhoneEntry.toStringExtended();
	outStream.writeMessage(entrySt,"","");
      }
    }
    
    outStream.writeMessage(LIST_END_MESSAGE,"","");
    outStream.flush();
  }
  
  /**
   * Put the PhoneEntry into the phone book, replacing any duplicate entry
   */
  synchronized void addToDirectory(PhoneEntry p) {
    // To make sure there are no duplicate entries
    removeFromDirectory(p.name);

    phoneTable.put(p.name, p);

    Hashtable typeTable = (Hashtable) typeVector.elementAt(p.type);

    if(typeTable == null) {
      //We've never handled this type before so we need to create the hashtable
      //for it and put that into the typeVector

      typeTable = new Hashtable();
      typeVector.setElementAt(typeTable,p.type);
    }

    typeTable.put(p.name,p);
    
    dirViewCon.entryAdded(p);
  }

  /**
   * Lookup the name in the phone book and remove it if it exists.
   *
   * When removing the PhoneEntry, remove it first from the main Hashtable
   * then also from the individual type Hashtable
   */
  synchronized boolean removeFromDirectory(String name) {
    PhoneEntry pEntry;
    boolean retVal = false;
    
    pEntry = (PhoneEntry) phoneTable.remove(name);

    if(pEntry != null) {
      //the entry was actually in the phoneTable and so we need to remove
      //it from the type Hashtable
      Hashtable typeTable = (Hashtable) typeVector.elementAt(pEntry.type);
      //note, since this entry was in the phoneTable, it's typeTable must exist
      typeTable.remove(name);
      dirViewCon.entryRemoved(pEntry);
      retVal = true;
    }
    return retVal;
  }

  synchronized PhoneEntry getPhoneEntry(String name) {
    return (PhoneEntry) phoneTable.get(name);
  }
  
  private static void printUsage() {
    System.out.println("Usage: java adaptive.core.DirectoryServer <options>");
    System.out.println("   -p <port>             Specify port number");
    System.out.println("                         Default: 6500");
    System.out.println("   -g                    Launch GUI interface");
    System.out.println("                         Default: GUI interface");
    System.out.println("   -v                    Print version");
    System.out.println("   -? -h                 Prints this");
    System.out.println("");
    System.out.println("NOTE: For conflicting options, options at the end of the command line always take precedence over earlier ones");
  }
  
  /** Starts a new directory at the specified port. */
  public static void main(String argv[]) {
      Directory dir = null;
      int dirPort = DEFAULT_PORT;
      boolean graphical = true;
      
      int currArg = 0;
    try {
      while(currArg < argv.length) {
	if(argv[currArg].length() >= 2 && argv[currArg].charAt(0) == '-') {
	  switch(argv[currArg].charAt(1)) {
	    case 'p':
	      if ((argv.length-currArg-1)>=1) {
		dirPort = Integer.parseInt(argv[++currArg]);
		currArg++;
	      } else {
		System.err.println("No port specified after -p");
		System.exit(1);
	      }
	      break;
	    case 'v':
	      System.out.println("Directory Server "+VERSION);
	      System.exit(0);
	      break;
	    case 'g': //check for the graphical flag
	      graphical = true;
	      currArg+=1;
	      break;
	    case 'h':
	    case '?':
	      printUsage();
	      System.exit(0);
	      break;
	    default:
	      System.err.println("Ignoring unknown switch: "+argv[currArg]);
	      currArg+=1;
	  }
	}
      }
      
      dir = new Directory(dirPort);

      if(graphical) {
	dir.setDirectoryViewCon(new DirectoryGUI(dir));
      } else {
	//dir.setDirectoryViewCon(new DirectoryText(dir));
      }
      //dir.setDaemon(true);
      dir.start();  //Spin off the Directory Server main processing thread
      dir.startViewCon(); //Start interface processing
                          //For the GUI interface, this method does nothing
                          //For the Text interface, this method takes over
                          //the console and uses the main thread for interface
                          //processing
    } catch (IOException ioe) {
      System.out.println("An IO Exception occurred while starting.");
      System.out.println("Probable cause: Specified port ("+dirPort+") already in use.");
      System.out.println("Exiting...");
    }
  }
}
