package comms.core;

import java.util.StringTokenizer;

/**
 *  Phone book entries for directory server
 */
public class PhoneEntry
{
  /** The name of the phone entry. */
  public String name;

  /** The phone number of the phone entry. */
  public PhoneNumber number;

  /** What type of entity is this */
  public int type;

  /** Any additional information related to type that needs to be in the directory server */
  public String optional;
  
  /** Creates a new phone entry with the specified name, IP address,
      and port number. */
  public PhoneEntry(String name, String IPAddress, int port) {
    this.name = name;
    this.number = new PhoneNumber(IPAddress, port);
    this.type = EntryTypes.DEFAULT;
    this.optional = null;
  }

  /** Creates a new phone entry with the specified name and phone number. */
  public PhoneEntry(String name, PhoneNumber number) {
    this.name = name;
    this.number = number;
    this.type = EntryTypes.DEFAULT;
    this.optional = null;
  } 

  /** Creates a new phone entry with the specified name, IP address,
      port number, type and optional info */
  public PhoneEntry(String name, String IPAddress, int port,
                    int type, String optional) {
    this.name = name;
    this.number = new PhoneNumber(IPAddress, port);
    this.type = type;
    this.optional = optional;
  }
  
  /** Creates a new phone entry with the specified name, phone number,
      type and optional info */
  public PhoneEntry(String name, PhoneNumber number,
                    int type, String optional) {
    this.name = name;
    this.number = number;
    this.type = type;
    this.optional = optional;
  }

  public PhoneEntry(String name,
		    String phoneEntryId,
		    boolean extended) throws IllegalArgumentException {
    StringTokenizer strtok = new StringTokenizer(phoneEntryId);
    
    if(extended) {
      if( strtok.countTokens() < 3 ) {
	throw new IllegalArgumentException("Not enough words in string \""+
					   phoneEntryId+"\"");
      }
    } else {
      if(strtok.countTokens() != 2) {
	throw new IllegalArgumentException("Too many words in string \""+
					   phoneEntryId+"\"");
      }
    }
    
    try {
      this.name = name;
      this.number = new PhoneNumber(strtok.nextToken(),
				    Integer.parseInt(strtok.nextToken()));
      
      if(extended) {
	this.type = Integer.parseInt(strtok.nextToken());
	this.optional = strtok.hasMoreTokens() ? strtok.nextToken("") : null;
      } else {
	this.type = EntryTypes.DEFAULT;
	this.optional = null;
      }
    } catch(NumberFormatException nfe) { 
      System.out.println(nfe);
      nfe.printStackTrace();
      throw new IllegalArgumentException("Unable to parse PhoneEntry: "+
					 phoneEntryId);
    }
  }
  
  public String getName() {
    return name;
  }

  public PhoneNumber getPhoneNumber() {
    return number;
  }

  public int getType() {
    return type;
  }

  public String getOptional() {
    return optional;
  }
  
  
  /**
   *  Creates a new phone entry from a given string.
   *  The format of the string should be of
   *  the form: "patton 128.2.111.155 6550"
   *
   *@return a new PhoneEntry created from the given string.
   */
  public static PhoneEntry parsePhoneEntry(String phoneEntryName,
                                           String phoneEntryId)
    throws IllegalArgumentException {
    
    StringTokenizer strtok = new StringTokenizer(phoneEntryId);
    
    if(strtok.countTokens() != 2)
      throw new IllegalArgumentException("Too many words in string \""+phoneEntryId+"\"");

    String phoneEntryIPAddress = strtok.nextToken();
    String phoneEntryPort = strtok.nextToken();

    //take out termination character

    //    phoneEntryName = phoneEntryName.substring(0, phoneEntryName.length() - 1);
    //    phoneEntryPort = phoneEntryPort.substring(0, phoneEntryPort.length() - 1);

    try
      {
	//	System.out.println(phoneEntryName + ' ' + phoneEntryIPAddress + ' ' + phoneEntryPort);
	//	System.out.println("Lengths:" + phoneEntryName.length() + ' ' + phoneEntryIPAddress.length() + ' ' + phoneEntryPort.length());
	return new PhoneEntry(phoneEntryName, phoneEntryIPAddress, Integer.parseInt(phoneEntryPort));
      }
    catch(NumberFormatException e) { 
      System.out.println(e);
      e.printStackTrace();
      throw new IllegalArgumentException("Unable to understand port number: "+phoneEntryPort); }
  }


  
  
  /**
   *  Creates a new phone entry from a given string.
   *  The format of the string should be of
   *  the form: "patton 128.2.111.155 6550 5 hello world"
   *
   *@return a new PhoneEntry created from the given string.
   */
  public static PhoneEntry parsePhoneEntryExtended(String phoneEntryName,
                                                   String phoneEntryId)
    throws IllegalArgumentException
  {
    StringTokenizer strtok = new StringTokenizer(phoneEntryId);
    
    if( strtok.countTokens() < 3 )
      throw new IllegalArgumentException("Not enough words in string \""+phoneEntryId+"\"");
    
    String phoneEntryIPAddress = strtok.nextToken();
    String phoneEntryPort = strtok.nextToken();
    String phoneEntryType = strtok.nextToken();
    String phoneEntryOptional =
      strtok.hasMoreTokens() ? strtok.nextToken("") : null;
    try {
      //	System.out.println(phoneEntryName + ' ' + phoneEntryIPAddress + ' ' + phoneEntryPort);
      //	System.out.println("Lengths:" + phoneEntryName.length() + ' ' + phoneEntryIPAddress.length() + ' ' + phoneEntryPort.length());
      return new PhoneEntry(phoneEntryName, phoneEntryIPAddress, Integer.parseInt(phoneEntryPort), Integer.parseInt(phoneEntryType), phoneEntryOptional);
    }
    catch(NumberFormatException e) { 
      System.out.println(e);
      e.printStackTrace();
      throw new IllegalArgumentException(
        "Unable to understand port number: "+phoneEntryPort); }
  }
  
  /** A string representation of a phone entry. */
  public String toString() {
    return (name+" "+number);
  }
  
  /** A string representation of an extended phone entry. */
  public String toStringExtended() {
    return (name+" "+number+" "+type+ (optional!=null ? " "+optional : "") ) ;  
  }

  /** Does some simple checks to see if this PhoneEntry appears valid.
      DOES NOT TRY TO VERIFY HOSTNAME OR PORT NUMBER, JUST THE FORM AND FORMAT
      OF THE INFORMATION
  */
  public boolean valid() {
    if(type < 0 || type >= EntryTypes.MAX_TYPE) {
      return false;
    }
    
    if(name == null ||
       name.equals("") ||
       name.indexOf(' ') != -1) {
      return false;
    }
    if(number == null) {
      return false;
    }
    
    return number.valid();
  }
}
