package comms.core;

/** 
  * @author Jeffrey Lam <Odin@CMU.edu>
  *
  */

import java.io.*;
import java.text.SimpleDateFormat;
import java.util.*;

public class Communications {
  private static final boolean debug = true;
  public static final String kDISCONNECTED = "DISCONNECTED";
  
  public static final String GUISERVER = "GuiServer",
    SUMAIN = "SU_Main",
    TEAMLEADER ="TeamLead",
    MAPSERVER = "MapServer",
    ENVMANAGER = "EnvManager";

  protected String name;
  protected MessageQueue messageQueue,
    // the next two are used for message logging:
    sentMessages, 
    receivedMessages;
  protected QueueAssign qAssign;
  protected ServerThread serverThread;
  protected ConnectionTable connectionTable;
  protected PhoneTable rolodex;
  protected boolean logMessages;
  protected boolean dumpLogAtEnd=true; // create log file at end of session
  protected Date upTime; // time when server started
  //protected String logfilename;

  /** start named server with specified port */
  public Communications(String name, int localPort) throws IOException {
    this.name = name;
    messageQueue = new MessageQueue();
    qAssign = new QueueAssign(messageQueue);
    connectionTable = new ConnectionTable();
    serverThread = new ServerThread(localPort, connectionTable, qAssign);
    rolodex = new PhoneTable();
    rolodex.addEntry(new PhoneEntry(name, serverThread.getHostname(),
				    serverThread.getLocalPort()));
    serverThread.start();
  }

  /** start named server with random port */
  public Communications(String name) throws IOException {
    this.name = name;
    messageQueue = new MessageQueue();
    qAssign = new QueueAssign(messageQueue);
    connectionTable = new ConnectionTable();
    rolodex = new PhoneTable();
    rolodex.addEntry(new PhoneEntry(name, serverThread.getHostname(),
				    serverThread.getLocalPort()));
    serverThread = new ServerThread(connectionTable, qAssign);
    serverThread.start();
  }

  public Communications(String name, String directoryServer,
			int directoryPort)
    throws IOException {

      this(name, directoryServer, directoryPort, true, true);

      /*this.name = name;
      messageQueue = new MessageQueue();
      connectionTable = new ConnectionTable();
      serverThread = new ServerThread(connectionTable, messageQueue);
      rolodex = new PhoneTable(new PhoneEntry(name, serverThread.getHostname(),
					      serverThread.getLocalPort()),
			       directoryServer, directoryPort);
      serverThread.start();*/
  }

  public Communications(String name, String directoryServer, int directoryPort,
			boolean keepLog, boolean dumpLogAtEnd) 
    throws IOException {

      this.logMessages = keepLog;
      this.name = name;
      this.dumpLogAtEnd = dumpLogAtEnd;
      messageQueue = new MessageQueue();
      qAssign = new QueueAssign(messageQueue);
      connectionTable = new ConnectionTable();
      serverThread = new ServerThread(connectionTable, qAssign);
      rolodex = new PhoneTable(new PhoneEntry(name, serverThread.getHostname(),
					      serverThread.getLocalPort()),
			       directoryServer, directoryPort);
      serverThread.start();
      if (keepLog == true) {
	 upTime = new Date();
	sentMessages = new MessageQueue();
	receivedMessages = new MessageQueue();
	if (dumpLogAtEnd == false) {
	  (new LogThread(name, upTime, sentMessages,
			 receivedMessages)).start();
	}
      }
  }

  public void addEntry(PhoneEntry p) {
    rolodex.addEntry(p);
  }

  public void addPhoneEntry(PhoneEntry p) {
    addEntry(p);
  }

  public void closeAllConnections() {
    for (Enumeration e = connectionTable.elements(); e.hasMoreElements();) {
      try {
	((Connection) e.nextElement()).close();
      } catch (IOException i) {
	System.out.println("Error closing connection: "+i);
      }
    }
  }

  /** closes named connection on connection list */
  public void closeConnection(String name) {
    try {
      connectionTable.getConnection(name).close();
    } catch (Exception e) {
    }
  }

  public String getLastConnectionName() {
    return serverThread.getLastConnectionName();
  }

  public int getLocalPort() {
    return serverThread.getLocalPort();
  }

  public String getDirectoryHostname() {
	 return rolodex.getDirectoryHostname();
  }

  public int getDirectoryPort() {
	 return rolodex.getDirectoryPort();
  }

  public String getLocalHost() {
    try{
      return serverThread.getHostname();
    }
    catch(Exception e) {
      return new String("");
    }
  }

  public Enumeration getPhoneEnumeration() {
    return  rolodex.keys();
  }

  public String getMyName() {
    return name;
  }

  /*public void printSentMessagesLog() {
    SimpleDateFormat date = 
      new SimpleDateFormat("MMM dd, yyyy HH:mm:ss:SSS z");
    System.out.println(date.format(upTime));
    logfile.println(date.format(upTime));
    long time1 = upTime.getTime();
    //for (int i = 0; i < sentMessages.size(); i++) {
    for (int i = 0; i < messageQueue.size(); i++) {
      Message m = (Message) messageQueue.elementAt(i);
    //Message m = (Message) sentMessages.elementAt(i);
      Date timestamp = m.getTimeStamp();
      long time2 = timestamp.getTime();
      System.out.println("+"+(time2 - time1)+" ms, "+date.format(timestamp)+
			 " "+m.getMessage());
      logfile.println("+"+(time2 - time1)+" ms, "+date.format(timestamp)+
			 " "+m.getMessage());
      time1 = time2;
    }
  }*/

  public Message readNextMessage() {
    //System.out.println("Communications.readNextMessage()");
    messageQueue.waitForMessage();
    if (logMessages == true) {
      Message m = messageQueue.remove();
      receivedMessages.add(m);
      return m;
    } else {
      return messageQueue.remove();
    }
  }

  public Message readNextMessage(MessageQueue m) {
    m.waitForMessage();
    if (logMessages == true) {
      Message mess = m.remove();
      receivedMessages.add(mess);
      return mess;
    } else {
      return m.remove();
    }
  }

  public void saveAllMessages(String filename) throws IOException {
    if (logMessages == false) {
      System.out.println("message logging not enabled");
      return;
    } else if (dumpLogAtEnd == false) {
      System.out.println("log thread is running");
      return;
    } else {
      PrintWriter logfile = 
	new PrintWriter(new BufferedWriter(new FileWriter(filename)), true);

      SimpleDateFormat date = 
	new SimpleDateFormat("MMM dd, yyyy HH:mm:ss:SSS z");
      //System.out.println(date.format(upTime));
      logfile.println("start time: "+date.format(upTime));

      int senti = 0,
	  receivedi=0;

      Message sentMessage, receiveMessage;
      try {
	sentMessage = (Message) sentMessages.elementAt(senti);
      } catch (ArrayIndexOutOfBoundsException n) {
	sentMessage = null;
      }
      try {
	receiveMessage = (Message) receivedMessages.elementAt(receivedi);
      } catch (ArrayIndexOutOfBoundsException n) {
	receiveMessage = null;
      }

      long time1 = upTime.getTime();
      while ((sentMessage != null) || (receiveMessage != null)) {
	if ((receiveMessage != null)&&(sentMessage == null)) {
	  Date receiveDate = receiveMessage.getTimeStamp();
	  long receiveTime = receiveDate.getTime();
	  /*System.out.println("+"+(receiveTime - time1)+"ms "+
			     date.format(receiveDate)+" from "+
			     receiveMessage.getCorrespondent()+" \""+
			     receiveMessage.getMessage()+"\"");*/
	  logfile.println("+"+(receiveTime - time1)+"ms "+
			  date.format(receiveDate)+" from "+
			  receiveMessage.getCorrespondent()+" \""+
			  receiveMessage.getMessage()+"\"");
	  receivedi++;
	  //receivedMessages.remove();
	  time1 = receiveTime;
	} else if ((sentMessage != null)&&(receiveMessage != null)) {
	  Date sentDate = sentMessage.getTimeStamp(),
	    receiveDate = receiveMessage.getTimeStamp();
	  long sentTime = sentDate.getTime(),
	    receiveTime = receiveDate.getTime();

	  if (sentTime < receiveTime) {
	    /*System.out.println("+"+(sentTime - time1)+"ms "+
			       date.format(sentDate)+" to "+
			       sentMessage.getCorrespondent()+" \""+
			       sentMessage.getMessage()+"\"");*/
	    logfile.println("+"+(sentTime - time1)+"ms "+
			    date.format(sentDate)+" to "+
			    sentMessage.getCorrespondent()+" \""+
			    sentMessage.getMessage()+"\"");
	    senti++;
	    //sentMessages.remove();
	    time1 = sentTime;
	  } else {
	    /*System.out.println("+"+(receiveTime - time1)+"ms "+
			       date.format(receiveDate)+" from "+
			       receiveMessage.getCorrespondent()+" \""+
			       receiveMessage.getMessage()+"\"");*/
	    logfile.println("+"+(receiveTime - time1)+"ms "+
			    date.format(receiveDate)+" from "+
			    receiveMessage.getCorrespondent()+" \""+
			    receiveMessage.getMessage()+"\"");
	    receivedi++;
	    //receivedMessages.remove();
	    time1 = receiveTime;
	  }
	} else {
	//if ((sentMessage != null)&&(receiveMessage == null)) {
	  Date sentDate = sentMessage.getTimeStamp();
	  long sentTime = sentDate.getTime();
	  /*System.out.println("+"+(sentTime - time1)+"ms "+
			     date.format(sentDate)+" to "+
			     sentMessage.getCorrespondent()+" \""+
			     sentMessage.getMessage()+"\"");*/
	  logfile.println("+"+(sentTime - time1)+"ms "+
			  date.format(sentDate)+" to "+
			  sentMessage.getCorrespondent()+" \""+
			  sentMessage.getMessage()+"\"");
	  senti++;
	  //sentMessages.remove();
	  time1 = sentTime;
	}

	try {
	  sentMessage = (Message) sentMessages.elementAt(senti);
	} catch (ArrayIndexOutOfBoundsException n) {
	  sentMessage = null;
	}
	try {
	  receiveMessage = (Message) receivedMessages.elementAt(receivedi);
	} catch (ArrayIndexOutOfBoundsException n) {
	  receiveMessage = null;
	}
      }
    }
  }

  public void saveAllMessages() throws IOException {
    //saveAllMessages(logfilename);
  }

  public void sendMessage(String name, Message m) throws IOException,
    NoPhoneNumberException {
    
    Connection c = connectionTable.getConnection(name);
    
    if (c == null || !c.isAlive()) {
      //the connection is down or never existed
      //so try to establish it
      if(c!= null) {
	// let's take the dead connection out of the table.
	connectionTable.deleteConnection(name);
      }

      //if the entity name is not valid, then a NoPhoneNumberException will
      //be thrown and the caller of sendMessage should deal with it
      try {
	c = new Connection(name, new CommsStream(rolodex.getPhoneNumber(name)),
			   qAssign.getQueue(name), qAssign);
      }
      catch (IOException ex) {
	// if we get here, the phone number might be wrong so lets remove
	// the phone entry and try again, if that fails then we've done
	// all we can and the caller of sendMessage should deal with
	// the IOException that occurs
	
	//remove the bad phone entry
	rolodex.remove(name);

	// try to make the connection with the new phone number from the
	//directory server
	c = new Connection(name, new CommsStream(rolodex.getPhoneNumber(name)),
			   qAssign.getQueue(name), qAssign);
      }

      //if we get here, we actually made the connection.
      connectionTable.addConnection(c);
      c.start();
      Message message = new Message(this.name);
      c.sendMessage(message);
      if (logMessages == true) {
	message.setCorrespondent(name);
	sentMessages.add(message);
      }
    }
    
    c.sendMessage(m);
    
    /*
      for (int tries = 0, success = 0;
      (tries < 3)&&(success == 0); tries++) {
      try {
      System.out.println("sending message to "+name);
      
      System.out.println("sending message succeeded");
      success = 1; // (success = true)
      } 
      catch (IOException s) {
      System.out.println("Message send failed: "+s);
      s.printStackTrace();
	//s.printStackTrace();
	success = 0;
	// remove bad phone number
	rolodex.remove(name);
	// redownload the phone entry
	c = new Connection(name, new 
	CommsStream(rolodex.getPhoneNumber(name)),
	qAssign.getQueue(name));
	connectionTable.addConnection(c);
	Message message = new Message(this.name);
	c.sendMessage(message);
	if (logMessages == true) {
	message.setCorrespondent(name);
	sentMessages.add(message);
	}
	c.start();
	}
	catch (Exception e) {
 	System.out.println("Another exception: "+e);
	e.printStackTrace();
	}
	}
    */
    
    if (logMessages == true) {
      m.setTimeStamp();
      m.setCorrespondent(name);
      sentMessages.add(m);
    }
  }

  /** make all messages from nSt go into message queue mq
      when rev is true, if the entity disconnects, they 
      will be reassigned to the default message queue automatically
      when rev is false, the 
      */
  public void setMessageQueue(String nSt, MessageQueue mq, boolean rev) {
    qAssign.setQueue(nSt, mq, rev);
    Connection con = connectionTable.getConnection(nSt);
    if(con != null) {
      con.setQueue(mq);
    }
  }
  
  public void setMessageQueue(String nSt, MessageQueue mq) {
    setMessageQueue(nSt,mq, false);
  }

  public MessageQueue getMessageQueue(String nST) {
    return qAssign.getQueue(nST);    
  }
  
  // reset nSt to the default message queue
  public void unsetMessageQueue(String nSt) {
    qAssign.unsetQueue(nSt);
    Connection con = connectionTable.getConnection(nSt);
    if(con != null) {
      con.setQueue(messageQueue);
    }
  }
  
  public void setupMessageLogging(String logfilename, Date upTime) {
    this.upTime = upTime;
    //this.logfilename = logfilename;
    sentMessages = new MessageQueue();
    receivedMessages = new MessageQueue();
    logMessages = true;
  }

  public void setLogDumping(boolean b) {
    dumpLogAtEnd =b;
  }

  public void shutdown() {
    closeAllConnections();
    stopServerThread();
    if (logMessages == true) {
      try {
	saveAllMessages();
      } catch (IOException i) {
	System.err.println(i.getMessage());
      }
    }
  }

  public void stopServerThread() {
    serverThread.stop();
    serverThread.close();
  }

  public boolean waitForMessage() {
    return messageQueue.waitForMessage();
  }

  public boolean waitForMessage(MessageQueue m) {
    return m.waitForMessage();
  }

  protected void finalize() throws Throwable {
    sentMessages = receivedMessages = messageQueue = null;
    qAssign = null;
    serverThread = null;
    connectionTable = null;
    rolodex = null;
    upTime = null;
    super.finalize();
  }
}
