package adaptive.modulemanager;

import java.awt.*;
import java.awt.event.*;
import java.util.*;
import adaptive.core.*;

/**
  * The main view for editing a system.  All user interaction is done through
  * the view.  The view gets it's data by observing a model.  Whenever the
  * model changes, the view empties it's panel, and redraws the new modules
  * from the model.
  * <p>
  * To view an open model, just set the view as an observer:
  * <code><br>
  * View v = new View(m,this);<br>
  * v.show();<br>
  * m.addObserver(v);<br>
  * </code>
  * Then whenever the model changes the view will be updated accordingly.
  *
  * @author Yatish Patel
  * @version 1.0
  */
public class View extends Frame implements Observer,ActionListener {
  
  /** Default background color for the layout canvas */
  private Color backgroundColor;

  /** Color of a selected line */
  private Color selectedLineColor=new Color(0xcc,0x00,0x00  ); //new Color(0x33,0,0x99);

  private ModulePanel modulePanel;

  private ScrollPane scrollPane;

  /** The size of the canvase */
  private Dimension canvasSize;
  
  /** The current module the cursor is over */
  private adaptive.modulemanager.Module moduleOver;

  private adaptive.modulemanager.Module moduleOverForProperties;
  
  /** The grid width */
  private int gridWidth=5;
  
  /** The grid height */
  private int gridHeight=5;

  /** The current model this view represents */
  private ManagerModel model;

  /** pointer to the ModuleManager window */
  private ModuleManager moduleManager;

  /** Is the grid active */
  private boolean isGridOn=true;
  
  /** Are modules moving without the mouse button being pressed */
  private boolean moveModules=false;
  
  private String name;

  private FrmSystemProperties frmSystemProperties;
  

  private PopupMenu mouse3Menu;
  private MenuItem mi;
 
  private Panel buttonPanel;
  private Button btnSave;
  private Button btnRefresh;
  
  

  /** The currentl line segment that's highlighted */
  //private LineSegment highLightedSegment;

  /**
    * Create a new View with the specified model.
    *
    * @param m The model to display in the view
    * @param p A pointer to the main ModuleManager window
    */
  public View(ManagerModel m,ModuleManager p) {
    moduleManager=p;
	 
    // Event listners
    WindowListener windowListener= new WindowListener();
    MouseListener mouseListener = new MouseListener();
    MouseMotionListener mouseMotionListener = new MouseMotionListener();	
    KeyListener keyListener = new KeyListener();
    ComponentListener componentListener = new ComponentListener();
    addWindowListener(windowListener);
	 
    model=m;
    canvasSize= new Dimension(400,350);
		
    modulePanel = new ModulePanel();
    modulePanel.setLayout(null);
    modulePanel.setSize(canvasSize.height,canvasSize.width);
    modulePanel.addMouseListener(mouseListener);
    modulePanel.addMouseMotionListener(mouseMotionListener);
    modulePanel.addKeyListener(keyListener);
	 
    scrollPane= new ScrollPane();
    scrollPane.add(modulePanel);
	 
    btnSave= new Button("Save");
    btnSave.addActionListener(this);
    btnSave.addKeyListener(keyListener);

    btnRefresh = new Button("Refresh");
    btnRefresh.addActionListener(this);
    btnRefresh.addKeyListener(keyListener);

    buttonPanel = new Panel();
    buttonPanel.add(btnSave);
    buttonPanel.add(btnRefresh);
    buttonPanel.addKeyListener(keyListener);
    addKeyListener(keyListener);
	 
    add(buttonPanel,"North");
    add(scrollPane,"Center");
    pack();
    setSize(400,350);
    setTitle(m.getSystemName());


    mouse3Menu = new PopupMenu("Hello");
    mi = new MenuItem("Place");
    mi.addActionListener(this);
    mi.setActionCommand("menu3:Place");
    mouse3Menu.add(mi);
    mi = new MenuItem("Map");
    mi.addActionListener(this);
    mi.setActionCommand("menu3:Map");
    mouse3Menu.add(mi);
    mi = new MenuItem("Select/Move");
    mi.addActionListener(this);
    mi.setActionCommand("menu3:SelectMove");
    mouse3Menu.add(mi);
    mi = new MenuItem("Resize");
    mi.addActionListener(this);
    mi.setActionCommand("menu3:Resize");
    mouse3Menu.add(mi);
    mouse3Menu.addSeparator();
    mi = new MenuItem("Copy");
    mi.addActionListener(this);
    mi.setActionCommand("menu3:Copy");
    mi.setEnabled(false);
    mouse3Menu.add(mi);
    mi = new MenuItem("Paste");
    mi.addActionListener(this);
    mi.setActionCommand("menu3:Paste");
    mi.setEnabled(false);
    mouse3Menu.add(mi);
    mi = new MenuItem("Cut");
    mi.addActionListener(this);
    mi.setActionCommand("menu3:Cut");
    mi.setEnabled(false);
    mouse3Menu.add(mi);
    mouse3Menu.addSeparator();
    mi = new MenuItem("Undo");
    mi.addActionListener(this);
    mi.setActionCommand("menu3:Undo");
    mi.setEnabled(false);
    mouse3Menu.add(mi);
    mouse3Menu.addSeparator();
    mi = new MenuItem("Properties");
    mi.addActionListener(this);
    mi.setActionCommand("menu3:Properties");
    mouse3Menu.add(mi);

    add(mouse3Menu);
	 
  }

  /**
    * Updates the view.  This is called by the model to tell the view
    * to change to reflect the changes in the model
    *
    * @param ob The model that's being observed
    * @param o Some object, i'm not really sure..
    */
  public void update(Observable ob,Object o) {
    // Remove all the modules in the panel
    modulePanel.removeAll();
    Enumeration e = model.getModuleNames();
    adaptive.modulemanager.Module m;
    // Add al the modules to the panel
    while (e.hasMoreElements()) {
      m=(Module)model.getModule((String)e.nextElement());

      if (!modulePanel.isAncestorOf(m)) {
	modulePanel.add(m);
	//System.out.println("adding to panel "+m);
      }
    }
    setTitle(model.getSystemName());
    if ( o !=null) {
      if (o instanceof String) {
	String s= (String)o;
	if (s.equals("openfile")) {
	  canvasSize= calculateCanvasSize();
	  modulePanel.setSize(canvasSize);
	}
      }
    }
    modulePanel.repaint();
    btnSave.setEnabled(!model.isSaved());
  }
  
  /**
    * Applies the grid to the specified points. 
    *
    * @param x the X pixel location
    * @param y the Y pixel location
    * @return A point that is the closest grid location to x,y
    */
  private Point applyGrid(int x, int y) {
    return new Point(applyGridX(x),applyGridY(y));
  }
  

  /**
    * Converts the coordinate into a grid coordinate.  Checks which grid
    * coordinate the x is closer to and returns that.
    *
    * @param x the raw coordinate
    * @return the grid coordinate closest to x
    */
  private int applyGridX(int x) {
    int gridExtraX;
    int mouseX;
		
    // If the grid has been turned on, then adjust the mouse
    // coordinates to match the grid
    if (isGridOn) {
      gridExtraX=x % gridWidth;

		  
      if (gridExtraX<=(gridWidth/2)) {
	mouseX=x-gridExtraX;
      } else {
	mouseX=x-gridExtraX+gridWidth;
      }
    } else {
      mouseX=x;
    }	 
    return mouseX;
		
  }

  /**
    * Processes any action events that are sent in.
    *
    * @param evt The action to process.
    */ 
  public void actionPerformed(ActionEvent evt) {
    Object c = evt.getSource();
    
    
    if (c instanceof MenuItem) {
      String command = ((MenuItem)c).getActionCommand();
      
      if (command.equals("menu3:Properties")) {
	
	// If this cursor is over a module, display a module properties window
	if (moduleOverForProperties!=null) {
	  // if there are selected modules, display a properties for all
	  // the selected modules
	  if (model.getNumberOfSelected()>1 &&
	      moduleOverForProperties.isSelected()) {
	    Point l=moduleOverForProperties.getLocationOnScreen();
	    (new ModulePropertiesWindow(model.getSelectedModules(),
					model,l.x,l.y)).show();
	  } else {
	    moduleOverForProperties.showProperties();
	  }
	} else {
	  editSystemProperties();
	}
	
      } else if(command.equals("menu3:Place")) {
	moduleManager.setCurrentTool(moduleManager.PLACE);
      } else if(command.equals("menu3:Map")) {
	moduleManager.setCurrentTool(moduleManager.MAP);
      } else if(command.equals("menu3:SelectMove")) {
	moduleManager.setCurrentTool(moduleManager.SELECT);
      } else if(command.equals("menu3:Resize")) {
	moduleManager.setCurrentTool(moduleManager.RESIZE);
      }
    } else if(c instanceof Button) {
      if (c==btnSave) {
	moduleManager.saveSystem(model,false);
      } else if(c==btnRefresh) {
	//	model.refreshModules();
	update(null,null);
	//	model.syncPorts();
      }
      
    }
    
    
  }
  
  
  /**
    * Converts the coordinate into a grid coordinate.  Checks which grid
    * coordinate the y is closer to and returns that.
    *
    * @param y the raw coordinate
    * @return the grid coordinate closest to y
    */
  private int applyGridY(int y) {
    int mouseY;
    int gridExtraY;
	 
    // If the grid has been turned on, then adjust the mouse
    // coordinates to match the grid
    if (isGridOn) {
      gridExtraY=y % gridHeight;		  
		
      if (gridExtraY<=(gridHeight/2)) {
	mouseY=y-gridExtraY;
      } else {
	mouseY=y-gridExtraY+gridHeight;
      }
      //System.out.println(event.getX()+","+event.getY()+" "+mouseX+","+mouseY);
		
      //System.out.println(event.getX()+","+event.getY()+" "+mouseX+","+mouseY);
		
    } else {
      mouseY=y;
    }	 
    return mouseY;
  }

  /**
    * Calculates the minimum canvas size needed to display all the modules
    * in the current model.
    *
    * @return A Dimension with the minimum canvas size needed to display all
    * the modules in the current model.
    */
  public Dimension calculateCanvasSize() {
    Enumeration e = model.getModuleNames();
    adaptive.modulemanager.Module m;
    Dimension d;
    Point p;
    int width=0, height=0;
    while (e.hasMoreElements()) {
      m = (Module)model.getModule((String)e.nextElement());
      d = m.getSize();
      p = m.getLocation();
      width=Math.max(width,d.width+p.x);
      height=Math.max(height,d.height+p.y);
    }
    return new Dimension(width,height);
  }

  /**
    * Brings up a System Properties window for the current model.
    */
  public void editSystemProperties() {
    if (frmSystemProperties==null || (!frmSystemProperties.isVisible())) {
      model.createInternalMapping();
      frmSystemProperties=new FrmSystemProperties(model);
    } 
	 
    frmSystemProperties.show();
  }
  

  /**
    * Returns the size of this view's modulePanel
    *
    * @return the size of this view's modulePanel
    */
  public Dimension getCanvasSize() {
    return modulePanel.getSize();
  }
  
  /**
    * Return the model of this view
    *
    * @return the model of this view
    */
  public ManagerModel getModel() {
    return model;
  }

  

  /**
    * Get the name for the module that's about to be placed.  This will either
    * prompt the user for a name, or choose a default name.  It checks to
    * make sure the new module name isn't already being used by another module.
    *
    * @param x The x location for the module
    * @param y The y location for the module
    * @param getName Wheter or not to ask the user for the module name
    */
  private void getModuleName(int x,int y,boolean getName)
    {
      // The default name will be the moduleclass plus a number
      String moduleClass=moduleManager.getBrowserModel().getSelectedModule().getClassType();
      String shortModuleClass;
      if (moduleClass.indexOf(".")!=0) {
	shortModuleClass=moduleClass.substring(moduleClass.indexOf(".",moduleClass.indexOf(".")+1)+1);
      } else
	shortModuleClass=moduleClass;

      String modName=shortModuleClass;
	 
      // We have to find a valid name for the module
      modName=model.getValidName(modName);
	 
      // If we want to prompt the user, show the dialog box
      if (getName) {
	Point p = getLocation();
	dlgModuleName dl=new dlgModuleName(this,p.x+x,p.y+y,1,modName,moduleManager.getBrowserModel().getSelectedModule()); 
	if (((p.x+x-150)>0) && ((p.y+y-40)>0)) 
	  dl.setLocation(p.x+x-150,p.y+y-40); 				 				
	dl.show();
      } else	
	/* ++++ Pretty sure this reference to the Browser Model is fine */
	model.placeModule(x,y,modName,
			  moduleManager.getBrowserModel().getSelectedModule(),false);

    }

  /**
    * Get a reference to the module manager
    *
    * @return a reference to the module manager
    */
  public ModuleManager getModuleManager() {
    return moduleManager;
  }
  
  /**
    * return the name of this view
    *
    * @return the name of this view
    */
  public String getName() {
    return name;
  }
  
  /**
    * Get the selected items in this view
    *
    * @return an AgentContainer with all the selected Modules
    */
  public ConfigAndLayout getSelectedItems() {
    return model.getSelectedModulesAsAgentContainer();
  }


  
  /**
    * Inserts the given agentContainer in the view
    *
    * @param ac the agent container to insert
    */
  public void insertConfigAndLayout(ConfigAndLayout cal) {
    // Inserts the modules into the model and selects all the modules
    model.displayConfigAndLayout(cal,new Point(10,10),true);
    model.notifyObservers();
	 
    // now move the modules until the user clicks to drop them

    // doesn't seem to work, and i'm too lazy to fix it..
    //  moveModules=true;   

    modulePanel.repaint();
    System.out.println("pasting in "+this);
  }
  

  /**
    * Paints the modules on the canvas and draws the lines to connect
    * them.
    *
    * @param g The graphics context to draw in
    */
  public void paint(Graphics g)
    {
      modulePanel.repaint();
    }

  /**
    * Print the system
    *
    * @param printJob the printjob to print too
    */
  public void print (PrintJob printJob) {
	 
    if (printJob!=null) {
      Graphics pg=printJob.getGraphics();
      if (pg!=null) {
	Enumeration e = model.getModules();
	Module m;
	while (e.hasMoreElements()) {
	  m = (Module)e.nextElement();
	  m.drawModule(pg,m.getLocation());
	}
	e = model.getLines();
	ConnectLine cl;
	while (e.hasMoreElements()) {
	  cl = (ConnectLine)e.nextElement();
	  cl.drawLine(pg,null);
	}
	paint(pg);
	pg.dispose();
      }
      printJob.end();
    } 
  }
  
  /**
    * Resize the scroll pane that contains the module Panel
    *
    * @param d the New size of the scroll pane
    */
  public void resizeScrollPane (Dimension d) {
    Dimension c = modulePanel.getSize();
    Point p = scrollPane.getLocation();
	 
    canvasSize.setSize(Math.max(d.width-p.x,c.width),
		       Math.max(d.height-p.y,c.height));
    modulePanel.setSize(canvasSize.width,canvasSize.height);	  		
    scrollPane.doLayout();
    pack();

  }

  /**
    * Select all the modules in the model.
    */
  public void selectAll() {
    model.selectAll();
  }
  
  /**
    * Sets this view's canvas to the specified dimension
    *
    * @param canvasSize the new canvas size
    */
  public void setCanvasSize(Dimension canvasSize) {
    modulePanel.setSize(canvasSize.width,canvasSize.height);				  
    pack();
  }

  /**
    * Set the name of this view.
    * 
    * @param n The new name.
    */
  public void setName(String n) {
    name=n;
  }
  
  

  /**
    * Set the cursor in the main drawing panel
    *
    * @param c The new cursor
    */
  public void setPanelCursor(Cursor c) {
    modulePanel.setCursor(c);
  }

  /**
    * Shows the popup menu for this view.
    *
    * @param x The x location for the menu
    * @param y The y location for the menu
    */
  public void showPopupMenu(int x,int y) {
    Component c=modulePanel.getComponentAt(x,y);
    if (c!=null && c instanceof Module) {
      // It did occur over a module.  So save it for possibly triggering
      // Module Property Window
      moduleOverForProperties=(Module)c;
    } else {
      moduleOverForProperties = null;
    }
    
    mouse3Menu.show(this,x,y);
  }
  
  /* Added by Andrea (7/29/99) to make my life easier right now.  I'm 
     probably ruining some kind of abstraction with this, but I'll find a 
     better solution later *
     public ModulePanel getModulePanel(){
     return modulePanel;
     }
  */

  class ComponentListener extends java.awt.event.ComponentAdapter {
    public void componentResized(java.awt.event.ComponentEvent evt)
      {
	Component c = evt.getComponent();
	resizeScrollPane(c.getSize());		  
      }
  }
  

  class KeyListener extends KeyAdapter {
    /*
     * Triggered whenever a key is pressed 
     */
    public void keyPressed(KeyEvent evt)
      {
	if (moduleOver!=null) {
	  (moduleOver).keyPressed(evt);			 
	}
	int keyCode=evt.getKeyCode();
	int modifiers= evt.getModifiers();
	
	// Alt-Q quits
	if (evt.isMetaDown() && keyCode==KeyEvent.VK_Q) {
	  moduleManager.quitProgram();
	}
	// Control-Q quits
	if (evt.isControlDown() && keyCode==KeyEvent.VK_Q) {
	  moduleManager.quitProgram();
	}

	// DEL either deletes the selected modules, or the connection line
	// currently highlighted
	if (keyCode==KeyEvent.VK_DELETE) {
	  if (model.getHighlightedSegment()!=null) {
	    model.deleteConnection(model.getHighlightedSegment().getParent());
	    setCursor(Cursor.getDefaultCursor());
	  } else
	    model.deleteSelectedModule();
	} else if (keyCode==KeyEvent.VK_ESCAPE) {
	  // ESC stops drawing the line
	  model.setDrawingLine(false);
	  model.setCurrentLine(null);
	  modulePanel.repaint();
	}
      }
  }

  class MouseListener extends java.awt.event.MouseAdapter {
    public void mousePressed(java.awt.event.MouseEvent event) {
      //System.out.println("Mouse Pressed: "+event);
      // Since the View is capturing all events and dispatching
      // them to the appropriate module, we have to see if the event 
      // occured over a module. if so, send it into the module.
      Component c=modulePanel.getComponentAt(event.getX(),
					     event.getY());
      if (c!=null) {		  
	if (c instanceof Module) {
	  // It did occur over a module.  Now we send it in. 
	  Point p = c.getLocation();
	  MouseEvent newEvent = new MouseEvent(c,
					       event.getID(),0,
					       event.getModifiers(), 
					       event.getX()-p.x,
					       event.getY()-p.y,
					       event.getClickCount(),
					       false);
	  ((Module)c).mousePressed(newEvent);
	  return;
	  
	}
      }
      
      // If the cursor is not over a line segment, start drawing a selection
      // rectangle
      if ((model.getHighlightedSegment()==null) && (moduleManager.isToolSelect())) {
	// Not over a module, so instead draw a rectangle to allow the
	// user to highlight all modules under the rectangle		
	model.setSelectRectangle(new SelectRectangle(event.getX(),event.getY(),
						     1,1));
	//modulePanel.repaint();
      }
    }
    
    public void mouseClicked(java.awt.event.MouseEvent event) {
      //      System.out.println("Mouse click: "+event);
      // If the user just pasted something then stop moving once
      // the mouse is clicked
      // This doesn't work
      /*
	if (moveModules) {
	moveModules=false;
	return;
	} */
      //Component c=modulePanel.getComponentAt(event.getX(),event.getY());
      
      int modifiers = event.getModifiers();
      if ((modifiers & InputEvent.BUTTON3_MASK) !=0) {
	showPopupMenu(event.getX(),event.getY());
	return;
      }
      
      // Since the View is capturing all events and dispatching
      // them to the appropriate module, we have to see if the event 
      // occured over a module. if so, send it into the module.
      
      //	if (c!=null) {		  
      //if (c instanceof Module) {

      //figure out if we're over a module
      Component c=modulePanel.getComponentAt(event.getX(),event.getY());
      if (c!=null && c instanceof Module) {
	// It did occur over a module.
	moduleOver=(Module)c;
      } else {
	moduleOver = null;
      }
      
      if (moduleOver!=null) {
	// It did occur over a module.  Now we send it in.
	Point p = moduleOver.getLocation();
	MouseEvent newEvent = new MouseEvent(moduleOver,event.getID(),0,
					     event.getModifiers(),
					     event.getX()-p.x,
					     event.getY()-p.y,
					     event.getClickCount(),false);
	((Module)moduleOver).mouseClicked(newEvent);
	return; 
	//  }
      }	
      
      
      
      if ((modifiers & InputEvent.BUTTON3_MASK) != 0) {
	if (((modifiers & InputEvent.SHIFT_MASK)!=0) 
	    && (model.getNumberOfSelected()>0)) {
	  Point l=moduleOver.getLocationOnScreen();
	  (new ModulePropertiesWindow(model.getSelectedModules(),
				      model,l.x,l.y)).show();
	} else 
	  moduleManager.setCurrentTool(moduleManager.getCurrentTool()+1);
	return;
      }
      
      int mouseX=applyGridX(event.getX());
      int mouseY=applyGridY(event.getY());
      
      
      
      // check to see what tool is being used to see what action must
      // take place
      if (moduleManager.isToolPlace()) {
	// The user is using the placement tool, so if a module is selected
	// in the moduleToolbar then place it where the click occured
	if (moduleManager.getBrowserModel().getSelectedModule()!=null) {
	  getModuleName(mouseX,mouseY,(modifiers & InputEvent.SHIFT_MASK)!=0);
	  // If control is not pressed, deselect the module in the toolbar
	  if ((modifiers & InputEvent.CTRL_MASK)==0) {
	    moduleManager.getBrowserModel().selectAgent(null);
	    moduleManager.setCurrentTool(moduleManager.MAP);
	  }
	}
      } else if (moduleManager.isToolMap()) {
	// the user is using the map tool, so place a bend wherever the
	// mouse click occured (if a line is being mapped)
	if (model.getCurrentLine()!=null) 
	  model.getCurrentLine().addBend(new Point(mouseX,mouseY));
      }
      
      /*
	if (((modifiers & InputEvent.CTRL_MASK)!=0) && 
	(modifiers & InputEvent.BUTTON1_MASK)!=0) {
	//deleteSelectedModule();
	}			  
	else if ((modifiers & InputEvent.BUTTON1_MASK) !=0) {
	if (moduleManager.getModuleToolbar().getSelectedModule()!=null) {
	//System.out.println("trying to place");
	getModuleName(mouseX,mouseY,(modifiers & InputEvent.SHIFT_MASK)!=0);
	moduleManager.getModuleToolbar().selectAgent(null);
	}
	} else if ((modifiers & InputEvent.BUTTON2_MASK) !=0) {
	if (model.getCurrentLine()!=null) 
	model.getCurrentLine().addBend(new Point(mouseX,mouseY));
	
	}
      */
      
      
    }
    
    public void mouseReleased(java.awt.event.MouseEvent event) {
      // When the mouse button is released, we no longer draw the select
      // rectangle
      if (model.getSelectRectangle()!=null) {
	model.setSelectRectangle(null);
	repaint();
      }

      // Since the View is capturing all events and dispatching
      // them to the appropriate module, we have to see if the event 
      // occured over a module. if so, send it into the module.
      Component c=modulePanel.getComponentAt(event.getX(),event.getY());
      if (c!=null) {		  
	if (c instanceof Module) {
	  // It did occur over a module.  Now we send it in.
	  Point p = c.getLocation();
	  MouseEvent newEvent = new MouseEvent(c,event.getID(),0,
					       event.getModifiers(),
					       event.getX()-p.x,
					       event.getY()-p.y,
					       event.getClickCount(),false);
	  ((Module)c).mouseReleased(newEvent);
	  return;
	}
		  
      }	
		
      // When we release the button, we don't need to drag the module any
      // more
      model.setModuleToDrag(null,0,0);
      if (model.isDraggingLine()) {
	modulePanel.repaint();
	model.setDraggingLine(false);
      }
    }
  }



  class MouseMotionListener extends java.awt.event.MouseMotionAdapter {
	 
    public void mouseDragged(java.awt.event.MouseEvent event) {
      int mouseX,mouseY;
      LineSegment highlightedSegment=model.getHighlightedSegment();
		
      mouseX=event.getX();
      mouseY=event.getY();
      // If the user is using the select/move tool, let him move the
      // module, connection line, or selected modules
      if (moduleManager.isToolSelect()) {

	if (!model.moveModule(mouseX,mouseY))
	  if (!model.moveLineSegment(applyGrid(mouseX,mouseY))) {
	    if (model.getSelectRectangle()!=null) {
	      model.getSelectRectangle().setSize
		(mouseX-model.getSelectRectangle().x,
		 mouseY-model.getSelectRectangle().y);	  
	      modulePanel.repaint();
	    }	
	  } 

      } else if (moduleManager.isToolResize()) {
	// If the user is using the resize tool, resize the module
	model.resizeModule(applyGrid(mouseX,mouseY));
      }
    }	
		
  

    public void mouseMoved(java.awt.event.MouseEvent event) {

      // When the user pastes from the clipboard the modules are
      // displayed on the screen.  We have to move the modules to
      // where ever the mouse cursor is and then drop them when
      // the user clicks.
      if (moveModules) {
	model.moveModule(event.getX(),event.getY());
	return;
      }
	 
      LineSegment highLightedSegment=model.getHighlightedSegment();
	 
      int mouseX=event.getX();
      int mouseY=event.getY();

      // Since the View is capturing all events and dispatching
      // them to the appropriate module, we have to see if the event 
      // occured over a module. if so, send it into the module.
      Component c=modulePanel.getComponentAt(event.getX(),event.getY());
      if (c!=null) {		  
	if (c instanceof Module) {
	  // It did occur over a module.  Now we send it in. 
	  moduleOver=(Module)c;
	  Point p = c.getLocation();
	  MouseEvent newEvent = new MouseEvent(c,event.getID(),0,
					       event.getModifiers(),
					       mouseX-p.x,event.getY()-p.y,
					       event.getClickCount(),false);
	  ((Module)c).mouseMoved(newEvent);
	  return;
	}
      }		
      moduleOver=null;
		
      // If the user is mapping an input/output we have to draw that
      // connection line on the screen
      if (model.getCurrentLine()!=null) {
	c = event.getComponent();
	model.getCurrentLine().eraseLine(c.getGraphics());
	model.getCurrentLine().drawLine(c.getGraphics(),
					new Point(event.getX(),event.getY()));
      } else if (model.isModuleSelected()) {
	// If the user is just moving the mouse cursor around, we have to
	// highlight a connection line whenever he has the mouse cursor
	// on top of it.
	Module m;
	ConnectLine cl;
	LineSegment ls=null;
	Object o;
	Enumeration e;
	Enumeration l;
					 
	// Check all the selected modules to see if we're on top
	// of a line segment. if we are we highlight it.
	Enumeration selModules = model.getSelectedModules();
	while (selModules.hasMoreElements()) {
	  m = (Module)selModules.nextElement();
	  e = m.getInputConnectLines();
	  while (e.hasMoreElements()) {
	    o=e.nextElement();
	    if (o instanceof ConnectLine) 
	      ls =((ConnectLine)o).getSegmentByPoint(new Point(event.getX(),event.getY()));
	    if (ls!=null) {
		
	      ls.draw(modulePanel.getGraphics(),Color.blue);
	      highLightedSegment=ls;
	      if (ls.isHorizontal()) 
		setCursor(Cursor.getPredefinedCursor(Cursor.S_RESIZE_CURSOR));
	      else
		setCursor(Cursor.getPredefinedCursor(Cursor.W_RESIZE_CURSOR));
	      model.setHighlightedSegment(ls);
	      return;
	    }
	  }
	  e = m.getOutputConnectLines();					  
	  while (e.hasMoreElements()) {
	    o=e.nextElement();
	    if (o instanceof Vector) {
	      l = ((Vector)o).elements();
	      while (l.hasMoreElements()) {	
		ls=((ConnectLine)l.nextElement()).getSegmentByPoint(new Point(event.getX(),event.getY()));	
		if (ls!=null) {
		  ls.draw(modulePanel.getGraphics(),Color.blue);
		  highLightedSegment=ls;
		  if (ls.isHorizontal()) 
		    setCursor(Cursor.getPredefinedCursor(Cursor.S_RESIZE_CURSOR));
		  else
		    setCursor(Cursor.getPredefinedCursor(Cursor.W_RESIZE_CURSOR));
		  model.setHighlightedSegment(ls);
		  return;
		}
	      }
	    }
								
	  }
	}

	if (highLightedSegment!=null) {
	  highLightedSegment.draw(modulePanel.getGraphics(),selectedLineColor);
	  setCursor(Cursor.getDefaultCursor());
	  model.setHighlightedSegment(null);
	  highLightedSegment=null;
	}
      }
    }
  }
  


  class WindowListener extends java.awt.event.WindowAdapter
  {
    public void windowClosing(java.awt.event.WindowEvent event)
      {
	moduleManager.closeView((View)event.getComponent());
	//dispose();
      }
    public void windowActivated(java.awt.event.WindowEvent event) {
      moduleManager.setActiveView((View)event.getComponent());
    }
	 
  }


}
