/**
 * The main window for the adaptive.modulemanager.Module Manager.  Brings up a window with the
 * menu bar and tool bar.  Also opens up the browser window, status window,
 * and module toolbar.  This also keeps track of all open views/models.
 *
 * When the module Manager is run, it checks to see if the network has been
 * enabled and attempts to connect to a running agentServer.  If it
 * successfully connects, there will be an icon in the Browser that says
 * "Network" in addition to the "Local" icon. 
 *
 * If no network is enabled, but the configuration file loaded in has
 * agents that orignate from the agentServer, the adaptive.modulemanager.Module Manager will try
 * to find an agent of the same class on the local filesystem.  If it finds
 * one it will be loaded, otherwise the agent will not be loaded.
 *
 * In theory the adaptive.modulemanager.Module Manager can be a GUIComponent in cyberrave.
 *
 * @author Yatish Patel
 * $Log: ModuleManager.java,v $
 * Revision 1.15.2.21  2000/06/15 18:47:45  telamon
 * fixed minimizing ModuleManager hid whole program bug
 *
 * Revision 1.15.2.20  2000/06/06 00:23:26  jrj
 * changed so it will synchronize the ports with the Agent when it loads
 * a file. also flipped around the appearance of the output ports.
 *
 * Revision 1.15.2.19  2000/05/20 22:17:10  jrj
 * Updated error message handling so an exception in Coordinator when
 * trying to execute a configuration will appear in ModuleManager's
 * status window
 *
 * Revision 1.15.2.18  2000/05/20 19:28:46  jrj
 * error message update
 *
 * Revision 1.15.2.17  2000/05/15 22:57:23  jrj
 * fixed typo
 *
 * Revision 1.15.2.16  2000/05/15 22:40:00  jrj
 * Made fixes to macros so that they work.  Changed the executing macro
 * so that it is recursive, can be manipulated as a whole, and its ports
 * can be asked for directly as if it were any other BasicPortAgent.
 * Made several other changes to simplify and assist these changes.
 * Finally purged the last remaining bit of things named clump.
 *
 */

package adaptive.modulemanager;

import java.awt.*;
import java.util.*;
import java.awt.event.*;
import adaptive.core.*;
import java.io.*;
//import comms.core.*;
import adaptive.core.net.*;

public class ModuleManager extends Frame implements ActionListener, ItemListener {

  public final int PLACE=1;
  public final int MAP=2;
  public final int SELECT=3;
  public final int RESIZE=4;
  private final int MAXTOOL=4;
  private final int MINTOOL=1;  

  private static final String PACKAGE_ROOT=".";
  private static final String NETWORK_HOST="localhost";
  private static final int NETWORK_PORT=6500;
  
  public static final String PROGRAM_NAME="Module Manager";
  public static final String VERSION="0.1.8a";
  
  /** A Hashtable of all the models in use indexed by systemName */
  private Hashtable models;
  
  /** A vector of all the views in use */
  private Hashtable views;

  /** The package root */
  private String networkPackageRoot;
  private String localPackageRoot;
  

  /** Network Server */
  private String networkHost;

  /** Network port */
  private int networkPort;
  
  private boolean usingNetwork=false;
  
  /** Name for this ModuleManager */
  private String name;
  
  /** The status window */
  public StatusWindow statusWindow;

  /** The browser */
  private Browser browser;

  /** The module Toolbar */
  private ModuleToolbar moduleToolbar;
  /** The checkboxitem in the menu for the status window */
  private CheckboxMenuItem chkStatusWindow;
  /** The checkbox item in the menu for the module toolbar */
  private CheckboxMenuItem chkModuleToolbar;
  /** The checkbox item in the menu for the browser */
  private CheckboxMenuItem chkBrowser;
  private CheckboxMenuItem chkGrid;
  
  /** The module Menubar */  
  private MenuBar menuBar;
  private Menu menuSystems,menuFile;
  
  private MenuItem miSave,miClose,miSaveAs,miSelectAll,miAbout;
  private MenuItem miCanvasSize,miPaste,miProperties,miCopy;
  
  private ConfigAndLayout clipBoard;

  /** Default background color for the layout canvas */
  private Color backgroundColor;
  
  /** the agent loader */
  private AgentLoader agentLoader;
  
  private View activeView;

  private Font moduleToolbarFont,statusWindowFont,browserFont;

  private Font font;
  private Button btnEditMode,btnRunMode;
  private Button btnMap,btnResize,btnPlace,btnSelect;
  private int currentTool;
  private Button btnCurrentTool;
  private boolean isEditor=true;
  private boolean isApplet = false;
  private DirectoryInfo dirInfo;

  /** runningAlone must be package visible */
  static boolean runningAlone=false;
  private boolean neverShown=true;
  private BrowserModel browserModel;
  private static boolean enableAll=false;
  private static boolean isDebugging=false;
  private static int debugLevel=0;

  //private String homeDir="";
  //private Dimension screenSize;
  private Properties properties;
  
  /*
    public ModuleManager(boolean isApplet) {
    this();
    }
  */

  public ModuleManager(String networkHost,int networkPort,String packageRoot,boolean usingNetwork) {
    this();
    neverShown=false;
    this.networkHost=networkHost;
    this.networkPort=networkPort;
    this.usingNetwork=usingNetwork;
    
    
    // Make sure there is a trailing slash on the packageroot
    if (packageRoot.charAt(packageRoot.length()-1)!='/') {
      packageRoot=new String(packageRoot+"/");
    }
    
    try {	
      // Create a new agent loader with the specified package root
      try {
	// Create the communications link to connect to the agentloader
	//	Communications commLink=null;
	//if (usingNetwork) {
	//  commLink=new Communications(name,networkHost,networkPort,false,true);
	//	}
	dirInfo = new DirectoryInfo(networkHost,networkPort,
						  "ModuleManager");
	dirInfo.retrieveDirInfo();
	PhoneEntry pe=dirInfo.getAgentServerEntry();
	//add check if there was a problem with the agent server and throw an exception
	if (pe==null){
	    System.err.println("Error contacting Agent Server--possibly not started");
	}

	agentLoader = new AgentLoader(packageRoot,pe.getHost(),pe.getPort());
	
	// Now we have to get the network package root from the agentloader
	if (usingNetwork) networkPackageRoot=agentLoader.getPackageRoot(false)+"/";
	// we can also retrieve our local package root from it
	localPackageRoot=agentLoader.getPackageRoot(true)+"/";
      }
      catch(Exception ex){
	ex.printStackTrace();
	usingNetwork=false;
      }
      // tell the browser what agentloader to use
      browser.setAgentLoader(agentLoader,localPackageRoot,networkPackageRoot,
			     usingNetwork);
      
      // Show the user the icons for the local and the network package root
      browser.displayFoldersInDirectory(null);
      
    } catch (Exception e) {
      System.err.println("Exception: "+e);
    }
    
    statusWindow.setVisible(true);
    //moduleToolbar.setVisible(true);
    browser.setVisible(true);

  }
  
  public ModuleManager() {
    Properties defaultProperties = new Properties();
    defaultProperties.put("numlastopened",String.valueOf(0));
    defaultProperties.put("maxlastopened",String.valueOf(5));
    defaultProperties.put("homedir",System.getProperty("user.home"));
    Dimension dim = this.getToolkit().getScreenSize();
    defaultProperties.put("screensizex",String.valueOf(dim.height));
    defaultProperties.put("screensizey",String.valueOf(dim.width));
    defaultProperties.put("resolution",String.valueOf(this.getToolkit().getScreenResolution()));
    
    properties = new Properties(defaultProperties);
    String propertiesFile=System.getProperty("user.home")+"/.modulemanagerrc";
    
    try {
      FileInputStream fis=new FileInputStream(propertiesFile);
      properties.load(fis);
      fis.close();
    } catch (Exception e) {
    }
    
    
    // calculate the location and size of the browser
    
    //*Set up the windows to look pretty:
    //if (properties.get("screensize
    Font font = new Font("SansSerif",Font.PLAIN,12);
    moduleToolbarFont=new Font("SansSerif",Font.PLAIN,12);
    statusWindowFont=moduleToolbarFont;
    browserFont=moduleToolbarFont;
    
    backgroundColor=Color.lightGray;
    setBackground(backgroundColor);
    menuBar= new MenuBar();	
    menuBar.setFont(font);
    setMenuBar(menuBar);
    
    //*Set up menu:
    Menu m;
    MenuItem mi;
    
    
    /////////////
    // File Menu
    ////////////
    menuFile = new Menu("File");
    menuBar.add(menuFile);

    mi = new MenuItem("New");
    mi.addActionListener(this);
    menuFile.add(mi);
    
    mi = new MenuItem("Open");
    mi.addActionListener(this);
    menuFile.add(mi);

    miSave = new MenuItem("Save");
    miSave.addActionListener(this);
    miSave.setEnabled(false);
    menuFile.add(miSave);

    miSaveAs = new MenuItem("Save As");
    miSaveAs.addActionListener(this);
    miSaveAs.setEnabled(false);
    menuFile.add(miSaveAs);

    miClose=new MenuItem("Close");
    miClose.addActionListener(this);
    miClose.setEnabled(false);
    menuFile.add(miClose);

    menuFile.addSeparator();
	 
    miProperties=new MenuItem("Properties");
    miProperties.addActionListener(this);
    miProperties.setEnabled(false);
    menuFile.add(miProperties);
	 
    menuFile.addSeparator();
    
    mi = new MenuItem("Print");
    mi.addActionListener(this);
    mi.setEnabled(true);
    menuFile.add(mi);
    
    menuFile.addSeparator();
    
    //*Checks how many open files there are and lists each in the File menu
    int numOpened = Integer.parseInt(properties.getProperty("numlastopened"));
    if (numOpened>0) {
      String fname;
      String filename;
      for (int no=1;no<=numOpened;no++) {
	filename=properties.getProperty("lastopened"+String.valueOf(no));
	if (filename.length()>30) {
	  fname=filename.substring(0,5)+"..."+
	    filename.substring(filename.length()-30);
	} else fname=filename;
	
	mi = new MenuItem(fname);
	mi.setActionCommand("open:"+filename);
	mi.addActionListener(this);
	menuFile.add(mi);
      }
      menuFile.addSeparator();
    }
    
    mi = new MenuItem("Quit");
    mi.addActionListener(this);
    menuFile.add(mi);
    

    ////////////
    // Edit Menu
    ////////////
    m= new Menu("Edit");
    menuBar.add(m);
    
    mi = new MenuItem("Cut");
    mi.addActionListener(this);
    mi.setEnabled(false);
    m.add(mi);
    
    miCopy = new MenuItem("Copy");
    miCopy.addActionListener(this);
    miCopy.setEnabled(false);
    m.add(miCopy);

    miPaste = new MenuItem("Paste");
    miPaste.addActionListener(this);
    miPaste.setEnabled(false);
    m.add(miPaste);

    miSelectAll = new MenuItem("Select All");
    miSelectAll.addActionListener(this);
    miSelectAll.setEnabled(false);
    m.add(miSelectAll);
    
    m.addSeparator();
    
    miCanvasSize = new MenuItem("Canvas Size");
    miCanvasSize.addActionListener(this);
    miCanvasSize.setEnabled(false);
    m.add(miCanvasSize);

    
    ////////////
    // View Menu
    ////////////
    m= new Menu("View");
    menuBar.add(m);

    chkModuleToolbar  = new CheckboxMenuItem("Module Toolbar",false);	
    chkModuleToolbar.addItemListener(this);
    m.add(chkModuleToolbar);

    chkStatusWindow  = new CheckboxMenuItem("Status Window",true);	
    chkStatusWindow.addItemListener(this);
    m.add(chkStatusWindow);

    chkBrowser = new CheckboxMenuItem("Browser Window",true);
    chkBrowser.addItemListener(this);
    m.add(chkBrowser);

    
    ////////////
    // Tool Menu
    ////////////
    m = new Menu("Tools");
    menuBar.add(m);

    mi= new MenuItem("Launch");
    mi.addActionListener(this);
    mi.setEnabled(true);
    m.add(mi);

    Menu mm= new Menu("Align");
    mi = new MenuItem("Horizontally by top");
    mi.addActionListener(this);
    mm.add(mi);
    mi = new MenuItem("Horizontally by bottom");
    mi.addActionListener(this);
    mm.add(mi);
    mi = new MenuItem("Vertically by left");
    mi.addActionListener(this);
    mm.add(mi);
    mi = new MenuItem("Vertically by right");
    mi.addActionListener(this);
    mm.add(mi);
    m.add(mm);

    m.addSeparator();

    chkGrid=new CheckboxMenuItem("Snap to Grid",true);
    chkGrid.addItemListener(this);
    m.add(chkGrid);
    
    
    ///////////
    // Systems
    ///////////
    menuSystems = new Menu("Systems");
    menuBar.add(menuSystems);

    
    ///////////
    // Help
    ///////////
    m = new Menu("Help");
    menuBar.setHelpMenu(m);

    miAbout=new MenuItem("About");
    miAbout.addActionListener(this);
    m.add(miAbout);

    //*That's all for the menus!!

    
    Panel toolbar= new Panel();
    toolbar.setLayout(null);
    toolbar.setBounds(0,50,400,30);
    
    btnPlace = new Button("Place");
    btnPlace.setBounds(0,0,50,30);
    btnPlace.setFont(font);
    btnPlace.addActionListener(this);
    toolbar.add(btnPlace);
    
    btnMap = new Button("Map");
    btnMap.setBounds(50,0,50,30);
    btnMap.setFont(font);
    btnMap.addActionListener(this);
    toolbar.add(btnMap);
    
    btnSelect = new Button("Select/Move");
    btnSelect.setBounds(100,0,90,30);
    btnSelect.setFont(font);
    btnSelect.setForeground(Color.white);
    btnSelect.setBackground(Color.darkGray);
    btnSelect.addActionListener(this);
    toolbar.add(btnSelect);
    btnCurrentTool=btnSelect;
    currentTool=SELECT;
    
    btnResize = new Button("Resize");
    btnResize.setBounds(190,0,50,30);
    btnResize.setFont(font);
    btnResize.addActionListener(this);
    toolbar.add(btnResize);
    
    setLayout(new BorderLayout());
    
    //*That sets up all the toolbar buttons


    Label l = new Label("Mode:");
    l.setFont(font);
    l.setBounds(250,0,50,30);
    //toolbar.add(l);

    btnEditMode = new Button("Edit");
    btnEditMode.addActionListener(this);
    btnEditMode.setFont(font);
    btnEditMode.setBounds(300,0,40,30);
    btnEditMode.setForeground(Color.white);
    btnEditMode.setBackground(Color.darkGray);
    toolbar.add(btnEditMode);

    btnRunMode = new Button("Run");
    btnRunMode.addActionListener(this);
    btnRunMode.setFont(font);
    btnRunMode.setBounds(340,0,40,30);
    toolbar.add(btnRunMode);
    
    add(toolbar);
    
    //setSize(400,80);
    
    
    // Open up the default windows, status,toolbar,browser
    statusWindow = new StatusWindow(this,statusWindowFont);
    statusWindow.setLocation(0,430);	
    
    moduleToolbar = new ModuleToolbar(this,moduleToolbarFont);
    moduleToolbar.setLocation(405,0);
    
    browserModel=new BrowserModel(this,debugLevel);
    browser = new Browser(browserModel,this,browserFont,debugLevel);
    browser.setLocation(500,0);
    
    
    // Event listners
    WindowListener windowListener= new WindowListener();
    
    addWindowListener(windowListener);
    //addMouseListener(mouseListener);
    //addMouseMotionListener(mouseMotionListener);
    //addKeyListener(keyListener);
    addComponentListener(new ComponentListener());
    
    statusWindow.setMessage("Loading modules into toolbar..");
    setCursor(Cursor.getPredefinedCursor(Cursor.CROSSHAIR_CURSOR));
    
    /*
      if (packageRoot.equals("240")) {
      // This is so I can use the module manager to make
      // my schematics for 240. :-) It will only load the
      // various shaped modules.  This does not call Ted's
      // agent loader, so if the loader is ever broken,
      // we can use this option instead to test the gui.
      browser.displayAgentsInDirectory(packageRoot);
      browser.displayFoldersInDirectory(packageRoot);
      return;
      }
    */

    // generate a name for this modulemanager
    // we append the time of creation of this object so we can have a
    // (hopefully) unique name
    Date d = new Date();
    //	 name = "Module_Manager_"+d.toString().replace(' ','_');
    
    // We get the number of seconds since 1/1/1970 and ignore the first
    // three numbers, since they only change about every year or so
    name = "MM_"+String.valueOf(d.getTime()).substring(3);
    
    setTitle("Module Manager "+String.valueOf(d.getTime()).substring(3));
    
    models=new Hashtable();
    views=new Hashtable();
    pack();
    
    statusWindow.setMessage("Module Manager successfully loaded.");
    
  }
  
  /**
    *	Close the status window and set the checkbox in the view menu.
    */
  public void closeStatusWindow() {
    statusWindow.setVisible(false);
    chkStatusWindow.setState(false);
  }
  
  /**
    *	Close the module toolbar and set the checkbox in the view menu.
    */
  public void closeModuleToolbar() {
    moduleToolbar.setVisible(false);
    chkModuleToolbar.setState(false);
  }
  
  /**
    * Close the browser window and set the checkbox in the view menu.
    */
  public void closeBrowser() {
    browser.setVisible(false);
    chkBrowser.setState(false);
  }
  
  /**
    * Close the currently active view.  If the view hasn't been saved, warn
    * the user.
    */
  public void closeView(View v) {
    if (v==null) return;
    DlgSave dlgSave;
    if (v.getModel().isSaved()) {
      removeView(v);
    } else {
      dlgSave =new DlgSave(this,v.getModel().getSystemName(),false);
      dlgSave.show();
      if (dlgSave.isSave()) {
	saveSystem(v.getModel(),false);
	removeView(v);
      } else if(dlgSave.isDiscard()) {
	removeView(v);
      }
      
    }
    
  }
  
  /**
    * Get the currently selected tool
    *
    * @return the currently selected tool
    */
  public int getCurrentTool() {
    return currentTool;
  }
  
  /**
    * Get a reference to the module toolbar
    *
    * @return a reference to the module toolbar
    */
  public ModuleToolbar getModuleToolbar() {
    return moduleToolbar;
  }
  
  /**
    * Get a reference to the status window
    *
    * @return a reference to the status window
    */
  public StatusWindow getStatusWindow() {
    return statusWindow;
  }
  
  /**
    * Get a reference to the browser window
    *
    * @return a reference to the browser window
    */
  //public Browser getBrowser() {
  //	 return browser;
  //}
  
  public BrowserModel getBrowserModel() {
    return browserModel;
  }
  
  public ManagerModel getModel(String name) {
    return (ManagerModel)models.get(name);
  }
  
  public void setBrowser(Browser b) {
    browser=b;
  }

  public String getName() {
    return name;
  }
  
  /**
    * Check to see if the tool selected is the Select/Move tool
    *
    * @return whether or not the select/move tool is selected
    */
  public boolean isToolSelect() {
    return currentTool==SELECT;
  }
  
  /**
    * Check to see if the tool selected is the Resize tool
    *
    * @return whether or not the resize tool is selected
    */
  public boolean isToolResize() {
    return currentTool==RESIZE;
  }
  

  /**
    * Check to see if the tool selected is the Place tool
    *
    * @return whether or not the place tool is selected
    */  
  public boolean isToolPlace() {
    return currentTool==PLACE;
  }
  
  /**
    * Check to see if the tool selected is the Map tool
    *
    * @return whether or not the map tool is selected
    */
  public boolean isToolMap() {
    return currentTool==MAP;
  }
  
  /**
    * Print out the usage information
    */
  private static void printUsage() {
    System.out.println("Usage: java ModuleManager <options>");
    System.out.println("   -n                    Enable Network");
    System.out.println("   -H <hostname>         Specify host name");
    System.out.println("                          Default: localhost");
    System.out.println("   -p <port>             Specify port number");
    System.out.println("                          Default: 6500");
    System.out.println("   -N <hostname> <port>  Enable network using hostname and port");
    System.out.println("                          Overrides -h -p");
    System.out.println("   -r <path>             Package root path");
    System.out.println("   -v                    Print version");
    System.out.println("   -? -h                 Prints this");
    System.exit(0);
  }
  
  /**
    * See if all is enabled
    */
  public boolean isEnableAll() {
    return enableAll;
  }
  
  /**
    * Main run method.
    */
  public static void main(String args[]) {
    boolean upperCaseN=false;
    boolean lowerCaseN=false;
    
    boolean usingNetwork=false;
    String networkHost=NETWORK_HOST;
    int networkPort=NETWORK_PORT;
    String packageRoot=PACKAGE_ROOT;
    
    String s= System.getProperty("MODULE_MANAGER_FLAGS");
    if (s!=null) {
      if (s.indexOf("DEBUG")>=0) {
	isDebugging=true;
      } else isDebugging=false;
      // whether to enable all features (including untested ones)
      if (s.indexOf("ENABLEALL")>=0) {
	enableAll=true;
      } else enableAll=false;
    }
    s = System.getProperty("DEBUG");
    if (s!=null) {
      try {
	debugLevel = Integer.parseInt(s);
      } catch (Exception e) {
	debugLevel=0;
      }
    }
	 
		
    
    //*Argument processing...
    if (args.length >0) {
      int currArg = 0;
      while(currArg < args.length) {
	if(args[currArg].length() >= 2 && args[currArg].charAt(0) == '-') {
	  switch(args[currArg].charAt(1)) {
	  case 'r':
	    if((args.length - currArg -1) >= 1) {
	      packageRoot = args[currArg +1];
	      currArg+=2;
	      // -r rootpath -n -p 1010 -h hostname 
	      // -N hostname port -p 33 -h joe
	    } else {
	      System.err.println("No root path specified after -r");
	      System.exit(1);
	    }
	    break;
	  case 'p':
	    if (!upperCaseN) {
	      if ((args.length-currArg-1)>=1) {
		networkPort=Integer.parseInt(args[++currArg]);
		currArg++;
	      } else {
		System.err.println("No port specified after -p");
		System.exit(1);
	      }
	    } else {
	      System.err.println("Ignoring -p because of -N");
	    }
	    break;
	  case 'v':
	    System.out.println(PROGRAM_NAME+" version "+VERSION);
	    break;
	  case 'h':
	  case '?':
	    printUsage();
	    break;
	  case 'H':
	    if (!upperCaseN) {
	      if ((args.length-currArg-1)>=1) {
		networkHost=args[++currArg];
		currArg++;
	      } else {
		System.err.println("No hostname specified after -H");
		System.exit(1);
	      }
	    } else {
	      System.err.println("Ignoring -h because of -N");
	    } 
	    break;
	  case 'n':
	    lowerCaseN=true;
	    if (!upperCaseN) {
	      usingNetwork=true;
	      currArg++;
	    } else {
	      System.err.println("Ignoring -n because of -N");
	    } 
	    break;
	  case 'N':
	    upperCaseN=true;
	    usingNetwork=true;
	    if ((args.length-currArg-1)>=2) {
	      networkHost=args[++currArg];
	      networkPort=Integer.parseInt(args[++currArg]);
	      currArg++;
	    } else {
	      System.out.println("Need both hostname and port after -N");
	      System.exit(1);
	    }
	    if (lowerCaseN) {
	      System.err.println("-N overrides -h and -p");
	    }
	    break;
	    
	  default:
	    System.err.println("Ignoring unknown switch: "+args[currArg]);
	  }
	}
      }
    }
    //*Done processing args...
 
    //Communications commLink=null;
    
    if (usingNetwork) {
      System.out.println("Network enabled. Host: " + networkHost +
			 ":" + networkPort);
    }
    runningAlone=true;
    (new ModuleManager(networkHost,networkPort,packageRoot,
		       usingNetwork)).setVisible(true);
  }
  
  /**
    * Open the module toolbar and set the checkbox in the view menu
    */
  private void openModuleToolbar() {
    moduleToolbar.setVisible(true);
    chkModuleToolbar.setState(true);
  }
  
  /**
    * Open the browser and set the checkbox in the view menu
    */
  private void openBrowser() {
    browser.setVisible(true);
    chkBrowser.setState(true);
  }
  
  public void openView(ManagerModel m,String name) {
    View v = new View(m,this);
    // Currently we can only have one view of a system open at any
    // give time
    if (models.containsKey(m.getSystemName())) {
      statusWindow.setError(true);
      statusWindow.setMessage("That system is already open.");
      Enumeration e=models.keys();
      System.out.println("Models open: ");
      while (e.hasMoreElements()) {
	System.out.println((String)e.nextElement());
      }
      v.dispose();
    } else {
      v.setName(name);
      /* The extra info here offsets the windows in a tiled manner to 
	 provide access to underlying open windows. */
      v.setLocation(getLocation().x+(5*models.size()), 
		    ((getSize().height-5)+23*models.size()));
      v.show();
      m.addObserver(v);
      // add the view and model to the list we store
      models.put(m.getSystemName(),m);
      views.put(v.getName(),v);
      
      statusWindow.setError(false);
      statusWindow.setMessage(m.getSystemName()+" loaded.");
      miClose.setEnabled(true);
      miSaveAs.setEnabled(true);
      miCanvasSize.setEnabled(true);
      miProperties.setEnabled(true);
      miSelectAll.setEnabled(true);
      miCopy.setEnabled(true);
      
      activeView=v;
      MenuItem mi = new MenuItem(m.getSystemName());
      mi.addActionListener(this);
      menuSystems.add(mi);
      //statusWindow.setMessage("Loading config for: "+ystemName+"\n"+
      //		      displayAgentContainer(agentContainer,
      //					    new Point(0,0),false));
    }
  }

  private void openSystem(String filename) throws Exception {
    int i = filename.lastIndexOf(System.getProperty("file.separator"));
    String sysName=filename.substring(i+1);
    ManagerModel m=new ManagerModel(this,
				    sysName.substring(0,sysName.indexOf(".")),
				    debugLevel);
    // Open up the config file
    openView(m,m.getSystemName());
    // Tell the model the view is watching it
    m.openSystem(filename);
  }
  
  /**
    * Opens up a new system.  Shows a dialog for the user to select
    * a saved system and creats a new model and view for that system.
    */
  private void openSystem() {
    FileDialog fd=new FileDialog(this,"Open System",FileDialog.LOAD);
    fd.setDirectory(localPackageRoot);
    fd.show();
    
    if ((fd.getDirectory()==null) || (fd.getFile()==null)) return;	  	
    String filename=fd.getDirectory()+fd.getFile();
    
    try {
      openSystem(filename);
      int i = Integer.parseInt(properties.getProperty("numlastopened"));
      int max = Integer.parseInt(properties.getProperty("maxlastopened"));
      int p;
      for (p=1;p<=i;p++) {
	if (filename.equals(properties.getProperty("lastopened"+
						   String.valueOf(p)))) break;
      }
      if (p>i) {
	if (i==max) {
	  menuFile.remove(10);
	  for (int y=1;y<max;y++) {
	    properties.put("lastopened"+String.valueOf(y),
			   properties.getProperty("lastopened" + 
						  String.valueOf(y+1)));
	  }
	} else i++;
	properties.put("lastopened"+String.valueOf(i),filename);
	properties.put("numlastopened",String.valueOf(i));

	String fname;
	if (filename.length()>30) {
	  fname=filename.substring(0,5)+"..."+
	    filename.substring(filename.length()-30);
	}  else fname=filename;
	
	MenuItem mi = new MenuItem(fname);
	mi.addActionListener(this);
	mi.setActionCommand("open:"+filename);
	menuFile.insert(mi,menuFile.getItemCount()-2);
      }
      
    } catch (Exception e) {
      statusWindow.setError(true);
      statusWindow.setMessage("Error loading config. See stderr for details");
      System.err.println("Error:"+e);
      e.printStackTrace();
    } 
  }
  
  
  
  /**
    *	Called to quit the program and uninitialize everything etc etc..
    */
  public void quitProgram() {
    System.out.println("Quitting ModuleManager");
    Enumeration e = models.keys();
    ManagerModel m;
    boolean discardAll=false;
    boolean saveAll=false;
    boolean cancel=false;
    DlgSave dlgSave;
	 
    while ((e.hasMoreElements()) && (!discardAll) && (!cancel)) {
      m = (ManagerModel)models.get((String)e.nextElement());
      //System.out.println(m+" is "+m.isSaved());
      if (!m.isSaved()) {
	// if the user already said save all the systems, just save it
	if (saveAll) {
	  saveSystem(m,false);
	} else {
	  dlgSave=new DlgSave(this,m.getSystemName(),true);
	  dlgSave.show();
	  saveAll=dlgSave.isSaveAll();
	  cancel=dlgSave.isCancel();
	  discardAll=dlgSave.isDiscardAll();
	  if ((dlgSave.isSave()) || (saveAll)) {
	    saveSystem(m,false);
	  } 
	}
      }
    }

    if (cancel) return;

    try {
      FileOutputStream fos = new FileOutputStream(properties.getProperty("homedir")+"/.modulemanagerrc");
      properties.save(fos,"Module Manager Configuration");
      //System.out.println(properties.toString());
      fos.close();
    } catch (Exception el) {
    }
	 

    if (runningAlone) {
      dispose();
      System.exit(0);
    } else {
      // When the system is running through cyberrave we can't dispose
      // or systemexit otherwise all of cyberrave will go down.  We have
      // to simply minimize everything and remove all view/model pointers
      // so they will get garbage collected.
      minimizeAllWindows();

		
      views.clear();
      models.clear();

    }
	 
	 
  }

  public void minimizeAllWindows() {
    statusWindow.setVisible(false);
    moduleToolbar.setVisible(false);
    browser.setVisible(false);	
		
    //setVisible(false);

	  
    View v;
    Enumeration e = views.keys();
    while (e.hasMoreElements()) {
      v = (View)views.get((String)e.nextElement());
      v.setVisible(false);		
    }
    // we have to remove the view from the systems menu list
    browserModel.setChanged();
    browserModel.notifyObservers("setVisible(false)");
		
  }
  
  /**
    * Remove a view from the list.  If it's the last view, set the menu
    * options appropriately.
    *
    * @param v The view to remove
    */
  public void removeView(View v) {
    if (v!=null) {
      // remove the the view from the vector of views
      views.remove(v.getName());
      // tell the model this view is no longer listening
      v.getModel().deleteObserver(v);
		
      // If there are no more observers of this model, then we don't
      // need to keep it. removing it from the hashtable should remove
      // the last reference to the model, which should hopefully allow
      // the garbage collector to trash the model
      if (v.getModel().countObservers()==0) {
	models.remove(v.getModel().getSystemName());
      }
		
      // we have to remove the view from the systems menu list
      int numViews = menuSystems.getItemCount();
      MenuItem mi;
		
      for (int x=0;x<numViews;x++) {
	mi = menuSystems.getItem(x);
	if (mi.getLabel().equals(v.getModel().getSystemName())) {
	  menuSystems.remove(mi);
	  numViews--;
	}
      }
      // bye bye!
      v.dispose();
    }
    if (views.size()==0) {
      miClose.setEnabled(false);
      miSaveAs.setEnabled(false);
      miSave.setEnabled(false);
      miCanvasSize.setEnabled(false);
      miProperties.setEnabled(false);
      miSelectAll.setEnabled(false);
      miCopy.setEnabled(false);
		
    }
    activeView=null;
	 
  }
  
  /**
    * Save the system in the currently active view.
    *
    * @param promptUser Whether or not to ask the user for a filename
    */
  public void saveSystem(ManagerModel m,boolean promptUser) {
    //if (activeView==null) return;
    
    String systemName=m.getSystemName();
    String filename=m.getSystemFile();
    
    if ((systemName.startsWith("Untitled")) || (promptUser)) {
      FileDialog fd=new FileDialog(this,"Save System",FileDialog.SAVE);
      fd.setDirectory(localPackageRoot);
      fd.setFile(filename);
      // filename filters haven't been implemented properly in
      // unix java ports.. go figure..
      //	 fd.setFilenameFilter("*.cfg");	 
      fd.show();
      if (fd.getDirectory()==null) return;				
      //filename=fd.getDirectory()+fd.getFile();
      //filename = filename.replace('.','_');
      //filename = filename.replace(' ','_');
      systemName=fd.getFile();
      if (systemName.endsWith(".cfg")) {
	systemName=systemName.substring(0,systemName.indexOf(".cfg"));
      }
      systemName=systemName.replace(' ','_');
      systemName=systemName.replace('.','_');
      filename=new String(fd.getDirectory()+systemName+".cfg");	
      //currentSystemFileName=filename;
      //currentSystemName=systemName;
      //saveMenuItem.setEnabled(true);
      
      // only set the new name if the user is not saving a macro, systemname
      // will have .'s in it if it's a macro
      if (m.getSystemName().indexOf('.')<0) 
	m.setSystemName(systemName);
      m.setSystemFile(filename);
    }
    m.saveSystem();
  }
  
  
  /**
    * Set the  active view.
    *
    * @param v the new active view
    */
  public void setActiveView(View v) {
    activeView=v;
    // If the model doesn't have a name then the user can't save it
    // unless he does a "save as"
    if (v.getModel().getSystemName().startsWith("Untitled")) {
      miSave.setEnabled(false);
    } else {
      miSave.setEnabled(true);
    }
  }
  
  /**
    * Set the current tool.
    *
    * @param tool the new current tool
    */
  public void setCurrentTool(int tool) {
    if (tool>MAXTOOL) tool=MINTOOL;
    if (tool<MINTOOL) tool = MAXTOOL;
	 
    btnCurrentTool.setBackground(Color.lightGray);
    btnCurrentTool.setForeground(Color.black);

    currentTool=tool;

    switch (tool) {
    case PLACE:
      btnCurrentTool=btnPlace;
      break;
    case MAP:
      btnCurrentTool=btnMap;
      break;
    case RESIZE:
      btnCurrentTool=btnResize;
      break;
    case SELECT:
      btnCurrentTool=btnSelect;
    }
    btnCurrentTool.setBackground(Color.darkGray);
    btnCurrentTool.setForeground(Color.white); 
		  
  }
  

  /**
    *	Called whenever an action is performed. 
    */
  public void actionPerformed(ActionEvent evt) {
    Object source = evt.getSource();
    
    if ((source instanceof Button) && (source!=btnCurrentTool)) {
      
      if (source == btnResize) {
	currentTool=RESIZE;
      } else if (source == btnPlace) {
	currentTool=PLACE;
      } else if (source == btnMap) {
	currentTool=MAP;
      } else if (source == btnSelect) {
	currentTool=SELECT;
      } else if ((source==btnEditMode)) {
	if (!isEditor) {
	  isEditor=true;
	  btnRunMode.setBackground(Color.lightGray);
	  btnRunMode.setForeground(Color.black);
	  btnEditMode.setBackground(Color.darkGray);
	  btnEditMode.setForeground(Color.white);
	  //openModuleToolbar(); what's this thing for anyway??
	  openBrowser();

	  /* Okay, let's try to close all those annoying windows 
	     that are a part of the run time view... 
	     The method to this maddness is the thought that all windows 
	     we want to close are the "Running xxx" window, and the 
	     windows that display what's going on inside each host (which 
	     should have a '.' in the name).  I'm assuming then that all 
	     the rest of the available views will want to be opened upon
	     returning to Edit Mode. */
	  String vName, mName;
	  View v;
	  Enumeration e=views.keys();
	  while (e.hasMoreElements()) {
	    vName = (String)e.nextElement();
	    v = (View)views.get(vName);
	    mName = v.getModel().getSystemName();
	    if(mName.startsWith("Running") || (mName.indexOf(".") > -1)){
	      removeView(v); //removes the model too  :)
	      views.remove(name);
	    } else { //open that model
	      ManagerModel m = (ManagerModel)models.get(mName);
	      v.setVisible(true);//openView(m, m.getSystemName());
	      activeView = v;
	    }
	  }
	} /* if(!isEditor) */
	return;
      } else if ((source==btnRunMode)) {
	if (isEditor) {
	  isEditor=false;
	  btnEditMode.setBackground(Color.lightGray);
	  btnEditMode.setForeground(Color.black);
	  btnRunMode.setBackground(Color.darkGray);
	  btnRunMode.setForeground(Color.white);
	  closeBrowser();
	  closeModuleToolbar();
	  
	  /* hide, but don't remove the edit view */
	  //activeView.setVisible(false);
	  if(activeView == null){
	    statusWindow.setError(true);
	    statusWindow.setMessage("Error: No system loaded to run.");
	  }else{
	    /* hide, but don't remove the edit view */
	    activeView.setVisible(false);
	    //testFunc(activeView.getModel());
	    //startRunWindow(activeView.getModel());
	    RunView rv = new RunView(activeView.getModel(),this,
				     activeView,debugLevel);
	  }
	}
	return;
      }  /* else if ((source==btnRunMode)) */
      // Unhighlight the old button
      btnCurrentTool.setBackground(Color.lightGray);
      btnCurrentTool.setForeground(Color.black);
      btnCurrentTool=(Button)source;
      btnCurrentTool.setBackground(Color.darkGray);
      btnCurrentTool.setForeground(Color.white);		
      
    }
    
    if (evt.getSource() instanceof MenuItem) {
      String label=((MenuItem)evt.getSource()).getLabel();
      if (label.equals("Quit")) {
	quitProgram();
      } else if (label.equals("Open")) {
	// When opening a file, we create a new model with the
	// new file information.
	
	openSystem();
      } else if (label.equals("Save")) {
	if (activeView!=null) 
	  saveSystem(activeView.getModel(),false);
	
	//saveModuleConfig(currentSystemFileName);
      } else 	if (label.equals("Save As")) {
	if (activeView!=null) 
	  saveSystem(activeView.getModel(),true);
	//saveModuleConfig(null);
	//(new dlgModuleName(this,0,0,2)).show();
      } else if (label.equals("New")) {
	//initializeDataStorage();
	//setTitle("Module Manager");
	//saveMenuItem.setEnabled(false);
	//modulePanel.repaint();
	
	ManagerModel m = new ManagerModel(this,debugLevel);
	int x=0;
	
	while (models.get("Untitled_"+String.valueOf(x))!=null) x++;
	
	m.setSystemName("Untitled_"+String.valueOf(x));
	View v = new View(m,this);
	v.setName(m.getSystemName());
	v.setLocation(10*views.size(),80+(20*views.size()));
	m.addObserver(v);
	
	models.put(m.getSystemName(),m);
	views.put(v.getName(),v);
	v.show();
	activeView=v;
	miClose.setEnabled(true);
	miSaveAs.setEnabled(true);
	miCanvasSize.setEnabled(true);
	miProperties.setEnabled(true);
	miSelectAll.setEnabled(true);
	miCopy.setEnabled(true);
	
	MenuItem mi = new MenuItem(m.getSystemName());
	mi.addActionListener(this);
	menuSystems.add(mi);
	
      } else if (label.equals("Close")) {
	closeView(activeView);
      } else if (label.equals("Properties")) {
	if (activeView!=null) {
	  activeView.editSystemProperties();
	}
      } else if (label.equals("Copy")) {
	//System.out.println("activeView:"+activeView);
	clipBoard=activeView.getSelectedItems();
	miPaste.setEnabled(true);
      } else if (label.equals("Paste")) {
	if (clipBoard!=null) {
	  //System.out.println("activeView:"+activeView);
	  activeView.insertConfigAndLayout(clipBoard);
	}
      } else if(label.equals("Select All")) {
	activeView.selectAll();
      } else if (label.equals("Horizontally by top")) {
	if (activeView!=null) {
	  activeView.getModel().alignSelected(ManagerModel.HORIZTOP);
	}
      } else if (label.equals("Horizontally by bottom")) {
	if (activeView!=null) {
	  activeView.getModel().alignSelected(ManagerModel.HORIZBOT);
	}
      } else if (label.equals("Vertically by left")) {
	if (activeView!=null) {
	  activeView.getModel().alignSelected(ManagerModel.VERTLEFT);
	}
      } else if (label.equals("Vertically by right")) {
	if (activeView!=null) {
	  activeView.getModel().alignSelected(ManagerModel.VERTRIGHT);
	}
      } else if(label.equals("About")) {
	(new FrmAbout(PROGRAM_NAME,VERSION)).show();
      } else if (label.equals("Canvas Size")) {
	if (activeView==null) return;
	Dimension canvasSize=new Dimension(activeView.getCanvasSize());
	CanvasSizeDialog csd=new CanvasSizeDialog(this,"Canvas Size",
						  true,canvasSize);
	csd.show();
	activeView.setCanvasSize(canvasSize);
      } else if (label.equals("Print")) {
	
	PrintJob printJob = getToolkit().getPrintJob(this,"Print Canvas",
						     null);
	activeView.print(printJob);
      } 
      else if (label.equals("Launch")) {
	if (activeView!=null) {
	  // bring up a list of coordinators
	  // can either choose a coordinator to send to, or
	  // query the status on the coordinator. 

 	  SystemConfiguration system=activeView.getModel().getSystemConfiguration();
	  
	  String error=null; 
	  if (system!=null){ 
	    ConfigurationExecutor exec=new ConfigurationExecutor(networkHost,networkPort,name,system); 
	    ExecutionWatcher watch=new ExecutionWatcher();
	    exec.addObserver(watch);
	    try{ 
	      exec.executeConfiguration(); 

	    } 
	    catch(java.io.IOException ioe){ 
	      System.err.println("Caught IO failure executing system:"); 
	      ioe.printStackTrace(); 
	      error=new String(ioe.toString()); 
	    } 
	    catch(InvalidSystemConfigurationException isce){ 
	      System.err.println("Caught System Configuration error executing system:"); 
	      isce.printStackTrace(); 
	      error=new String(isce.toString()); 
	    }
	    catch(Exception e){
	      System.err.println("Caught exception executing system:");
	      e.printStackTrace();
	      error=new String(e.toString());
	    }
	  } 
	  else{ 
	    error=new String("Error launching system: null SystemConfiguration--try saving system");
	  }
	  if (error!=null){
	    statusWindow.setError(true);
	    statusWindow.setMessage(error);
	  }

	}
	
      }/*end if Launch*/
      else {
	MenuItem mi = (MenuItem)evt.getSource();
	if ((mi.getActionCommand()!=null) && 
	    (mi.getActionCommand().startsWith("open:"))) {
	  try {
	    openSystem(mi.getActionCommand().substring(5));
	  } catch (Exception e) {
	    statusWindow.setError(true);
	    statusWindow.setMessage("Error loading config. " + 
				    "See stderr for details");
	    System.err.println("Error:"+e);
	    e.printStackTrace();
	  }
	}
	
	// If the menuitem is a name of a view, bring that view to the 
	// front
	if (views.containsKey(label)) {
	  View view = (View)views.get(label);
	  view.show();
	}
      }
      
    } 
  }
  
  /**
    *	Called whenever an item state has changed.
    */
  public void itemStateChanged(ItemEvent e) {	 
      if (e.getSource() instanceof CheckboxMenuItem)
	{
	  CheckboxMenuItem chk=(CheckboxMenuItem)e.getSource();
	  boolean state= chk.getState();
	  if (chk==chkStatusWindow)
	    {
	      if (state) statusWindow.setVisible(true);
	      else
		statusWindow.setVisible(false);
	    }
	  if (chk==chkModuleToolbar)
	    {
	      if (state) moduleToolbar.setVisible(true);
	      else
		moduleToolbar.setVisible(false);
	    }
	  if (chk==chkBrowser)
	    {
	      if (state) browser.setVisible(true);
	      else
		browser.setVisible(false);
	    }	  
	}/* else if(e.getSource() instanceof List){//this no longer needed, superceded by ConfigurationExecutor
	  PhoneEntry pe=dirInfo.getAgentServerEntry(); 

	  Vector v=new Vector(); 
	  v.setSize(2); 
	  v.setElementAt(pe.getHost(),0); 
	  v.setElementAt(new Integer(pe.getPort()),1); 

	  Object[] o = e.getItemSelectable().getSelectedObjects(); 
	  pe = dirInfo.getCoordinatorEntry((String)o[0]); 

	  CoordinatorInfo ci = new CoordinatorInfo(pe.getHost(),pe.getPort(),activeView.getModel().getSystemConfiguration(),v,isApplet,networkHost); 
	  ci.start(); 
	}*/
    }


  class ComponentListener extends java.awt.event.ComponentAdapter {
    public void componentHidden(ComponentEvent event) {
      statusWindow.setVisible(false);
      moduleToolbar.setVisible(false);
      browser.setVisible(false);			
    }
    public void componentShown(ComponentEvent event) {
      statusWindow.setVisible(true);
      //moduleToolbar.setVisible(true);
      browser.setVisible(true);			
    }
  }

  /*******************************************/
  //This may need to go, but I'm adding it for help with saving Macros.  
  //See ManagerModel:1740 for more details...
  public Enumeration getModels() {
    return models.elements();
  }
  /*******************************************/
  
  // Event listener classes
  class WindowListener extends java.awt.event.WindowAdapter
  {
    public void windowClosing(java.awt.event.WindowEvent event)
      {
	System.out.println("Called window closing");
      	quitProgram();
      }

    /*
      public void windowClosed(WindowEvent event) {
      statusWindow.setVisible(false);
      moduleToolbar.setVisible(false);
      browser.setVisible(false);		
		
      }*/
	 
    /**
      *	Called whenever the window is minimized.  We want to minimize all the
      *	extra little windows that go with the program so they don't cluter
      *	the screen.
      */
    public void windowIconified(WindowEvent event)
      {
	System.out.println("Called window inconified");
	/*
	  statusWindow.setVisible(false);
	  moduleToolbar.setVisible(false);
	  browser.setVisible(false);
	*/
	minimizeAllWindows();
      }
    /*
      public void windowOpened(WindowEvent event) {
      if (chkStatusWindow.getState()) statusWindow.setVisible(true);
      if (chkModuleToolbar.getState()) moduleToolbar.setVisible(true);
      if (chkBrowser.getState()) browser.setVisible(true);	
      
      }*/
    
    /**
      *	Called whenever the window is de-minimized.  We want to restore the
      *	little windows if they were previously open.
      */
    public void windowDeiconified(WindowEvent event)
      {
	System.out.println("Called window Deiconified");
	if (chkStatusWindow.getState()) statusWindow.setVisible(true);
	if (chkModuleToolbar.getState()) moduleToolbar.setVisible(true);
	if (chkBrowser.getState()) browser.setVisible(true);
      }
  }
  
    class ExecutionWatcher extends Thread implements Observer{
	//this is used to watch the execution for errors and put the text of those errors in the status window
	public void update(Observable o, Object arg){
	    Exception e;
	    //  System.out.println(this +" received "+arg);
	    if (arg instanceof Exception){
		e=(Exception)arg;
		System.err.println("Exception during execution:"+e);

		statusWindow.setError(true);
		statusWindow.setMessage(e.toString());
	    }
	    if (arg instanceof String){
		if (ConfigurationExecutor.FINISHED.equals((String)arg)){
		    this.stop();
		}
	    }
	}
    }
  
  
}


/*
  class CFGFilter implements FilenameFilter {
  public CFGFilter() {
  super();
  
  };
  
  public boolean accept(File file,String name) {
  //System.out.println("accept");
  
  //System.out.println(name);
  
  return (name.endsWith(".cfg"));
  }
  }
*/
