/*
	Module Class
	By: Yatish Patel
*/

package adaptive.modulemanager;
import java.awt.*;
import java.util.*;
import java.awt.event.*;
import adaptive.core.*;
import java.lang.reflect.*;
import java.io.*;
import java.net.UnknownHostException;

/**
 * adaptive.modulemanager.Module class
 *
 * GUI representation of a PBModule.  The information about the PBModule is stored in agentDescription

 * Draws the module on the screen and stores all the input and output
 * mappings for the module. Default shape is a rectangle.
 *
 * If you want to draw a module of a different shape than simply
 * extend this class and override the following methods:
 * <ul>
 * <li> drawModule
 * <li> getInputLineLocation
 * <li> getOutputLineLocation
 * </ul>
 *
 * @author Yatish Patel <yatish@cmu.edu>
 * @version 1.0
 * $Log: Module.java,v $
 * Revision 1.9.2.12  2000/06/06 00:23:25  jrj
 * changed so it will synchronize the ports with the Agent when it loads
 * a file. also flipped around the appearance of the output ports.
 *
 * Revision 1.9.2.11  2000/06/05 05:55:14  jrj
 * more macro fixes.  Changed how and when ModuleManager sets up the ports on a macro so it is now possible to remove an output and keep it removed.  It will also always add unmapped inputs to the macro's ports.  Adding an output is there, but the capability is not put in ModuleManager yet.
 *
 * Revision 1.9.2.10  2000/05/20 19:11:33  jrj
 * fixed a couple more macro bugs.  Took preliminary steps toward making
 * fully recursive macros in MacroDescription.  Have not yet made changes
 * that would damage save file.
 *
 * Revision 1.9.2.9  2000/05/15 22:39:59  jrj
 * Made fixes to macros so that they work.  Changed the executing macro
 * so that it is recursive, can be manipulated as a whole, and its ports
 * can be asked for directly as if it were any other BasicPortAgent.
 * Made several other changes to simplify and assist these changes.
 * Finally purged the last remaining bit of things named clump.
 *
 */
public class Module extends Component implements java.io.Serializable 
{
  public boolean dispatching=false;
  public int Xoffset,Yoffset;

  protected AgentDescription agentDescription;
  
  /** Whether or not to paint this module */
  private boolean paintModule;	
  
  /** The class name that gets displayed on the module.  Might be
      truncated to fit within the module borders */
  protected String displayedClass;
  
  /** Parent frame for this module. */
  //public Object parent;
  protected Object owner;
  
  /** Agent this module represents */
    //note by jrj:  use of agent will be deprecated.  We want to make it so calls to agent are to agentDescription instead, but need to make some changes still to agentDescription before we can do so.
  protected Agent agent;
  private Class agentClass;
    
  /** All the connectLines into and out of this module */
  protected Vector lines;

  //  protected Mapping inputMapping;
  // output mappings are not needed. but i'll get rid of them later..
  // protected Mapping outputMapping;
  
  /** Store all the connectionlines that map to this modules inputs */
  protected Vector inputConnectLines;
  protected Vector outputConnectLines;
  
  /** The size of the module */
  protected Dimension moduleSize;

  //////////////////////////////////////////
  protected int inputLineSpacing=5;
  protected int outputLineSpacing=5;
  // a rectangle where the user can click to enlarge/shrink the
  // module 
  protected Shape resizeShape;
  
  // a rectangle where the user can click to move the module
  protected Shape moveShape;
  protected boolean resizing;
  protected int selectedPort;
  protected int selectedPortType;
  /** font used to display labels in the module */
  protected Font labelBoxFont;
  /** wheter or not the module is highlighted */
  protected boolean isSelected=false;
  
  /** whether or not the module has been initialized */
  protected boolean initialized=false;

  protected LineRectangleElement oldLre;

  // xpad and ypad are used when the user clicks and drags. we need
  // to know where in the box the user clicked when he started to
  // resize the module.
  protected int xpad;
  protected int ypad;

  /** Minimum value for the input line spacing */
  protected final int minInputLineSpacing = 5;
  /** Minimum value for the output line spacing */
  protected final int minOutputLineSpacing = 5;
  protected final int Xinset = 5;
  protected final int Yinset = 1;
  
  // The space inbetween two input or two output lines.
  protected final int labelBoxX = 10;
  protected final int labelBoxY = 10;

  /** The color of the lines/text in the module when it's selected */
  protected final Color selectedColor=new Color(0x33,0,0x99);
  /** The color of the lines/text when the module is not selected */
  protected final Color normalColor = Color.black;
  /** The background color when the module is not selected */
  protected Color backgroundColor;
  /** The background color when the module is selected */
  protected final Color backSelectedColor = new Color(0x66,0x66,0xff);

  private ModulePropertiesWindow modulePropertiesWindow=null;

  private boolean numberPressed=false;
  
  // private MouseMotionListener mouseMotionListener;
  private MouseListener mouseListener;

  // private int groupID;

  protected String name;  //added by Andrea b/c my agentDescription gets lost.
  int debugLevel;
  

  //*Constructor 1
  public Module(int debugLevel) {
    this(false,debugLevel);
  }

  //*Constructor 2
  public Module(boolean dontRunInit,int debugLevel) {
    // backgroundColor=new Color(0x66,0x99,0xFF);
    this.debugLevel=debugLevel;
    moduleSize=new Dimension(80,80);
    setSize(moduleSize);
    //setBackground(Color.white);
		
    labelBoxFont = new Font("Serif",Font.BOLD,10);	
    //setBackground(backgroundColor);
    setForeground(normalColor);
    // Event Listeners
    agentDescription=new AgentDescription();
    mouseListener = new MouseListener();
    backgroundColor=new Color(0x99,0xcc,0xff);
    if (!dontRunInit) init();
  }
  
  //*Constructor 3
  public Module(Agent a,int debugLevel)
    {
      this(false,debugLevel);
      agent=a;
      agentClass=a.getClass();
      owner=null;

      // agentDescription=new AgentDescription(a);

            agentDescription.setType(a.getClass().getName());
      //  System.out.println("clas="+moduleClass); 
      agentDescription.setName(a.getID());  

      agentDescription.setNumInputs(a.getNumInputs());  
      agentDescription.setNumOutputs(a.getNumOutputs());


      try {
        agentDescription.setRunLocation(
          adaptive.core.net.LocalHost.getFullLocalHostName());
      }
      catch (UnknownHostException uhe) {
        agentDescription.setRunLocation(agentDescription.RUN_LOCAL);
      }
      
      agentDescription.setSource(agentDescription.AGENT_SERVER);
      agentDescription.setModuleMode(agent.getModuleModeCapability());
      init();
    }


  /* Added by Andrea so that we can make Modules from AgentDescriptions */
  public Module(AgentDescription ad,int debugLevel){
    this(false,debugLevel);
    agentDescription = ad;
  }

  /**
   * copy constructor
   */
  public Module(Module m,int debugLevel) {	 	
    this(false,debugLevel);
    agentDescription=(AgentDescription)m.agentDescription.clone();
    //classType = agentDescription.getType();
    agent=m.getAgent();
    displayedClass=m.displayedClass;
    owner=m.owner;
    moduleSize=new Dimension(m.moduleSize);
    init();	    
    backgroundColor=m.backgroundColor;
	 
  }
   
  public Object clone() {
      //adaptive.modulemanager.Module m = new Module(this,debugLevel);
      return cloneSerializableObject(this);
    // inputMapping=(Mapping)m.inputMapping.clone();
    //outputMapping=(Mapping)m.outputMapping.clone();
    // should clone more here. but nothing really calls module clone.
    // so no need to worry. (yet)
    // I'm cloning the connect line vectors for the hell of it..
    // in reality this won't work because of pointers.. oh-well
    //inputConnectLines=(Vector)m.inputConnectLines.clone();
    //outputConnectLines=(Vector)m.outputConnectLines.clone();
    //lines=(Vector)lines.clone();

      //return m;
  }

  public Object clone(boolean s,String g) {
    return this.clone();
  }
  

  /**
    * Clones a serializeable object.  If the object isn't serializable then
    * return null.
    *
    * @param o The object to Clone
    * @return A clone of the object.
    */
  protected Object cloneSerializableObject(Object o) {
    try {
      ByteArrayOutputStream bout = new ByteArrayOutputStream();
      ObjectOutputStream out = new ObjectOutputStream(bout);
      out.writeObject(o);
      out.close();
      ByteArrayInputStream bin = new ByteArrayInputStream(bout.toByteArray());
      ObjectInputStream in = new ObjectInputStream(bin);
      Object ret = in.readObject();
      in.close();
      return ret;
    } catch (Exception e)
      {
        return null; 
      }
	 
  }
  

  /**
    * I need this because i can't init the vectors properly in the
    * constructor. 
    */
  protected void init()
    {
      // If a module that extends this class has some local things
      // to initialize before we calculate this information, let
      // them do it here.
      localInit();

      // these mappings will soon be unneeded.
      //		inputMapping=new Mapping(agentDescription.getNumInputs());
      //		outputMapping=new Mapping(agentDescription.getNumOutputs());

      lines = new Vector(agentDescription.getNumInputs()+agentDescription.getNumOutputs());
      addLineRectangles();
      // We need to store the port types for in inputs and outputs
      //inputPortTypes=new Vector(agentDescription.getNumInputs());
      //outputPortTypes=new Vector(numOutputs);
      // We need to store the port names for in inputs and outputs
      //inputPortNames=new Vector(agentDescription.getNumInputs());
      //outputPortNames=new Vector(agentDescription.getNumOutputs());
      int x;
      // If the module was created from an instantiated agent
      // then we can get the input/output port types and names from it
      /*
	if (agent!=null) {
	for (x=0;x<agentDescription.getNumInputs();x++) {
	inputPortTypes.addElement(agent.getInputPortType(x));
	inputPortNames.addElement(agent.getInputPortName(x));
	}
	for (x=0;x<agentDescription.getNumOutputs();x++) {
	outputPortTypes.addElement(agent.getOutputPortType(x));
	outputPortNames.addElement(agent.getOutputPortName(x));
	}
	}
      */
      inputConnectLines=new Vector(agentDescription.getNumInputs());
      inputConnectLines.setSize(agentDescription.getNumInputs());
      // I need to put elements in the vector so i can assign them by
      // index later on. I'm just putting random objects in.

      // Maybe i should store all the inputs and outputs together in a
      // hashtable indexed by IOPort?  Would that be more efficent? not
      // sure.. because there are times i would like all the connection liens
      // for only the inputs. (ie, when i draw the lines in the gui) 

      //for (x=0;x<agentDescription.getNumInputs();x++)
      //	inputConnectLines.addElement(new Object());
      outputConnectLines = new Vector(agentDescription.getNumOutputs());
      outputConnectLines.setSize(agentDescription.getNumOutputs());
      //for (x=0;x<agentDescription.getNumOutputs();x++)
      //	outputConnectLines.addElement(new Object());


      paintModule=true;
      selectedPort=-1;
      resizing=false;
      // Defaults the hostname to local
      //hostName="local";
      //agentLocation=0;
      initialized=true;
    }		  

  /**
    * Local class initialization called before running the
    * main classes init function.  By default this is empty
    * and it should be overidden in classes that extend
    * this one.
    */
  protected void localInit() {
    System.err.println("ERROR! ERROR! ERROR!");
    System.err.println("  You fool! You're trying to create a");
    System.err.println("  module that has no shape!");
    System.err.println("\n  Try creating a RectangleModule instead.");
  }


  /**
    * adds rectangles to the vector that outline the input/output lines
    */
  protected void addLineRectangles()
    {
      // sometimes this gets called when the module hasn't
      // been initialized. not sure why. so this is here to
      // prevent the null pointer
      //if (lines==null) return;
      int x;
      Point lineLoc;
      Point currentLoc=getLocation();
      LineRectangleElement lre1;
      lines.removeAllElements();
      for (x=0;x<agentDescription.getNumInputs();x++) {
	lineLoc=getInputLineLocation(x);
	lineLoc.x=lineLoc.x-currentLoc.x;
	lineLoc.y=lineLoc.y-currentLoc.y;
	lre1=new LineRectangleElement(new Rectangle(lineLoc.x,lineLoc.y-2,Xinset,5),x,IOPort.INPUT);
	lines.addElement(lre1);
	//System.out.println(lre1.toString());
	/*System.out.println("in rect= X="+lineLoc.x+
	  " Y= "+(lineLoc.y-2)+
	  "   X= "+(lineLoc.x+Xinset)+
	  " Y= "+(lineLoc.y+2));*/
      }
      for (x=0;x<agentDescription.getNumOutputs();x++) {
	lineLoc=getOutputLineLocation(x);
	lineLoc.x=lineLoc.x-currentLoc.x;
	lineLoc.y=lineLoc.y-currentLoc.y;
	lines.addElement(new LineRectangleElement(new Rectangle(lineLoc.x-Xinset,lineLoc.y-2,Xinset,5),x,IOPort.OUTPUT));
	//System.out.println("out rect= X="+(lineLoc.x-Xinset)+
	//	      " Y= "+(lineLoc.y-2)+
	//	      "   X= "+(lineLoc.x)+
	//	      " Y= "+(lineLoc.y+2));
	//
      }


    }

  /**
    * Returns this module in the form of an AgentDescription class.
    * If moduleList is not null, then it checks to make sure the module
    * connected to the input of this module is in the moduleList, if not
    * that input connection is not saved.

    * DEPRECATED 5/19/2000: use getAgentDescription instead

    * @see adaptive.core.AgentDescription	  
    * @return this module in the form of an AgentDescription class.
    */
  public AgentDescription toAgentDescription() {
    return agentDescription;
  }
  
  /**
    * Adjust all the connectlines connected to this module. This is
    * usually called after a module movement or a module resize
    */
  public void adjustConnectLines() {
    ConnectLine cl;
    Object o;
    Enumeration e = getInputConnectLines();
    while (e.hasMoreElements()) {
      o=e.nextElement();
      if (o instanceof ConnectLine) {
	((ConnectLine)o).adjustPointOf(this);
	// since the connectline moved, we have to
	// redo the hashtable that has the line segments
	// and update their values.
	// ((ConnectLine)o).createLineSegments();
      }
		
    }
    e = getOutputConnectLines();
    Enumeration l;
    ConnectLine k;
    while (e.hasMoreElements()) {
      o=e.nextElement();
      if (o instanceof Vector) {
	l = ((Vector)o).elements();
	while (l.hasMoreElements()) {
	  k = (ConnectLine)l.nextElement();
	  k.adjustPointOf(this);
	  //k.createLineSegments();
	       
	}
      }
    }
	 
  }

  /**
    * Draws the module on the screen.  This is needed so we can print
    * the whole screen. It will get called by ModuleManager with the
    * location of the module, so we know where to draw it.  
    *
    * By default the module doesn't have a location since it's canvas
    * is controlled by java and it's an object in my panel.
    */
  public void drawModule(Graphics g, Point offset) {
    drawModuleGraphics(g,offset,isSelected);
    drawText(g,offset,isSelected);
  }
  
  /**
    * Draws the text inside the Module. Should be overridden by
    * the class that extends this. 
    */
  protected void drawText(Graphics g, Point offset,boolean highlight) {
  }


  /**
    * Draw the outline and/or other graphics the module requires
    * to be displayed on the canvas. Should be overridden by the
    * class that extends this.
    *
    * @param g The graphics context of the module
    * @param offset The offset of the module on the screen
    * @param highlight whether or not to highlight to module
    */
  protected void drawModuleGraphics(Graphics g, Point offset,boolean highlight) 
    { 	  
    }

  /**
    * Draw a port line.
    *
    * @param g The graphics context for this module.
    * @param type The type of port input/output
    * @param port The port number
    * @param highlight Is the line highlighted
    */
  protected void drawPortLine(Graphics g, int type, int port, boolean highlight)
    {
      drawPortLine(g,new Point(0,0),type,port,highlight);
    }

  /**
    * Draw a port line.
    *
    * @param g The graphics context for this module.
    * @param o The offset of the module
    * @param type The type of port input/output
    * @param port The port number
    * @param highlight Is the line highlighted
    */
  protected void drawPortLine(Graphics g, Point o,int type, int port,boolean highlight)
    { 				
    }		  

  public AgentDescription getAgentDescription() {
    return agentDescription;
  }
  
  /**
    * Get the name of the module.
    *
    * @return Name of the module.
    */
  public String getName() {
    return agentDescription.getName();
  }

  /**
   * Get the run mode choices for this module
   *
   * @return run mode capability of this module
   */
  public int getRunModeCapability() {
    if (agent==null) return 0;
    return agent.getModuleModeCapability();
  }

  /**
   * Get the current run mode for this module
   *
   * @return current run mode of this module
   */
  public int getRunMode() {
    return agentDescription.getModuleMode();
  }

  /**
    * Get the type of class this module represents.
    *
    * @return the type of class of this module.
    */
  public String getClassType() {
    return agentDescription.getType();
  }

  /**
    * Get the agent this module represents.
    *
    * @return the agent this module represents.
    */
  public Agent getAgent() {
    return agent;
  }


  /**
    * Get the class of the agent this module represents.
    *
    * @return the class of the agent this module represents.
    */
  public Class getAgentClass() {
    return agentClass;
  }

  /**
    * Get the hostname where this agent will run.
    *
    * @return the hostname where this agent will run.
    */
  public String getHostName() {
    return agentDescription.getRunLocation();
  }



  /**
    * Get the location of this agent. Local/Network
    *
    * @return the location of this agent.
    */
    public int getAgentLocation() {//DEPRECATED 5/19/2000 jrj
    return agentDescription.getSource();
    }
    public int getAgentSource(){
	return agentDescription.getSource();
    }

  /**
    * Get the internal parameters for this agent.
    *
    * @return the internal parameters for this agent.
    */
  public Object getInternalParameters() {
    return agentDescription.getState();
  }

  /**
    * Get the line rectangle the specified coordinates are over. 
    * If the there is no line rectangle underneath the point,
    * a null is returned.
    *
    * @param x The X coordinate for the point.
    * @param y The Y coordinate for the point.
    * @return the LineRectangleElement the point is over, or null.
    */
  protected LineRectangleElement getRectangleIn(int x, int y)
    {
      Enumeration e=lines.elements();
      LineRectangleElement lre;
      //System.out.println("X="+x+" Y="+y);
      while (e.hasMoreElements()) {
	lre=(LineRectangleElement)e.nextElement();
	//System.out.println(lre.toString());
	if (lre.contains(x,y)) {
	  //System.out.println("Found: "+lre.getPort()+" "+lre.getType());
	  return lre;
	}
      }
      return null;		  
    }

  public Enumeration getInputConnectLines()
    {
      return inputConnectLines.elements();
    }

  /**
    * Return the connection lines that are mapped from the output.  
    * Each output can have multiple connection lines so we have to
    * return the elements from each vector.
    *
    * Now that i looked at this closer, it's actually easier to pass
    * the vector of vectors and let the person deal with it rather than
    * passing the connectlines themselves.  Maybe i'll implement it later..
    */
  public Enumeration getOutputConnectLines() {
    return outputConnectLines.elements();
  }

  /**
    * Get the input ConnectLine for the specified port
    *
    * @param port which port to get the ConnectLine of
    * @return the ConnectLine for the specified port
    */
  public ConnectLine getInputConnectLine(int port) {
    return (ConnectLine)inputConnectLines.elementAt(port);
  }
    /***********************************************************************
     * getOutputConnectLines
     * @input portNumber
     * @output Vector listing all ConnectLines that are connected to output portNumber
     *********/
    public Vector getOutputConnectLines(int port){ 
	Vector results=new Vector();
	Enumeration outputs=getOutputConnectLines();
	ConnectLine cl;
	IOPort ioport;
	while (outputs.hasMoreElements()){
	    cl=(ConnectLine)outputs.nextElement();
	    ioport=cl.getOutputPort();
	    if (ioport.getPort()==port){
		results.addElement(cl);
	    }
	}
	return results;
    }

  /**
    *	Get the mapping for the input port specified.
    *
    * @param port The port to get the mapping for
    * @return the mapping for the specified input port
    */
  public IOPort getInputMapping(int port) {
    //return inputMapping.getMap(i);	
    ConnectLine cl= (ConnectLine)inputConnectLines.elementAt(port);
    if (cl==null) return null; 
    else return cl.getOutputPort();
  }

  /**
    *	Get the location of where we should draw the line to connect the
    *	specified input port.  
    *
    *	@return a Point in the owners coordinate system.
    */
  protected Point getInputLineLocation(int inputNumber)
    {
      System.err.println("wrong");
      return null;
    }

  /**
    *	Get the location of where we should draw the line to conenct the
    *	specified output port.
    *
    *	@return a Point in the owner's coordinate system.
    */
  protected Point getOutputLineLocation(int outputNumber)
    {
      System.err.println("wrong");
      return null;
    }

  /**
    *	Get the location of where we should draw the line to connect the
    *	specified input or output port.
    *
    */
  public Point getLineLocation(String port)
    {
      int i = (new Integer(port.substring(1))).intValue();
      if (port.charAt(0)=='I')
	return getInputLineLocation(i);
      if (port.charAt(0)=='O')
	return getOutputLineLocation(i);
      return null;
    }

  public Point getLineLocation(int port, int type)
    {
      if (type==IOPort.INPUT) return getInputLineLocation(port);
      if (type==IOPort.OUTPUT) return getOutputLineLocation(port);
      return null;
    }
	

  /**
    *	Get the number of inputs.
    */
  public int getNumInputs()
    {
      return agentDescription.getNumInputs();
    }


  /**
    *	Get the number of outputs.
    */
  public int getNumOutputs()
    {
      return agentDescription.getNumOutputs();
    }

  /**
    * Return the port type for the specified input port
    *
    * @param port The input port number
    * @return the port type for input port <port>
    */
  public Class getInputPortType(int port) {
    Class retval=agentDescription.getInputPortType(port);
    if (retval==null) retval=agent.getInputPortType(port); //temp measure
    return retval;
  }

  /**
    * Return the port type for the specified output port
    *
    * @param port The output port number
    * @return the port type for output port <port>
    */
  public Class getOutputPortType(int port) {
    Class retval=agentDescription.getOutputPortType(port);
    if (retval==null) retval=agent.getOutputPortType(port); //temp measure
    return retval;
  }

  /**
    * Return the port type for the specfied port
    *
    * @param type whether it's an input or output
    * @param num the port number
    */
  public Class getPortType(int type, int num) {
    if (type==IOPort.INPUT) 
      return getInputPortType(num);
    else if(type == IOPort.OUTPUT) 
      return getOutputPortType(num);
    return null;
  }  

  /** ****************************************************************
    * <!
    * Method: getInputPortName()
    *
    * Description:
    * >
    * Returns the name of the specified input port
    * 
    * @param port the input port number
    * 
    * @return the port name for the input port
    *
    * @exception 
    ******************************************************************/
  public String getInputPortName(int port) {
    String s= agentDescription.getInputPortName(port);
    if (s==null)
      return agent.getInputPortName(port);
    else
      return s;
  }
  /***************************************************************************
   * syncPorts
   * synchronize the port types and names stored in the agent description 
   *      with those defined by the Agent
   * @param Agent the agent to synchronize with
   * @return true=finished, no errors
   *         false=agent wasn't the right type or other errors occurred 
   *********/
  boolean syncPorts(Agent a){
    boolean retval=false;
    retval= agentDescription.syncPorts(a);
    if (retval) {
      resizeModule(moduleSize.width,moduleSize.height);//if there is any change in the ports, resize so the ports get displayed correctly
      this.agent=a;
    }
    return retval;
  }
  boolean syncPorts(){
    boolean retval=false;
    if (agent!=null){
      retval=this.syncPorts(agent);
    }
    return retval;
  }
  boolean syncPortTypes(Agent a){
    boolean retval=false;
    retval= agentDescription.syncPortTypes(a);
    if (retval) {
      resizeModule(moduleSize.width,moduleSize.height);//if there is any change in the ports, resize so the ports get displayed correctly
      this.agent=a;
    }
    return retval;
  }
  boolean syncPortTypes(){
    boolean retval=false;
    if (this.agent!=null){
      retval=this.syncPortTypes(this.agent); 
    }
    return retval;
  }


  /** <!-- added by Andrea -->
    * See how many inputs are allocated
    */
  public int getNumInputLines(){
    //System.out.println(inputConnectLines.size()+" inputLines");
    return inputConnectLines.size();
  }
  /** <!-- added by Andrea -->
    * See how many outputs are allocated
    */
  public int getNumOutputLines(){
    //System.out.println(outputConnectLines.size()+" outputLines");
    return outputConnectLines.size();
  }

  /** ****************************************************************
    * <!
    * Method: getOutputPortName()
    *
    * Description:
    * >
    * Returns the name of the specified output port
    * 
    * @param port the output port number
    * 
    * @return the port name for the output port
    *
    * @exception 
    ******************************************************************/
  public String getOutputPortName(int port) {
    // First check to see if the agentdescription has a name for the port
    // if it does, return that, if not, get the port name from the agent
    // class
    String s = agentDescription.getOutputPortName(port);
    if (s==null) {
      return agent.getOutputPortName(port);
    }
    else
      return s;
  }

  /** ****************************************************************
    * <!
    * Method: getPortName()
    *
    * Description:
    * >
    * Returns the name of the specified port
    * 
    * @param port the port type/number
    * 
    * @return the port name for port
    *
    * @exception 
    ******************************************************************/
  public String getPortName(String port) {
    int i = (new Integer(port.substring(1))).intValue();
    if (port.charAt(0)=='I')
      return getInputPortName(i);
    if (port.charAt(0)=='O')
      return getOutputPortName(i);
    return null;
  }
  
  /** ****************************************************************
    *
    * <!
    * Method: getPortName()
    *
    * Description:
    * >
    * Returns the name of the specified port
    * 
    * @param type whether it's an input or output
    * @param num the port number
    * 
    * @return the port name for port
    *
    * @exception 
    ******************************************************************/
  public String getPortName(int type, int num) {
    if (type==IOPort.INPUT) 
      return getInputPortName(num);
    else if(type == IOPort.OUTPUT) 
      return getOutputPortName(num);
    return null;
  }
  
 

  public Dimension getPreferredSize() {
    return new Dimension(moduleSize.width,moduleSize.height);
  }
  public Dimension getMinimumSize() {
    return getPreferredSize();
  }
  
  /**
    * Returns true if the output port leads somewhere
    *
    * @param port the port to check
    * @return true if the port is mapped, false if not
    */
  public boolean isOutputMapped(int port) {
    Vector v = (Vector)outputConnectLines.elementAt(port);
    return ((v!=null) && (v.size()>0));
  }

  /**
    * Returns true if the input port is mapped
    * 
    * @param port to the port to check
    * @return true if the port is mapped, false if not
    */
  public boolean isInputMapped(int port) {
    return inputConnectLines.elementAt(port)!=null;
  }
  
  /**
    * Returns true if this module is selected, false if not
    * 
    * @return true if this module is selected, false if not
    */
  public boolean isSelected() {
    return isSelected;
  }

  /**
    * Is the point specified within the resize region.
    *
    * @param x X coordinate of the point to check.
    * @param y Y coordinate of the point to check.
    * @return true if the point is within the resize region otherwise false.
    */
  protected boolean insideResizeRegion(int x, int y) {
    if (resizeShape!=null) {		  
      return ((Rectangle)resizeShape).contains(x,y);
    }
		
    else
      return false;
  }

  /**
    * Is the point specified within the move region.
    *
    * @param x X coordinate of the point to check.
    * @param y Y coordinate of the point to check.
    * @return true if the point if within the move region otherwise false.
    */
  protected boolean insideMoveRegion(int x, int y) {
    if (moveShape!=null)
      return ((Rectangle)moveShape).contains(x,y);
    else
      return false;
  }

  /**
    * Is the point specified over one of the input or output
    * lines.
    *
    * @param x X coordinate of the point to check.
    * @param y Y coordinate of the point to check.
    * @return true if the point is over one of the input or output lines
    */	 
  protected boolean overInOutLine(int x, int y) {
    LineRectangleElement lre=getRectangleIn(x,y);
    Graphics g= getGraphics();
    if (lre!=null) {
      if (oldLre!=null) {
	g.setColor(backgroundColor);
	drawPortLine(g,oldLre.getType(),oldLre.getPort(),true); 
	g.setColor(normalColor);
	drawPortLine(g,oldLre.getType(),oldLre.getPort(),false);
	oldLre=null;
      }
      g.setColor(Color.blue);
      drawPortLine(g,lre.getType(),lre.getPort(),true);
      g.setColor(normalColor);
      oldLre=lre;
      selectedPort=lre.getPort();
      selectedPortType=lre.getType();	
      String type=getPortType(selectedPortType,selectedPort).getName();
      // now we want whatever is after the last .
      type = type.substring(type.lastIndexOf('.')+1);
      type = new String(String.valueOf(selectedPort)+" = "+type);
      String p_name = getPortName(selectedPortType,selectedPort);
      ((ManagerModel)owner).getModuleManager().getStatusWindow().setStatusInfo(this);
      ((ManagerModel)owner).getModuleManager().getStatusWindow().setPortType(type);
      ((ManagerModel)owner).getModuleManager().getStatusWindow().setPortName(p_name);
    } else if (oldLre!=null) {
      g.setColor(backgroundColor);
      drawPortLine(g,oldLre.getType(),oldLre.getPort(),true); 
      g.setColor(normalColor);
      drawPortLine(g,oldLre.getType(),oldLre.getPort(),false);
      oldLre=null;
      // When a number key gets pressed, we assume the user wants to
      // go into a slightly advanced mode.  See the keyPressed method
      if (!numberPressed) selectedPort=-1;
      selectedPortType=-1;
    }
    return (oldLre!=null);
  }

  /**
    * Paint method that draws the module on the canvas. All it does
    * is call drawModule.
    *
    * @param g The graphics context of the module.
    */
  public void paint(Graphics g)
    {
      //	 System.out.println("paint");
      if (paintModule)
	drawModule(g,new Point(0,0));
    }

  public void removeOutputMapping(ConnectLine cl)
    {
      int port=cl.getOutputPort().getPort();
      Vector v = (Vector)outputConnectLines.elementAt(port);
      v.removeElement(cl);
      if (v.size()==0) 
	outputConnectLines.setElementAt(null,port);
	 
      //outputConnectLines.setElementAt(null,cl.getOutputPort().getPort());
      //outputMapping.removeMap(cl.getOutputPort().getPort());
    }
	 
  public void removeInputMapping(ConnectLine cl)
    {
      inputConnectLines.setElementAt(null,cl.getInputPort().getPort());
      //inputMapping.removeMap(cl.getInputPort().getPort());
      agentDescription.setInputMapping(null,cl.getInputPort().getPort());
    }

  /**
    * Resize the module to the spcified size.
    *
    * @param x X coordinate of the point to check.
    * @param y Y coordinate of the point to check. 
    */
  public void  resizeModule(int x, int y) {
  }


  public void setAgentDescription(AgentDescription ad) {
    agentDescription=ad;
  }

  /**
    * Change the selected attribute for this module.
    *
    */
  public void setSelected(boolean h) 
    {
      isSelected=h;
      repaint();
    }
  


  /**
    *	Set the mapping for the port specified.
    */
  
  public void setMapping(IOPort source,IOPort destination){
	 
    if (source.isInput()) {
      //inputMapping.setMap(source.getPort(),destination);
      agentDescription.setInputMapping
	(new InputLocation(destination.getModule().getName(),
			   destination.getPort(),source.getPort()),
	 source.getPort());
    } 
    //else if (source.isOutput()) {
    //outputMapping.setMap(source.getPort(),destination);
    //}
  }
  


  /**
    * Set the location of this agent. Local/Network
    *
    * @param location the new location of this agent.
    */
  public void setAgentLocation(int location) {
    agentDescription.setSource(location);
  }

  /**
    * Set the agent class of this module.
    *
    * @param c the new agent class.
    */
  public void setAgentClass(Class c) {
    agentClass=c;
  }

  /**
    * Set the hostname where this agent will run.
    *
    * @param hn the new hostname for this agent.
    */
  public void setHostName(String hn) {
    agentDescription.setRunLocation(hn);
  }

  /**
    * Sets the internal parameters of this agent.  This is
    * usually saved after the agents have been running for
    * awhile.  Basically only the agents themselves will 
    * change their internal parameters. (i think)
    *
    * @param o The new internal parameters for the agent.
    */
  public void setInternalParameters(Object o) {
    agentDescription.setState(o);
  }

  /**
    * Set the agent this module represents.
    *
    * @param a the new agent for this module.
    */
  public void setAgent(Agent a) {
    agent=a;
    agentDescription.setModuleMode(agent.getModuleModeCapability());    
  }

  /** 
    * Turn off painting of this module
    */
  public void setPaint(boolean p) {
    paintModule=p;
  }

  /**
    * Set the mapping for the input port specified.
    */
  /*
    public void setInputMapping(int i,Module m,int port)
    {
    setMapping(new IOPort(null,i,IOPort.INPUT),
    new IOPort(m,port,IOPort.OUTPUT));
    }
  */

  public void setInputMapping(ConnectLine cl)
    {
      // this puts information in the mapping so we know where
      // a certian input is mapped to.  We are passing the line that
      // connects the two ports together.  the line knows which side
      // is the input, and which side is the output.

      // we are going to insert the information of the output port
      // into the mapping for this module's input port.


      //in order to know which input port we are specifiying the info 
      // for, we need to get it from the line.

      // input is the source (on this module), output is the destination
      setMapping(cl.getInputPort(),cl.getOutputPort());

      // add the line to the vector of connectlines
      inputConnectLines.setElementAt(cl,cl.getInputPort().getPort());
    }

  public void setOutputMapping(ConnectLine cl)
    {
      // this is the same as setInputMapping, so see that for more
      // info.
      //		IOPort p = cl.getInputPort();
      setMapping(cl.getOutputPort(),cl.getInputPort());

      if (outputConnectLines.elementAt(cl.getOutputPort().getPort()) 
	  instanceof Vector) {
	Vector v = (Vector)outputConnectLines.elementAt(cl.getOutputPort().getPort());
	v.addElement(cl);
      } else {
	Vector v = new Vector(3,3);
	v.addElement(cl);
	outputConnectLines.setElementAt(v,cl.getOutputPort().getPort());
      }
    }

  public void setOffset(int x,int y) {
    Xoffset=x;
    Yoffset=y;
  }
  
  /**
    * Sets the owner of the module
    */
  public void setOwner(Object p)
    {
      owner=p;
    }

  /**
    *	Sets the name of the module.
    */
  public void setName(String s)
    {
      name = s;
      agentDescription.setName(s);
    }

  /**
   * Sets the current run mode of this module
   *
   * @param rm the current run mode
   */
  public void setRunMode(int rm) {
    agentDescription.setModuleMode(rm);
  }
  
  public void setMouseListener(boolean setOn) {
    if (setOn) {
      //System.out.println("adding listener");
		
      addMouseListener(mouseListener);
    } else {
      removeMouseListener(mouseListener);
    }	 
  }
  
  public void setInputPortName(String n,int p){
    agentDescription.setInputPortName(n,p);
  }

  public void setOutputPortName(String n,int p) {
    agentDescription.setOutputPortName(n,p);
  }

  public void showProperties() {
    // We pass the location of the click in the module plus the
    // location of the module on the screen so we can show the
    // properties window exactly where the user clicked.
    Point l=getLocationOnScreen();
    if (modulePropertiesWindow==null) {
      modulePropertiesWindow=new ModulePropertiesWindow(this,
							(ManagerModel)owner,
							l.x,l.y);
    }
    modulePropertiesWindow.show();
  }
  
  
  public void update(Graphics g) {
    //System.out.println("update");
    if (paintModule)
      drawModule(g,new Point(0,0));
  }

  
  public void mouseClicked(java.awt.event.MouseEvent event)
    {

      Point p = new Point(event.getX(),event.getY());
      int modifiers=event.getModifiers();
      // The component the mouse was clicked in. (current module)
      Component c = event.getComponent();
      // The location on the screen of the module.  
      Point s = c.getLocationOnScreen();
    
      // If right clicking, show module proporties no matter what tool is
      // being used
	 
      // check what tool is being used 
      if (((ManagerModel)owner).getModuleManager().isToolSelect()) {
	// using the selection tool
	isSelected=!isSelected;
	repaint();
	if (owner instanceof BrowserModel)
	  ((BrowserModel)owner).selectAgent((Module)c);
	else if (owner instanceof ManagerModel) {
	  ((ManagerModel)owner).selectModule((Module)c,(modifiers & InputEvent.CTRL_MASK)!=0);			  
	}
      } else if (((ManagerModel)owner).getModuleManager().isToolMap()) {
	// The user is using the map tool
	if (owner instanceof ManagerModel) {
	  if (selectedPort>=0) {
				//System.out.println("click draw line");
	    if (selectedPortType==-1) {
	      // the user is using a shortcut by holding down a number key.
	      // we need to find out if he means an input or output. We
	      // assume that if the click is in the left hand side of the 
	      // module it's an input, if it's in the right, it's an output
	      if (event.getX()<=(moduleSize.width/2)) 
		selectedPortType=IOPort.INPUT;
	      else
		selectedPortType=IOPort.OUTPUT;
	    }
	  
	    ((ManagerModel)owner).drawLine((Module)c,selectedPort,selectedPortType);
	  }
	
	}
      } 
    }
  
  public void mousePressed(java.awt.event.MouseEvent event){
    // The event being sent in is in the owners coordinate 
    // system. we need to convert it to our coordinate system.	 
    //Point l=getLocation();	 
    //Point p=new Point(event.getX()-l.x,event.getY()-l.y);
    Point p = new Point(event.getX(),event.getY());
    //owner.dispatchEvent(event);
    // System.out.println(event.getX()+","+event.getY());
    if ((!resizing) && (insideResizeRegion(event.getX(),event.getY()))) {
      // System.out.println("inside!");
      xpad=moduleSize.width-p.x;
      ypad=moduleSize.height-p.y;
      resizing=true;
      ((ManagerModel)owner).setModuleToResize((Module)event.getComponent(),
					      xpad,ypad);
    } else if (owner instanceof ManagerModel) {
      Xoffset=p.x;
      Yoffset=p.y;
      //Point g=((Module)event.getSource()).getLocation();
      ((ManagerModel)owner).setModuleToDrag((Module)event.getComponent(),
					    p.x,p.y);
      
    }
  }
  
  public void mouseReleased(java.awt.event.MouseEvent event)
    {
      //System.out.println("M release");
      
      resizing=false;
      if (owner instanceof ManagerModel) {
	((ManagerModel)owner).setModuleToDrag(null,0,0);
	((ManagerModel)owner).setModuleToResize(null,0,0);
	//enableMouseMotionListener();
      }
      
      
    }

  public void mouseDragged(java.awt.event.MouseEvent event)
    {
      Point p = getLocation();
      
      if (((ManagerModel)owner).getModuleManager().isToolResize()) {
	// If the user clicks in the resize rectangle then he's
	// trying to resize it.
	if (insideResizeRegion(event.getX(),event.getY())) {
	  // xpad and ypad get recorded in the mousePressed event. 
	  // xpad/ypad+ the location of the mouse is the new size
	  // of the module.
	  resizeModule(event.getX()+xpad,event.getY()+ypad); 
	}
      }
	 
    }

  public void setCursor(Cursor c) {
    //((ModuleManager)owner).setPanelCursor(c);
    getParent().setCursor(c);
  }
  
		
  public void mouseMoved(java.awt.event.MouseEvent event) {
    // The event being sent in is in the owners coordinate 
    // system. we need to convert it to our coordinate system.	 
    //Point l=getLocation();	 
    //Point p=new Point(event.getX()-l.x,event.getY()-l.y);
    Point p = new Point(event.getX(),event.getY());
    if (owner instanceof ManagerModel) {		  
      ModuleManager moduleManager=((ManagerModel)owner).getModuleManager();
      moduleManager.getStatusWindow().setStatusInfo
	((Module)event.getComponent());
      if ((insideResizeRegion(p.x,p.y)) && (moduleManager.isToolResize()))
	setCursor(Cursor.getPredefinedCursor(Cursor.SE_RESIZE_CURSOR));
      else if ((insideMoveRegion(p.x,p.y)) && (moduleManager.isToolSelect()))
	setCursor(Cursor.getPredefinedCursor(Cursor.MOVE_CURSOR));
      else if ((moduleManager.isToolMap()) && (overInOutLine(p.x,p.y)))
	setCursor(Cursor.getPredefinedCursor(Cursor.CROSSHAIR_CURSOR));
      else
	setCursor(Cursor.getPredefinedCursor(Cursor.DEFAULT_CURSOR));
    }
  }

  public void keyPressed(KeyEvent event) {
    
    // Shortcut, if the user is holding down a number key while over a
    // module, then we assume he means to map to that number input
    if ((event.getKeyCode()>=0x30) && (event.getKeyCode()<=0x39)) {
      selectedPort=event.getKeyCode()-0x30;
      selectedPortType=-1; // we don't know the port type yet	
      //System.out.println("port="+selectedPort);
      numberPressed=true;		
    }
  }
  

  class FocusListener extends java.awt.event.FocusAdapter {
    public void focusGained(java.awt.event.FocusEvent event) {
      // If the owner is of type ModuleManager, than we
      // want to tell the status window the user has his mouse
      // on this module
      /*
	if (owner instanceof ModuleManager) {
	((ModuleManager)owner).statusWindow.setModuleName(moduleName);
	((ModuleManager)owner).statusWindow.setModuleClass(moduleClass.substring(moduleClass.lastIndexOf(".")+1));
	}
      */
    }
    public void focusLost(java.awt.event.FocusEvent event)
      {
	/*
	  if (owner instanceof ModuleManager) {
	  ((ModuleManager)owner).statusWindow.setModuleName("");
	  ((ModuleManager)owner).statusWindow.setModuleClass("");
	  }
	*/
	oldLre=null;
	repaint();
      }
  }



  public void componentResized(java.awt.event.ComponentEvent event)
    {
      if (event.getSource()==this) {
      }
    }

  class MouseListener extends MouseAdapter {
    public void mouseClicked(java.awt.event.MouseEvent event) {
      //System.out.println("real listener!"); 
      ((Module)event.getComponent()).mouseClicked(event);		
    }
  }

  /*
    class KeyListener extends KeyAdapter {
	 
    public void keyPressed(KeyEvent event) {
		
    // Shortcut, if the user is holding down a number key while over a
    // module, then we assume he means to map to that number input
    if ((event.getKeyCode()>=0x30) && (event.getKeyCode()<=0x39)) {
    selectedPort=event.getKeyCode()-0x30;
    selectedPortType=-1; // we don't know the port type yet	
    System.out.println("port="+selectedPort);
				
    } 
    owner.dispatchEvent(event);
    }
    }
  */
}
