
package adaptive.modulemanager;

import java.util.*;
import adaptive.core.*;
import java.awt.*;
import java.io.*;

/**
 * The Model portion of the ModuleManager.  Contains all the data about
 * the modules and connections.
 *
 * This is my feeble attempt at implementing MVC.
 *
 * @author Yatish Patel
 * @version 1.0
 * $Log: ManagerModel.java,v $
 * Revision 1.13.2.15  2000/06/06 00:23:24  jrj
 * changed so it will synchronize the ports with the Agent when it loads
 * a file. also flipped around the appearance of the output ports.
 *
 * Revision 1.13.2.14  2000/06/05 05:54:18  jrj
 * more macro fixes.  Changed how and when ModuleManager sets up the ports on a macro so it is now possible to remove an output and keep it removed.  It will also always add unmapped inputs to the macro's ports.  Adding an output is there, but the capability is not put in ModuleManager yet.
 *
 * Revision 1.13.2.13  2000/06/03 19:27:49  jrj
 * Made macros fit MacroDescription a little better.  Fixed an error in mapping inputs on macros.  Added a few diagnostic messages during the system verification step.
 *
 * Revision 1.13.2.12  2000/05/20 19:11:32  jrj
 * fixed a couple more macro bugs.  Took preliminary steps toward making
 * fully recursive macros in MacroDescription.  Have not yet made changes
 * that would damage save file.
 *
 * Revision 1.13.2.11  2000/05/15 22:39:58  jrj
 * Made fixes to macros so that they work.  Changed the executing macro
 * so that it is recursive, can be manipulated as a whole, and its ports
 * can be asked for directly as if it were any other BasicPortAgent.
 * Made several other changes to simplify and assist these changes.
 * Finally purged the last remaining bit of things named clump.
 *
 */
public class ManagerModel extends Observable {

  /** Align a horizontal group of modules by the top of the adaptive.modulemanager.Module */
  public static final byte HORIZTOP = 0;
  /** Align a horizontal group of modules by the bottom of the adaptive.modulemanager.Module */
  public static final byte HORIZBOT = 1;
  /** Align a vertical group of modules by the left of the module */
  public static final byte VERTLEFT = 2;
  /** Align a vertical group of modules by the right of the module */
  public static final byte VERTRIGHT= 3;

  /** Color of a selected line */
  private Color selectedLineColor=new Color(0xcc,0x00,0x00  ); //new Color(0x33

  /** List of all modules in the canvas */
  private Hashtable modules;
  /** List of all the selected modules */
  private Hashtable selectedModules;
  /** List of all the lines that connect modules */
  private Vector lines;
  
  /** Name of the current system */
  private String systemName;

  /** Drectory and file of the current system */
  private String systemFile;
  

  /** Whether or not we are currently drawing a line */
  private boolean drawingLine=false;

  /** The currentline we are drawing */
  private ConnectLine currentLine;

  /** Is a module currently selected */
  private boolean moduleSelected=false;

  /** Are we dragging a segment of a line around */
  private boolean draggingLine;

  public SelectRectangle selectRectangle;

  /** The currentl line segment that's highlighted */
  private LineSegment highlightedSegment;
	 
  /** adaptive.modulemanager.Module that is currently being dragged around */
  private adaptive.modulemanager.Module moduleToDrag;

  /** adaptive.modulemanager.Module that is currently being resized */
  private adaptive.modulemanager.Module moduleToResize;

  /** Xoffset of where the user clicked when he started to drag a module */
  private int Xoffset=0;

  /** Yoffset of where the user clicked when he started to drag a module */
  private int Yoffset=0;
	 
  private IOPort portSelected;

  /** Default name for the module so the user won't have to name it */
  private int defaultName;



  /** Currently selected agent from the toolbar */
  private adaptive.modulemanager.Module selectedAgent;

  /** The highest layout version supported by this program */
  private final int MAX_LAYOUT_VERSION = 1;

  /** testing */
  //private StatusWindow statusWindow;
  
  //private Browser browser;
  
  private ModuleManager moduleManager;
  
  private boolean saved;

  /** author of the system */
  private String author="";
  /** system version */
  private String version="";
  /** description of system */
  private String description="";
  /** Date of creation */
  private String date="";
  /** Things that were changed since last revision */
  private String lastRevision="";

  private ConfigAndLayout systemConfiguration;

  /** Debug level indicates how much info to print during operation */
  private int debugLevel=0;
  
  /**
    * Constructs an empty model.
    *
    * @param mm A reference to the main ModuleManager
    */
  public ManagerModel(ModuleManager mm,int debugLevel) {
    modules=new Hashtable(50);
    lines=new Vector(40,10);
    selectedModules=new Hashtable(10);
    systemName="Untitled";
    systemFile="Untitled.cfg";
    //statusWindow=new StatusWindow(null);
    moduleManager=mm;
    saved=true;
    systemConfiguration=new ConfigAndLayout();
	 this.debugLevel=debugLevel;
  }

  /**
    * Constructs a model with the data contained in the ConfigAndLayout object.
    *
    * @param mm A reference to the main ModuleManager
    * @param cal The configuration this model should represent
    */
  public ManagerModel(ModuleManager mm, ConfigAndLayout cal,int debugLevel) {
    this(mm,cal,"",debugLevel);
    //	 agentContainer=ac;
    //openSystemac);
  }
  
  /**
    * Contructs an empty model with a given name.
    */
  public ManagerModel(ModuleManager mm, String name,int debugLevel) {
    this(mm,debugLevel);
    systemName=name;
  }
  
  /**
    * Contructs a model with the data contained in the ConfigAndLayout
    *
    * @param mm A refrence to the main ModuleManager
    * @param cal The configuration this model should represent
    * @param p I don't know why this is here
    */
  public ManagerModel(ModuleManager mm, ConfigAndLayout cal,String p,
							 int debugLevel) {
    this(mm,debugLevel);
    systemConfiguration=cal;
    openSystem(cal);
  }  

  /**
    * Add a module to the list of modules that are selected. The module is
    * only added if it isn't already in the list.
    *
    * @param m The module to add
    */
  public void addSelectedModule(Module m) {
    if (!selectedModules.containsKey(m.getName())) {
      if (!m.isSelected()) m.setSelected(true);
      selectedModules.put(m.getName(),m);
      highlightLines(m,true);			  
    }	 
  }

  /**
    * Aligns all selected models according to the method. 
    * <p>
    * The types of alignment available are: 
    * <ul>
    * <li>HORIZTOP: Aligns a horizontal group of modules by the top.
    * <li>HORIZBOT: Aligns a horizontal group of modules by the bottom.
    * <li>VERTLEFT: Aligns a vertical group of modules by the left.
    * <li>VERTRIGHT: Aligns a vertical group of modules by the right.
    * </ul>
    * 
    * @param method How to align the modules.
    */
  public void alignSelected(byte method) {
    if (selectedModules.size()>0) {
      Enumeration e = selectedModules.keys();
      adaptive.modulemanager.Module m;
      m = (Module)selectedModules.get((String)e.nextElement());
      Dimension d = m.getSize();
      Point p = m.getLocation();
      int val=0;
      switch (method) {
      case HORIZTOP: val = p.y; break;
      case HORIZBOT: val = p.y+d.height; break;
      case VERTLEFT: val = p.x; break;
      case VERTRIGHT: val = p.x+d.width; break;
      }


      while (e.hasMoreElements()) {
		  m = (Module)selectedModules.get((String)e.nextElement());
		  d=m.getSize();
		  p=m.getLocation();
		  switch (method) {
		  case HORIZTOP: m.setLocation(p.x,val); break;
		  case HORIZBOT: m.setLocation(p.x,val-d.height); break;
		  case VERTLEFT: m.setLocation(val,p.y); break;
		  case VERTRIGHT: m.setLocation(val-d.width,p.y); break;
		  }
		  m.adjustConnectLines();
      }
      saved=false;
      setChanged();
      notifyObservers();
    }
	
	 
  }

  /** 
    * Clones this Model 
    *
    * @param prefix A prefix to put on all the names. (isn't used anymore)
    * @return A clone of this model
    */
  public Object clone(String prefix) {
    // this might not be the most efficent way of doing this, but for now it's the easiest way.
    ManagerModel mm= new ManagerModel(moduleManager,toSystemConfiguration(),prefix,debugLevel);
    return mm;
  }

  /**
    * Takes a list of modules and creates an agent container out of it.
    * This is used to save files and to copy/paste.  If a module in the
    * moduleList has a connect to a module that is not contained in the
    * list, the connect is removed and not saved.  This way there are
    * no funky errors when the system gets loaded again.  The only time
    * this should happen is when an agentContainer is created out off
    * a copy/cut command.
    *
    * This way it's easy to create
    * a ConfigAndLayout out of selected modules and then relayout them out
    * when the paste is done.
    * 
    * @param systemName The name of the system
    * @param moduleList A Hashtable of all the modules
    *
    * @return A ConfigAndLayout that represents the modules in the list
    */
  private ConfigAndLayout createConfigAndLayout (String systemName,Hashtable moduleList) {
    try {
      Enumeration e = moduleList.keys(); //modules.elements();
      adaptive.modulemanager.Module m;
      IOPort io;
      Point p ;
      AgentDescription ad,ad1;
      Enumeration eLines;
      Object o;
      SystemConfiguration sc=new SystemConfiguration();
      sc.setType(systemName);
      sc.setVersion(version);
      sc.setLastRevision(lastRevision);
      sc.setAuthor(author);
      sc.setDescription(description);
      sc.setDate(date);
      sc.setPortsPrepared(systemConfiguration.systemConfiguration.haveExternalPortsBeenPrepared());
      
      //This loop counts how many wires are laid out for all modules??
      int layoutCount=0;
      while (e.hasMoreElements()) {
		  layoutCount++;
		  layoutCount=layoutCount+((Module)moduleList.get(e.nextElement())).getNumInputs();
      }

      //System.out.println("layoutcount="+layoutCount);
      
      Hashtable allLayouts = new Hashtable();      
      String[] guiLayout = new String[layoutCount+1];
      Dimension size;
      layoutCount=1;
      e = moduleList.keys(); // elements();  modules.elements();

      // Loop through all the modules on the canvas
      guiLayout[0]="LayoutFileVersion "+String.valueOf(MAX_LAYOUT_VERSION);
      while (e.hasMoreElements()) {
	m=(Module)moduleList.get(e.nextElement());
	ad= (AgentDescription)m.getAgentDescription().clone();
	
	// if the agent is set to run locally, we set the host
	// to the hostname of the local computer
	
	if (ad.getRunLocation().equals(AgentDescription.RUN_LOCAL)) {
	  ad.setRunLocation(adaptive.core.net.LocalHost.getFullLocalHostName());
	}
	
	if (debugLevel<-3) {
	  System.out.println("Adding agent "+ad.getName()+" Host: "+
			     ad.getRunLocation()+".");
	}
	
	//sc.addAgent(ad.getRunLocation(),ad);
	Enumeration locations=ad.getAllRunLocations();
	String tempLocation;
	while (locations.hasMoreElements()){
	  tempLocation=(String)locations.nextElement();
	  sc.addAgent(tempLocation,ad);
	} 


	
	if (ad instanceof MacroDescription) {
	  // by the way, the layout for nested macros aren't stored.
	  // should do something about this... i suppose..  FIX
	  allLayouts.put(ad.getName(),((Macro)m).getLayoutObject());
	}
	
	// Put the location of the module, and the size of the module
	p=m.getLocation();
	size=m.getSize();
	//layout.println(m.getName()+" "+p.x+","+p.y);
	
	guiLayout[layoutCount++]=new String(m.getName()+" "+m.getClassType()+
					    " "+m.getNumInputs()+" "+
					    m.getNumOutputs()+" "+p.x+" "+p.y+
					    " "+size.width+" "+size.height);
	eLines=m.getInputConnectLines();
	ConnectLine cl;
	while (eLines.hasMoreElements()) {
	  o=eLines.nextElement();
	  if (o instanceof ConnectLine) {
	    cl=(ConnectLine)o;
	    if (moduleList.containsKey(cl.getOutputPort().
				       getModule().getName())) {
	      guiLayout[layoutCount++]=new String(((ConnectLine)o).toString());
	    } else {
	      guiLayout[layoutCount++]=new String("none");
	    }
	  }
	  else {
				//layout.println("none");
	    guiLayout[layoutCount++]=new String("none");
	  }
	}
	
      }
      //		  pw.close();
      //layout.close();
      // save the layout
      
      //sc.gui=guiLayout;
      allLayouts.put(" .Main",guiLayout);
      ConfigAndLayout cal = new ConfigAndLayout(sc,allLayouts);
      return cal;
      
    } catch (Exception e) {
      System.err.println("error in save"+e);
      e.printStackTrace();
    }
    return null;

  }
  
  /**
    * Cycles through all the modules and finds free inputs and outputs.
    * Then it adds them to the internal mapping.  If the mapping already
    * exists in the listing, then it isn't added.
    * DEPRECATED --use method in MacroDescription when creating one from a SystemConfiguration.  This isn't important otherwise.
    */
  public void createInternalMapping() {
    Enumeration agents = modules.elements();
    adaptive.modulemanager.Module m;
    int i,o;
    PortLocation pl;
    SystemConfiguration sc= systemConfiguration.systemConfiguration;
    int nextInput=sc.getIALsize();
    int nextOutput=sc.getOALsize();
    int x;
    
    if (debugLevel<-3) System.out.println("internal mapping");
    
    while (agents.hasMoreElements()) {
      m = (Module)agents.nextElement();
      //System.out.println("processing "+m);
      i = m.getNumInputs();
      for (x=0;x<i;x++) {
	  // If there is an IOPort object stored in the input 
	  // mapping, that means that port is being used
	if (!(m.getInputMapping(x) instanceof IOPort)) {
	  if (debugLevel<-2) System.out.println(m.getName()+" input "+x+" not mapped");
	  
	  if (debugLevel>3) 
	    System.out.println("## Making pl, ad=="+m.getAgentDescription());

	  pl=new PortLocation(m.getName(),x,m.getAgentDescription());
	  
	  if (!sc.ialContains(pl)) {
	    sc.addInputPort(nextInput++,m.getInputPortName(x),pl);
	  } else {
	    sc.setInputPortName(m.getInputPortName(x),pl.getPortNumber());
	  }
	}
      }
      
      o = m.getNumOutputs();
      // look for free outputs
      for (x=0;x<o;x++) {
	if (!m.isOutputMapped(x)) {
	  /*
	    if (m instanceof Macro) {
	    pl =((Macro)m).getOutputAgent(x);
	    pl.setAgentName(m.getName()+"."+pl.getAgentName());
	    } else {
	    pl=new PortLocation(m.getName(),x);
	    } */	
	  if (debugLevel>3)
	    System.out.println("## Making pl, ad=="+m.getAgentDescription());

	  pl=new PortLocation(m.getName(),x,m.getAgentDescription());
	  
	  if (!sc.oalContains(pl)) {
	    sc.addOutputPort(nextOutput++,m.getOutputPortName(x),pl);
	  } else {
	    sc.setOutputPortName(m.getOutputPortName(x),pl.getPortNumber());
	  }
	  
	}
      }
    }
    // System.out.println("sci="+sc.getNumInputs()+" sco="+sc.getNumOutputs());
    //	 System.out.println("sci="+systemConfiguration.systemConfiguration.getNumInputs()+" sco="+sc.getNumOutputs());
    
  }
  /*********
   * syncPorts
   * synchronize the port types stored in each module with the ports of the agent defining it
   *********/
  boolean syncPorts(){
    boolean retval=true;
    Enumeration mods=getModules();
    Module m;
    while(mods.hasMoreElements()){
      m=(Module)mods.nextElement();
      retval&=m.syncPorts();
    }
    return retval;
  }
  boolean syncPortTypes(){
    boolean retval=true;
    Enumeration mods=getModules();
    Module m;
    while(mods.hasMoreElements()){
      m=(Module)mods.nextElement();
      retval &= m.syncPortTypes();
    }
    return retval;
  }

  /**
    * Delete a connection line and remove the mapping from the modules it
    * was connected to.
    *
    * @param cl ConnectLine to remove
    */
  public void deleteConnection(ConnectLine cl) {
    cl.getOutputPort().getModule().removeOutputMapping(cl);
    cl.getInputPort().getModule().removeInputMapping(cl);
    lines.removeElement(cl);
    highlightedSegment=null;
    //setCursor(Cursor.getDefaultCursor());
    //modulePanel.repaint();
    saved=false;
    setChanged();
    notifyObservers();
  }


  /**
    * Deletes all the modules that are selected.
    */	  
  public void deleteSelectedModule()
    {
      if (selectedModules.size()==0) return;
      moduleSelected=false;
      //System.out.println("Deleting: "+selectedModule.getName());
		  

      adaptive.modulemanager.Module selectedModule;
      Enumeration f = selectedModules.elements();
      Enumeration e;
      while (f.hasMoreElements()) {
	selectedModule=(Module)f.nextElement();
	// Remove the module from the hashtable
	modules.remove(selectedModule.getName());
	systemConfiguration.systemConfiguration.deleteInputAgent(selectedModule.getName());
	systemConfiguration.systemConfiguration.deleteOutputAgent(selectedModule.getName());
	
	//System.out.println("deleting: "+selectedModule.getName());
	
	// Remove the module from the panel 
	//modulePanel.remove(selectedModule);
	// dehighlight all the lines connected to it
	// highlightLines(selectedModule,false);
	Object o;
	ConnectLine cl;
	Enumeration points;
	
	// We also have to delete all the connection lines that
	// lead to the module, and all the points the connection
	// line has in it.
	e = selectedModule.getInputConnectLines();
	while (e.hasMoreElements()) {
	  o=e.nextElement();
	  if (o instanceof ConnectLine) {
	    cl=(ConnectLine)o;
	    lines.removeElement(cl);
	    cl.getOutputPort().getModule().removeOutputMapping(cl);
	  }
	}
	e = selectedModule.getOutputConnectLines();
	Enumeration l;
	while (e.hasMoreElements()) {
	  o=e.nextElement();
	  if (o instanceof Vector) {
	    l= ((Vector)o).elements();
	    while (l.hasMoreElements()) {
	      cl=(ConnectLine)l.nextElement();
	      lines.removeElement(cl);
	      cl.getInputPort().getModule().removeInputMapping(cl);
	    }
	  }			
	}
      }
      
      selectedModules.clear();
      //modulePanel.repaint();
      saved=false;
      setChanged();
      notifyObservers();
    }

  public String displayConfigAndLayout(ConfigAndLayout cal,Point offset,boolean select) {
    return displayConfigAndLayout(cal,offset,select,"");
  }

  /**
    * Display an agentContainer on the screen.  Processes all the modules
    * inside, makes all the appropriate connections, and displays them
    * with the same layout as they were saved in plus the given offset.
    *
    * @param agentContainer The container to load from
    * @param offset A point offset to act as 0,0 of the saved system.
    * @param select Whether or not to select the modules in the container
    */
  public String displayConfigAndLayout(ConfigAndLayout cal,Point offset,boolean select,String prefix) {
    //System.out.println("displaying system "+cal.systemConfiguration.getType());
    //System.out.println("layout="+cal.layout.toString());

    SystemConfiguration sc= cal.systemConfiguration;
    
    // Random Variables
    Dimension window = new Dimension(100,100); //this.getSize();
    Dimension d;
    Enumeration e,f;
    AgentDescription agentDescription;
    adaptive.modulemanager.Module m,otherModule;
    String nameAndPort,agentName,outputPort;
    int width,height,y=60;
    //Hashtable agentTable = new Hashtable(50);
    InputLocation il;
    
    Hashtable changedNames=new Hashtable();
    String n;	 
    Hashtable macroList=new Hashtable();
    
    width=window.width-10;
    height=window.height-40;
    
    // if we are going to select the modules we are about to show, clear the
    // other selected modules so they don't interfere
    //if (select) deleteSelectedModule();
    e = selectedModules.elements();
    adaptive.modulemanager.Module l;
    
    while (e.hasMoreElements()) {
      l = (Module)e.nextElement();
      l.setSelected(false);
      highlightLines(l,false);
    }
    selectedModules.clear();
    
    // Pass the File Stream to the SystemLoader and get the
    // hashtable of the agents back
    //SystemLoader.loadConfig(agentLoader,fis,agentTable);
    
    // Cycle through all the agents in the table and add
    // them to the screen
    
    Agent a;
    
    Enumeration hostNames=sc.getHostNames();
    Enumeration agentNames;
    String hostName,macroName, thisAgent;
    int dot;
    
    while (hostNames.hasMoreElements()) {
      hostName=(String)hostNames.nextElement();

      agentNames = sc.getAgentNames(hostName);
      while (agentNames.hasMoreElements()) {
	thisAgent = (String)agentNames.nextElement();
	agentDescription=sc.getAgent(hostName,thisAgent);
	dot = agentDescription.getName().indexOf(".");
	//System.out.println("display = "+agentDescription.getName());

	m=null;
	//create gui Module for agentDescription
	if (agentDescription instanceof MacroDescription) {

	  m = new Macro((MacroDescription)agentDescription,moduleManager,debugLevel);
	  ((Macro)m).setPrefix(systemName);
	  //System.out.println("created "+agentDescription.getName());
	  ((Macro)m).setLayout((Hashtable)cal.layout.get(agentDescription.getName()));
	  ((Macro)m).generatePortTypes(moduleManager.getBrowserModel());

	} 
	else {

	    if(agentDescription.getName().indexOf(":") > -1){
		m = new StubModule(agentDescription.getName(),this,false,
				   debugLevel);
	    }
	    else{
		m = (Module)modules.get(agentDescription.getName());
	    }
	    if(m==null){
		m = moduleManager.getBrowserModel().getModule(agentDescription.getType(),agentDescription.getSource());
	    }
	    if (m!=null) {
		m = (Module)m.clone();
		m.setAgentDescription(agentDescription);
	    }
	}
	
	//if the module was successfully created, place it in the layout
	if (m!=null) {
	  m.syncPortTypes();
	  // When we name the module we have to make sure there are no
	  // other modules in this model with the same name, if there is one,
	  // we have to find a better name, and store the changed name so we
	  // can perserve the mappings
	  if (!validModuleName(agentDescription.getName())) {
	    n=getValidName(agentDescription.getName());
	    changedNames.put(agentDescription.getName(),prefix+n);
	    m.setName(prefix+n);
	  }
	  else {
	    m.setName(prefix+agentDescription.getName());
	    changedNames.put(agentDescription.getName(),
			     prefix+agentDescription.getName());
	  }
	  
	  m.setOwner(this);
	  m.setVisible(true);
	  m.setPaint(false);
	  if (select) {
	    m.setSelected(select);
	    selectedModules.put(m.getName(),m);
	  }
	  //modulePanel.add(m);		  
	  modules.put(m.getName(),m);
	} 
	else {
	  // The agent wasn't found
	  moduleManager.getStatusWindow().setWarning(true);
	  moduleManager.getStatusWindow().
	    setMessage("Warning: Attempted to load: "+
		       agentDescription.getType()+
		       ".  Agent Class not found.");
	}
      }
    }
    
    // Now we have to cycle through again and make all
    // the mappings
    
    //    System.out.println("changedNames="+changedNames.toString());
    //System.out.println("modules="+modules.toString());

    e = modules.elements();
    while(e.hasMoreElements()) {
      m = (Module)e.nextElement();
      agentDescription=m.getAgentDescription();
      
      // If the agent doesn't exist in the list of changed names, then
      // the system was unable to find it, so we don't map any of it's
      // inputs..
      if (!changedNames.containsKey(agentDescription.getName())){
	  System.err.println("not found, so we won't map its inputs:"+agentDescription.getName());

	  continue;
      }
      
      m = (Module)modules.get(changedNames.get(agentDescription.getName()));
      
      
      //System.out.println("Module "+m.getName());
      
      //f = agentDescription.getInputs().elements();
      try{
	for (int im=0;im<agentDescription.getNumMappedInputs();im++) {
	  
	  il=agentDescription.getInputMapping(im);//(InputLocation)f.nextElement();
	  if (il==null) continue;
	  //  System.out.println("       Mapping input:"+il.toString());
	  if (il.getAgentID().charAt(0)=='.') {
	    // ignore this mapping, it goes to a agent/macro that's one level
	    // higher than this one
	    continue;
	  }
	  
	  // If the module exists in the modulelist, make the mapping.
	  // If it doesn't, warn the user and don't map.
	  if (changedNames.containsKey(il.agentID)) {
	    if (modules.containsKey(changedNames.get(il.agentID))) {
	      otherModule=(Module)modules.get(changedNames.get(il.agentID));
	      //  System.out.println("## checking stuff: "+m+";"+il+";"+otherModule);
	      if (m.getPortType(IOPort.INPUT,il.inputNum).isAssignableFrom(otherModule.getPortType(IOPort.OUTPUT,il.outputNum))) {
		IOPort source = new IOPort(m,il.inputNum,IOPort.INPUT);
		IOPort destination = new IOPort((Module)modules.get(changedNames.get(il.agentID)),il.outputNum,IOPort.OUTPUT);
		// m.setInputMapping(il.inputNum,(Module)modules.get(changedNames.get(il.agentID)),il.outputNum); 
		m.setInputMapping(new ConnectLine(source,destination));
		if (debugLevel<-3) System.out.println("Dummy:"+m.getInputConnectLine(il.inputNum).toString2());
		
		//System.out.println("Mapped:"+il);
	      } else {
		moduleManager.getStatusWindow().setWarning(true);
		moduleManager.getStatusWindow().
		  setMessage("Warning: Attempted to map output "+
			     String.valueOf(il.outputNum)+" of ["+
			     il.agentID+"] to input "+String.valueOf
			     (il.inputNum)+
			     " but failed due to port type mismatch.");
	      }
	      
	    } else {
	      moduleManager.getStatusWindow().setWarning(true);
	      moduleManager.getStatusWindow().
		setMessage("Warning: adaptive.modulemanager.Module ["+il.agentID+
			   "] previously mapped to input "+
			   String.valueOf(il.inputNum)+" of [" 
+m.getName()+"] doesn't exist anymore.");
	      
	    }
	  } else { 
	    /*Here's where I'll put the STUB class stuff... (test for RunMode)*/
	    moduleManager.getStatusWindow().setWarning(true);
	    moduleManager.getStatusWindow().
	      setMessage("Warning: Unable to map to "+il.agentID+
			 " because the agent class file was not found.");
	  }
	  
	}
      }catch(Exception ee){
	moduleManager.getStatusWindow().setWarning(true);
	moduleManager.getStatusWindow().
	  setMessage("Warning: Unable to map ports on the agent because of an exception(see stderr for details)");
	System.err.println("Caught exception in mapping ports:");
	ee.printStackTrace();
      }
    }
    return loadLayout((String[])cal.layout.get(" .Main"),offset,changedNames);
    
  }

  
  /**
    * Start/stop drawing a line to connect modules
    *
    * @param m The module where the user clicked
    * @param port The port number
    * @param type The type of port
    */
  public int drawLine(Module m,int port,int type) {
    
    // If we're already drawing a line, then that means we've reached
    // the end of the line, and we have to make the connections
    if (drawingLine) {
      //moduleManager.getStatusWindow().setError(false);
      
      // we have to check to see if the module we are trying to conenct
      // to already has a mapping to it.
      if (type==IOPort.INPUT) {
	// Only input ports cannot have mutiple mappings
	
	if (m.getInputMapping(port)!=null) {
	  moduleManager.getStatusWindow().setMessage("Error: Cannot map "+
						     "to input port "+type);
	  moduleManager.getStatusWindow().setMessage("       Multiple inputs"+
						     " not permitted.");
	  moduleManager.getStatusWindow().setError(true);
	  return -1;
	}
	
      } 
      // Then check to see if the user tried to map input->input
      // or output->output
      if (currentLine.getStartPort().getType()==type) {
	if (type==IOPort.INPUT)
	  moduleManager.getStatusWindow().setMessage("Error: Cannot map "+
						     "input to input.");
	else
	  moduleManager.getStatusWindow().setMessage("Error: Cannot map "+
						     "output to output.");
	moduleManager.getStatusWindow().setError(true);
	return -2;
      }
      // Then we check to see if the port types match
      // input superclass of output is ok
      
      if (currentLine.getStartPort().getType()==IOPort.INPUT) {
	if(!currentLine.getStartPortType()
	   .isAssignableFrom(m.getPortType(type,port)))
	  {
	    moduleManager.getStatusWindow().setMessage("Error: Unable to "+
						       "map due to Port type "+
						       "mismatch");
	    moduleManager.getStatusWindow().setError(true);
	    return -3;
	  }
      } else {
	if (!m.getPortType(type,port).isAssignableFrom(currentLine
						       .getStartPortType())) 
	  {
				//if (!currentLine.getStartPortType().isAssignableFrom(m.getPortType(type,port))) {
	    moduleManager.getStatusWindow().setMessage("Error: Unable to "+
						       "map due to Port type"+
						       " mismatch");
	    moduleManager.getStatusWindow().setError(true);
	    return -4;
	  }
      } 
      
      if (currentLine.getNumberOfBendPoints() % 2 !=0)  {
	// If the user tried to map an input/output with a vertical
	// line, just add a bend to make the line horizontal.  Makes
	// things cleaner.
	//currentLine.eraseLine(modulePanel.getGraphics());
	currentLine.addBend(m.getLineLocation(port,type));
      }
      
      if ((currentLine.getNumberOfBendPoints()==0)) {
	// Since the user didn't put any bends in the line, we have to
	// insert one so that if he tries to move one of the modules there
	// will be a place for the connect line to grow
	
	Point s = currentLine.getStart();
	Point f = m.getLineLocation(port,type);
	
	// We'll just insert the mid point of the two lines. If there's
	// something there, too bad.
	// We have to add it twice because we need a vertical line
	//currentLine.eraseLine(modulePanel.getGraphics());
	currentLine.addBend(new Point((s.x+f.x)/2,(s.y+f.y)/2));
	currentLine.addBend(new Point((s.x+f.x)/2,(s.y+f.y)/2));
	
      }		
      drawingLine=false;
      //selectPort(new IOPort(m,port,type));
      
      // Sometimes the user might not align things properly, so
      // we first erase the line, set the finish point, and then
      // draw the line again.
      //currentLine.eraseLine(modulePanel.getGraphics());
      currentLine.setFinish(m.getLineLocation(port,type));
      currentLine.setFinishPort(new IOPort(m,port,type));
      /* m.getPortType(type,port) */
      lines.addElement(currentLine);
      
      currentLine.getInputPort().getModule().setInputMapping(currentLine);
      currentLine.getOutputPort().getModule().setOutputMapping(currentLine);
      
      // try and register the mapping 
      //boolean worked=currentLine.getInputPort().getModule().getAgent().registerInputProvider(currentLine.getInputPort().getPortNumber(),currentLine.getOutputPort().getModule().getAgent(),currentLine.getOutputPort().getPortNumber());
      
      
      IOPort portSelected=currentLine.getInputPort();
      IOPort p = currentLine.getOutputPort();
      // Print out the status in the status window
      moduleManager.getStatusWindow()
	.setMessage(portSelected.getTypeAsString()+" "+
		    String.valueOf(portSelected.getPort())+" on module "+
		    portSelected.getModule().getName()+ " mapped to "+
		    p.getTypeAsString()+" "+ 
		    String.valueOf(p.getPort())+" on module "+
		    p.getModule().getName()+".");
      portSelected=null;
      
      
      // Put the points of the line in the linesbypoint hashtable so
      // we can quickly retrieve what line corresponds to what point
      //
      // We are not using the points yet, so i'm not going to waste
      // time adding them into the vector.  In the future these may
      // be used for something.
      /*
	linesByPoint.put(currentLine.getStart(),currentLine);
	linesByPoint.put(currentLine.getFinish(),currentLine);
	Enumeration e = currentLine.getBendPoints();
	while (e.hasMoreElements()) 
	linesByPoint.put((Point)e.nextElement(),currentLine);
      */
      //currentLine.drawLine(modulePanel.getGraphics(),null);
      currentLine=null;
      saved=false;
      setChanged();
      notifyObservers();
      
    } else {
      // If we just started drawing the line, set it's direction
      // and start port
      drawingLine=true;
      currentLine= new ConnectLine(m.getLineLocation(port,type));
      currentLine.setStartPort(new IOPort(m,port,type));
      // whenever we start drawing a line, it's always horizontally
      currentLine.setDirection(currentLine.HORIZ);
    }
    return 0;
  }
  
  
  public SystemConfiguration getSystemConfiguration() {
    toSystemConfiguration();
    return systemConfiguration.systemConfiguration;
  } 
  public void setSystemConfiguration(SystemConfiguration sc){
    //added this so the system properties window can make its changes and save them only if the OK button is pushed
    this.systemConfiguration.systemConfiguration=sc;
  }
  
  public ConnectLine getCurrentLine() {
    return currentLine;
  }
  
  public int getDefaultName() {
    return defaultName++;
  }
  
  /**
    * Gets a valid name for the module given a suggested name.  Names cannot
    * have spaces or periods in them so it replaces all spaces and periods with
    * underscores.  Names must also be unique so if a module with the given
    * name already exists, a number is appended until a unique name is created.
    *
    * @param modName the suggested name for the module
    */
  public String getValidName(String modName) {
    modName.trim().replace(' ','_');
    modName.replace('.','_');
    modName=new String(modName+"_"+String.valueOf(defaultName++));
    while (!validModuleName(modName))
      modName=new String(modName+"_"+String.valueOf(defaultName++));
    return modName;
  }
  

  public LineSegment getHighlightedSegment() {
    return highlightedSegment;
  }

  public Enumeration getLines() {
    return lines.elements();
  }
  
  public ModuleManager getModuleManager() {
    return moduleManager;
  }
  
  public Enumeration getModules() {
    return modules.elements();
  }

  public adaptive.modulemanager.Module getModule(String s) {
    return (Module)modules.get(s);
  }

  public Enumeration getModuleNames() {
    return modules.keys();
  }
  
  public adaptive.modulemanager.Module getModuleToResize() {
    return moduleToResize;
  }
  
  
  public Enumeration getSelectedModules() {
    return selectedModules.elements();
  }

  public ConfigAndLayout getSelectedModulesAsAgentContainer() {
    if (selectedModules.size()==0) return null;
    return createConfigAndLayout("clipboard",selectedModules);
  }
  

  public adaptive.modulemanager.Module getSelectedAgent() {
    return selectedAgent;
  }
  
  public SelectRectangle getSelectRectangle() {
    return selectRectangle;
  }

  public int getNumberOfSelected() {
    return selectedModules.size();
  }
  
  /**
    * Return this model's systemName
    *
    * @return this model's systemName
    *
    */
  public String getSystemName() {
    return systemName;
  }
  
  public String getSystemFile() {
    return systemFile;
  }
  
  public String getAuthor() {
    return author;
  }
  
  public String getDescription() {
    return description;
  }
  
  public String getVersion() {
    return version;
  }
  
  public String getDate() {
    return date;
  }
  
  public String getLastRevision() {
    return lastRevision;
  }

  /**
    * Highlights or dehighlights the connection lines going into and out
    * of a specific module.
    *
    * @param m The module to highlight the lines of
    * @param highlight True - highlights the lines, False - dehighlights the lines
    */
  public void highlightLines(Module m,boolean highlight)
    {
      Color c;
      if (highlight) c = selectedLineColor; else c = Color.black;
      Object o;
      Enumeration e = m.getInputConnectLines();
      while (e.hasMoreElements()) {
	o=e.nextElement();
	if (o instanceof ConnectLine) 
	  ((ConnectLine)o).setColor(c);
      }
      e = m.getOutputConnectLines();	
      Enumeration l;
      while (e.hasMoreElements()) {
	o=e.nextElement();
	if (o instanceof Vector) {
	  l = ((Vector)o).elements();
	  while (l.hasMoreElements()) {
	    ((ConnectLine)l.nextElement()).setColor(c);
	  }
	}
      }
      
      
    }
  
  /**
    * Initializes all the vectors, hash tables, etc so people can start
    * inserting modules and messing around.
    *
    * Kind of a cheesy name isn't it? oh-well..
    */
  private void initializeDataStorage() 
    {
      // we have to remove all the components from the screen if there 
      // are any

      //-modulePanel.removeAll(); 

	 
      // In theory if these were already initialized and we just re-init
      // them all, the java garbage collector will take care of the
      // unused data, right? lets hope so... 
      portSelected=null;
      //availableModuleList= new Vector(100,10);

      //lines = new Vector(40,10);
      //linesByPoint = new Hashtable(100);
      //modules = new Hashtable(50);
      //selectedModules = new Hashtable(10);

      currentLine=null;
      //selectedModule=null;
      moduleSelected=false;
      moduleToDrag=null;

      defaultName=0;
      highlightedSegment=null;
      draggingLine=false;
      systemFile=null;
      systemName=null;	 
      selectRectangle=null;
	 
    }

  public boolean isDraggingLine() {
    return draggingLine;
  }
  

  public boolean isModuleSelected() {
    return moduleSelected;
  }
  
  public boolean isSaved() {
    return saved;
  }
  

  /**
    * Loads the layout of the modules and the connection lines from
    * an array of strings.
    *
    * @param configFile The configuration information stored in an array
    */
  private String loadLayout(String[] configFile,Point offset,Hashtable changedNames) {
    String s,name,l,moduleAgentClass;
    int x,comma,space;
    ConnectLine cl;
    Point p;			
    IOPort i;
    adaptive.modulemanager.Module m;
    StringTokenizer st;
    int width=0;
    int height=0;
    int w,h,startLine=0;
    int numInputs=0,numOutputs=0;
    int fileLayoutVersion=0;
    String returnString="Layout successfully loaded.";
    /*
      for (int x32=0;x32<configFile.length;x32++) {
      System.out.println(configFile[x32]);
      }
    */
    
    if (configFile==null) {
      System.err.println("Warning: No layout in save file.");
      return "Warning: No layout in save file.";
    }
    // Check the version of the layout file
    if (configFile[0].startsWith("LayoutFileVersion")) {
      startLine=1;
      st = new StringTokenizer(configFile[0]," ");
      st.nextToken();
      fileLayoutVersion=Integer.parseInt(st.nextToken());
      if (fileLayoutVersion>MAX_LAYOUT_VERSION) {
	System.err.print  ("Error: The layout of this configuration file was");
	System.err.println(" created by a newer version of adaptive.modulemanager.Module Manager.");
	System.err.print  ("       This version cannot read it.  Please use");
	System.err.println("       the newer version.");
	moduleManager.getStatusWindow().setError(true);
	return "Error loading layout. Check stderr for details.";
      }		
    }
    
    
    // Now cycle through the layout file and retrieve the 
    // module locations and the connectline information
    for (int y=startLine;y<configFile.length;y++) {
      try {
	s = configFile[y];
	//System.out.println("%% "+s);
	
	st = new StringTokenizer(s," ");
	name = st.nextToken();
	
	if (fileLayoutVersion==1) {
	  moduleAgentClass=st.nextToken();
	  numInputs=Integer.parseInt(st.nextToken());
	  numOutputs=Integer.parseInt(st.nextToken());
	} else {
	  moduleAgentClass=null;
	}
	
	
	// If the module name is not in the hashtable, that means the cfg
	// file was changed but the layout wasn't.  This could have happened
	// while running the system and an evaluator agent swaped agents
	// around.  Since the module isn't there, we skip it's layout.
	if (!changedNames.containsKey(name)) {
	  System.err.print("Warning: No module "+name+" exists. Skipping");
	  System.err.println(" it's layout. More warnings may follow.");
	  returnString="Warnings encountered while loading " + 
	    "layout. Check stderr for details.";
	  moduleManager.getStatusWindow().setWarning(true);
	  // Skip over the lines that contain the points for the connection
	  // lines
	  y=y+numInputs;
	  continue;
	}
	m=(Module)modules.get(changedNames.get(name));
	name=(String)changedNames.get(name);
	// Check to see if the agent classes match. If not, don't process
	// the layout. 
	if (fileLayoutVersion==1) {
	  if (!m.getClassType().equals(moduleAgentClass)) {
	    System.err.print("Warning: Agent class for "+name+" doesn't");
	    System.err.println("   match expected value.");
	    System.err.println("   Skipping it's layout.");
	    moduleManager.getStatusWindow().setWarning(true);
	    y=y+numInputs;
	    returnString="Warnings encountered while loading " +
	      "layout. Check stderr for details.";
	    continue;
	  }
	}
	
	// To maintain support for old version of the layout
	if (moduleAgentClass==null) {
	  numInputs=m.getNumInputs();
	}
	
	
	p = new Point(Integer.parseInt(st.nextToken()),Integer.parseInt(st.nextToken()));
	p.translate(offset.x,offset.y);
	m.setLocation(p);	
	w=Integer.parseInt(st.nextToken());
	h=Integer.parseInt(st.nextToken());
	
	
	// Since we aren't storing the canvas size when the system was saved,
	// we have to calculate the biggest X, and Y needed to display all
	// the modules
	
	width=Math.max(width,p.x+w+5);
	height=Math.max(height,p.y+h+5);		  
	
	m.repaint();		 
	
	for (x=0; ((x<m.getNumInputs()) && (x<numInputs)); x++) {
	  l = configFile[++y];
	  if (l.startsWith("none")) {
	  } else {
	    cl = new ConnectLine(l,offset);
	 // we have to set the ports in the connectline. in order to do that
	 // we need to know which end is the start
	    //				i = m.getInputMapping(x);
	    
	  // If there is no input mapping for this port, just skip the
			 // creation of the connectline. 
	    if (m.isInputMapped(x)) {
	      
	      if ((cl.getStart().x==m.getInputLineLocation(x).x) && 
		  (cl.getStart().y==m.getInputLineLocation(x).y) ) {
		//System.out.println("=");
		cl.setFinishPort(m.getInputMapping(x));
		cl.setStartPort(new IOPort(m,x,IOPort.INPUT));
	      }
	      else {
		cl.setStartPort(m.getInputMapping(x));
		cl.setFinishPort(new IOPort(m,x,IOPort.INPUT));
	      }							
	      lines.addElement(cl);
	      cl.createLineSegments();
	      
	      m.setInputMapping(cl);
	      
	      // add the connect line to the other module also
	      m.getInputMapping(x).getModule().setOutputMapping(cl);
	    }
	  }
	}
	m.resizeModule(w,h);
      } catch (Exception e) {
	System.out.println(e);
	e.printStackTrace();
	System.err.print  ("Warning: Exception occured while processing the");
	System.err.println(" layout. Some or all of the");
	System.err.println("         layout may be incorrect.");
	//e.printStackTrace();		  
	returnString="Warnings incountered while loading layout. Check stderr for details.";
	
      }
      
    }
    setChanged();
    
    //(this);
    //canvasSize.setSize(width,height);
    //modulePanel.setSize(canvasSize.width,canvasSize.height);
    //pack();
    
    return returnString;
    
  }

  public boolean moveLineSegment(Point p) {

    if (highlightedSegment!=null){
      // If a line segment is highlighted that means the
      // user is most likely trying to move it around..
      draggingLine=true;
      //highlightedSegment.draw(modulePanel.getGraphics(),backgroundColor);

      // When we set the x and y here of the line segment, it modifies
      // the points.  But since everything is a pointer, the points also
      // get modified in the original ConnectLine.
      highlightedSegment.setX(p.x);
      highlightedSegment.setY(p.y);
      saved=false;
      setChanged();
      notifyObservers();
      return true;
    }
    return false;
	 
	 
  }
  

  public boolean moveModule(int x, int y) {
    //System.out.println("trying to move module");
    if (moduleToDrag!=null) {
      //Point p = moduleToDrag.getLocation();		
	
      if (selectedModules.containsKey(moduleToDrag.getName())) {
		  Enumeration e=selectedModules.elements();
		  adaptive.modulemanager.Module m;
		  while (e.hasMoreElements()) {
			 m=(Module)e.nextElement();
			 m.setLocation(x-m.Xoffset,y-m.Yoffset);
			 m.adjustConnectLines();
			 
		  }
      } else {
		  //moduleToDrag.setLocation(applyGridX(x-moduleToDrag.Xoffset),applyGridY(y-moduleToDrag.Yoffset));
		  //moduleToDrag.adjustConnectLines();
		  //System.out.println("dragging :" +moduleToDrag);
		  
		  moduleToDrag.setLocation(x-moduleToDrag.Xoffset,
											y-moduleToDrag.Yoffset);
		  moduleToDrag.adjustConnectLines();
      }
		
      //System.out.println("xy="+x+" "+y+" location="+(x-Xoffset+p.x)+" "+(y-Yoffset+p.y));		
		
      // find the line connecting this module (if it exists) and
      // modify the end/start point to match the new location
      // ConnectionLine = (ConnectLine)linesByPoint.get(
      //Module m = (Module)event.getComponent();
      //moduleToDrag.adjustConnectLines();
      //moduleToDrag.dispatching=false;
      //modulePanel.repaint();
      saved=false;
      setChanged();
      notifyObservers();
      return true;
    } 
    return false;
	 
  }
  public void openSystem(ConfigAndLayout ac) {
    openSystem(ac,"");
  }

  public void openSystem(ConfigAndLayout cal,String prefix) {
    SystemConfiguration sc=cal.systemConfiguration;
    this.systemConfiguration=cal;
    displayConfigAndLayout(cal,new Point(0,0),false,prefix);

    if ((version=sc.getVersion())==null) version="";
    if ((author=sc.getAuthor())==null) author="";
    if ((date=sc.getDate())==null) date="";
    if ((description=sc.getDescription())==null) description="";
    if ((lastRevision=sc.getLastRevision())==null) lastRevision="";
    systemName=sc.getType();	 
    systemFile="";
    saved=true;
    setChanged();
    notifyObservers("openfile");
  }
  

  public void openSystem(String filename) throws Exception {
	 
    ObjectInputStream ois =
      new ObjectInputStream(new java.util.zip.GZIPInputStream(new FileInputStream(filename)));
	 
    SystemConfiguration systemConfiguration;
    Hashtable layout;
	 
    systemConfiguration = (SystemConfiguration)ois.readObject();
    layout = (Hashtable)ois.readObject();
    ois.close();
    openSystem(new ConfigAndLayout(systemConfiguration,layout));
    systemFile=filename;
  }
  
  public void placeModule(int x,int y,String name,Module moduleToPlace) {
    placeModule(x,y,name,moduleToPlace,false);
  }
  

  /**	  
    * Place the module on the canvas. Sets the x,y and name.
    *
    * @param x The X location for the module
    * @param y The Y location for the module
    * @param name The name of the module
    */
  public void placeModule(int x,int y,String name,Module moduleToPlace,boolean symbolic)
    {
      //if (selectedAgent==null) return;
		 
      // If the module name isn't being used in the hashtable
      // than allow the user to add it.
      if (modules.get(name)==null) {
		  name=name.trim().replace(' ','_');
		  // The agent MUST have a name in order to map inputs/outputs
		  //selectedAgent.initialize(name,agentLoader);
		  adaptive.modulemanager.Module m;
		  //System.out.println("placing "+moduleToPlace);

		  m = (Module)moduleToPlace.clone(symbolic,name+".");
		  // If someone placed a macro, we have to prefix the name of the macro
		  // on the names of all the agents inside the macro.  this way we won't
		  // get name conflicts.
		  if (m instanceof Macro) {
			 //System.out.println("Placing macro");
		    ((Macro)m).setPrefix(name);

			 //((Macro)m).setType(m.getName());
			 //((Macro)m).generatePortTypes();
		  }
		  //System.out.println(m);
		  m.setName(name);
		  m.setOwner(this);
		  m.setLocation(x,y);
		  //modulePanel.add(m);
		  //m.repaint();
		  m.setPaint(false);
		  //modulePanel.repaint();
		
		
		  // Add the module to list of modules
		  modules.put(m.getName(),m);
		  saved=false;
		  setChanged();
		  notifyObservers();
		
      } else {
		  // Otherwise display an error about duplicate
		  // module names. (they aren't allowed)
		  moduleManager.getStatusWindow().setMessage("Error: adaptive.modulemanager.Module with name "+name+" already exists!");
		  moduleManager.getStatusWindow().setMessage("       Request to add module denied. Try a different name.");
		  moduleManager.getStatusWindow().setError(true);
      }
    }

  /*
    public void prefixNames(String n) {
    n = n+".";
    Enumeration e = modules.keys();
    adaptive.modulemanager.Module m;
    while (e.hasMoreElements()) {
    m = (Module)modules.get(e.nextElement());
    if (m instanceof Macro) 
    prefixNames(n);
    else 
    m.setName(n+m.getName());
    }
    }
  */

  /**
    * Rename a module.  Removes the module from the hashtable, sets the new
    * name, and adds the module back to the hashtable with indexed by the
    * new name.
    *
    * @param module The module to rename
    * @param newName The new name of the module
    */
  public void renameModule(Module module,String newName) {
    systemConfiguration.systemConfiguration.renameInputAgent(module.getName(),newName);
    systemConfiguration.systemConfiguration.renameOutputAgent(module.getName(),newName);
    modules.remove(module.getName());
    module.setName(newName);
    modules.put(newName,module);

	 // the module that was renamed has to go to all modules/macros mapped to
	 // it and fix their inputlocations
	 Enumeration e=module.getOutputConnectLines();
	 Vector v;
	 Enumeration lines;
	 ConnectLine cl;
	 IOPort port;
	 while (e.hasMoreElements()) {
		v = (Vector)e.nextElement();
		if (v!=null) {
		  lines=v.elements();
		  while (lines.hasMoreElements()) {
			 cl = (ConnectLine)lines.nextElement();
			 // All we need to do it move the mapping, and add it again.
			 // this will update the data in the AgentDescription for the
			 // module.
			 cl.getInputPort().getModule().removeInputMapping(cl);
			 cl.getInputPort().getModule().setInputMapping(cl);
		  }
		}
		
	 }
	 

    saved=false;
    setChanged();
    notifyObservers();
  }


  /**
    * Remove a module from the selected list.
    *
    * @param m The module to remove
    */
  public void removeSelectedModule(Module m) {
    if (selectedModules.containsKey(m.getName())) {		
      selectedModules.remove(m.getName());		
      highlightLines(m,false);
    }  
  }
  
  private void replaceMacroHelper(ConnectLine cl,Macro c) {
    //System.out.println("replace helper "+cl);
    int inputPort;
    int outputPort;
    adaptive.modulemanager.Module inputModule,outputModule;
    inputModule=cl.getInputPort().getModule();
    inputPort=cl.getInputPort().getPort();
    outputPort=cl.getOutputPort().getPort();
    // if the connect lines output port no longer exists on the new
    // macro, get rid of the connection
    // remove the input mapping
    inputModule.removeInputMapping(cl); //remove the mapping to the old macro
    if (outputPort+1>c.getNumOutputs()) {
      //inputModule.removeInputMapping(cl);
		
      // warn the user 
      moduleManager.getStatusWindow().setMessage("Removing mapping from "+inputModule.getName()+" to "+c.getName()+" because output port "+String.valueOf(outputPort) +" no longer exists on "+c.getName()+".");
		
    } else {
      // if the mapping still seems valid, try and make the connection
      // on the new module. check port types befor making the mapping.
      if (inputModule.getPortType(IOPort.INPUT,inputPort).isAssignableFrom(c.getPortType(IOPort.OUTPUT,outputPort))) {
		  // connect the line to the new macro
		  cl.setOutputPortModule(c);
		  // set the mapping in the module it's connected to
		  inputModule.setInputMapping(cl);
		  c.setOutputMapping(cl);
      } else {
		  moduleManager.getStatusWindow().setMessage("Removing mapping from "+inputModule.getName()+" port "+String.valueOf(inputPort)+" to "+c.getName()+" port "+String.valueOf(outputPort)+" because of a port type mismatch.");
		  
		  // remove the connect line so it won't get drawn
		  lines.removeElement(cl);
      }
    }
  }

  
  /**
    * Replace a MacroDescription with a new MacroDescription. This has to
    * recheck all the mappings in the model to make sure nobody is getting
    * their input from an output on this macro that doesn't exist any more.
    *
    * @param cd the New MacroDescription
    */
  public void replaceMacro(MacroDescription cd,Hashtable layout) {
    
    // FIX
    // do a diff on port types/ number of ports an c and newmacro  and warn
    // the user if somethings changed to allow him to cancel the save.
    

    //System.out.println("replace Macro");
    Macro c = (Macro)modules.get(cd.getName());
    Macro newMacro = new Macro(cd,moduleManager,debugLevel);
    newMacro.setLayout(layout);
    newMacro.setPrefix(c.getPrefix());
    newMacro.setOwner(this);
    //System.out.println("owner:"+this);
    newMacro.setPaint(false);
    newMacro.setVisible(true);
    newMacro.setLocation(c.getLocation());
    if (c.isSelected()) removeSelectedModule(c);
    if (c.isSelected()) addSelectedModule(newMacro);
    newMacro.setSelected(c.isSelected());
    newMacro.generatePortTypes(moduleManager.getBrowserModel());
    modules.remove(cd.getName());
    // check to see where all the outputs are going
    ConnectLine cl;
    int inputPort;
    int outputPort;
    adaptive.modulemanager.Module inputModule,outputModule;
    Enumeration cls;
    Object o;
    Enumeration e   = c.getOutputConnectLines();
    //System.out.println(e);
    // outputs can be mapped to more than 1 input
    while (e.hasMoreElements()) {
      o = e.nextElement();
      if (o!=null) {
		  //System.out.println(o);
		  if (o instanceof Vector) {
			 cls = ((Vector)o).elements();
			 while (cls.hasMoreElements()) {
				cl = (ConnectLine)cls.nextElement();
				replaceMacroHelper(cl,newMacro);
			 }  
		  } else {
			 replaceMacroHelper((ConnectLine)o,newMacro);
		  }
      }
    }
	 
    cls = c.getInputConnectLines();
    while (cls.hasMoreElements()) {		
      cl = (ConnectLine)cls.nextElement();
      //System.out.println("input "+cl);
      if (cl==null) continue;
      outputModule=cl.getOutputPort().getModule();
      inputPort=cl.getInputPort().getPort();
      outputPort=cl.getOutputPort().getPort();
      // if the connect lines input port no longer exists on the new
      // macro, get rid of the connection
      // remove the input mapping
      outputModule.removeOutputMapping(cl);
      if (inputPort+1>c.getNumInputs()) {
		  moduleManager.getStatusWindow().setMessage("Removing mapping from "+outputModule.getName()+" to "+newMacro.getName()+" because input port "+String.valueOf(outputPort) +" no longer exists on "+newMacro.getName()+".");	  
      } else {
		  // if the mapping still seems valid, try and make the connection
		  // on the new module. check port types befor making the mapping.
		  if (newMacro.getPortType(IOPort.INPUT,inputPort).isAssignableFrom(outputModule.getPortType(IOPort.OUTPUT,outputPort))) {
			 // connect the line to the new macro
			 cl.setInputPortModule(newMacro);
			 // set the mapping in the module it's connected to
			 outputModule.setOutputMapping(cl);
			 //System.out.println(outputModule.getName()+" adding "+cl.toString2());
			 newMacro.setInputMapping(cl);
			 //System.out.println("mapped:"+cl.toString2());
		  } else {
			 moduleManager.getStatusWindow().setMessage("Removing mapping from "+outputModule.getName()+" port "+String.valueOf(outputPort)+" to "+newMacro.getName()+" port "+String.valueOf(inputPort)+" because of a port type mismatch.");
			 
			 // remove the connect line so it won't get drawn
			 lines.removeElement(cl);
		  }
      }
    }
    modules.put(cd.getName(),newMacro);
    newMacro.adjustConnectLines();
    saved=false;
    setChanged();
    notifyObservers();
  }
  
  public boolean resizeModule(Point p) {
    if (moduleToResize!=null) {
      Point l=moduleToResize.getLocation();
      moduleToResize.resizeModule(p.x-l.x+Xoffset,p.y-l.y+Yoffset);
      saved=false;
      setChanged();
      notifyObservers();
      return true;
    }
    return false;
  }
  

  /*
   * Save the module mappings to a config file. The file written will be
   * [systemName].cfg
   *
   * @param systemName The name of the system the modules represent.
   */
  public void saveSystem()
    {
      if (debugLevel<-3) System.out.println("saving "+systemName);
      //      createInternalMapping();

      // if the name has a dot in it, it's a macro. if it's a macro we don't
      // save it to a file, we save it in it's master systemConfig

      //NOTE: this is bad practice here and should be changed to a boolean passed in telling whether this is a macro or not
      if (systemName.indexOf('.')>0) {
		  toSystemConfiguration();
		  // Create the macroDescription out of the current systemConfig
		  //	System.out.println(systemConfiguration.systemConfiguration);
		  MacroDescription cd = new MacroDescription(systemConfiguration.systemConfiguration);
		  if (debugLevel<-4) System.out.println(cd);
		  int dot = cd.getName().lastIndexOf('.');
		  String macroName = cd.getName().substring(dot+1);
		  String parentName=cd.getName().substring(0,dot);
	   
		  // Set the macros name to just its name. not the full name.
		  cd.setName(macroName);
		  ManagerModel model;

		  for(Enumeration jim=moduleManager.getModels(); jim.hasMoreElements();){
			 model = (ManagerModel)jim.nextElement();
			 for(Enumeration bob = model.getModules(); bob.hasMoreElements();){
				Module module = (Module)bob.nextElement();
				if(macroName.compareTo(module.getName()) == 0) {
				  // Then I've found the macro I'm looking for  :)
				  cd.setSource(module.getAgentLocation());
				  cd.setRunLocation(module.getHostName());
				}
			 }
		  }
		  //		  System.out.println("\nConstructed macro is :\n"+cd.toString());
		  // Check to see if there is a model open for the 
		  // parent of this macro	 
		  model=moduleManager.getModel(parentName);
		  //System.out.println("model: "+model+"; looking for: "+parentName);

		  if (model!=null) {
			 // replace the old macrodescription with the new one
			 model.replaceMacro(cd,systemConfiguration.layout);
			 //System.out.println(model.getSystemName()+" updated.");
			 moduleManager.getStatusWindow().setMessage(parentName+" updated.");
			 saved=true;
			 setChanged();
			 notifyObservers();
			 // now recursively save the system if the parent is a macro
			 if (parentName.indexOf(".")>0)
				model.saveSystem();
		  } else {
			 // FIX we need to do something if the model isn't open
			 /* So this is a total guess, hopefully it'll work well enough 
				 and I can talk to Ted when he gets back... 
				 There's *got* to be a better way to do this, but for now I'm 
				 going to go through all models in the moduleManager and look 
				 for a module named parentName.  It has to be open somewhere for 
				 this model to be open for saving... */
			 for(Enumeration jim = moduleManager.getModels(); jim.hasMoreElements();) {
				model = (ManagerModel)jim.nextElement();
				for(Enumeration bob = model.getModules(); bob.hasMoreElements();){
				  adaptive.modulemanager.Module module = (Module)bob.nextElement();
				  if(parentName.compareTo(module.getName()) == 0) {
					 // Then I've found the macro I'm looking for  :)
					 module.setAgentDescription((AgentDescription)cd);
				  }
				}
			 }
		  }
	   
	   
	   
	   
      } else {
	try {
	  ObjectOutputStream oos = new ObjectOutputStream(new java.util.zip.GZIPOutputStream(new FileOutputStream(systemFile)));
	  ConfigAndLayout cal=toSystemConfiguration();
	  //System.out.println("sci="+cal.systemConfiguration.getNumInputs());
	  
	  //System.out.println("serial="+(cal.systemConfiguration instanceof Serializable));
	  if (debugLevel<-4) System.out.println(cal.systemConfiguration.toString());
	  
	  oos.writeObject(cal.systemConfiguration);
	  oos.writeObject(cal.layout);
	  oos.close();
	  moduleManager.getStatusWindow().setMessage("File "+systemName+".cfg saved succesfully.");
	  //setTitle("Module Manager - "+systemName);
	  saved=true;
	  setChanged();
	  notifyObservers();
	} catch (Exception e) {
	  moduleManager.getStatusWindow().setMessage("Unable to save to "+systemName+".cfg");
	  System.err.println("Exception "+e);
	  e.printStackTrace();
	}
      }
      
      
    }

  public void selectAll() {
	 
    Enumeration keys=modules.keys();
    while (keys.hasMoreElements()) 
      addSelectedModule((Module)modules.get(keys.nextElement()));
	 
  }
  

  /**
    * Select a module that will be dropped on the screen
    *
    * @param m The module to drop
    */
  public void selectAgent(Module m)
    {
      if (m!=null) {
	//	try {
	//selectedAgent=agentLoader.newInstance(a.getClass().getName());
	selectedAgent=m;
	//} catch (Exception e) {
	// System.err.println("error"+e);
	//}
	moduleManager.getStatusWindow().setMessage("Click to place module of type "+m.getAgentClass().getName()); 
      } else {
	selectedAgent=null;
      }
    }


  /**
    * Selects a module on the screen and highlights all input/output
    * lines connected to the module. If the ctrl key was held down
    * multiple should be set true and the method will allow selection
    * of multiple modules.  If multiple is false any selected modules
    * will be deselected and only the current module will be selected.
    * <p>
    * If multiple is true, and m is in the list, only that module will
    * be deselected.
    * <p>
    * If multiple is false, and m is in the list, only that module will
    * remain selected.
    * <p>
    * If multiple is true, and m is not in the list, it will be added.
    * <p>
    * If multiple is false and m is not in the list, i dunno..
    *
    * @param m The module that has been selected
    * @param multiple Whether or not the ctrl was held down while selecting
    */
  public void selectModule(Module m,boolean multiple)
    {
      if (multiple) {				

		  if (selectedModules.containsKey(m.getName())) {
			 selectedModules.remove(m.getName());
			 m.setSelected(false);
			 highlightLines(m,false);
		  } else {
			 selectedModules.put(m.getName(),m);
			 highlightLines(m,true);			  
		  }
      } else {
		  boolean wasSelected = selectedModules.containsKey(m.getName());
		  // First remove all modules from the selected list
		  Enumeration e = selectedModules.elements();
		  adaptive.modulemanager.Module l;
			  
		  while (e.hasMoreElements()) {
			 l = (Module)e.nextElement();
			 l.setSelected(false);
			 highlightLines(l,false);
		  }
		  selectedModules.clear();

		  if (!wasSelected) {
			 selectedModules.put(m.getName(),m);
			 highlightLines(m,true);
		  }
				
      }
      moduleSelected=(selectedModules.size()>0);
      //modulePanel.repaint();
      setChanged();
      notifyObservers();

    }


  /*
   * This gets called by the module whenever an input or output is 
   * selected. This method should then check to see if it was the 
   * first or second selection and make the appropriate mapping(s).
   * 
   * @param p The Port that was selected.
   *
   * @see IOPort
   *
   */
  public void selectPort(IOPort p){
    //System.out.println("port selected: "+p.toString());
    
    if (portSelected==null)
      {
	portSelected=p;
	moduleManager.getStatusWindow()
	  .setMessage(p.getTypeAsString()+" "+ 
		      String.valueOf(p.getPort())+" on module "+ 
		      p.getModule().getName()+ " selected.");	
      }
    else
      {
	
	String inputModID,outputModID;
	Agent inputAgent,outputAgent;
	int inputPort,outputPort;
	
	// We need to know which port is the input port since the user
	// can click on the output or input first.
	if (p.isInput()) {
	  inputAgent=p.getModule().getAgent();
	  inputPort=p.getPort();
	  outputAgent=portSelected.getModule().getAgent();
	  outputPort=portSelected.getPort();
	} else {
	  inputAgent=portSelected.getModule().getAgent();
	  inputPort=portSelected.getPort();
	  outputAgent=p.getModule().getAgent();
	  outputPort=p.getPort();
	}
	
	// try and register the mapping 
	boolean worked=inputAgent.setInputMapping(inputPort,
						  outputAgent,outputPort);
	
	
	// We have to set the mapping in both the modules.
	p.getModule().setMapping(p,portSelected);
	portSelected.getModule().setMapping(portSelected,p);
	//modulePanel.repaint();
	// Print out the status in the status window
	moduleManager.getStatusWindow()
	  .setMessage(worked+" "+
		      portSelected.getTypeAsString()+" "+
		      String.valueOf(portSelected.getPort())+" on module "+
		      portSelected.getModule().getName()+ " mapped to "+
		      p.getTypeAsString()+" "+ 
		      String.valueOf(p.getPort())+" on module "+
		      p.getModule().getName()+".");
	portSelected=null;
      }
  }
  
  
  
  public void setCurrentLine(ConnectLine cl) {
    currentLine=cl;
  }
  
  public void setDrawingLine(boolean b) {
    drawingLine=false;
  }
  
  public void setDraggingLine(boolean b) {
    draggingLine=b;
  }

  public void setHighlightedSegment(LineSegment ls) {
    highlightedSegment=ls;
  }
 

  public void setAuthor(String a) {
    author=a;
    setChanged();
    notifyObservers();
  }
  
  public void setDate(String d) {
    date=d;
    setChanged();
    notifyObservers();
  }
  
  public void setVersion(String v) {
    version=v;
    setChanged();
    notifyObservers();
  }
  
  public void setDescription(String d) {
    description=d;
    setChanged();
    notifyObservers();
  }
  
  public void setLastRevision(String lr) {
    lastRevision=lr;
    setChanged();
    notifyObservers();

  }   
  
  /**
    * Select a module to be dragged around. The location of the click inside
    * the module is sent to the module so it knows how to move with respect
    * to the mouse cursor
    *
    * @param module adaptive.modulemanager.Module to drag
    * @param x The x of the click
    * @param y The y of the click
    */
  public void setModuleToDrag(Module m,int x,int y)
    {
      // If there are selected modules, and we are dragging one
      // of them, we move them all
      //System.out.println("dragging:"+m);
	 
      moduleToDrag=m;
      if (m==null) return;
	 
      if (selectedModules.containsKey(m.getName())) {
	Point l;
	l=m.getLocation();
	x=l.x+x;
	y=l.y+y;
	// We have to tell each module the offset so we can move
	// them properly
	Enumeration e = selectedModules.elements();
	adaptive.modulemanager.Module k;
	while (e.hasMoreElements()) {
	  k=(Module)e.nextElement();
	  l=k.getLocation();
	  k.setOffset(x-l.x,y-l.y);
	}		
      } else {
	m.setOffset(x,y);
	//System.out.println("offset="+x+" "+y);
      }
      
      
      moduleToDrag=m;
      //Point p = m.getLocation();
      //Xoffset=x;
      //Yoffset=y;

	 
    }

  /**
	 * Tell this model whether it's saved or not. If the model data is
	 * not saved, all the views are updated with the new data
	 *
	 * @param saved Whether it's saved or not
	 */
  public void setSaved(boolean saved) {
	 this.saved=saved;
	 setChanged();
	 notifyObservers();
  }
  

  /**
    * Set a module to resize
    *
    * @param m The module to resize
    * @param x The offset of the mouse cursor
    * @param y The offset of the mouse cursor
    */
  public void setModuleToResize(Module m,int x,int y) {
    moduleToResize=m;
    Xoffset=x;
    Yoffset=y;
  }
  
  public void setSelectRectangle(SelectRectangle sr) {
    selectRectangle=sr;
  }
  
  /**
    * Sets this models systemName
    *
    * @param name the New system name
    */
  public void setSystemName(String name) {
    systemName=name;
    systemConfiguration.systemConfiguration.setType(name);
    setChanged();
    notifyObservers();
  }
  
  public void setSystemFile(String filename) {
    systemFile=filename;
    setChanged();
    notifyObservers();
  }

  /**
    * Returns an agentContainer that represents the whole model.  If the
    * stored agentcontainer is valid, then return that. otherwise create
    * a new agentcontainer and store it.
    *
    * @return an agentcontainer that represents the whole system.
    */
  public ConfigAndLayout toSystemConfiguration() {
    if (!isSaved()) {
		
      ConfigAndLayout cal = createConfigAndLayout(systemName,modules);
      SystemConfiguration sc=cal.systemConfiguration;
      SystemConfiguration oldSC=systemConfiguration.systemConfiguration;

      for (int x=0;x<oldSC.getNumInputs();x++) {
	sc.addInputPort(x,oldSC.getInputPortName(x),(PortLocation)oldSC.getInputAgent(x).clone());
	//sc.setInputAgent((PortLocation)oldSC.getInputAgent(x).clone(),x);
	//sc.setInputPortName(oldSC.getInputPortName(x),x);
      }
      for (int x=0;x<oldSC.getNumOutputs();x++) {
	sc.addOutputPort(x,oldSC.getOutputPortName(x),(PortLocation)oldSC.getOutputAgent(x).clone());
	//sc.setOutputAgent((PortLocation)oldSC.getOutputAgent(x).clone(),x);
	//sc.setOutputPortName(oldSC.getOutputPortName(x),x);
      }
      systemConfiguration=cal;
      //System.out.println(systemConfiguration);
		
      // if this model is a macro in another system it won't have a system
      // file. if so, we set the model saved.
      if (systemFile.equals("")) saved=true;
    }
    return systemConfiguration;
  }
  

  /**
    * Returns if the specified module name is valid.  If there is a
    * module with that name already it's not valid. There cannot be
    * two modules with the same name.
    *
    * @param name The name of the module
    */
  public boolean validModuleName(String name) {
    return ((modules.get(name)==null) && (name.indexOf(' ')<0));
  }  
  /*********
   * refreshModules
   * reload each module from the BrowserModel, which we'll have reload the agent from disk(or agent server)
   *********/
  public void refreshModules(){
    Module m,m2;
    System.out.println("Refreshing modules");
    BrowserModel browser=this.moduleManager.getBrowserModel();
    Enumeration names=this.modules.keys();
    String name;
    AgentDescription ad;
    while (names.hasMoreElements()){
      name=(String)names.nextElement();
      m = (Module)modules.get(name);
      ad=m.getAgentDescription();
      System.out.println(ad.getType()+" "+ad.getSource());
      m2=browser.refreshModule(ad.getType(),ad.getSource());
      m.setAgent(m2.getAgent());//reload the agent into this module
      

      // this.modules.put(name,m);
    }
  }
}
