package adaptive.modulemanager;

import java.util.*;
import adaptive.core.*;
import java.awt.Color;
import java.awt.Point;
import java.awt.event.*;

/**
  * <p>
  * Macros are groups or "macros" of agents that are represented by only one
  * module on the screen.  Any saved system can be loaded as a macro and then
  * inserted into any other system.  This allows the user to create systems
  * with a lot of abstraction. 
  * <p>
  * Macros can also be pegged or non-pegged.  A pegged macro will always
  * remain together.  They will always run on the same machine.  Non-pegged
  * macros are allowed to break apart and have pieces of them run on other
  * machines.
  * Note:  all macros are non-pegged
  * <p>
  * Extends  adaptive.modulemanager.RectangleModule just to give it a shape. 
  *
  * @author Yatish Patel
 * $Log: Macro.java,v $
 * Revision 1.1.2.5  2000/06/05 05:54:04  jrj
 * more macro fixes.  Changed how and when ModuleManager sets up the ports on a macro so it is now possible to remove an output and keep it removed.  It will also always add unmapped inputs to the macro's ports.  Adding an output is there, but the capability is not put in ModuleManager yet.
 *
 * Revision 1.1.2.4  2000/06/03 19:27:35  jrj
 * Made macros fit MacroDescription a little better.  Fixed an error in mapping inputs on macros.  Added a few diagnostic messages during the system verification step.
 *
 * Revision 1.1.2.3  2000/05/20 19:11:32  jrj
 * fixed a couple more macro bugs.  Took preliminary steps toward making
 * fully recursive macros in MacroDescription.  Have not yet made changes
 * that would damage save file.
 *
 * Revision 1.1.2.2  2000/05/15 22:39:57  jrj
 * Made fixes to macros so that they work.  Changed the executing macro
 * so that it is recursive, can be manipulated as a whole, and its ports
 * can be asked for directly as if it were any other BasicPortAgent.
 * Made several other changes to simplify and assist these changes.
 * Finally purged the last remaining bit of things named clump.
 *
  */
public class Macro extends RectangleModule {

  //private ManagerModel model;

  // contains all the agents inside this macro. (AgentDescription's)
  //private Hashtable agentList;

  private Hashtable layout;
  
  private boolean pegged;

  private Vector inputPortTypes;
  private Vector outputPortTypes;
  private String prefix;

  protected ModuleManager moduleManager;
  
  /**
    * Constructs a new Macro from the specified SystemConfiguration.
    * Creates a MacroDescription from the SystemConfiguration and uses that
    * to create the macro.
    *
    * @param mm A reference to the modulemanager
    * @param sc The SystemConfiguration to use
    * @param l The Hashtable that contains the layout for this macro
    */
  public Macro(ModuleManager mm,SystemConfiguration sc,Hashtable l,
					int debugLevel) {
	 this(new MacroDescription(sc),debugLevel);
	 moduleManager=mm;
	 layout=l;
  }
  
  /**
   * Constructs a new Macro with nothing in it.
   */
  public Macro(int debugLevel) {
	 super(false,debugLevel);
	 backgroundColor=new Color(0x55,0xdd,0x77);
	 agentDescription.setState(new Hashtable());
	 layout=null;
  }
  
  /**
   * Constructs a new Macro with nothing in it.
   *
   * @param mm A reference to the modulemanager
   */
  public Macro(ModuleManager mm,int debugLevel) {
	 this(debugLevel);
	 moduleManager=mm;
  }
  
  /**
   * Constructs a new Macro with the specified MacroDescription.
   *
   * @param ad The MacroDescription to use
   */
    public Macro(MacroDescription ad,int debugLevel) {
	 this(debugLevel);
	 setAgentDescription(ad);
  }

  /**
   * Constructs a new Macro with the specified MacroDescription.
   *
   * @param ad The MacroDescription to use
   * @param mm A reference to the modulemanager
   */
    public Macro(MacroDescription ad,ModuleManager mm, int debugLevel) {
	 this(mm,debugLevel);
	 setAgentDescription(ad);
  }

    /**
     * Sets the MacroDescription for this Macro.  This also resizes the
     * Vectors that contain the input and output port types/names.
     *
     * @param c The new MacroDescription
     */
  public void setAgentDescription(AgentDescription c) {
      MacroDescription cd=(MacroDescription)c;
      agentDescription=cd;
      inputPortTypes=new Vector(cd.getNumInputs());
      outputPortTypes=new Vector(cd.getNumOutputs());
      //System.out.println(this+" inputporttypes are defined!");
      init();
  }
 
    /* Cycles through all the input and output ports and generates port types
     * for them.  This involves retrieving the Agent that maps to the port
     * and getting the port type from it.  In order to get the agent a
     * reference to the BrowserModel is needed.
     *
     *
     * @param bm A reference to the BrowserModel
     */
  public void generatePortTypes(BrowserModel bm) {
	 MacroDescription cd = (MacroDescription)agentDescription;
	 PortLocation pl;
	 adaptive.modulemanager.Module m;
	 AgentDescription ad;
	 System.out.println("generatePortTypes: "+this);
	
		
	 for (int x=0;x<cd.getNumInputs();x++) {
		pl=cd.getInputAgent(x);
		//		System.out.println(pl.getAgentName());

		pl = cd.getAgentDescriptionFor(pl,0);
		ad = pl.getAgentDescription();
		//		System.out.println("port:"+ad);
		// ++++ This is one place where we have an agentDescription, but need a adaptive.modulemanager.Module 
		// There needs to be a better way to figure out the shape 
		if(ad.getType().endsWith("Stub")) {
		  m = new StubModule(ad.getName(),(new ManagerModel(moduleManager,new ConfigAndLayout(new SystemConfiguration((MacroDescription)agentDescription), (Hashtable)layout),debugLevel)),false,debugLevel);
		} else {//if(m == null) {
		  m = bm.getModule(ad.getType(),ad.getSource());
		}
		if (m!=null) {
		  if (cd.getInputPortName(x)==null) {
			 if (ad.getInputPortName(pl.getPortNumber())==null) {
				cd.setInputPortName(m.getInputPortName(pl.getPortNumber()),x);
			 } else {
				cd.setInputPortName(ad.getInputPortName(pl.getPortNumber()),x);
			 }
			 if (debugLevel<-4) System.out.println(x+"="+cd.getInputPortName(x)+"|");
		  } else {
			 if (debugLevel<-4) System.out.println("not null "+x+"="+cd.getInputPortName(x)+"|");
		  }
		  
		  inputPortTypes.addElement(m.getInputPortType(pl.getPortNumber()));
		  } else {
		  System.err.println("ERROR:Unable to find agent of type "+ad.getType());
		  }
	 }
	 for (int x=0;x<cd.getNumOutputs();x++) {
		pl=cd.getOutputAgent(x);
		if (debugLevel<-4) System.out.println("output="+pl.getAgentName());
		pl = cd.getAgentDescriptionFor(pl,1);
		ad=pl.getAgentDescription();
		if (debugLevel<-4) System.out.println("ad="+ad);
		m = bm.getModule(ad.getType(),ad.getSource());
		if (m!=null) {
		  if (cd.getOutputPortName(x)==null) {
			 cd.setOutputPortName(m.getOutputPortName(pl.getPortNumber()),x);
		  }
		  outputPortTypes.addElement(m.getOutputPortType(pl.getPortNumber()));
		  }else {
		  System.err.println("ERROR:Unable to find agent of type "+ad.getType());
		  }
	 }
	 
  }
  


  /**
   * Clones this macro.
   *
   * @param s dunno
   * @param p The prefix for the macro.
   * @return A clone of this macro.
   *********/
  public Object clone(boolean s,String p) {
	 Macro c = new Macro((MacroDescription)(agentDescription.clone()),
								debugLevel);

	 c.moduleManager=moduleManager;
	 c.prefix=p;
	 Enumeration e = inputPortTypes.elements();
	 c.inputPortTypes=new Vector();
	 c.outputPortTypes=new Vector();
	 //	 System.out.println("macro clone");
	 Class cl;
	 while (e.hasMoreElements()) {
		cl=(Class)e.nextElement();
		c.inputPortTypes.addElement(cl);
		//System.out.println("inputtype="+c.toString());
	 }
	 e = outputPortTypes.elements();
	 while (e.hasMoreElements()) {
		cl=(Class)e.nextElement();
		c.outputPortTypes.addElement(cl);
	 }
	 
	 c.setLayout(layout);
	 return c;
	 
  }

  /**
   * Clones this macro.
   *
   * @return A clone of this macro.
   */
  public Object clone() {	
	 return clone(false, this.getPrefix());
  }

  /**
   * Returns whether or not this macro is pegged.
   *
   * @return whether or not this macro is pegged.
   */
  public boolean isPegged() {
	 return pegged;
  }


  /**
   * Return the port type for the specified input port
   *
   * @param port The input port number
   * @return the port type for input port <port>
   */
  public Class getInputPortType(int port) {
      Class retval=null;
      retval=agentDescription.getInputPortType(port);
      if (retval==null) retval=(Class)inputPortTypes.elementAt(port);//temporary measure until port types are stored in AgentDescription--use generatePortTypes to set this up beforehand
      return retval;
  }

  /**
   * Gets the layout for this macro.  The layout also contains the
   * layout for all macros inside this macro.
   *
   * @return The layout for this macro.
   */
  public Hashtable getLayoutObject() {
	 return layout;
  }
  
  /**
   * Sets the layout for this macro.
   *
   * @param l The new layout.
   */
  public void setLayout(Hashtable l) {
	 layout=l;
  }
  
  /**
   * Sets the type of this macro
   *
   * @param type The new type
   */
  public void setType(String type) {
	 agentDescription.setType(type);
  }
  

  /**
   * Return the port type for the specified output port
   *
   * @param port The output port number
   * @return the port type for output port <port>
   */
  public Class getOutputPortType(int port) {
      Class retval=null;
      retval=agentDescription.getOutputPortType(port);
      if (retval==null) retval=(Class)outputPortTypes.elementAt(port);//temporary measure
      return retval;
  }

  /**
   * Gets a PortLocation that contains the AgentDescription and port for
   * the agent that corresponds to input number <code>port</code>.
   *
   * @param port The input port to look at
   * @return A PortLocaiton with the AgentDescription and port number.
   */
  public PortLocation getInputAgent(int port) {
	 MacroDescription cd=(MacroDescription)agentDescription;
	 return cd.getInputAgent(port);
  }

  /**
   * Gets a PortLocation that contains the AgentDescription and port for
   * the agent that corresponds to output number <code>port</code>.
   *
   * @param port The output port to look at
   * @return A PortLocation with the AgentDescription and port number.
   */
  public PortLocation getOutputAgent(int port) {
	 MacroDescription cd=(MacroDescription)agentDescription;
	 return cd.getOutputAgent(port);
  }
  
  public String getPrefix() {
	 return prefix;
  }
  
  /**
   * Remove an output mapping for this macro.  Removes it from the
   * connection line vector, the outputMapping, and the macro's agent
   * description.
   *
   * @param cl The ConnectLine to remove
   */
  public void removeOutputMapping(ConnectLine cl) {
	 super.removeOutputMapping(cl);
  }
	 
  /**
   * Remove an input mapping for this macro.  Removes it from the
   * connection line vector, the inputMapping, the macro's agent description
   * and the real agents agentDescription
   *
   * @param cl The ConnectLine to remove
   */
    public void removeInputMapping(ConnectLine cl) {
	 super.removeInputMapping(cl);

	 // now we also have to remove the mapping on the real agent--now handled by MacroDescription, so the call to the superclass will work
	 /*int i = cl.getInputPort().getPort();
	 MacroDescription cd = (MacroDescription)agentDescription; 
	 PortLocation pl= cd.getAgentDescriptionFor(cd.getInputAgent(i),MacroDescription.INPUT);
	 AgentDescription ad = pl.getAgentDescription(); 
	 int agentInput = pl.getPortNumber(); 
	 ad.setInputMapping(null,agentInput); */
    }
  

  public void setOutputPortType(Class type,int port) {
	 outputPortTypes.setElementAt(type,port);
	 agentDescription.setOutputPortType(type,port);
  }

  public void setInputPortType(Class type,int port) {
	 inputPortTypes.setElementAt(type,port);
	 agentDescription.setOutputPortType(type,port);
  }
  
  public void setPegged(boolean b) {
	 pegged=b;
  }  

  public void setPrefix(String s) {
	 prefix=s;
  }

  /**
   * Set a mapping for one of this Macro's ports.  This sets the mapping
   * in the MacroDescription for this macro, and the AgentDescription for
   * the agent the port leads too.
   * <p>
   * Bug:  Doesn't map correctly if there are macros within macros.  This
   * only maps the high level macro, and the low level agent, not all the
   * macros inbetween.
   *
   * @param source The port on this Macro to map too
   * @param destination Where to map too.
   */
    public void setMapping(IOPort source,IOPort destination)
  {
	 //System.out.println("Mapping:"+destination+" to "+myPort);
	
	 MacroDescription cd = (MacroDescription)agentDescription;
	 if (source.isInput())	 {
	     cd.setInputMapping(new InputLocation(destination.getModule().getName(),destination.getPort(),source.getPort()),source.getPort());
	     /*
	 	//inputMapping.setMap(source.getPort(),destination);
		PortLocation pl= cd.getAgentDescriptionFor(cd.getInputAgent(source.getPort()),MacroDescription.INPUT);
		AgentDescription ad = pl.getAgentDescription(); 
		//System.out.println("real agent="+ad.toString(""));
		int agentInput = pl.getPortNumber(); 

		//System.out.println("before map="+agentDescription.toString(""));
		
		// The mapping bug:  macros within macros don't get mapped because
		// it only maps the high level macro, and the low level agent, not
		// all the macros inbetween

		  agentDescription.setInputMapping(new InputLocation(destination.getModule().getName(),destination.getPort(),source.getPort()),source.getPort()); 
		  // we have to set the mapping in the actual agent also
		  ad.setInputMapping(new InputLocation("."+pl.getAgentName()+destination.getModule().getName(),destination.getPort(),agentInput),agentInput); 
		  //System.out.println("after map="+agentDescription.toString(""));
		  //}
	     */
	 }
	 // if (myPort.charAt(0)=='O')
	 //	outputMapping.setMap(source.getPort(),destination);
  }
  

  public void mouseClicked(MouseEvent event) {
    // capture all the double click events. if a user double clicks on top
    // of a macro, show the internals of the macro.
    if (event.getClickCount()>=2) {
      ConfigAndLayout cal = new ConfigAndLayout();
      cal.layout=(Hashtable)layout;
      //cal.systemConfiguration = new SystemConfiguration();
      //cal.systemConfiguration.setType(agentDescription.getName());
      MacroDescription cd = (MacroDescription)agentDescription;

      cal.systemConfiguration = new SystemConfiguration(cd);

      //		System.out.println("Creating new model from:");
      //System.out.println(cal.systemConfiguration.toString());

      ManagerModel model = new ManagerModel(moduleManager,
					    prefix+"."+getName(),debugLevel);
      
      moduleManager.openView(model,prefix+"."+getName());
      model.openSystem(cal);

      // we need to rename the model so it has the full name
      model.setSystemName(prefix+"."+getName());
      // but if the user wants to save it, then default to just the macro
      // name
      model.setSystemFile(getName());

    } else {
      super.mouseClicked(event);
    }
  }
  
}
  

  

