package adaptive.modulemanager;

import java.util.*;
import adaptive.core.*;

public class BrowserModel extends Observable {

  /** The AgentLoader will give us classes given the string */
  private AgentLoader agentLoader;
  
  /** The local package Root for the agentLoader */
  private String localPackageRoot;
  
  /** The network package root for the agentloader */
  private String networkPackageRoot;
  
  /** Hashtables to store all the agent classes that we've loaded */
  private Hashtable localAvailableModules;
  private Hashtable networkAvailableModules;

  /** Is the network enabled */
  private boolean usingNetwork=false;

  /** Reference back to the adaptive.modulemanager.Module Manager that started this browser */
  private ModuleManager moduleManager;

  /** The module that is currently selected in the browser window */
  private adaptive.modulemanager.Module selectedModule;

  private int debugLevel;
  

  /** Takes a reference to the ModuleManager which created this browser.
    *
    * @param m The adaptive.modulemanager.Module Manager which created this browser
    * @param debugLevel The level of debuging information requested
    */
  public BrowserModel(ModuleManager m,int debugLevel) {
	 localAvailableModules = new Hashtable(50);
	 networkAvailableModules = new Hashtable();
	 moduleManager=m;
	 this.debugLevel=debugLevel;
  }


  public String getNetworkPackageRoot() {
	 return networkPackageRoot;
  }
  
  public String getLocalPackageRoot() {
	 return localPackageRoot;
  }
  
  public void setChanged() {
	 super.setChanged();
  }

  public synchronized String[] listDirectoriesIn(String directory,boolean isLocal) {
	 try {
		return agentLoader.listDirectoriesIn(directory,isLocal);
	 } catch (Exception e) {
		e.printStackTrace();
		return null;
	 }
  }
  
  public synchronized Vector getAgentsIn(String directory,boolean isLocal) {
	 Vector moduleList= new Vector(50);
	 try {
		
		String[] classList = agentLoader.listAgentsIn(directory,isLocal);
		
		String s=null;
		Class c;
		Agent a;
		Object o;
		Module t;
		
		for (int x=0;x <classList.length;x++) {
		  
		  s = new String(classList[x]);
		  
		  // First we check the hashtable to see if we've already
		  // retrieved the class for the specified agent
		  try {
			 if (isLocal) 
				t=(Module)localAvailableModules.get(s);
			 else
				t=(Module)networkAvailableModules.get(s);
			 
			 if (t==null) {
				
				a = agentLoader.newInstance(s,isLocal);
				// Make a new module with the class
				t = getModuleFromAgent(a);
				t.setOwner(this);
				// Need to instantiate the agent to get the port
				// numbers and types here.
				
				// Add it to the hashtable
				// The hashtable is indexed by both directory/name and
				// class name
				if (isLocal)
				  localAvailableModules.put(s,t);
				else
				  networkAvailableModules.put(s,t);
				setChanged();		
			 }
			 moduleList.addElement(t.clone());
			 // System.out.println("module:"+t);
		  } catch (InstantiationException e) {
			 System.err.println("Error loading agent:");
			 System.err.println(e);
		  } catch (Exception e) {
			 System.err.println("Error loading agent:");
			 System.err.println(e);
			 e.printStackTrace();
		  }
		}
	 } catch (Exception e) {
		e.printStackTrace();
	 }
	 notifyObservers();
	 return moduleList;
  }


  public synchronized Vector getMacrosIn(String directory,boolean isLocal) {
    Vector macroList= new Vector(20);
    
    try {
      String s;
      String[] allFiles=agentLoader.listFilesIn(directory,isLocal);
      ConfigAndLayout cal;
      Macro macro;
      SystemConfiguration ac;
      
      if (isLocal) 
	for (int x=0;x<allFiles.length;x++) {
	  s= new String(allFiles[x]);
	  if (s.endsWith(".cfg")) {
	    //System.out.println(directory+s);
	    try {
	      cal=agentLoader.getConfigAndLayout(directory+s,isLocal);
	      ac = cal.systemConfiguration;
	      if (isLocal) 
		macro=(Macro)localAvailableModules.get(ac.getType());
	      else
		macro=(Macro)networkAvailableModules.get(ac.getType());
	      
	      if (macro==null) {
		//FIX macro = new Macro(new ManagerModel(moduleManager,ac),ac.isPegged());
		
		
		macro= new Macro(moduleManager,ac,cal.layout,debugLevel);
		macro.generatePortTypes(this);
		
		
		if (isLocal)
		  localAvailableModules.put(ac.getType(),macro);
		else
		  networkAvailableModules.put(ac.getType(),macro);
		setChanged();
	      }
	      macroList.addElement(macro.clone());
	      
	    } catch (Exception e) {
	      e.printStackTrace();
	    } 
	  }
	}
    } catch (Exception e) {
      e.printStackTrace();
    }
    notifyObservers();
    return macroList;
  }
    /*********
     getModule
     gets a module, checking cached resources first
     input: agent type, source of the agent
     output:  Module for that agent
    ******/
  public synchronized adaptive.modulemanager.Module getModule(String s, int location) {
	 adaptive.modulemanager.Module m;
	 // Check to see if where the agent should be retrieved from
	 if (location==AgentDescription.LOCAL_FILE) {
	   m=(Module)localAvailableModules.get(s);
	 } 
	 else {
	   // if the agent comes from the agentserver, check to see if the
		  // network is enabled
	   if (usingNetwork) 
	     // if so, get the agent from the network list of modules
	     m=(Module)networkAvailableModules.get(s);
	   else {
	     m=(Module)localAvailableModules.get(s);
	   }
	 }
	 		
	 if (m==null) {
	   m=loadModule(s,location);
	 }
	 return m;
  }
  /*********
   * loadModule
   * load a module from disk or ModuleServer without checking the cached resources
   *********/
  public synchronized adaptive.modulemanager.Module loadModule(String s,int location){
    adaptive.modulemanager.Module m;
    try {
      if (location==AgentDescription.LOCAL_FILE) {
	m=getModuleFromAgent(agentLoader.newInstance(s,true));
	m.setOwner(this);
	localAvailableModules.put(s,m);
      } else {
	// if the agent comes from the agentserver, check to see if the
	// network is enabled
	if (usingNetwork) {
	  // if so, get the agent from the network list of modules
	  m=getModuleFromAgent(agentLoader.newInstance(s,false));
	  m.setOwner(this);
	  networkAvailableModules.put(s,m);
	} else {
	  // there is no network, warn the user, and try and get the
	  // agent from the local filesystem
	  moduleManager.getStatusWindow().setWarning(true);
	  moduleManager.getStatusWindow().
	    setMessage("WARNING: Network is not enabled!\n"+
		       "Attemping to locate "+s+"\n"+
		       "         on the local file system.");
	  // there is no network, warn the user, and try and get the
	  // agent from the local filesystem
	  m=getModuleFromAgent(agentLoader.newInstance(s,true));
	  m.setOwner(this);
	  localAvailableModules.put(s,m);
	}
      }			 
    } catch (ClassNotFoundException e) {
      moduleManager.getStatusWindow().setWarning(true);
      moduleManager.getStatusWindow().
	setMessage("Unable to locate agent class for:"+s);
      m=null;
    } catch (InstantiationException e) {
			 System.err.println("Error loading agent:");
			 System.err.println(e);
			 m=null;
    } catch (Exception e) {
      System.err.println("error getModule:"+s);
      e.printStackTrace();
      m=null;
    }		
    
    return m;
  }
  /*
   * refreshModule
   * loadModule without checking cache
   */
  public synchronized Module refreshModule(String name,int location){  
    return loadModule(name,location);
  }

 
  private adaptive.modulemanager.Module getModuleFromAgent(Agent a) {
	 if (a instanceof Driver) {
		return new CircleModule(a,debugLevel);
	 } else
		return new RectangleModule(a,debugLevel); 
  }  

  public boolean isUsingNetwork() {
	 //	 System.out.println("BM usingNetwork="+usingNetwork);
	 return usingNetwork;
  }
  
  /**
	* Set the current AgentLoader.
	*
	* @param al The agentLoader to set to.
	*/
  public void setAgentLoader(AgentLoader al,String lpr,String npr,boolean un) {
	 agentLoader=al;
	 localPackageRoot=lpr;
	 networkPackageRoot=npr;
	 usingNetwork=un;
  }
 
	 /**
	  * Selects the specified agent and highlights it so the user knows
	  * what he clicked.
	  */
	 public synchronized void selectAgent(Module m)
	 {
		// if the same agent was selected than deselect it
		if (selectedModule==m) {
		  //parent.selectAgent(null);
		  selectedModule=null;
		  return;
		}
		//else 
		  //parent.selectAgent(m);
		
		// If something else is selected, unselect it before selecting
		// the new module
		if (selectedModule!=null) {
		  selectedModule.setSelected(false);
		}
		
		selectedModule=m;
		if (m!=null) m.setSelected(true);
		
	 }
  
  public synchronized adaptive.modulemanager.Module getSelectedModule() {
	 //System.out.println("sel="+selectedModule);
	 return selectedModule;
  }
 
  public synchronized void setSelectedModule(Module m) {
	 selectedModule=m;
  }
  
}
