package adaptive.core.util;

/*****************************************************************************
 * <! Copyright 1999, Institute for Complex Engineered Systems,
 *                    Carnegie Mellon University
 *
 * PROJECT: Adaptable 
 *
 * FILE: Queue.java 
 * >
 *
 * A linked list FIFO queue useful for multithreaded programs.  The
 * queue methods are synchronized. The queue maintains a number of
 * preallocated node so as to speed inserts. There are three variations
 * of dequeueing. queueOut() returns the first element available or
 * null if no elments are available. queueOutBlocking() returns the
 * first element available and will wait until an element is available
 * before returning.  queueOutBlocking(long) waits until an element is
 * available or until the timeout expires, which ever happens
 * first.
 * 
 * @author Theodore Q Pham <A HREF="mailto:telamon@CMU.EDU">telamon@CMU.EDU</A>
 *         <br> </br>
 *
 * @version  1.00 4/16/99<br> </br>
 *
 * <!
 * REVISION HISTORY:
 *
 * $Log: Queue.java,v $
 * Revision 1.1  1999/04/20 04:36:11  telamon
 * improved Queue to do blocking and waking. moved Queue, QNode and LocalHost
 *
 * >
 ****************************************************************************/
public class Queue {
  public final static int kDEFAULTNODES = 5;
  private int numNodesAvail;
  private QNode freehead;
  private int size;
  private int numWaiting;
  private QNode head;
  private QNode tail;

  
  /***************************************************************************
   *
   * Creates a queue with the default number of preallocated nodes
   *
   **************************************************************************/
  public Queue() {
    this(kDEFAULTNODES);
  }
  
  /***************************************************************************
   *
   * Creates a queue with the specified number of preallocated nodes
   *
   * @param preAlloc the number of preallocated nodes
   *
   **************************************************************************/
  public Queue(int preAlloc) {
    freehead = null;
    numNodesAvail = preAlloc;
    for(int i=0;i<preAlloc;i++) {
      freehead = new QNode(null,freehead); 
    }
    head = null;
    tail = null;
    size = 0;
    numWaiting = 0;
  }

  
  /***************************************************************************
   *
   * Get the current size of the queue.
   *
   * @return the current size of the queue
   *
   **************************************************************************/
  public final int getSize() {
    return size;
  }

  /***************************************************************************
   *
   * Get the current size of the free list.
   *
   * @return the current size of the free list
   *
   **************************************************************************/
  public final int getFreeSize() {
    return numNodesAvail;
  }
  
  /***************************************************************************
   *
   * Return whether the queue is empty.
   *
   * @return <code>true</code> if the queue is empty;
   *         <code>false</code> otherwise
   *
   **************************************************************************/
  public final boolean isEmpty() {
    return size == 0;
  }
  
  /***************************************************************************
   *
   * Insert an object onto the queue.
   *
   * @param data the object to insert onto the queue
   *
   **************************************************************************/
  public final synchronized void queueIn(Object data) {
    QNode newNode;
    if(freehead != null) {
      newNode = freehead;
      freehead = freehead.next;
      numNodesAvail--;
    }
    else {
      newNode = new QNode();
    }
    newNode.theData = data;
    newNode.next = null;
    if(tail == null) {
      head = newNode;
    }
    else {
      tail.next = newNode;
    }
    tail = newNode;
    size++;
    if(numWaiting > 0 && size==1) this.notify();
  }

  
  /***************************************************************************
   *
   * Remove the first element on the queue, blocking if necessary.
   *
   * @return the first element on the queue
   *
   * @exception InterruptedException if the wait was interrupted.
   *
   **************************************************************************/
  public final synchronized Object queueOutBlocking()
    throws InterruptedException {
    QNode retNode;
    Object retVal;
    
    if(head == null) {
      numWaiting++;
      try {
        do {
          this.wait();
        } while(head == null);
      }
      catch(InterruptedException ie) {
        //we'll only get in here if the this.wait() got interrupted
        //so we no longer are a waiter.
        numWaiting--;
        throw ie;
      }
      numWaiting--;
    }
    
    retNode = head;
    if(head == tail) {
      tail = null;
    }
    head = head.next;
    size--;
    
    retVal = retNode.theData;
    
    retNode.theData = null;
    retNode.next = freehead;
    freehead = retNode;
    numNodesAvail++;
    if(numWaiting > 0 && size > 0) this.notify();
    return retVal; 
  }

  /***************************************************************************
   *
   * Remove the first element on the queue, blocking if necessary.
   * If the method blocks, it will only block for the specified timeout.
   *
   * @return the first element on the queue, or null if the timeout expires
   *
   * @exception InterruptedException if the wait was interrupted.
   * @exception IllegalArgumentException if the specified timeout is negative
   *
   **************************************************************************/
  public final synchronized Object queueOutBlocking(long timeout)
    throws InterruptedException, IllegalArgumentException {
    if(timeout < 0) throw new IllegalArgumentException();
    
    QNode retNode;
    Object retVal;
    long startTime;
    
    if(head == null) {
      numWaiting++;
      try {
        do {
          startTime = System.currentTimeMillis();
          this.wait(timeout);
          timeout -= (System.currentTimeMillis() - startTime);
          if(timeout <= 0 && head == null) {
            numWaiting--;
            return null;
          }
        } while(head == null);
      }
      catch(InterruptedException ie) {
        //we'll only get in here if the this.wait() got interrupted
        //so we no longer are a waiter.
        numWaiting--;
        throw ie;
      }
      numWaiting--;
    }
    
    retNode = head;
    if(head == tail) {
      tail = null;
    }
    head = head.next;
    size--;
    
    retVal = retNode.theData;
    
    retNode.theData = null;
    retNode.next = freehead;
    freehead = retNode;
    numNodesAvail++;
    if(numWaiting > 0 && size > 0) this.notify();
    return retVal; 
  }

  /***************************************************************************
   *
   * Returns the first element on the queue.
   *
   * @return the first element on the queue or null if there is none
   *
   **************************************************************************/
  public final synchronized Object queueOut() {
    QNode retNode;
    Object retVal;

    if(head != null) {
      retNode = head;
      if(head == tail) {
	tail = null;
      }
      head = head.next;
      size--;
    
      retVal = retNode.theData;
      
      retNode.theData = null;
      retNode.next = freehead;
      freehead = retNode;
      numNodesAvail++;

      return retVal; 
    }
    else {
      return null;
    }
  }

  
  /***************************************************************************
   *
   * Reduce the number of free preallocated nodes to at most
   * the specified amount
   *
   * @param howManyAvail the maximum number of free nodes
   *
   * @exception IllegalArgumentException if howManyAvail is negative
   *
   **************************************************************************/
  public final synchronized void shrinkFree(int howManyAvail)
    throws IllegalArgumentException {
    if(howManyAvail < 0) throw new IllegalArgumentException();
    
    while(howManyAvail < numNodesAvail) {
      freehead = freehead.next;
      numNodesAvail--;
    }
  }

  public synchronized void printQueue() {
    QNode currNode = head;
    System.out.println("Queue has "+size+" nodes used.");
    System.out.println("Start Used");
    while(currNode != null) {
      System.out.println(currNode.theData);
      currNode = currNode.next;
    }
    System.out.println("End Used");

    currNode = freehead;
    System.out.println("Queue has "+numNodesAvail+" nodes free.");
    System.out.println("Start Free");
    while(currNode != null) {
      System.out.println(currNode);
      currNode = currNode.next;
    }
    System.out.println("End Free");
  }
}


/***************************************************************************
 *
 * Support class for Queue representing one node.
 *
 **************************************************************************/
class QNode {
  Object theData;
  QNode next;

  
  /***************************************************************************
   *
   * Create a node with null next and data pointers.
   *
   **************************************************************************/
  QNode() {
    this(null,null);
  }

  
  /***************************************************************************
   *
   * Create a node with the specified data and next references.
   *
   * @param data the data reference
   * @param n the next reference
   *
   **************************************************************************/
  QNode(Object data, QNode n) {
    theData = data;
    next = n;
  }
}
