package adaptive.core.net;

import java.net.Socket;
import java.net.SocketImpl;
import java.net.SocketException;
import java.net.UnknownHostException;
import java.net.InetAddress;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.IOException;

/*****************************************************************************
 * <! Copyright 1999, Institute for Complex Engineered Systems,
 *                    Carnegie Mellon University
 *
 * PROJECT: Adaptable 
 *
 * FILE: ObjectSocket.java 
 * >
 *
 * Socket that provides buffered object input and output streams.
 * Don't forget to use flush when sending objects.
 * 
 * @author Yatish Patel
 * <A HREF="mailto:ypatel@andrew.cmu.edu">ypatel@andrew.cmu.edu</A>
 * <br> </br>
 *
 * @version  1.00 4/16/99<br> </br>
 *
 * <!
 * REVISION HISTORY:
 *
 * $Log: ObjectSocket.java,v $
 * Revision 1.3  1999/04/18 20:48:16  telamon
 * added more constructors
 *
 * >
 ****************************************************************************/
public class ObjectSocket extends Socket {
  ObjectInputStream objectInputStream = null;
  ObjectOutputStream objectOutputStream = null;
  boolean closed = false;
  
  /**
   * Creates an unconnected socket, with the system-default
   * type of SocketImpl.
   */
  protected ObjectSocket() {
    super();
  }

  /**
   * Creates an unconnected Socket with a user-specified SocketImpl. 
   * The impl parameter is an instance of a SocketImpl the subclass
   * wishes to use on the Socket.
   *
   * @param impl The socket implementation to use.
   *
   * @exception SocketException if an error in the the underlying
   *                            protocol occurs
   */
  protected ObjectSocket(SocketImpl impl) throws SocketException {
    super(impl);
  }

  /**
   * Creates a stream socket and connects it to the specified port
   * number on the named host. 
   *
   * @param host the host name.
   * @param port the port number.
   *
   * @exception UnknownHostException if the host name does not identify
   *                                 a valid host
   * @exception IOException if an I/O error occurs when creating the socket
   */
  public ObjectSocket(String host,
                int port) throws UnknownHostException, IOException {
    super(host,port);
  }

  /**
   * Creates a stream socket and connects it to the specified port
   * number at the specified IP address.
   *
   * @param address the IP address.
   * @param port the port number.
   *
   * @exception IOException if an I/O error occurs when creating the socket
   */
  public ObjectSocket(InetAddress address,
                      int port) throws IOException {
    super(address,port);
  }
  
  /**
   * Creates a socket and connects it to the specified remote host on
   * the specified remote port. The Socket will also bind() to the
   * local address and port supplied.
   *
   * @param host the name of the remote host
   * @param port the remote port
   * @param localAddr the local address the socket is bound to
   * @param port the local port the socket is bound to
   *
   * @exception IOException if an I/O error occurs when creating the socket
   */
  public ObjectSocket(String host,
                int port,
                InetAddress localAddr,
                int localPort) throws IOException {
    super(host,port,localAddr,localPort);
  }

  /**
   * Creates a socket and connects it to the specified remote address
   * on the specified remote port. The Socket will also bind() to the
   * local address and port supplied.
   *
   * @param address the remote address
   * @param port the remote port
   * @param localAddr the local address the socket is bound to
   * @param port the local port the socket is bound to
   *
   * @exception IOException if an I/O error occurs when creating the socket
   */
  public ObjectSocket(InetAddress address,
                      int port,
                      InetAddress localAddr,
                      int localPort) throws IOException {
    super(address,port,localAddr,localPort);
  }

  /**
   * Gets an ObjectInputStream for this socket.  The ObjectInputStream
   * sits on top of a BufferedInputStream.
   *
   * @return an ObjectInputStream for this socket.
   */
  public ObjectInputStream getObjectInputStream() throws IOException {
    if (objectInputStream==null) {
      BufferedInputStream bis =
        new BufferedInputStream(super.getInputStream());
      objectInputStream=new ObjectInputStream(bis);
    }
    return objectInputStream;
  }

  /**
   * Gets an ObjectOutputStream for this socket.  The ObjectOutputStream
   * sits on top of a BufferedOutputStream.
   *
   * @return an ObjectOutputStream for this socket.
   */
  public ObjectOutputStream getObjectOutputStream() throws IOException {
    if (objectOutputStream==null) {
      BufferedOutputStream bos =
        new BufferedOutputStream(super.getOutputStream());
      objectOutputStream=new ObjectOutputStream(bos);
      objectOutputStream.flush();
    }
    return objectOutputStream;
  }
  
  /**
   * Gets an ObjectInputStream for this socket.  The ObjectInputStream
   * sits on top of a BufferedInputStream.
   *
   * @return an ObjectInputStream for this socket.
   * @deprecated This methods returns a buffered ObjectInputStream, not
   *             a normal InputStream. Use getObjectInputStream() instead
   */
  public InputStream getInputStream() throws IOException {
    return getObjectInputStream();
  }
  
  /**
   * Gets an ObjectOutputStream for this socket.  The ObjectOutputStream
   * sits on top of a BufferedOutputStream.
   *
   * @return an ObjectOutputStream for this socket.
   * @deprecated This methods returns a buffered ObjectOutputStream, not
   *             a normal OutputStream. Use getObjectOutputStream() instead
   */
  public OutputStream getOutputStream() throws IOException {
    return getObjectOutputStream();
  }

  /**
   * Closes this socket. 
   *
   * @exception  IOException  if an I/O error occurs when closing this socket.
   * @since      JDK1.0
   */
  public synchronized void close() throws IOException {
    objectOutputStream = null;
    objectOutputStream = null;
    closed = true;
    super.close();
  }
  
  /**
   * Cleans up if the user forgets to close it.
   * @exception  IOException  if an I/O error occurs when closing this socket.
   */
  protected void finalize() throws IOException {
    if(!closed) {
      close();
    }
  }
}
