package adaptive.core.net;

import comms.core.CommsInputStream;
import comms.core.CommsOutputStream;
import comms.core.PhoneCodes;
import comms.core.EntryTypes;
import java.net.Socket;
import java.io.IOException;
import java.io.DataInputStream;
import java.util.Hashtable;
import java.util.Enumeration;


/*****************************************************************************
 * <! Copyright 1999, Institute for Complex Engineered Systems,
 *                    Carnegie Mellon University
 *
 * PROJECT: Adaptable 
 *
 * FILE: DirectoryInfo.java 
 * >
 *
 * A light weight utility class for accessing the CyberRAVE directory server.
 * It has three purposes:
 *
 *   1) Used in ModuleManager to retrieve the AgentServer location
 *      and a list of available Coordinator servers.
 *
 *   2) Used in Coordinators to register their location with the directory
 *      server. And to unregister themselves when shutting down.
 * 
 *   3) Used by AgentServer to register their location with the directory
 *      server. And to unregister themselves when shutting down.
 * 
 * @author Theodore Q Pham <A HREF="mailto:telamon@CMU.EDU">telamon@CMU.EDU</A>
 *         <br> </br>
 *
 * @version  1.00 10/01/99<br> </br>
 *
 * <!
 * REVISION HISTORY:
 *
 * $Log: DirectoryInfo.java,v $
 * Revision 1.1.2.2  2000/05/22 23:45:11  telamon
 * synched with the change comms classes
 *
 * Revision 1.1.2.1  1999/10/06 01:11:30  telamon
 * Initial checking of new comms stuff
 *
 * >
 ****************************************************************************/
public class DirectoryInfo implements PhoneCodes {
  public static String AGENTSERVER = "AgentServer";
  
  private String dirHost;
  private int dirPort;
  private String myName;

  private Hashtable coordinators;
  private PhoneEntry agentServerEntry;
  
  private Socket socket;
  private CommsInputStream cis;
  private CommsOutputStream cos;
  
  /** Records the directory location and port */
  public DirectoryInfo(String dirHost, int dirPort, String myName) {
    this.dirHost = dirHost;
    this.dirPort = dirPort;
    this.myName = myName;
  }

  /** retrieves the AgentServer and Coordinators */ 
  public void retrieveDirInfo() throws IOException {
    connectToDirectory();

    if( coordinators == null ) {
      coordinators = new Hashtable();
    }
    else {
      coordinators.clear();
    }
    agentServerEntry = null;
    
    cos.writeMessage(AGENTSERVER, UNKNOWN, UNKNOWN);
    cos.writeMessage(MATCH_EX, UNKNOWN, UNKNOWN);
    cos.flush();
    String[] resp = cis.readMessage();
    String response = resp[0];
    if( !FAIL.equals(response) ) {  
      try {
        agentServerEntry = new PhoneEntry(response);
      }
      catch(IllegalArgumentException iae) {
        throw new IOException(iae.getMessage());
      }
    }
    
    cos.writeMessage( Integer.toString(EntryTypes.ADAPTABLE_COORDINATOR), UNKNOWN, UNKNOWN );
    cos.writeMessage( MATCH_TYPE_EXTENDED, UNKNOWN, UNKNOWN );
    cos.flush();
    PhoneEntry currEntry;
    resp = cis.readMessage();
    response = resp[0];
    //System.out.println("RES: "+response);
    
    if( !FAIL.equals(response) ) {
      for(;!DONE.equals(response);
	  resp = cis.readMessage(), response = resp[0]) {
        try {
          currEntry = new PhoneEntry(response);
        }
        catch(IllegalArgumentException iae) {
          throw new IOException(iae.getMessage());
        }
        coordinators.put(currEntry.getName(),currEntry);
      }
    }
    disconnectFromDirectory();
  }
  
  /** put this Entity into the directory servers listings */
  public void registerWithDirectory(int localPort, int type)
    throws IOException {
    connectToDirectory();
    cos.writeMessage(myName, UNKNOWN, UNKNOWN);
    cos.writeMessage(REGISTER_EX, UNKNOWN, UNKNOWN);
    cos.writeMessage(LocalHost.getFullLocalHostName()+" "+0+" "+
                     type+" "+localPort, UNKNOWN, UNKNOWN);
    cos.flush();
    disconnectFromDirectory();
  }

  /** remove this Entity from the directory servers listings */
  public void unregisterWithDirectory() throws IOException {
    connectToDirectory();
    cos.writeMessage(myName, UNKNOWN, UNKNOWN);
    cos.writeMessage(REMOVE, UNKNOWN, UNKNOWN);
    cos.flush();
    disconnectFromDirectory();
  }

  private void connectToDirectory() throws IOException {
    socket = new Socket(dirHost, dirPort);
    cis = new CommsInputStream(socket.getInputStream());
    cos = new CommsOutputStream(socket.getOutputStream());
    cos.flush();
  }

  private void disconnectFromDirectory() {
    try {
      cis.close();
    }
    catch(IOException ioe) {
    }
    try {
      cos.close();
    }
    catch(IOException ioe) {
    }
    try {
      socket.close();
    }
    catch(IOException ioe) {
    }
  }

  /** returns null if there are no coordinators or the DirectoryInfo
      hasn't been retrieved yet */
  public Enumeration getCoordinators() {
    if( coordinators == null ) {
      return null;
    }
    if( coordinators.isEmpty() ) {
      return null;
    }
    return coordinators.keys();
  }

  /** returns null if there is no coordinator by that name or the DirectoryInfo
      hasn't been retrieved yet */
  public PhoneEntry getCoordinatorEntry(String coordinatorName) {
    if( coordinators == null ) {
      return null;
    }
    PhoneEntry matched = (PhoneEntry) coordinators.get(coordinatorName);
    if(matched == null) {
      return null;
    }
    return matched;
  }

  /** returns null if there is no AgentServer or the DirectoryInfo
      hasn't been retrieved yet */
  public PhoneEntry getAgentServerEntry() {
    return agentServerEntry;
  }

  public static void main(String[] args) {
    try {
      if(args.length < 5) {
        System.exit(0);
      }
      
      DirectoryInfo dirInfo;
      /* dir host, dir port, my name */
      dirInfo = new DirectoryInfo(args[2],Integer.parseInt(args[3]),args[0]);

      if("register".equals(args[4])) {
        System.out.println("Registering with Directory Server");
        /* put me in the directory, with type set to argument 1 */
        dirInfo.registerWithDirectory(5400, Integer.parseInt(args[1]) );
        DataInputStream ois = new DataInputStream( System.in );
        System.out.println("Registration done");
        System.out.println(
          "Hit return to quit and unregister from the Direcotry Server");
        ois.readLine();
        dirInfo.unregisterWithDirectory();
        System.out.println("Unregistered. Closing down");
      }
      else if( "list".equals(args[4]) ) {
        System.out.println("Retrieving info from Directory Server\n");
        dirInfo.retrieveDirInfo();

        PhoneEntry currEntry = dirInfo.getAgentServerEntry();

        if(currEntry != null) {
          System.out.println("AgentServer: ");
          System.out.println(currEntry);
        }
        else {
          System.out.println("WARNING: No AgentServer found");
        }

        Enumeration enum = dirInfo.getCoordinators();
        String coordName;
        
        if(enum != null) {
          System.out.println("\nCoordinators: ");
          while( enum.hasMoreElements() ) {
            coordName = (String) enum.nextElement();
            System.out.println(dirInfo.getCoordinatorEntry(coordName));
          }
        }
        else {
          System.out.println("\nWARNING: No Coordinators found");
        }
        System.out.println("\nDone. Closing down");
      }
    }
    catch(Exception e) {
      e.printStackTrace();
    }
  }
}
