package adaptive.core;

import java.io.Serializable;
import java.util.Hashtable;
import java.util.Enumeration;
import java.util.Vector;
/*****************************************************************************
 * <! Copyright 2000, Institute for Complex Engineered Systems,
 *                    Carnegie Mellon University
 *
 * PROJECT: Adaptable 
 *
 * FILE: SystemConfiguration.java
 * >
 * 
 * Stores all the configuration information for a system. 
 *
 * Structure: There is one hashtable for each host where agents will be 
 * running.  Each host hashtable contains the AgentDescriptions for all 
 * the Agents running on that host.  
 *
 * When Macros are added the MacroDescription for the macro is put on
 * every host where it resides.  The MacroDescription contains all the
 * information about what's inside the macro regardless of where the 
 * individual pieces are supposed to run.
 *
 * When something is mapped to a macro, the InputLocation contains the
 * name of the macro and its port number, not the name of the specific
 * piece inside the macro where the mapping leads.
 *
 *
 * <!
 * REVISION HISTORY:
 *
 * $Log: SystemConfiguration.java,v $
 * Revision 1.3.2.7  2000/06/05 05:48:41  jrj
 * more macro fixes.  Wrote a couple more functions to help once we start start storing port types in the AgentDescription and to keep the port information in sync with the PBAgent definition.
 *
 * Revision 1.3.2.6  2000/05/20 19:11:18  jrj
 * fixed a couple more macro bugs.  Took preliminary steps toward making
 * fully recursive macros in MacroDescription.  Have not yet made changes
 * that would damage save file.
 *
 * Revision 1.3.2.5  2000/05/15 22:39:39  jrj
 * Made fixes to macros so that they work.  Changed the executing macro
 * so that it is recursive, can be manipulated as a whole, and its ports
 * can be asked for directly as if it were any other BasicPortAgent.
 * Made several other changes to simplify and assist these changes.
 * Finally purged the last remaining bit of things named clump.
 *
 ****************************************************************************/

public class SystemConfiguration implements Serializable {

  static final long serialVersionUID = 268711020416024183L;
  
  private String type;
 
 /** author of the system */
  private String author;
  /** system version */
  private String version;
  /** description of system */
  private String description;
  /** Date of creation */
  private String date;
  /** Things that were changed since last revision */
  private String lastRevision;

  /** All the agents hashed by hostname */
  private Hashtable agentList; /** hashtable of hashtables */

  /** mainly used for macros */
  //note(by jrj) these should be changed to Vectors, as they make more sense that way, but I'm holding off to preserve the save file format
  private Hashtable inputPortNames;
  private Hashtable outputPortNames;

  /** The list of what agent/input corresponds with what free input in the
	  system , used for macros */
  //jrj--same comment as the names hashtables
  private Hashtable inputAgentList;
  /** Same with the outputs */
  private Hashtable outputAgentList;

  private boolean pegged;


  protected Vector allAgents; //hack to make multi-host macros work correctly--jrj 5/9/2000
    protected boolean externalPortsPrepared=false;//variable used when converting from a SystemConfiguration to a MacroDescription to tell whether the ports have already been set up.  This is used so that once the port have been set up, we can actually remove output ports and not have them reappear.

  /**
   * Construct a new empty SystemConfiguration 
   */ 
  public SystemConfiguration() {
      System.out.println("Creating empty systemconfiguration");
	 type="";
	 author="";
	 version="1.0";
	 description="";
	 date="";
	 lastRevision="";
	 agentList = new Hashtable();
	 inputPortNames = new Hashtable();
	 outputPortNames = new Hashtable();
	 inputAgentList = new Hashtable();
	 outputAgentList = new Hashtable();
	 allAgents=new Vector();
	 externalPortsPrepared=false;
  }

  /**********
   * Construct a new SystemConfiguration with a given type.
   *
   * @param type The type of this SystemConfiguration
   *********/
  public SystemConfiguration(String type) {
	 this();
	 System.out.println("Created typed systemconfiguration");
	 this.type=type;
  }
  
  /**********
   * Constructs a SystemConfiguration out of the data from a MacroDescription.
   * Once completed, the SystemConfiguration is independent from the
   * MacroDescription.  Everything is cloned to obtain new references.
   *
   * @param cd The MacroDescription to use
   *********/
  public SystemConfiguration(MacroDescription cd) {
	 this();
	 type = cd.getName();
	 
	 Enumeration e =cd.getAgents();
	 AgentDescription ad;  
	 System.out.println("Creating configuration out of "+cd.getName());
	 while (e.hasMoreElements()) {  
	     ad = (AgentDescription)e.nextElement();		   
	     ad = (AgentDescription)ad.clone();  
	     //  System.out.println("Looking at "+ad.getName()); 

	     Enumeration locations=ad.getAllRunLocations();  
	     String loc;  
	     while (locations.hasMoreElements()){  
		 loc=(String)locations.nextElement();    
		 addAgent(loc,ad);  
	     }  
	 }

	 for (int x=0;x<cd.getNumInputs();x++) {
	   addInputPort(x,cd.getInputPortName(x),(PortLocation)cd.getInputLocation(x).clone());
	 }

	 for (int x=0;x<cd.getNumOutputs();x++) {
	   addOutputPort(x,cd.getOutputPortName(x),(PortLocation)cd.getOutputLocation(x).clone());
	 }
	 this.externalPortsPrepared=true;
	 System.out.println("preparedPorts="+this.externalPortsPrepared);
  }
  
  /**
  * Add an agent to the Configuration.
  *
  * @param hostname The hostname where the agent will run
  * @param ad The AgentDescription for the agent
  */
  public void addAgent(String hostname,AgentDescription ad) {
    Hashtable agents;
    AgentDescription agent=(AgentDescription)ad.clone();
    agent.setRunLocation(hostname); //added jrj--3/20/2000, since it makes the most sense to me to have this here, rather than have what calls addAgent call this as well
    // System.out.println("Adding "+agent.getName()+" to run on "+hostname);
    if (agentList.containsKey(hostname)){
      agents=(Hashtable)agentList.get(hostname);
    }
    else{
      agents=new Hashtable();
      agentList.put(hostname,agents);
    }
    if (!allAgents.contains(ad)){//hack to make multi-host macros work(want it to be listed under only one host so it doesn't appear multiple times in ModuleManager.  It will still execute correctly because I forced the SystemExecutor to check every agent.
      agents.put(agent.getName(),agent);       
      //      System.out.println("Added "+agent.getName());
      allAgents.addElement(ad);
    }
  }
  
  /**
   * Return an enumeration of all the hostnames in this configuration.
   *
   * @return an enumeration of all the hostnames in this configuration.
   */
  public Enumeration getHostNames() {
	 return agentList.keys();
  }

  /**
    * Return the names of all the agents in this configuration.
    *
    * @return an enumeration of all the agents in this configuration.
    */
  public Enumeration getAgentNames(String hostName) {
	 return ((Hashtable)agentList.get(hostName)).keys();
  }
  
  /**
    * Get an AgentDescription for a specific agent.
    *
    * @param hostName the location where the agent is running
    * @param agentName the name of the agent
    * @return an AgentDescription for the agent
    */
  public AgentDescription getAgent(String hostName,String agentName) {
	 return (AgentDescription)((Hashtable)agentList.get(hostName)).get(agentName);	 
  }

  /**
   * Get an enumeration of all the agents on a specifc host
   *
   * @param hostName The hostName to query
   * @return an enumeration of all the agents on a specific host
   */
  public Enumeration getAgents(String hostName) {
	 return ((Hashtable)agentList.get(hostName)).elements();
  }
    public Enumeration getAllAgents(){//this should be removed once I find a better way to handle multi-host macros, so don't use it.  Iterate through the hosts and agents on each instead.
    return allAgents.elements();
  }
  
  
  /////////////////////
  //methods to add/remove/alter inputs and outputs to the system
  /////////////////////
    /*********
     * havePortsBeenExported()
     * returns true if the unmapped ports have already been set to be external to the system
     *********/
    public boolean haveExternalPortsBeenPrepared(){
	System.out.println("external ports already prepared?"+this.externalPortsPrepared);
	return this.externalPortsPrepared;
    }
    public void setPortsPrepared(boolean b){
	this.externalPortsPrepared=b;
    }
  /**
    * Get the number of (unmapped)inputs to this configuration.
    *
    * @return the number of inputs to this configuration.
    */
  public int getNumInputs() {
	 return inputAgentList.size();
  }
  
  /**
    * Get the number of outputs that have been designated as system outputs.
    *
    * @return the number of outputs that have been designated as system outputs
    */
  public int getNumOutputs() {
	 return outputAgentList.size();
  }
  /***************
  addInputAgent
  adds a new input agent with the specified name and source(from an agent internal to the system)
  ****************/
  public boolean addInputPort(int portNum,String name,PortLocation pl){
    setInputAgent(pl,portNum);
    setInputPortName(name,portNum);
    return true;
  }
  /***************
  addOutputAgent
  adds a new output agent with the specified name and source(from an agent internal to the system)
  ****************/
  public boolean addOutputPort(int portNum,String name,PortLocation pl){
    setOutputAgent(pl,portNum);
    setOutputPortName(name,portNum);
    return true;
  }
  public boolean swapInputPorts(int portNum1,int portNum2){
    boolean retval=false;
    String tempName;
    PortLocation tempPl;
    if (portNum1>=0 && portNum1<getNumInputs() && portNum2>=0 && portNum2<getNumInputs()){
      tempName=getInputPortName(portNum1);
      tempPl=getInputAgent(portNum1);
      //      System.out.println(tempName+" "+tempPl);
      addInputPort(portNum1,getInputPortName(portNum2),getInputAgent(portNum2));
      addInputPort(portNum2,tempName,tempPl);
      retval=true;
    }
    return retval;
  } 
  public boolean swapOutputPorts(int portNum1,int portNum2){
    boolean retval=false;
    String tempName;
    PortLocation tempPl;
    if (portNum1>=0 && portNum1<getNumOutputs() && portNum2>=0 && portNum2<getNumOutputs()){
      tempName=getOutputPortName(portNum1);
      tempPl=getOutputAgent(portNum1);
      //System.out.println(tempName+" "+tempPl);
      addOutputPort(portNum1,getOutputPortName(portNum2),getOutputAgent(portNum2));
      addOutputPort(portNum2,tempName,tempPl);
      retval=true;
    }
    return retval;
  }
  /**
    * Set the specified system input to point to the specified agent/port
    *
    * @param pl The agent/port to map to
    * @param port the System input port to map to pl
    */
  public void setInputAgent(PortLocation pl,int port) {
      if (inputAgentList==null) inputAgentList=new Hashtable();
    if (pl!=null){
      inputAgentList.put(new Integer(port),pl);	 
    }
  }
  
  /**
    * Set the specified system input to point to the specified agent/port
    *
    * @param pl The agent/port to map to
    * @param port the system input port to map to pl
    */
  public void setOutputAgent(PortLocation pl,int port) {
      if (outputAgentList==null)outputAgentList=new Hashtable();
    if (pl!=null){
      outputAgentList.put(new Integer(port),pl);
    }
  }
	 
  /**
    * Get the name of a system input port
    *
    * @param port The port number to get the name of
    * @return the name of a system input port
    */
  public String getInputPortName(int port) {
    //if (inputPortNames!=null)
    return (String)inputPortNames.get(new Integer(port));
    // else return null;
  }
  /**
    * Get the name of a system output port
    * 
    * @param port The port number to get the name of
    * @return the name of a system output port
    */
  public String getOutputPortName(int port) {
    //if (outputPortNames!=null)
    return (String)outputPortNames.get(new Integer(port));
    //else return null;
  }
  public boolean setInputPortName(String n,int port) {
    boolean retval=false;
	 // for compatibility with old containers...
	 //if (inputPortNames==null) inputPortNames=new Hashtable();
    if (n!=null) {
      inputPortNames.put(new Integer(port),n);
      retval=true;
    }
    return retval;
  }
  public boolean setOutputPortName(String n,int port) {
    boolean retval=false;
	 //if (outputPortNames==null) outputPortNames=new Hashtable();
    if (n!=null) {
      outputPortNames.put(new Integer(port),n);
      retval=true;
    }
    return retval;
  }  


  // Why are there private accessor methods???? 
  private Hashtable getInputPortNames() {
	 return inputPortNames;
  }
  
  private Hashtable getOutputPortNames() {
	 return outputPortNames;
  }

  private void setInputPortNames(Hashtable in) {
	 inputPortNames=in;
  }
  
  private void setOutputPortNames(Hashtable on) {
	 outputPortNames=on;
  }

  
  /**
    * Removes an system input agent from this configuration.  The agent
    * will not be removed from the system.  This will only remove the
    * references to the agent that were in the input agent list.
    *
    * @param name the Name of the agent to remove 
    * deprecated cause this doesn't really do what we want--jrj--use deleteInputPort()
    */
  public void deleteInputAgent(String name) {
    Hashtable h = new Hashtable();
    PortLocation pl;
    int newcount=0;
    for (int x=0;x<getNumInputs();x++) {		
      pl = (PortLocation)inputAgentList.get(new Integer(x));
      if (!pl.getAgentName().equals(name)) {
	h.put(new Integer(newcount++),pl);
      }
    }
    inputAgentList=h;
  }
  /**
    * Removes an system output agent from this configuration.  The agent
    * will not be removed from the system.  This will only remove the
    * references to the agent that were in the output agent list.
    *
    * @param name the Name of the agent to remove
    * deprecated cause this doesn't really do what we want--jrj--use deleteOutputPort
    */
  public void deleteOutputAgent(String name) {
	 //System.out.println("deleting:"+name);
	 //System.out.println("from:"+outputAgentList.toString());
	 Hashtable h = new Hashtable();
	 PortLocation pl;
	 int newcount=0;
	 for (int x=0;x<getNumOutputs();x++) {		
		pl = (PortLocation)outputAgentList.get(new Integer(x));
		if (!pl.getAgentName().equals(name)) {
		  h.put(new Integer(newcount++),pl);
		}
	 }
	 outputAgentList=h;
	 //System.out.println("result:"+outputAgentList);
  }
  /*******
   Removes this port and moves any following ports up one in the list
   I have to do this in a dumb inefficient way because of the stupid way they are stored.--jrj
  ****/
  public void deleteOutputPort(int portNum){
    if (portNum>=0){
      int size=getNumOutputs();
      String name;
      PortLocation pl;
      for (int i=portNum;i<size-1;i++){
	addOutputPort(i,getOutputPortName(i+1),getOutputAgent(i+1));
      }
      outputAgentList.remove(new Integer(size-1));
      outputPortNames.remove(new Integer(size-1));
    }
  }  
  /*******
   Removes this port and moves any following ports up one in the list
   I have to do this in a dumb inefficient way because of the stupid way they are stored.--jrj
  ****/
  public void deleteInputPort(int portNum){
    if (portNum>=0){
      int size=getNumInputs();
      String name;
      PortLocation pl;
      for (int i=portNum;i<size-1;i++){
	addInputPort(i,getInputPortName(i+1),getInputAgent(i+1));
      }
      inputAgentList.remove(new Integer(size-1));
      inputPortNames.remove(new Integer(size-1));
    }
  }
  /**
    * Rename any references to the given agent in the input list.  Cycles
    * through the list of input Agents and renames a specific agent.
    *
    * @param oldName the previous name of the agent.
    * @param newName the new name for the agent.
    */
  public void renameInputAgent(String oldName,String newName) {
    Enumeration e = inputAgentList.elements();
    PortLocation pl;
    while (e.hasMoreElements()) {
      pl = (PortLocation)e.nextElement();
      if (pl.getAgentName().equals(oldName)) pl.setAgentName(newName);
    }
    
  }

  /**
    * Rename any references to the given agent in the output list.  Cycles
    * through the list of input Agents and renames a specific agent.
    *
    * @param oldName the previous name of the agent.
    * @param newName the new name for the agent.
    */
  public void renameOutputAgent(String oldName,String newName) {
    Enumeration e = outputAgentList.elements();
    PortLocation pl;
    while (e.hasMoreElements()) {
      pl = (PortLocation)e.nextElement();
      if (pl.getAgentName().equals(oldName)) pl.setAgentName(newName);
    }
    
  }
  
  /**
    * Gets the information for the system input port specified
    *
    * @param port The system input number to query
    * @return the information for the system input port
    */
  public PortLocation getInputAgent(int port) {
    return (PortLocation)inputAgentList.get(new Integer(port));
  }
  
  /** 
    * Gets the information for the system output port specified
    *
    * @param port The system output number to query
    * @return the information for the system output port
    */
  public PortLocation getOutputAgent(int port) {
    return (PortLocation)outputAgentList.get(new Integer(port));
  }
  
  /**
    * Get the number of system input ports
    *
    * @return the number of system input ports
    */
  public int getIALsize() {
    return inputAgentList.size();
  }
  
  /**
    * Get the number of system output ports
    * 
    * @return the number of system output ports
    */
  public int getOALsize() {
    return outputAgentList.size();
  }
  
  /**
    * Print all the agents in the system input list
    */
  public void printInputAgentList() {
    Enumeration e = inputAgentList.keys();
    while (e.hasMoreElements()) {
      Object o = e.nextElement();
      System.out.println(o+" "+inputAgentList.get(o));
    }
    
  }
  
  
  /**
    * Checks to see if the specified agent/port is in the input agent list
    * for this SystemConfiguration.  If so, this sets the portnumber in the
    * PortLocaiton passed in to the port number it maps to and returns true.
    *
    * @param pl The agent/port to look for.
    *
    * @return true if it's in the list; false if not.
    */
  public boolean ialContains(PortLocation pl) {	 
    Enumeration e=inputAgentList.keys();
    Integer i;
    while (e.hasMoreElements()) {
      i = (Integer)e.nextElement();
      if (((PortLocation)inputAgentList.get(i)).equals(pl)) {
	pl.setPortNumber(i.intValue());
	return true;
      }
    }
    return false;
  }
  
  /**
    * Checks to see if the specified agent/port is in the output agent list
    * for this SystemConfiguration.  If so, this sets the portnumber in the
    * PortLocaiton passed in to the port number it maps to and returns true.
    *
    * @param pl The agent/port to look for.
    *
    * @return true if it's in the list; false if not.
    */
  public boolean oalContains(PortLocation pl) {
    Enumeration e=outputAgentList.keys();
    Integer i;
    while (e.hasMoreElements()) {
      i = (Integer)e.nextElement();
      if (((PortLocation)outputAgentList.get(i)).equals(pl)) { 
	pl.setPortNumber(i.intValue());
	return true;
      }
    }
    return false;
  }

  //Misc methods

  /**
    * Returns whether or not the configuration is pegged.
    *
    * @return whether or not the configuratino is pegged.
    * I think this should be deprecated--jrj 5/15/2000
    */
  public boolean isPegged() {
    return pegged;
  }
  
  /**
    * Set whether the configuration is pegged.
    *
    * @param value Is the configuration pegged
    */
  public void setPegged(boolean value) {
    pegged=value;
  }
  
  public String getAuthor() {
    return author;
  }
  
  public String getDescription() {
    return description;
  }
  
  public String getVersion() {
    return version;
  }
  
  public String getDate() {
    return date;
  }
  
  public String getLastRevision() {
    return lastRevision;
  }
  
  public String getType() {
    return type;
  }
  
  public void setAuthor(String a) {
    author=a;
  }
  public void setDate(String d) {
    date=d;
  }
  
  public void setVersion(String v) {
    version=v;
  }
  
  public void setDescription(String d) {
    description=d;
  }
  
  public void setLastRevision(String lr) {
    lastRevision=lr;
  }  
  public void setType(String sysType) {
    type = sysType;
  }

  public Object clone() {
      System.out.println("Cloning systemconfiguration");
	 SystemConfiguration sc = new SystemConfiguration();
	 sc.type=type;
	 sc.author=author;
	 sc.version=version;
	 sc.description=description;
	 sc.date=date;
	 sc.lastRevision=lastRevision;
	 
	 sc.agentList=new Hashtable();
	 sc.inputPortNames=new Hashtable();
	 sc.outputPortNames=new Hashtable();
	 sc.inputAgentList = new Hashtable();
	 sc.outputAgentList = new Hashtable();
	 Enumeration e = agentList.keys();
	 Enumeration f;
	 Hashtable h;
	 Hashtable newH;
	 AgentDescription ad;
	 String key;

	 while (e.hasMoreElements()) {
		key = (String)e.nextElement();
		h = (Hashtable)agentList.get(key);
		newH = new Hashtable(h.size());
		f = h.elements();
		while (f.hasMoreElements()) {
		  ad=(AgentDescription)f.nextElement();
		  newH.put(ad.getName(),ad.clone());
		}
		sc.agentList.put(key,newH);
	 }

	 try {

		// inputPortName and outputPortnames are hashtables of Strings
		// a shallow copy will work.
		sc.inputPortNames=(Hashtable)inputPortNames.clone();
		sc.outputPortNames=(Hashtable)outputPortNames.clone();
		/**		e = inputPortNames.keys();
		String name;
		while (e.hasMoreElements()) {		
		  key=(String)e.nextElement();
		  name=(String)inputPortNames.get(key);
		  sc.inputPortNames.put(key,name);
		}
		
		e = outputPortNames.keys();
		while (e.hasMoreElements()) {
		  key=(String)e.nextElement();
		  name=(String)outputPortNames.get(key);
		  sc.outputPortNames.put(key,name);
		}*/


		PortLocation pl;
		Integer key2;  
		e = inputAgentList.keys();
		while (e.hasMoreElements()) {
		  key2=(Integer)e.nextElement();  //fixed cast error jrj 7/8
		  pl = (PortLocation)inputAgentList.get(key2);
		  sc.inputAgentList.put(key2,pl.clone());
		}
		
		e = outputAgentList.keys();
		while (e.hasMoreElements()) {
		  key2=(Integer)e.nextElement();
		  pl = (PortLocation)outputAgentList.get(key2);
		  sc.outputAgentList.put(key2,pl.clone());
		}
	 } catch (Exception ex) {
		System.err.print("Exception occured while trying to clone");
		System.err.println(" the SystemConfiguration!");
		ex.printStackTrace();
	 }
	 sc.externalPortsPrepared=this.externalPortsPrepared;
	 System.out.println("copied externalPortsPrepared"+sc.externalPortsPrepared);
	 return sc;
	 
  }

  public String toString() {
	 StringBuffer sb= new StringBuffer("SystemConfiguration: "+type+"\n");
	 Enumeration hostNames=agentList.keys();
	 String hostName;
	 Enumeration agents;
	 
	 while (hostNames.hasMoreElements()) {
		hostName=(String)hostNames.nextElement();
		sb.append("Host: "+hostName+"\n");
		agents=((Hashtable)agentList.get(hostName)).elements();
		while (agents.hasMoreElements()) {
		  sb.append(((AgentDescription)agents.nextElement()).toString(" ")+"\n");
		}
	 }
	 
	 return sb.toString();
  }

  public String toString(String s) {
	 return toString();
  }
  



}
