package adaptive.core;

import java.util.*;
import comms.core.*;

public class SocketTable {
  private static SocketTable singletonSocketTable = null;

  /** hashed by source name string. stores the message 
      queue that will be used for that source. the Thread
      responsible for that queue, and a hashtable of message types*/
  private Hashtable sourceTable;
  /** hashed by pbsagent references. stores a list of messages and sources
      that the agent is registered for. used to unregister the agent later*/
  private Hashtable pbsTable;
  /** the commLink that the messages are coming in over */
  private Communications commLink;

  
  /** Constructor */
  //public SocketTable(Communications c) {
  private SocketTable(Communications c) {
    super();
    commLink = c;
    sourceTable = new Hashtable();
    pbsTable = new Hashtable();
  }

  public static SocketTable getSingletonInstance(Communications c) {
    if (singletonSocketTable == null) {
      singletonSocketTable = new SocketTable(c);
    }
    return singletonSocketTable;
  }

  /** PBSAgent's will indirectly call this method to signal that they
      would like messType messages from the source entity */
  public synchronized void registerForMessage(String source, String messType, 
					      PBSAgent pbsa) {
    PBSEntry pbse = new PBSEntry(source,messType);
    
    //Start removal bookkeeping
    Vector pbsTV = (Vector) pbsTable.get(pbsa);
    if(pbsTV == null) {      
      //this is the first message this agent has registered for
      //so allocate the bookkeeping for this agent.
      pbsTV = new Vector();
      pbsTable.put(pbsa, pbsTV);
    }
    //put the registration entry into the removal bookkeeping structure
    pbsTV.addElement(pbse);
    //end removal bookkeeping

    //Start delivery bookkeeping
    SourceTableEntry ste = (SourceTableEntry) sourceTable.get(source);
    if(ste == null) {
      //this is the first time messages are being requested from this
      //source.
      MessageQueue mq = new MessageQueue();
      //tell the commlink to put messages from source into mq
      commLink.setMessageQueue(source,mq);
      ste = new SourceTableEntry(mq, commLink, sourceTable);
      sourceTable.put(source, ste);
      //start the thread to process this new message queue
      ste.startThread();
    }
    ste.add(messType, pbsa);
    //end delivery bookkeeping
  }

  /** call this method to shrink the trimToSize all vectors
      associated with the SocketTable */
  public void trimToSize() {
    //go through all vectors and set their capacities equal to their size
    Enumeration enum;
    PBSAgent pbsaKey;
    Vector curVec;
    enum = pbsTable.keys();
    
    //trim all the removal bookkeeping vectors
    while(enum.hasMoreElements()) {
      pbsaKey = (PBSAgent) enum.nextElement();
      curVec = (Vector) pbsTable.get(pbsaKey);
      curVec.trimToSize();
    }
    
    String theKey;
    //trim all the delivery bookkeeping vectors through
    //SourceTableEntries' trimToSize() call.
    SourceTableEntry curSTE;
    enum = sourceTable.keys();
    while(enum.hasMoreElements()) {
      theKey = (String) enum.nextElement();
      curSTE = (SourceTableEntry) sourceTable.get(theKey);
      curSTE.trimToSize();
    }
  }
  
  /** PBSAgent stopMessages() calls this method to signal that the
      pbsa agent no longer wishes to receive any messages */
  public void unregisterForMessage(PBSAgent pbsa) {
    //grab the removal bookkeeping vector and use it to remove every
    //messagetype the agent is registered for
    Vector pbsaVec = (Vector) pbsTable.get(pbsa);
    PBSEntry pbsE;
    SourceTableEntry ste;

    if(pbsaVec != null) {
      for(int i=0; i < pbsaVec.size();i++) {
	//grab the current removal entry
	pbsE = (PBSEntry) pbsaVec.elementAt(i);
	//grab the sourcetableentry assocaited with that removal entry
	ste = (SourceTableEntry) sourceTable.get(pbsE.source);	
	if(ste != null) {
	  //remove the message type entry from the sourcetableentry
	  ste.remove(pbsE.type, pbsE.source, pbsa);
	}
      }
    }
  }
}

class SourceTableEntry {
  MessageQueue theMQueue;
  SocketThread theSockThread;
  /** each entry is a vector of pbsagents that listen for that
      specific message type */
  Hashtable messTypes;
  /** so we can remove ourselves when no message types are required from
      this source */
  Hashtable sourceTable;

  public SourceTableEntry( MessageQueue q, Communications c,
			   Hashtable srcT) {
    theMQueue = q;
    messTypes = new Hashtable();
    theSockThread = new SocketThread(this,c);
    sourceTable = srcT;
  }

  public synchronized void trimToSize(){
    Enumeration enum = messTypes.keys();
    String theKey;
    Vector curVec;
    while(enum.hasMoreElements()) {
      theKey = (String) enum.nextElement();
      curVec = (Vector) messTypes.get(theKey);
      curVec.trimToSize();
    }
  }

  public synchronized void startThread() {
    theSockThread.start();
  }

  public synchronized void add(String type, PBSAgent pbsa) {
    Vector agentReceivers = (Vector) messTypes.get(type);
    if(agentReceivers == null) {
      agentReceivers = new Vector();
      messTypes.put(type,agentReceivers);
    }
    agentReceivers.addElement(pbsa);
  }

  public synchronized void remove(String type, String source, PBSAgent pbsa) {
    Vector agentReceivers = (Vector) messTypes.get(type);
    if(agentReceivers != null) {
      agentReceivers.removeElement(pbsa);
      if(agentReceivers.size() == 0) {
	//no more agents care about this message type from this
	//source, so let go of the vector
	messTypes.remove(type);

	if(messTypes.isEmpty()) {
	  //no more messages types registered for this source, so
	  //revert this source to the default message queue, take
	  //ourselves out of the main sourceTable and kill the thread
	  //managing the queue
	  
	  //take ourself out of the main sourceTable
	  sourceTable.remove(source);
	  
	  //revert the source
	  theSockThread.commLink.unsetMessageQueue(source);
	  
	  //stop the thread processing this source and let it go
	  theSockThread.stop();
	  theSockThread.sourceTableEntry = null;
	  theSockThread = null;
	}
      }
    }
  }
  
  public synchronized void propagateMessage(String type, Message m) {
    // System.out.println("Propagating message: "+m.getMessage());
    
    Vector agentReceivers = (Vector) messTypes.get(type);
    PBSAgent curPBSA;
    if(agentReceivers != null) {
      //there are agents that want this message.
      for(int i = 0; i < agentReceivers.size(); i++) {
	curPBSA = (PBSAgent) agentReceivers.elementAt(i);
	curPBSA.deliverMessage(m);
      }
    }
  }
}

class PBSEntry {
  String source;
  String type;

  PBSEntry(String s, String t) {
    source = s;
    type = t;
  }
}

class SocketThread extends Thread {
  Communications commLink;
  SourceTableEntry sourceTableEntry;

  public SocketThread(SourceTableEntry stEntry, Communications comm) {
    super();
    sourceTableEntry = stEntry;
    commLink = comm;
  }

  public void run() {
    Message m;
    String type;
    int index;
    while(true) {
      //get a messsage.
      //System.out.println("Curr priority: "+
      //			 Thread.currentThread().getPriority());
      
      //System.out.println("MessageQueuesize: "+
      //		   sourceTableEntry.theMQueue.size());
      
      m = commLink.readNextMessage(sourceTableEntry.theMQueue);
      index = m.getMessage().indexOf((int) ' ');
      if(index == -1) {
	//the message is one token
	type = m.getMessage();
      }
      else {
	//the message is multiple tokens
	type = m.getMessage().substring(0,index);
      }
      //call stEntry's propagate message
      sourceTableEntry.propagateMessage(type,m);
      m = null;
      type = null;
      index = 0;
    }
  }
}
