package adaptive.core;

import java.util.*;
import java.io.*;
import comms.core.*;

/*****************************************************************
 * <!
 * PROJECT: Adaptable 
 *
 * FILE: PBSAgent.java
 * 
 * ABSTRACT:
 * >
 * 
 * Class <code>PBSAgent</code> is an extension to the
 * <code>BasicPortAgent</code> class. Extend this class when your 
 * agent needs to receive/send socket messages.
 *
 * @see Agent
 * @see BasicPortAgent
 *
 * @author Ted Pham <A HREF="mailto:telamon@CMU.EDU">telamon@CMU.EDU</A>
 *         <br> </br>
 *
 * @version  1.00 11/27/98<br> </br>
 *
 * <!
 * REVISION HISTORY:
 * Version  Date       Who        What
 * -------  -------    -------    --------------------  
 * 1.00     11/27/98   telamon    Created
 * >
 *****************************************************************/
public abstract class PBSAgent extends BasicPortAgent {
  private Communications commLink;
  private SocketTable socketTable;

  private Communications getCommLink() {
    return commLink;
  }

  private SocketTable getSocketTable() {
    return socketTable;
  }

  private void setCommLink(Communications c) {
    commLink = c;
  }

  private void setSocketTable(SocketTable st) {
    socketTable = st;
  }

  public void runLoop()
  {
  }
    

  /******************************************************************
   * <!
   * Method: PBSAgent()
   *
   * Description:
   * >
   *
   * No argument constructor required for dynamic loading.
   * Calls super().
   *
   * @param 
   *
   * @return 
   *
   * @exception 
   ******************************************************************/
  public PBSAgent() {
    super();
  }

  /******************************************************************
   * <!
   * Method: setup()
   *
   * Description:
   * >
   *
   * Psuedo-constructor to prepare agent for runtime. 
   * Calls the <code>BasicPortAgent</code>'s setup to do normal setup.
   * Additionally initializes Socket related bookkeeping.
   * Calls initMessages() to allow SocketAgents to register for incoming
   * messages.
   *
   * @param agentID the agent's name
   * @param myLoader for use by composed agents
   * @param internalState the agent's saved internal state
   * @param c the communications object
   * @param st bookkeeping
   *
   * @return  <code>true</code> if setup completed successfully; <br>
   *          <code>false</code> otherwise </br>
   *
   * @exception 
   ******************************************************************/
  public final boolean setup(String agentID, AgentLoader myLoader,
			     Object internalState, Communications c,
			     SocketTable st) {
    if(c == null || st == null ) {
      return false;
    }
    boolean retVal;    
    // do the normal agent setup
    retVal = super.setup(agentID, myLoader, internalState);
    
    if(!retVal) {
      return false;
    }
    setCommLink(c);
    setSocketTable(st);
    initMessages();
    return true;
  }

  /******************************************************************
   * <!Description: >
   *
   * Call this method to send messages via the Communications link.
   * 
   * @see cyberscout.core.comms.Message
   *
   * @param 
   *
   * @return 
   *
   * @exception IOException thrown when there is an error in the connection
   * @exception NoPhoneNumberException thrown when the entity you're 
   *                                   sending a message to does not exist
   *                                   in the directory
   *
   ******************************************************************/   
  public final synchronized void sendMessage(String who, Message m) throws IOException, NoPhoneNumberException {
    getCommLink().sendMessage(who, m);
  }

  /******************************************************************
   * <!Description: >
   *
   * This method returns the name associated with the Communications
   * channel.
   * 
   * @see cyberscout.core.comms.Message
   *
   * @param 
   *
   * @return 
   *
   * @exception 
   *
   ******************************************************************/   
  public final synchronized String getMyName(){      
    return getCommLink().getMyName();
  }
  
  /******************************************************************
   * <!Description: >
   *
   * This method will be called by SystemExecutor threads to deliver
   * messages that your agent has previously registered
   * for. <code>deliverMessage()</code> uses internal syncrhonization
   * to guarantee that <code>runLoop()</code> is not running when
   * deliverMessage() is executed. You can not override this method.
   * To handle incoming messages, override
   * <code>receiveMessage()</code>. <code>receiveMessage</code> is
   * called inside <code>deliverMessage()</code> after proper
   * synchronization has occurred.
   * 
   * @param 
   *
   * @return 
   *
   * @exception 
   ******************************************************************/
  public final void deliverMessage(Message m) {
    synchronized(runLock) {
      receiveMessage(m);      
    }
    triggerOutputEvent();//some of our drivers use receiveMessage to set the outputPorts, so we need to call this for events to work
  }

  /******************************************************************
   * <!Description: >
   *
   * This method will be called by deliverMessage when a message
   * arrives. This happens through one of SystemExecutors threads.
   * <b>Tip:</b> If your agent just sets its outputs based on the
   * sockets messages, consider directly setting outputs in this
   * method and saving a thread.
   * 
   * @param 
   *
   * @return 
   *
   * @exception 
   ******************************************************************/
  public void receiveMessage(Message m) { 
  }

  /******************************************************************
   * <!Description: >
   *
   * This method is called as the last step in setup(). Your agent
   * should override this method and make calls to
   * <code>registerForMessages()</code> and <code>sendMessages()</code>.
   * so that it can communicate with entities outside of the Port
   * Based Agent architecture.
   *
   * @see #registerForMessages
   * @see #sendMessages
   * @see cyberscout.core.comms.Message 
   *
   * @param 
   *
   * @return 
   *
   * @exception 
   ******************************************************************/
  protected void initMessages(){ 
  }

  /******************************************************************
   * <!Description:>
   *
   * This method is called inside stop(). Your agent
   * should override this method to do whats necessary to
   * end communication with outside entities.
   *
   * @see #registerForMessages
   * @see #sendMessages
   * @see cyberscout.core.comms.Message 
   *
   * @param 
   *
   * @return 
   *
   * @exception 
   ******************************************************************/
  protected void endMessages(){
  }

  /******************************************************************
   * <!Description: >
   *
   * Call this method inside initMessages to register to 
   * receive types of socket messages.
   * 
   * @param 
   *
   * @return 
   *
   * @exception 
   ******************************************************************/
  protected final void registerForMessage(String source, String type){
    socketTable.registerForMessage(source,type,this);
  }
   
  /******************************************************************
   * <!Description: >
   *
   * Call this method to permanently stop message deliver.
   * use stopMessages() instead.
   * 
   * @param 
   *
   * @return 
   *
   * @exception 
   ******************************************************************/
  private final void unregisterForMessages(){
    socketTable.unregisterForMessage(this);
  }
 
  /******************************************************************
   * <!Description: >
   *
   * This method is automatically called inside stop() to clean up
   * message handling. Only call this method if you want to
   * never receive messages again.
   * 
   * @param 
   *
   * @return 
   *
   * @exception 
   ******************************************************************/
  private final void stopMessages() {
    endMessages();
    unregisterForMessages();
  }
  /******************************************************************
   * <!Description: >
   *
   * Get the name of the communications link we're using.
   * 
   * @param 
   *
   * @return 
   *
   * @exception 
   ******************************************************************/
  protected final String getCommName() {
    return getCommLink().getMyName();  
  }

  /******************************************************************
   * <! 
   * Method: getID() 
   * 
   * Description: 
   * > 
   * 
   * Returns the Agent's name. 
   * 
   * @param  
   * 
   * @return the Agent's name
   *
   * @exception 
   ******************************************************************/
  public final String getID() {
    return super.getID();
  }

  /******************************************************************
   * <!
   * Method: setID()
   *
   * Description:
   * >
   *
   * Set the Agent's name.
   *
   * @param agentID The Agent's name.
   *
   * @return 
   *
   * @exception 
   ******************************************************************/
  public final void setID( String agentID ) {
    super.setID(agentID);
  }

  /******************************************************************
   * <!
   * Method: getNumInputs()
   *
   * Description:
   * >
   *
   * Get the number of inputs this agent needs.
   *
   * @param 
   *
   * @return the number of inputs this agent needs. 
   *
   * @exception 
   ******************************************************************/

  public final int getNumInputs() {
    return super.getNumInputs();
  }

  /******************************************************************
   * <!
   * Method: getNumOutputs()
   *
   * Description:
   * >
   *
   * Get the number of outputs this agent provides.
   *
   * @param 
   *
   * @return the number of outputs this agent provides. 
   *
   * @exception 
   ******************************************************************/
  public final int getNumOutputs() {
    return super.getNumOutputs();
  }

  /******************************************************************
   * <!
   * Method: setNumInputsOutputs()
   *
   * Description:
   * >
   *
   * Call to set the agent's number of inputs and outputs. Should be
   * called inside initialize(). Otherwise the agent will have 0 inputs
   * and 0 outputs.
   *
   * @param numin Number of inputs
   * @param numout Number of outputs
   *
   * @return 
   *
   * @exception 
   ******************************************************************/
  protected final void setNumInputsOutputs( int numin, int numout ) {
    super.setNumInputsOutputs(numin, numout);
  }

  /******************************************************************
   * <!
   * Method: isInputIndex()
   *
   * Description:
   * >
   *
   * Checks if <code>which</code> is a valid input port index.
   *
   * @param which the input port index
   *
   * @return <code>true</code> if valid;
   *         <br><code>false</code> otherwise</br>
   *
   * @exception 
   ******************************************************************/
  public final boolean isInputIndex(int which) {
    return super.isInputIndex(which);
  }
  
  /******************************************************************
   * <!
   * Method: getInputPort()
   *
   * Description:
   * >
   *
   * Get the specified InputPort.
   *
   * @param which the input port index
   *
   * @return the InputPort Object at the specified port number;
   *         <code>null</code> if the index is invalid
   *
   * @exception 
   ******************************************************************/
  public final InputPort getInputPort(int which) {
    return super.getInputPort(which);
  }

  /******************************************************************
   * <!
   * Method: setInputPort()
   *
   * Description:
   * >
   *
   * Set the input port at index to be the given InputPort.
   *
   * @param inp the InputPort object
   * @param which the input port index
   *
   * @return <code>true</code> if the input port was set;
   *         <br><code>false</code> if the index is invalid<br>
   *
   * @exception 
   ******************************************************************/
  public final boolean setInputPort(InputPort inp, int which) {
    return super.setInputPort(inp, which);
  }
  
  /******************************************************************
   * <!
   * Method: getInputPortType()
   *
   * Description:
   * >
   *
   * Get the type of the InputPort at the specified index.
   *
   * @param which the input port index
   *
   * @return the Class Object for the input port's type;
   *         <br><code>null</code> if the index is invalid</br>
   *
   * @exception 
   *
   ******************************************************************/
  public final Class getInputPortType( int which ) {
    return super.getInputPortType(which);
  }

  /******************************************************************
   * <!
   * Method: setInputPortType()
   *
   * Description:
   * >
   *
   * Set the InputPort type. If the input port doesn't yet exist,
   * create it.
   *
   * @param stp the fully qualified class name of the input type 
   * @param which the input port index
   *
   * @return <code>true</code> if the input port type was set;
   *         <br><code>false</code> if the index is invalid</br>
   *
   * @exception ClassNotFoundException thrown when stp does not
   *            indicate a valid java class
   ******************************************************************/
  public final boolean setInputPortType( String stp, int which ) throws ClassNotFoundException{
    return super.setInputPortType(stp, which);
  }
  
  /******************************************************************
   * <!
   * Method: setInputPortType()
   *
   * Description:
   * >
   *
   * Set the InputPort type. If the input port doesn't yet exist,
   * create it.
   *
   * @param otp the port should be set to this Object's type 
   * @param which the input port index
   *
   * @return <code>true</code> if the input port type was set;
   *         <br><code>false</code> if the index is invalid</br>
   *
   * @exception 
   ******************************************************************/
  public final boolean setInputPortType( Object otp, int which ) {
    return super.setInputPortType(otp, which);
  }

  /******************************************************************
   * <!
   * Method: setInputPortType()
   *
   * Description:
   * >
   *
   * Set the InputPort type. If the input port doesn't yet exist,
   * create it.
   *
   * @param tp the Class object for the type the InputPort should
   *           be set to
   * @param which the input port index
   *
   * @return <code>true</code> if the input port type was set;
   *         <br><code>false</code> if the index is invalid</br>
   *
   * @exception 
   ******************************************************************/
  public final boolean setInputPortType( Class tp, int which ) {
    return super.setInputPortType(tp, which);
  }
  
  /******************************************************************
   * <!
   * Method: getInputMapping()
   *
   * Description:
   * >
   *
   * Get the input mapping for the specified input port.
   *
   * @param which the input port index
   *
   * @return a space seperated string containing the providing agent's
   *         name and output number; <br><code>null</code> if no
   *         provider is mapped to the specified input port or the if
   *         specified input port doesn't exist
   *
   * @exception 
   ******************************************************************/
  public String getInputMapping(int which) {
    return super.getInputMapping(which);
  }

  /******************************************************************
   * <!
   * Method: setInputMapping()
   *
   * Description:
   * >
   *
   * Set the output port that will provide data for the specified
   * input port.
   *
   * @param which the index of the input port
   * @param provider the
   *
   * @return <code>true</code> if the mapping was made;
   * <br><code>false</code> otherwise. Failure results when the port
   * index are invalid or the provider argument is <code>null</code>
   * </br>
   *
   * @exception 
   ******************************************************************/
  public final boolean setInputMapping(int which, Agent provider,
				       int whichOut) {
    return super.setInputMapping(which,provider,whichOut);
  }

  /******************************************************************
   * <!
   * Method: getInput()
   *
   * Description:
   * >
   *
   * Get the current value of the Agent's specified input.
   *
   * @param which the desired input number
   *
   * @return the current input value;
   *
   * @exception 
   ******************************************************************/
  public final Object getInput( int which ) {
    return super.getInput(which);
  }
  
  /******************************************************************
   * <!
   * Method: isOutputIndex()
   *
   * Description:
   * >
   *
   * Checks if <code>which</code> is a valid output port index.
   *
   * @param which the output port index
   *
   * @return <code>true</code> if valid;
   *         <br><code>false</code> otherwise</br>
   *
   * @exception 
   ******************************************************************/
  public final boolean isOutputIndex(int which) {
    return super.isOutputIndex(which);
  }

  /******************************************************************
   * <!
   * Method: getOutputPort()
   *
   * Description:
   * >
   *
   * Gets the specified OutputPort.
   *
   * @param which the output port index
   *
   * @return the OutputPort Object;
   *         <br><code>null</code> if index is invalid</br>
   *
   * @exception 
   ******************************************************************/
  public final OutputPort getOutputPort(int which) {
    return super.getOutputPort(which);
  }

  /******************************************************************
   * <!
   * Method: setOutputPort()
   *
   * Description:
   * >
   *
   * Set Agent's specified OutputPort.
   *
   * @param outp the OutputPort object
   * @param which index of the output port to set
   *
   * @return <code>true</code> if the OutputPort was set;
   *         <br><code>false</code> if the specified output port
   *         number is invalid</br>
   *
   * @exception 
   ******************************************************************/
  public final boolean setOutputPort(OutputPort outp, int which) {
    return super.setOutputPort(outp,which);
  }

  /******************************************************************
   * <!
   * Method: getOutputPortType()
   *
   * Description:
   * >
   *
   * Get the specified output port's type.
   *
   * @param which index of the output port
   *
   * @return the Class object for the output port's type;
   *         <br><code>null</code> if the output port index is
   *         invalid</br>
   *
   * @exception 
   ******************************************************************/
  public final Class getOutputPortType( int which ) {
    return super.getOutputPortType(which);
  }

  /******************************************************************
   * <!
   * Method: setOutputPortType()
   *
   * Description:
   * >
   *
   * Set the specified output port's type. If the OutputPort doesn't
   * yet exist, create it.
   *
   * @param stp fully qualified name representing of the type
   * @param which index of the output port to set
   *
   * @return <code>true</code> if the port type was set;
   *         <br><code>false</code> if the output port index is
   *         invalid</br>
   *
   * @exception ClassNotFoundException thrown if stp doesn't indicate a valid
   *            type
   ******************************************************************/
  public final boolean setOutputPortType( String stp, int which ) 
    throws ClassNotFoundException {
    return super.setOutputPortType(stp, which);
  }
  
  /******************************************************************
   * <!
   * Method: setOutputPortType()
   *
   * Description:
   * >
   *
   * Set the specified output port's type. If the OutputPort doesn't
   * yet exist, create it.
   *
   * @param otp port's type should be set to this Object's type
   * @param which index of the output port to set
   *
   * @return <code>true</code> if the port type was set;
   *         <br><code>false</code> if the output port index is
   *         invalid</br>
   *
   * @exception 
   ******************************************************************/
  public final boolean setOutputPortType( Object otp, int which ) {
    return super.setOutputPortType(otp, which);
  }

  /******************************************************************
   * <!
   * Method: setOutputPortType()
   *
   * Description:
   * >
   *
   * Set the specified output port's type. If the OutputPort doesn't
   * yet exist, create it.
   *
   * @param tp the type to set to
   * @param which index of the output port to set
   *
   * @return <code>true</code> if the port type was set;
   *         <br><code>false</code> if the output port index is
   *         invalid</br>
   *
   * @exception 
   ******************************************************************/
  public final boolean setOutputPortType( Class tp, int which ) {
    return super.setOutputPortType(tp, which);
  }

  /******************************************************************
   * <!
   * Method: getOutput()
   *
   * Description:
   * >
   *
   * Get the current value of the Agent's specified output.
   *
   * @param which the desired output number
   *
   * @return the current output value; <br> <code>null</code> if the
   *         specified output does not exist </br>
   *
   * @exception 
   ******************************************************************/
  public final Object getOutput(int which) {
    return super.getOutput(which);
  }
  
  /******************************************************************
   * <!
   * Method: setOutput()
   *
   * Description:
   * >
   *
   * Set the value of the Agent's specified output.
   *
   * @param value the new output value
   * @param which the desired output port number
   *
   * @return <code>true</code> if the value was writen to the port;
   *         <br><code>false</code> if the specified port is invalid
   *         or there is a type mismatch</br>
   *
   * @exception 
   ******************************************************************/
  public final boolean setOutput( Object value, int which ) {
    return super.setOutput( value, which);
  }

  /******************************************************************
   * <!Description: >
   *
   * Put's the Agent to sleep for the specified amount of time. This
   * method automatically releases the internal runLock monitor. DO
   * NOT USE ANY OTHER METHODS THAT WOULD BLOCK YOUR RUNLOOP.
   *
   * @param sleepTime how long to sleep for   
   *
   * @return 
   *
   * @exception InterruptedException when the thread is interrupted 
   ******************************************************************/
  public final void sleep(long sleepTime) throws InterruptedException {
    super.sleep(sleepTime);    
  }
         
  /******************************************************************
   * <!
   * Method: start()
   *
   * Description:
   * >
   *
   * Called to start execution.
   *
   * @param 
   *
   * @return 
   *
   * @exception 
   ******************************************************************/
  public final void start() {
    super.start();
  }

  /******************************************************************
   * <!
   * Method: pause()
   *
   * Description:
   * >
   *
   * Called to pause execution. <code>unpause()</code> may be called
   * later to resume execution.
   *
   * @param 
   *
   * @return 
   *
   * @exception 
   ******************************************************************/
  public final void pause() {
    super.pause();
  }

  /******************************************************************
   * <!
   * Method: unpause()
   *
   * Description:
   * >
   *
   * Called to unpause execution. <code>pause()</code> should have
   * been called before. If <code>pause()</code> wasn't called before,
   * then do nothing.
   *
   * @param 
   *
   * @return 
   *
   * @exception 
   ******************************************************************/
  public final void unpause() {
    super.unpause();
  }

  /******************************************************************
   * <!
   * Method: stop()
   *
   * Description:
   * >
   *
   * Called to end execution. <b>Note:</b> Do not deallocate resources
   * or clear state in this method. Do so in <code>destroy()</code>.
   * <code>dumpState</code> may be called after this method has been
   * called.
   *
   * @param 
   *
   * @return 
   *
   * @exception 
   ******************************************************************/
  public final void stop() {
     synchronized(runLock) {
      if(!stopped) {	
	if(usesThread) {
	  mainLoop.stop();
	}
	stopMessages();
	stopped = true;
      }
    }
    System.out.println(getID() + " stopped.");
  }

  /******************************************************************
   * <!
   * Method: saveState()
   *
   * Description:
   * >
   *
   * Returns the Agent's internal state. This will be serialized and
   * saved for later restoration. <b>Note:</b> State may not have been
   * explicitly frozen and Agent may not have been stopped prior to
   * this call. However, this method is internally synchronized such
   * that it will only occur between successive calls to
   * <code>runLoop()</code>.
   *
   * @param 
   *
   * @return a Serializable Object representing the Agent's saved state
   *
   * @exception 
   ******************************************************************/
  public final Object saveState() {
    return super.saveState();
  }
  
  /******************************************************************
   * <!
   * Method: isStateFrozen()
   *
   * Description:
   * >
   *
   * Whether the state should currently be frozen. This method,
   * <code>freezeState()</code>, and <code>unfreezeState()</code> are
   * internally synchronized so that they will never freeze or
   * unfreeze state in the middle of a call to
   * <code>runLoop()</code>. The Agent implementer should check to see
   * if state is frozen at the start of <code>runLoop()</code> and trust
   * that it will not change
   *
   * @param 
   *
   * @return an Object representing the Agent's saved state
   *
   * @exception 
   ******************************************************************/
  public final boolean isStateFrozen() {
    return super.isStateFrozen();
  }

  /******************************************************************
   * <!
   * Method: freezeState()
   *
   * Description:
   * >
   *
   * Called to prevent the Agent from changing it's internal state.
   * The Agent should continue to update it's outputs, but internal
   * data structures such as neural net weights or adaptive filter
   * coefficients should not be changed while state is frozen. See
   * <code>isStateFrozen()</code> for notes on synchronization.
   *
   * @see #isStateFrozen
   *
   * @param 
   *
   * @return 
   *
   * @exception 
   ******************************************************************/
  public final void freezeState() {
    super.freezeState();
  }

  /******************************************************************
   * <!
   * Method: unfreezeState()
   *
   * Description:
   * >
   *
   * Called to undo a state freeze. The Agent is then free to change
   * it's internal state again. See <code>isStateFrozen()</code> for
   * notes on synchronization.
   *
   * @see #isStateFrozen
   *
   * @param 
   *
   * @return 
   *
   * @exception 
   ******************************************************************/
  public final void unfreezeState() {
    super.unfreezeState();
  }

  /******************************************************************
   * <!
   * Method: destroy()
   *
   * Description:
   * >
   *
   * Called at end of Agent's run cycle. Agent should release all
   * resources it's holding and free up memory where necessary. If
   * stop hasn't been called before destroy, destroy will call stop
   * before proceeding with deallocation.
   *
   * @param 
   *
   * @return 
   *
   * @exception 
   ******************************************************************/
  public final void destroy() {
    super.destroy();
  }

  /******************************************************************
   * <!
   * Method: run()
   *
   * Description:
   * >
   * Satisfies Runnable interface. The main work loop. Derived
   * classes should not override this method. 
   * Override runLoop instead. This is public because java demands
   * that it be public.
   *
   * @param 
   *
   * @return 
   *
   * @exception 
   ******************************************************************/
  public final void run() {
    super.run();
  }

  /******************************************************************
   * 
   * <! Description: >
   * tells us to not use a seperate thread for this agent.
   *
   * @param 
   *
   * @return 
   *
   * @exception 
   ******************************************************************/
  protected final void setNoThread(){
    super.setNoThread();
  }
  
  /****************************************************************************
   * <!
   * Method: getSleepTime()
   *
   * Description:
   * >
   *
   * Get the current per loop sleep time value.
   *
   * @param 
   *
   * @return the current per loop sleep time value.
   *
   * @exception 
   ***************************************************************************/
  public final long getSleepTime() {
    return super.getSleepTime();
  }
  
  /****************************************************************************
   * <!
   * Method: setSleepTime()
   *
   * Description:
   * >
   *
   * Set's the current per loop sleep value.
   *
   * @param sleepTime the new current sleep value
   *
   * @return 
   *
   * @exception 
   ***************************************************************************/
  protected final void setSleepTime(long sleepTime) {
    super.setSleepTime(sleepTime);
  }

  /******************************************************************
   * <!
   * Method: getInputPortName()
   *
   * Description:
   * >
   *
   * Get the type of the InputPort at the specified index.
   *
   * @param which the input port index
   *
   * @return the Class Object for the input port's name;
   *         <br><code>null</code> if the index is invalid</br>
   *
   * @exception 
   *
   ******************************************************************/
  public String getInputPortName( int which ) {
	 return super.getInputPortName(which);
  }

  /******************************************************************
   * <!
   * Method: setInputPortName()
   *
   * Description:
   * >
   *
   * Set the InputPort name. If the input port doesn't yet exist,
   * create it.
   *
   * @param nm the String for the name the InputPort should
   *           be set to
   * @param which the input port index
   *
   * @return <code>true</code> if the input port name was set;
   *         <br><code>false</code> if the index is invalid</br>
   *
   * @exception 
   ******************************************************************/
  public boolean setInputPortName( String nm, int which ) {
	 return super.setInputPortName(nm,which);
  }

  /******************************************************************
   * <!
   * Method: getOutputPortName()
   *
   * Description:
   * >
   *
   * Get the specified output port's type.
   *
   * @param which index of the output port
   *
   * @return the String for the output port's name;
   *         <br><code>null</code> if the output port index is
   *         invalid</br>
   *
   * @exception 
   ******************************************************************/
  public String getOutputPortName( int which ) {
	 return super.getOutputPortName(which);
  }
  

  /******************************************************************
   * <!
   * Method: setOutputPortName()
   *
   * Description:
   * >
   *
   * Set the specified output port's name. If the OutputPort doesn't
   * yet exist, create it.
   *
   * @param nm the name to set to
   * @param which index of the output port to set
   *
   * @return <code>true</code> if the port type was set;
   *         <br><code>false</code> if the output port index is
   *         invalid</br>
   *
   * @exception 
   ******************************************************************/
  public boolean setOutputPortName( String nm, int which ) {
	 return super.setOutputPortName(nm,which); 
  }
  





}
