package adaptive.core;

import java.util.*;
/*****************************************************************
 * <!
 * PROJECT: Adaptable 
 *
 * FILE: PBAMacro.java
 * 
 * ABSTRACT: > 
 * Agent that represents a macro of agents.  This way the macro can be
 * addresssed as a whole rather then having to talk to each little piece.
 *
 *
 * @author  yatish?  
 * @author Jonathan Jackson <jackson4@andrew.cmu.edu>
 *
 * $Log: PBAMacro.java,v $
 * Revision 1.1.2.4  2000/06/03 19:25:55  jrj
 * fixed an error in mapping inputs on macros.  Added a few diagnostic messages during the system verification step.
 *
 * Revision 1.1.2.3  2000/05/20 19:11:17  jrj
 * fixed a couple more macro bugs.  Took preliminary steps toward making
 * fully recursive macros in MacroDescription.  Have not yet made changes
 * that would damage save file.
 *
 * Revision 1.1.2.2  2000/05/15 22:39:38  jrj
 * Made fixes to macros so that they work.  Changed the executing macro
 * so that it is recursive, can be manipulated as a whole, and its ports
 * can be asked for directly as if it were any other BasicPortAgent.
 * Made several other changes to simplify and assist these changes.
 * Finally purged the last remaining bit of things named clump.
 *
 ********/

public class PBAMacro extends BasicPortAgent { 
  /** Stores the macro this agent represents */  
  protected MacroDescription macroDescription;
  protected Hashtable internalAgentTable;//agent's ID-->Agent
  protected Hashtable internalAgentDescTable;//agent's ID-->AgentDescription
  protected PortLocation[] inputPortLocations;
  protected PortLocation[] outputPortLocations;

  public PBAMacro(MacroDescription cd,SystemExecutor callBack) {
    setID(cd.getName());
    //System.out.println("Creating macro for "+getID());
    this.macroDescription=cd;

    internalAgentTable=new Hashtable();
    internalAgentDescTable=new Hashtable();

    Enumeration e;
    AgentDescription agentDes;
    Agent agentInst;
    String agentName;
    int i;
    PortLocation pl;

    try{
	//System.out.println("Instantiating agents inside "+getID());
      e = macroDescription.getAgents();
    //iterate through each agent in MacroDescription and instantiate them as necessary(instantiates the agent if it runs on this location or instantiates a stub if it runs elsewhere but is connected to an agent running on this location)
      while (e.hasMoreElements()){
	agentDes=(AgentDescription)e.nextElement();
	agentInst=callBack.instantiateAgent(agentDes);
	if (agentInst!=null){
	  internalAgentTable.put(agentInst.getID(),agentInst);
	  internalAgentDescTable.put(agentInst.getID(),agentDes);
	}
      }
      //now connect internal ports
      //System.out.println("Connecting ports inside "+getID());
      e=internalAgentTable.keys();
      while (e.hasMoreElements()){
	agentName=(String)e.nextElement();
	agentDes=(AgentDescription)internalAgentDescTable.get(agentName);
	agentInst=(Agent)internalAgentTable.get(agentName);
	connectPorts(agentInst,agentDes);
      }
    }catch(Exception ex){
      System.err.println("Caught an exception instantiating a macro("+getID()+"):");
      ex.printStackTrace();
      this.destroy();
      return;
    }
    //set up input and output port locations external to the macro so that they connect to the proper internal agent
    int numI=macroDescription.getNumInputs();
    int numO=macroDescription.getNumOutputs();
    this.setNumInputsOutputs(numI,numO);

    //System.out.println("Preparing ports to connect outside "+getID());
    inputPortLocations=new PortLocation[macroDescription.getNumInputs()];
    outputPortLocations=new PortLocation[macroDescription.getNumOutputs()];
    //System.out.println("#in="+macroDescription.getNumInputs());
    //System.out.println("#out="+macroDescription.getNumOutputs());  
    for (i=0;i<macroDescription.getNumInputs();i++){
	//pl=macroDescription.getInputAgent(i);
      pl=macroDescription.getInputLocation(i);
      //System.out.println("setting up input port "+i+":"+pl);
      setInputPortAgent(i,pl);
    }  
    for (i=0;i<macroDescription.getNumOutputs();i++){
      pl=macroDescription.getOutputLocation(i);
      //System.out.println("setting up output port "+i+":"+pl);
      setOutputPortAgent(i,pl);
    }
    //System.out.println("finished connecting external ports for "+getID());
  }

  protected boolean connectPorts(Agent agentInst,AgentDescription agentDes){
    boolean regGood=true;
    InputLocation inputLoc;
    Agent provider;
    for(int j = 0; j < agentDes.getNumMappedInputs() && regGood;j++) {
      inputLoc = agentDes.getInputMapping(j);
      //System.out.println("Input "+j+" is from "+agentDes.getInputMapping(j));
      //System.out.println("trying to map:"+inputLoc.toString());
      provider = (Agent) internalAgentTable.get(inputLoc.getAgentID());
      if (provider!=null){//the provider should be null if the input is connected to an external source
	regGood = false;
	//System.out.println("   provider="+provider);
	regGood = agentInst.setInputMapping(inputLoc.getInputNum(),
					    provider,
					    inputLoc.getOutputNum());
      }
      else{
	//	System.out.println("Null provider for "+inputLoc);
      }
    }
    return regGood;
  }
  public final void start() {
    Enumeration e=internalAgentTable.elements();
    Agent agent;
    while (e.hasMoreElements()){
      agent=(Agent)e.nextElement();
      System.out.println("About to start "+agent.getID());
      agent.start();
      System.out.println("Started "+agent.getID());
    }
  }
  
  public final void stop() {
    Enumeration e=internalAgentTable.elements();
    Agent agent;
    while (e.hasMoreElements()){
      agent=(Agent)e.nextElement();
      agent.stop();
    }
  }
  
  public final void pause() {
    Enumeration e=internalAgentTable.elements();
    Agent agent;
    while (e.hasMoreElements()){
      agent=(Agent)e.nextElement();
      agent.pause();
    }
  }
  
  public final void unpause() {
    Enumeration e=internalAgentTable.elements();
    Agent agent;
    while (e.hasMoreElements()){
	  agent=(Agent)e.nextElement();
	  agent.unpause();
    }
  }
  public final void destroy(){
    Enumeration e=internalAgentTable.elements();
    Agent agent;
    while (e.hasMoreElements()){
      agent=(Agent)e.nextElement();
      agent.destroy();
    }
    internalAgentTable=null;
    internalAgentDescTable=null;
    macroDescription=null;
  }
  
  
  
  //internal helper classes for associating a particular internal agent's port with one of the macros ports
  protected PortLocation getInputPortLocation(int which){
    return inputPortLocations[which];
  }
  protected Agent getInputPortAgent(PortLocation pl){
    return (Agent)internalAgentTable.get(pl.getAgentName());
  }
  protected PortLocation getOutputPortLocation(int which){
    return outputPortLocations[which];
  }
  protected Agent getOutputPortAgent(PortLocation pl){
    return (Agent)internalAgentTable.get(pl.getAgentName());
  }

  protected boolean setInputPortAgent(int which,PortLocation pl){
    Agent agent=getInputPortAgent(pl);
    InputPort input;
    inputPortLocations[which]=pl;
    input=agent.getInputPort(pl.getPortNumber());
    setInputPort(input,which);
    return true;
  }
  protected boolean setInputPortAgent(int which,Agent agent,int agentPort){
    //I don't think this function is actually used
    PortLocation pl=new PortLocation(agent.getID(),agentPort,(AgentDescription)internalAgentDescTable.get(agent.getID()));
    return setInputPortAgent(which,pl);
  } 
  protected boolean setOutputPortAgent(int which,PortLocation pl){
    Agent agent=getOutputPortAgent(pl);
    OutputPort output;
    outputPortLocations[which]=pl;
    output=agent.getOutputPort(pl.getPortNumber());
    setOutputPort(output,which);
    return true;
  }
  protected boolean setOutputPortAgent(int which,Agent agent,int agentPort){
    PortLocation pl=new PortLocation(agent.getID(),agentPort,(AgentDescription)internalAgentDescTable.get(agent.getID()));
    outputPortLocations[which]=pl;
    return true;
  }
    //overwrite these methods from BasicPortAgent, echoing the call to the proper agent within the macro as well
    public boolean setInputPort(InputPort inp, int which){
	//    System.out.println("Setting input port "+which+" on "+this.getID()+" to be "+inp);
      Agent agentInst;
      PortLocation input;
      if(!isInputIndex(which) ||inp==null) {
	return false;
      }
      inputPorts.setElementAt(inp,which);//set this so BasicPortAgent's getInputPort can access the inputPort directly, as if this were a normal module

      //also set the InputPort on the agent actually receiving that input
      input=getInputPortLocation(which);
      agentInst=getInputPortAgent(input);
  
      agentInst.setInputPort(inp,input.getPortNumber());
      //      System.out.println("Success on input port "+which);
      return true;
    }
   

  public boolean setInputMapping(int which, Agent provider, int whichOut){
    //check for bad input index,
    //no provider
    //and bad output index on provider
      //      System.out.println(getID()+" initializing input port "+which+" to come from port"+whichOut+" on "+provider.getID());
    if( !isInputIndex(which)
	|| (provider == null )
	|| !provider.isOutputIndex(whichOut)) {
      return false;
    }
    InputPort curInput = getInputPort( which );
    if( ( getInputPortAgent(getInputPortLocation(which)).getModuleMode() & Agent.EVENT_ONLY ) != 0 ) {
      System.out.println( "adding " + this + " as a listener to "
+ provider.getOutputPort( whichOut ) );
      provider.getOutputPort( whichOut ).addTriggerListener( this, which );
      //I might need to register the actual agent receiving the agent instead of this macro, but I like this better if it works
    }
    //    System.out.println("current input "+which+":"+curInput);
    return curInput.setMapping(provider.getOutputPort(whichOut));   
  }
  public boolean setOutputPort(OutputPort outp, int which){
    Agent agentInst;
    PortLocation output;
    if(!isOutputIndex(which)) {
      return false;
    }
    outputPorts.setElementAt(outp,which);//set this so BasicPortAgent's getInputPort can access the outputPort directly, as if this were a normal module
    
    //also set the OutputPort on the agent actually receiving that input
    output=getOutputPortLocation(which);
    agentInst=getOutputPortAgent(output);
    agentInst.setOutputPort(outp,output.getPortNumber());
    return true;	
  }



  protected void initialize() {
    this.setNumInputsOutputs(0,0);
  }
  //these methods need to have an empty placeholder, since the macro has nothing to run itself, only its agents do
  //jrj--note I may have goofed here slightly.  It is possible that some of these need to echo the proper call down to its constituent agents, but we'll see.
  public boolean setOutput(Object value, int which){
    return true;
  }
  protected void allocateInternals() {
    //nothing internal to allocate
  }

  protected boolean processInternalState(Object state) {
    Hashtable groupState=(Hashtable)state;
    Enumeration keys=groupState.keys();
    String id;
    BasicPortAgent agent;
    Object localState;
    while (keys.hasMoreElements()){
      id=(String)keys.nextElement();
      agent=(BasicPortAgent)internalAgentTable.get(id);
      localState=groupState.get(id);
      agent.processInternalState(localState);
    }
    return true;
  }
  
  protected boolean initOutputVals() {
      //its output values are initialized by the modules that actually create those outputs
    return true;
  }

  protected void runLoop() {    
	 // Macro agents don't run. only the agents inside the macros
	 // run
  }

  protected Object dumpState() {
    Hashtable groupState=new Hashtable();
    Enumeration agents=internalAgentTable.elements();
    BasicPortAgent agent;
    while (agents.hasMoreElements()){
      agent=(BasicPortAgent)agents.nextElement();
      groupState.put(agent.getID(),agent.dumpState());
    }
    return groupState;
  }

  protected void deAllocateInternals() {
    //nothing to deallocate
  }
  public int getModuleModeCapability(){
    return 0;//this is a nonsense constant
  }
  
}
