package adaptive.core;

import java.util.*;

/*****************************************************************
 * <!
 * PROJECT: Adaptable 
 *
 * FILE: OutputPort.java
 * 
 * ABSTRACT:
 * >
 * Output port for agents.
 * 
 * @author Charles Tennent <A HREF="mailto:tennent@andrew.cmu.edu">tennent@andrew.cmu.edu</A>
 * @author <br>Ted Pham <A HREF="mailto:telamon@CMU.EDU">telamon@CMU.EDU</A></br>
 *
 * @version  1.10 11/20/98
 *
 * <!
 * REVISION HISTORY:
 * Version  Date        Who        What
 * -------  --------    -------    ------------------------------
 * 1.00     11/18/98    cht        Created.
 * 1.10     11/20/98    telamon    Restructured for indirection
 *                                 and security.
 * >
 *****************************************************************/

class OutputPort extends Port{

  private Object currentOutput;
  private int myIndex;
  private String myOwnerID;
  private boolean proxy;
  /** a Hashtable of strings indicating which remote hosts want data from this
      port */
  private Hashtable proxyList;
  /** a Hashtable that is used to store the event listeners for this port */
  private Hashtable triggerSupport = new Hashtable( 5 );
  
  /** a boolean to keep track of clean/dirty, used for event aggregation */
  private boolean cleanFlag = true;

  /******************************************************************
   * <!
   * Method: OutputPort()
   *
   * Description:
   * >
   * Constructor that sets type, index and ownerID
   * 
   * @param index the output port's index in owner
   * @param ownerID the owner's ID
   * @param portType the output port's type
   *
   * @return 
   *
   * @exception 
   ******************************************************************/
  OutputPort(int index, String ownerID, Class portType) {
    super(portType);
    myIndex = index;
    myOwnerID = ownerID;
    currentOutput = null;
    proxy = false;
  }

  /******************************************************************
   * <!
   * Method: OutputPort()
   *
   * Description:
   * >
   * Constructor that sets name, index and ownerID
   * 
   * @param index the output port's index in owner
   * @param ownerID the owner's ID
   * @param portName the output port's name
   *
   * @return 
   *
   * @exception 
   ******************************************************************/
  OutputPort(int index, String ownerID, String portName) {
    super(portName);
    myIndex = index;
    myOwnerID = ownerID;
    currentOutput = null;
    proxy = false;
  }

  /******************************************************************
   * <!
   * Method: setOutput()
   *
   * Description:
   * >
   * Set the current output of the agent.
   * 
   * @param co the new output value
   *
   * @return true if the value was set;
   *         false if the object type does not match the port type
   *
   * @exception 
   ******************************************************************/

  public synchronized boolean setOutput( Object co ) {
    //System.err.println("Inside setoutput");
    //System.err.println("The new vals class: "+ co.getClass());
    //System.err.println("The ports class: "+getType());
    if( getType().isAssignableFrom(co.getClass())) {
      //System.err.println("Output set correctly.");
      currentOutput = co;
      cleanFlag = false;
      //System.out.println( "outputPort is not clean" );
      return true;
    }
    return false;
  }

  /******************************************************************
   * <!
   * Method: getOutput()
   *
   * Description:
   * >
   * Get the current output value.
   * 
   * @param 
   *
   * @return the current output Object
   *
   * @exception 
   ******************************************************************/

  public synchronized Object getOutput() {
    return currentOutput;
  }
    
  /******************************************************************
   * <!
   * Method: setIndex()
   *
   * Description:
   * >
   * Set the port's index.
   * 
   * @param index the port's new index
   *
   * @return 
   *
   * @exception 
   ******************************************************************/

  synchronized void setIndex(int index) {
    myIndex = index;
  }

  /******************************************************************
   * <!
   * Method: getIndex()
   *
   * Description:
   * >
   * Get port's index.
   * 
   * @param 
   *
   * @return the port's index
   *
   * @exception 
   ******************************************************************/

  synchronized int getIndex() {
    return myIndex;
  }
  
  /******************************************************************
   * <!
   * Method: setOwnerID()
   *
   * Description:
   * >
   * Set the port's ownerID.
   * 
   * @param owner the new ownerID
   *
   * @return 
   *
   * @exception 
   ******************************************************************/

  synchronized void setOwnerID(String ownerID) {
    myOwnerID = ownerID;
  }

  /******************************************************************
   * <!
   * Method: getOwnerID()
   *
   * Description:
   * >
   * Get port's ownerID.
   * 
   * @param 
   *
   * @return the port's owner
   *
   * @exception 
   ******************************************************************/

  synchronized String getOwnerID() {
    return myOwnerID;
  }

  /******************************************************************
   * <!
   * Method: setProxy()
   *
   * Description:
   * >
   * Set the outport's proxy field.
   * 
   * @param owner true to proxy; <br> false otherwise <\br>
   *
   * @return 
   *
   * @exception 
   ******************************************************************/

  synchronized void setProxy(boolean p) {
    proxy = p;
  }

  /******************************************************************
   * <!
   * Method: getProxy()
   *
   * Description:
   * >
   * Get port's proxy value.
   * 
   * @param 
   *
   * @return true if the port is being proxied; <br> false otherwise <\br>
   *
   * @exception 
   ******************************************************************/

  synchronized boolean getProxy() {
    return proxy;
  }

  /******************************************************************
   * <!
   * Method: initProxy()
   *
   * Description:
   * >
   * Initialize proxy bookeeping data and sets proxy to true;
   * 
   * @param 
   *
   * @return 
   *
   * @exception 
   ******************************************************************/

  synchronized void initProxy() {
    proxyList = new Hashtable();
    proxy = true;
  }
  
  /******************************************************************
   * Add a listener for trigger events on this port
   * @param l The object that should be listening for trigger events
   * @param portNum The InputPort corresponding to this OutputPort for
   *                listener
   *****************************************************************/
  synchronized void addTriggerListener( TriggerListener l, int portNum ) {
    triggerSupport.put( l, new Integer( portNum ) );
  } 

  /******************************************************************
   * Remove a listener for trigger events on this port
   * @param l The object that should not be listening for trigger events
   *****************************************************************/
  synchronized void removeTriggerListener( TriggerListener l ) {
    triggerSupport.remove( l );
  }

  /******************************************************************
   * Returns the table of listeners and their input ports
   *****************************************************************/
  synchronized Hashtable getListenerTable() {
    return triggerSupport;
  }
  
  /**
   * sets the clean status of this output port
   * @param clean the new value for the clean flag
   */
  synchronized void setClean( boolean clean ) {
    cleanFlag = clean;
  }
  
  /**
   * returns the clean status for this output port
   * @return true if this output port is clean
   */
  synchronized boolean isClean() {
    return cleanFlag;
  }

    public String toString(){
	return ("OutputPort{class="+getType()+"}");
    }
}
