package adaptive.core;

import java.util.Hashtable;
import java.util.Enumeration;
import java.io.IOException;
import java.net.Socket;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.ObjectOutputStream;
import java.io.ObjectInputStream;
import java.io.BufferedInputStream;
import java.io.InvalidClassException;
import java.io.StreamCorruptedException;
import java.io.OptionalDataException;
import java.io.FileNotFoundException;
import adaptive.agentserver.AgentServer;

/*****************************************************************************
 * <! Copyright 1998, Institute for Complex Engineered Sytems,
 *                    Carnegie Mellon University
 *
 * PROJECT: Adaptable 
 *
 * FILE: NetAgentLoader.java 
 * >
 *
 * Class responsible for loading Agents and support classes over the
 * network. Java automatically calls ClassLoader.loadClass() for an
 * agent's support classes so the NetAgentLoader and LocalAgentLoader
 * must be split up into two seperate class loaders. They will both be
 * contained in AgentLoader.
 * 
 * @author Theodore Q Pham <A HREF="mailto:telamon@CMU.EDU">telamon@CMU.EDU</A>
 *         <br> </br>
 *
 * @version  1.00 12/12/98<br> </br>
 *
 * <!
 * REVISION HISTORY:
 *
 * $Log: NetAgentLoader.java,v $
 * Revision 1.5.2.7  2000/03/02 07:57:09  telamon
 * NetAgentLoader looks at local file system first
 *
 * Revision 1.5.2.6  2000/02/28 17:40:17  telamon
 * turned off debugging output in NetAgentLoader.java
 *
 * Revision 1.5.2.5  2000/02/28 00:39:04  telamon
 * less debugging in LocalAgentLoader and NetAgentLoader
 *
 * Revision 1.5.2.4  2000/02/27 22:38:45  telamon
 * safety checking, NetExecutor.java
 *
 * Revision 1.5.2.3  2000/01/27 18:57:41  telamon
 * fixed LinkedList for move to util directory
 *
 * Revision 1.5.2.2  1999/10/08 18:59:44  telamon
 * added a missing flush in NetAgentLoader
 *
# Revision 1.5.2.1  99/10/08  13:58:03  telamon
# NetAgentLoader updated for new communications
# 
 * Revision 1.5  1999/04/06 19:40:10  telamon
 * moving comms out of cyberscout
 *
 * Revision 1.4  1999/04/06 15:11:59  yatish
 * Massive changes to the Save File format along with the addition of Macro
 * support.
 *
 * Revision 1.3  1999/03/07 19:20:44  telamon
 * changed Agent constructors so they don't System.exit() on initialize failure
 *
 * Revision 1.2  1999/03/01 06:15:52  yatish
 * Added port names to the agent container and description.  Added the
 * ability to get an agentcontainer to local/net agent loader. (not fully
 * implemented in net yet)
 *
 * Revision 1.1  1999/02/05 21:15:17  telamon
 * agentloader related
 *
 * >
 ****************************************************************************/
public class NetAgentLoader extends ClassLoader{
  private final static boolean debug = false;
  String agentHost;
  int agentPort;
  Hashtable agentTable;
  Socket socketToServer;
  /** This will be null when manually disconnected.
      However, unintended disconnections may occur,
      so explictly check it's state before using it. */
  BufferedInputStream bis;
  ObjectInputStream ois;

  BufferedOutputStream bos;
  ObjectOutputStream oos;
  
  Object serverLock;
  String packageRoot;
  
  boolean tempB = false;
  
  /***************************************************************************
   *
   * Constructor that connects to the AgentServer.
   *
   * @param host the AgentServer hostname
   * @param port the AgentServer port number
   *
   * @return 
   *
   * @exception AgentLoaderException thrown if the directory server cannot
   *                                 be contacted
   *
   **************************************************************************/
  public NetAgentLoader(String host, int port) throws AgentLoaderException{
    super();
    agentHost = host;
    agentPort = port;
    agentTable = new Hashtable();
    serverLock = new Object();
    socketToServer = null;
    bis = null;
    ois= null;

    /*connect to the agent server to ensure that it is up and running.
      note, we're not holding the server lock, but since were in the
      constructor, that's okay */
    connect();
    /* we only call this once */
    getServerPackageRoot();

    /* presumably the code will use the loader soon so
       we'll leave the connection on though the server may choose to
       timeout the connection so must check alive() status before using */
  }
  
  /***************************************************************************
   *
   * Method to see if the server data stream is alive.
   * NOTE: THIS SHOULD ONLY
   * BE CALLED WHILE HOLDING THE SERVERLOCK
   *
   * @param 
   *
   * @return <code>true</code> if the datastream is alive;
   *         <br><code>false</code> otherwise<br>
   *
   * @exception 
   *
   **************************************************************************/
  private boolean alive() {
    if(debug) {
      System.err.println("Got into alive.");
    }
    try {
      if(ois== null) {
        return false;
      }
      else {
        //explicitly check the stream's status 
        ois.available();
        return true;
      }
    }
    catch(IOException ioe) {
      //the connection died so cleanup
      if(debug) {
        System.err.println("Server datastream is dead. Cleaning up.");
      }
      cleanupSockets();
      return false;
    }
  }
  
  /***************************************************************************
   *
   * Connect to the agent server and prepare for future queries.
   * NOTE: THIS SHOULD ONLY BE CALLED WHILE HOLDING THE SERVERLOCK.
   *
   * @param 
   *
   * @return 
   *
   * @exception AgentLoaderException thrown if can't connect to the Agent
   *                                 Server
   *
   **************************************************************************/
  private void connect() throws AgentLoaderException {
    //make a connnection to agent server
    if(debug) {
      System.err.println("Got into connect.");
    }
    
    try {
      if(debug) {
        System.err.println("AgentServer: "+agentHost+" "+agentPort);
      }
      
      socketToServer = new Socket( agentHost, agentPort );
      bis = new BufferedInputStream(socketToServer.getInputStream());
      ois = new ObjectInputStream(bis);

      bos = new BufferedOutputStream(socketToServer.getOutputStream());
      oos = new ObjectOutputStream(bos);
      oos.flush();
      if(debug) {
        System.err.println("Socket to AgentServer established. Handshaking...");
      }

      //sending hello string
      oos.writeObject(AgentServerProtocol.kHELLOCLIENT);
      oos.flush();

      //getting reply
      boolean error = false;
      
      String reply;
      try {
        reply = (String) ois.readObject();

        if(!AgentServerProtocol.kHELLOSERVER.equals(reply)) {
          cleanupSockets();
          throw new AgentLoaderException("Error: AgentServer rejected connection.");
        }
        else {
          if(debug) {
            System.err.println("Handshaking complete. Ready for use...");
          }
          
        }
      }
      catch(ClassNotFoundException cnfe) {
        error = true;
      }
      catch(InvalidClassException ice) {
        error = true;
      }
      catch(StreamCorruptedException sce) {
        error = true;
      }
      catch(OptionalDataException ode) {
        error = true;
      }
      catch(IOException ioe2) {
        error = true;
        ioe2.printStackTrace();
      }
      /* bad response */
      if(error == true) {
        cleanupSockets();
        throw new AgentLoaderException("Error: Corrupted handshake");
      }
    }
    catch(IOException ioe) {
      ioe.printStackTrace();
      cleanupSockets();
      throw new AgentLoaderException("Error: Couldn't connect to AgentServer");
    }
  }
  
  /***************************************************************************
   *
   * Request and receive the AgentServer's PackageRoot.
   * Note this should only be called during initialization.
   * OTHERWISE YOU NEED TO HOLD THE SERVER LOCK WHEN CALLING.
   *
   * @param 
   *
   * @return the AgentServer's package root 
   *
   * @exception AgentLoaderException thrown if the server doesn't respond
   *
   **************************************************************************/
  private void getServerPackageRoot() throws AgentLoaderException {
    if(!alive()) {
      connect();
    }
    byte response;
    try {
      oos.write(AgentServerProtocol.kREQ_PACKAGEROOT);
      oos.flush();

      response = (byte) ois.read();

      if(response == AgentServerProtocol.kOKAY) {
        try {
          packageRoot = (String) ois.readObject();
        }
        catch(Exception except) {
          if(except instanceof IOException) {
            throw (IOException) except;
          }
        }
      }
      else {
        throw new AgentLoaderException(
          "No AgentServer package root available");
      }
    }
    catch(IOException ioe) {
      cleanupSockets();
      throw new AgentLoaderException("Connection to server broken...");
    }
  }
  
  /***************************************************************************
   *
   * Close the link to the AgentServer.
   *
   * @param 
   *
   * @return 
   *
   * @exception 
   *
   **************************************************************************/
  public void disconnect() {
    synchronized(serverLock) {
      try {
        if(alive()) {
          oos.write(AgentServerProtocol.kBYE);
          oos.flush();
        }
      }
      catch (IOException ioe) {
      }
      cleanupSockets();
    }
  }

  /* make sure to call this only while holding serverLock */
  private void cleanupSockets() {
    try {
      if(ois!= null) {
        ois.close();
      }
    }
    catch (IOException ioe) {      
    }

    try {
      if(oos!= null) {
        oos.close();
      }
    }
    catch (IOException ioe) {      
    }

    try {
      if(socketToServer != null) {
        socketToServer.close();
      }
    }
    catch (IOException ioe) {      
    }
    
    ois= null;
    bis= null;
    oos=null;
    bos=null;
    socketToServer = null;
  }
  
  /***************************************************************************
   *
   * Returns the package root of the AgentServer. Browsing should
   * start from here and file system access is restricted to subpaths of
   * the package root.
   *
   * @param 
   *
   * @return the AgentServer's package root
   *
   * @exception 
   *
   **************************************************************************/
  public String getPackageRoot() {
    //this was set in the constructor
    return packageRoot;    
  }
  
  /***************************************************************************
   *
   * Retrieve a class from the AgentServer or local file system if it's
   * designated core (javax.*, java.*, adaptive.core.*, cyberscout.core.*)
   *
   * @param name the class to retrieve from the server
   *
   * @return the Class object for the desired class
   *
   * @exception ClassNotFoundException thrown if the class could not
   *                                   be found on the server
   * @exception AgentLoaderException thrown if communications
   *                                 failed
   *
   **************************************************************************/
  protected Class getClass(String cname, boolean agentCheck, boolean resolve)
    throws ClassNotFoundException, AgentLoaderException {
    Class retVal = null;
    
    boolean errorMess = false;


    synchronized(serverLock) {
      //check if we already have this class loaded
      retVal = (Class) agentTable.get(cname);
      if(retVal == null) {
        
        try {
          retVal = findSystemClass(cname);
          if(debug) {
            System.out.println("Class loaded from System: "+retVal.getName());
          }
        }
        catch (ClassNotFoundException cnfe) {
          //class is not on the local file system so look for it
          //on the ModuleServer
          
          if( cname.startsWith( "java") ||
              cname.startsWith( "javax") ) {
            throw new ClassNotFoundException(
              "Can't load java or javax class "+
              cnfe.getMessage() +
              " from ModuleServer");
          }
          else {
            try {
              //retrieve class from server
              //check alive status
              if(!alive()) {
                connect();
              }
              
              oos.write(AgentServerProtocol.kREQ_CLASS);
              oos.writeObject(cname);
              oos.flush();
              
              byte reply;
              reply = (byte) ois.read();
              
              if(reply == AgentServerProtocol.kNO) {
                throw new ClassNotFoundException(cname+" not on AgentServer");
              }
              else {
                try {
                  byte[] classBytes;
                  classBytes = (byte[]) ois.readObject();
                  retVal = defineClass(null,classBytes,0,classBytes.length);
                }
                catch(Exception except) {
                  if(except instanceof IOException) {
                    throw (IOException) except;
                }
                  else {
                    if(debug) {
                      System.err.println(except.getMessage());
                    }
                    throw new AgentLoaderException(except.getMessage());
                  }
                }
              }
            }
            catch(IOException ioe) {
              cleanupSockets();
              throw new AgentLoaderException(
                "AgentServer connection stream broken.");
            }
          }
        }
        agentTable.put(retVal.getName(), retVal);
      }
        
      if(agentCheck) {
        if(!adaptive.core.Agent.class.isAssignableFrom(retVal)) {
          throw
            new AgentLoaderException(retVal.getName()+
                                     " is not an agent type.");
        }
      }
    }
        
    /* if we get here, we definitely got a valid class file */
    if(resolve) {
      resolveClass(retVal);
    }
    //System.out.println("The class loader for " +
    //                   cname +
    //                   " " +
    //                   retVal.getClassLoader());
    return retVal;
  }
  
  /***************************************************************************
   *
   * Method to create a new Agent given it's fully qualified class name
   *
   * @param agentType the Agent's classname
   *
   * @return a new Agent object
   *
   * @exception ClassNotFoundException thrown if the agentType doesn't
   *                                   resolve to a class
   * @exception AgentLoaderException thrown if specified agentType isn't
   *                                 of type Agent or AgentServer comms failed
   * @exception InstantiationException thrown if class cannot be instantiated
   * @exception IllegalAccessException thrown if the java securitymanager
   *                                   has been violated
   *
   **************************************************************************/
  public Agent newInstance(String agentType) throws ClassNotFoundException,
    InstantiationException, IllegalAccessException, AgentLoaderException {
    if(debug) {
      System.out.println("New Agent instance request: "+agentType);
    }
    
    Class c = getClass(agentType,true,false);
    Agent holder;
    try {      
      holder = (Agent) c.newInstance();
      if(holder.isBadInit()) {
        throw new AgentLoaderException(
          c.getName()+": setNumInputsOutputs() not called in initialize()");
      }
    }
    catch(ClassCastException cce) {
      throw new AgentLoaderException(c.getName()+" is not an agent type.");
    }
    //if we get here then we definitely have an agent class and
    //we've created a new instance of it
    return holder;
  }
  
  /***************************************************************************
   *
   * Support class loading method. Overrides ClassLoader.loadClass()
   *
   * @see java.lang.ClassLoader#loadClass
   *
   * @param name the support class's name
   * @param resolve whether to resolve the class or not
   *
   * @return the Class for the given name
   *
   * @exception ClassNotFoundException thrown when the name isn't
   *                                   a valid class or network communications
   *                                   fail
   *
   **************************************************************************/
  protected Class loadClass( String cname, boolean resolve ) throws 
    ClassNotFoundException {
    if(debug) {
      System.out.println( "loadclass name " + cname );
    }
    
    Class c;
    try {
      c = getClass(cname,false,resolve);
    }
    catch(AgentLoaderException ale) {
      throw new ClassNotFoundException(ale.getMessage());
    }
    return c;
  }
  
  /***************************************************************************
   *
   * List the directories available in the specified directory.
   * The Directory must be a sub directory of the package root or
   * package root itself.
   *
   * @param dir the directory to look in
   *
   * @return an array of Strings containing the directory names
   *
   * @exception FileNotFoundException thrown if dir isn't a dir
   * @exception IllegalAccessException thrown if java securitymanager is
   *                                   violated
   * @exception AgentLoaderException thrown if server communication failed
   *
   ***************************************************************************/
  public String[] listDirectoriesIn(String dir) throws FileNotFoundException, IllegalAccessException, AgentLoaderException {
    if(debug) {
      System.err.println("List dirs: "+dir);
    }
    
    if(isIllegalAccess(dir)) {
      throw new
        IllegalAccessException(dir+" not a subdirectory of package root.");
    }

    String[] retVal = null;

    synchronized(serverLock) {
      try {
        if(!alive()) {
          connect();
        }

        oos.write(AgentServerProtocol.kREQ_LISTDIR);
        oos.writeObject(dir);
        oos.flush();

        byte reply;
        reply = (byte) ois.read();
        
        if(reply == AgentServerProtocol.kOKAY) {
          try {
            retVal = (String[]) ois.readObject();
          }
          catch(Exception except) {
            if(except instanceof IOException) {
              throw (IOException) except;
            }
            retVal = null;
          }
        }
        else if( reply == AgentServerProtocol.kNO) {
          throw new FileNotFoundException(dir);
        }
        else {
          throw new IllegalAccessException(
            dir+" not a subdirectory of package root.");
        }
      }
      catch(IOException ioe) {
        cleanupSockets();
        throw new AgentLoaderException(
          "AgentServer communication stream broken.");
      }
    } //end synchronized
    if(retVal == null) {
      throw new AgentLoaderException("Corrupted directory list reply");
    }
    return retVal;
  }

  /***************************************************************************
   *
   * Get the files available in the specified directory
   *
   * @param dir the directory to search
   *
   * @return an array of Strings containing the file names
   *
   * @exception FileNotFoundException thrown if the dir isn't a directory
   * @exception IllegalAccessException thrown if java securitymanager is
   *                                   violated
   * @exception AgentLoaderException thrown if server communication failed
   *
   **************************************************************************/
  public String[] listFilesIn(String dir) throws FileNotFoundException, IllegalAccessException, AgentLoaderException {

    if(debug) {
      System.err.println("List files: "+dir);
    }
    
    if(isIllegalAccess(dir)) {
      throw new
        IllegalAccessException(dir+" not a subdirectory of package root.");
    }

    String[] retVal = null;

    synchronized(serverLock) {
      try {
        if(!alive()) {
          connect();
        }

        oos.write(AgentServerProtocol.kREQ_LISTFILES);
        oos.writeObject(dir);
        oos.flush();

        byte reply;
        reply = (byte) ois.read();
        
        if(reply == AgentServerProtocol.kOKAY) {
          try {
            retVal = (String[]) ois.readObject();
          }
          catch(Exception except) {
            if(except instanceof IOException) {
              throw (IOException) except;
            }
            retVal = null;
          }
        }
        else if( reply == AgentServerProtocol.kNO) {
          throw new FileNotFoundException(dir);
        }
        else {
          throw new IllegalAccessException(
            dir+" not a subdirectory of package root.");
        }
      }
      catch(IOException ioe) {
        cleanupSockets();
        throw new AgentLoaderException(
          "AgentServer communication stream broken.");
      }
    } //end synchronized
    if(retVal == null) {
      throw new AgentLoaderException("Corrupted file list reply");
    }
    return retVal;
  }

  /***************************************************************************
   *
   * List the Agents available in the specified directory.
   * The names returned are fully qualified java classnames with
   * respect to the package root.
   *
   * @param dir The directory to search.
   *
   * @return an array of strings of the Agent's fully qualified
   *         classnames
   *
   * @exception FileNotFoundException thrown if the specified
   *                                  directory is not a directory
   * @exception IllegalAccessException thrown if the java security manger is
   *                                   violated
   * @exception AgentLoaderException thrown if server communication failed
   *
   **************************************************************************/
  public String[] listAgentsIn(String dir) throws FileNotFoundException, IllegalAccessException, AgentLoaderException {

    if(debug) {
      System.err.println("List agents: "+dir);
    }
    
    if(isIllegalAccess(dir)) {
      throw new
        IllegalAccessException(dir+" not a subdirectory of package root.");
    }

    String[] retVal = null;

    synchronized(serverLock) {
      try {
        if(!alive()) {
          connect();
        }

        oos.write(AgentServerProtocol.kREQ_LISTAGENTS);
        oos.writeObject(dir);
        oos.flush();

        byte reply;
        reply = (byte) ois.read();
        
        if(reply == AgentServerProtocol.kOKAY) {
          try {
            retVal = (String[]) ois.readObject();
          }
          catch(Exception except) {
            if(except instanceof IOException) {
              throw (IOException) except;
            }
            retVal = null;
          }
        }
        else if( reply == AgentServerProtocol.kNO) {
          throw new FileNotFoundException(dir);
        }
        else {
          throw new IllegalAccessException(
            dir+" not a subdirectory of package root.");
        }
      }
      catch(IOException ioe) {
        cleanupSockets();
        throw new AgentLoaderException(
          "AgentServer communication stream broken.");
      }
    } //end synchronized
    if(retVal == null) {
      throw new AgentLoaderException("Corrupted agent list reply");
    }
    return retVal;
  }
  
  /***************************************************************************
   *
   * Checks if the specified path is a file and exists.
   *
   * @param fname the specified path
   *
   * @return <code>true</code> if the path is a file and exists;
   *         <br><code>false</code> otherwise<br>
   *
   * @exception IllegalAccessException thrown if pname is not a sub path
   *                                   of the package root
   * @exception AgentLoaderException thrown if server communication failed
   *
   **************************************************************************/
  public boolean pathIsFile(String pname) throws IllegalAccessException, AgentLoaderException {
    
    if(debug) {
      System.err.println("pathIsFile: "+pname);
    }

    if(isIllegalAccess(pname)) {
      throw new
        IllegalAccessException(pname+" not a subdirectory of package root.");
    }
    boolean retVal;
    synchronized(serverLock) {
      try {      
        if(!alive()) {
          connect();
        }

        oos.write(AgentServerProtocol.kREQ_PATHISFILE);
        oos.writeObject(pname);
        oos.flush();

        byte reply;
        reply = (byte) ois.read();

        if(reply == AgentServerProtocol.kDENIED) {
          throw new IllegalAccessException(
            pname+" not a subdirectory of package root.");
        }
        else if(reply == AgentServerProtocol.kOKAY) {
          retVal = true;
        }
        else {
          retVal = false;
        }
      }
      catch(IOException ioe) {
        cleanupSockets();
        throw new AgentLoaderException(
          "AgentServer communication stream broken");
      }
    } //end synchronized
    return retVal;
  }

  /***************************************************************************
   *
   * Checks if the specified path is a directory and exists.
   *
   * @param fname the specified path
   *
   * @return <code>true</code> if the path is a directory and exists;
   *         <br><code>false</code> otherwise<br>
   *
   * @exception IllegalAccessException thrown if pname is not a sub path
   *                                   of the package root
   * @exception AgentLoaderException thrown if server communication failed
   *
   **************************************************************************/
  public boolean pathIsDir(String pname) throws IllegalAccessException,
    AgentLoaderException {
    
    if(debug) {
      System.err.println("pathIsDir: "+pname);
    }

    if(isIllegalAccess(pname)) {
      throw new
        IllegalAccessException(pname+" not a subdirectory of package root.");
    }
    boolean retVal;
    synchronized(serverLock) {
      try {      
        if(!alive()) {
          connect();
        }

        oos.write(AgentServerProtocol.kREQ_PATHISDIR);
        oos.writeObject(pname);
        oos.flush();

        byte reply;
        reply = (byte) ois.read();

        if(reply == AgentServerProtocol.kDENIED) {
          throw new IllegalAccessException(
            pname+" not a subdirectory of package root.");
        }
        else if(reply == AgentServerProtocol.kOKAY) {
          retVal = true;
        }
        else {
          retVal = false;
        }
      }
      catch(IOException ioe) {
        cleanupSockets();
        throw new AgentLoaderException(
          "AgentServer communication stream broken");
      }
    } //end synchronized
    return retVal;
  }

  /***************************************************************************
   *
   * Checks if the specified path exists.
   *
   * @param fname the specified path
   *
   * @return <code>true</code> if the path exists;
   *         <br><code>false</code> otherwise<br>
   *
   * @exception IllegalAccessException thrown if pname is not a sub path
   *                                   of the package root
   * @exception AgentLoaderException thrown if server communication failed 
   *
   **************************************************************************/
  public boolean pathExists(String pname) throws IllegalAccessException, AgentLoaderException {
    
    if(debug) {
      System.err.println("pathIsDir: "+pname);
    }

    if(isIllegalAccess(pname)) {
      throw new
        IllegalAccessException(pname+" not a subdirectory of package root.");
    }
    boolean retVal;
    synchronized(serverLock) {
      try {      
        if(!alive()) {
          connect();
        }

        oos.write(AgentServerProtocol.kREQ_PATHEXISTS);
        oos.writeObject(pname);
        oos.flush();

        byte reply;
        reply = (byte) ois.read();

        if(reply == AgentServerProtocol.kDENIED) {
          throw new IllegalAccessException(
            pname+" not a subdirectory of package root.");
        }
        else if(reply == AgentServerProtocol.kOKAY) {
          retVal = true;
        }
        else {
          retVal = false;
        }
      }
      catch(IOException ioe) {
        cleanupSockets();
        throw new AgentLoaderException(
          "AgentServer communication stream broken");
      }
    } //end synchronized
    return retVal;
  }

  /***************************************************************************
   *
   * Get a InputStream for the specified filename.
   *
   * @param fname the input file's name
   *
   * @return the InputStream for the specified file
   *
   * @exception IllegalAccessException thrown if fname is not a subpath
   *                                   of the package root
   * @exception FileNotFoundException thrown if fname does not exist
   * @exception AgentLoaderException thrown if server communication failed   
   *
   **************************************************************************/
  public InputStream getFileInputStream(String fname) throws IllegalAccessException, FileNotFoundException, AgentLoaderException {
        
    if(debug) {
      System.err.println("getFileInputStream: "+fname);
    }

    if(isIllegalAccess(fname)) {
      throw new
        IllegalAccessException(fname+" not a subdirectory of package root.");
    }
    InputStream retVal = null;
    synchronized(serverLock) {
      try {      
        if(!alive()) {
          connect();
        }

        oos.write(AgentServerProtocol.kREQ_GETFILEINSTREAM);
        oos.writeObject(fname);
        oos.flush();

        byte reply;
        reply = (byte) ois.read();

        if(reply == AgentServerProtocol.kDENIED) {
          throw new IllegalAccessException(
            fname+" not a subdirectory of package root. Or NOT IMPLEMENTED");
        }
        else if(reply == AgentServerProtocol.kOKAY) {
          throw new AgentLoaderException("NOT implemented");
        }
        else if(reply == AgentServerProtocol.kNO) {
          throw new FileNotFoundException(fname);
        }
      }
      catch(IOException ioe) {
        cleanupSockets();
        throw new AgentLoaderException(
          "AgentServer communication stream broken");
      }
    } //end synchronized
    return retVal;
  }

  /***************************************************************************
   *
   * Get an OutputStream for the specified filename.
   *
   * @param fname the output file's name
   * @param append whether to append to the end of an existing file
   *
   * @return the OutputStream for the specified file
   *
   * @exception IllegalAccessException thrown if fname is not a subpath of
   *                                   the package root
   * @exception IOException thrown if fname cannot be opened for writing
   * @exception AgentLoaderException thrown if server communication failed
   *
   **************************************************************************/
  public OutputStream getFileOutputStream(String fname, boolean append) throws IllegalAccessException, IOException, AgentLoaderException {
        
    if(debug) {
      System.err.println("getFileOutputStream: "+fname);
    }

    if(isIllegalAccess(fname)) {
      throw new
        IllegalAccessException(fname+" not a subdirectory of package root.");
    }
    OutputStream retVal = null;
    synchronized(serverLock) {
      try {      
        if(!alive()) {
          connect();
        }

        oos.write(AgentServerProtocol.kREQ_GETFILEOUTSTREAM);
        oos.writeObject(fname);
        oos.write(append ? 1 : 0);
        oos.flush();

        byte reply;
        reply = (byte) ois.read();

        if(reply == AgentServerProtocol.kDENIED) {
          throw new IllegalAccessException(
            fname+" not a subdirectory of package root. Or NOT IMPLEMENTED");
        }
        else if(reply == AgentServerProtocol.kOKAY) {
          throw new AgentLoaderException("NOT implemented");
        }
        else if(reply == AgentServerProtocol.kNO) {
          throw new AgentLoaderException(
            "Responst to getFileOutputStream was NO. This should not happen. The only proper responses are DENIED and YES, but for the sake of completeness it is here");
        }
      }
      catch(IOException ioe) {
        cleanupSockets();
        throw new AgentLoaderException(
          "AgentServer communication stream broken");
      }
    } //end synchronized
    return retVal;
  }

  /***************************************************************************
   *
   * Get an OutputStream for the specified filename.
   *
   * @param fname the output file's name
   *
   * @return the OutputStream for the specified file
   *
   * @exception IllegalAccessException thrown if fname is not a subpath of
   *                                   the package root
   * @exception IOException thrown if fname cannot be opened for writing
   * @exception AgentLoaderException thrown if server communication failed
   *
   **************************************************************************/
  public OutputStream getFileOutputStream(String fname) throws IllegalAccessException, IOException, AgentLoaderException {
    return getFileOutputStream(fname,false);
  }

  /**************************************************************************
	*
	* Get an AgentContainer given the filename
	*
	* @param fname The file name for the agentContainer
	*
	* @return the agentContainer in fname
	*
   * @exception IllegalAccessException thrown if fname is not a subpath of
   *                                   the package root
   * @exception IOException thrown if fname cannot be opened for writing
   *
   **************************************************************************/
  public SystemConfiguration getSystemConfiguration(String fname) throws IllegalAccessException, IOException {

	 return null;
	 
  }

 public ConfigAndLayout getConfigAndLayout(String fname) throws IllegalAccessException, IOException {
	return null;
 }
  

  
  /***************************************************************************
   *
   * Deletes the specified path.
   *
   * @param pname the full path to the file or directory you want to delete
   *              a directory must be empty for the delete to succeed
   *
   * @return <code>true</code> if the file or directory was deleted;
   *         <br><code>false</code> otherwise<br>
   *
   * @exception IllegalAccessException thrown if the path is not a sub path
   *                                   of the package root
   * @exception AgentLoaderException thrown if server communication failed
   *
   **************************************************************************/
  public boolean delete(String pname) throws IllegalAccessException,
    AgentLoaderException {
    
    if(debug) {
      System.err.println("delete: "+pname);
    }
    
    if(isIllegalAccess(pname)) {
      throw new
        IllegalAccessException(pname+" not a subdirectory of package root.");
    }

    boolean retVal;

    synchronized(serverLock) {
      try {
        if(!alive()) {
          connect();
        }

        oos.write(AgentServerProtocol.kREQ_DELETE);
        oos.writeObject(pname);
        oos.flush();

        byte reply;
        reply = (byte) ois.read();

        if(reply == AgentServerProtocol.kDENIED) {
          throw new IllegalAccessException(
            pname+" not a subdirectory of package root.");
        }        
        else if(reply == AgentServerProtocol.kOKAY) {
          retVal = true;
        }
        else { /* reply == AgentServerProtocol.kNO */
          retVal = false;
        }
      }
      catch(IOException ioe) {
        cleanupSockets();
        throw new AgentLoaderException(
          "AgentServer communication stream broken.");
      }
    } //end synchronized
    return retVal;
  }
  
  /***************************************************************************
   *
   * Creates the specified directory if it doesn't already exist 
   *
   * @param pname the full path to the directory to create
   *
   * @return <code>true</code> if the directory (or directories) could
   *         be created;
   *         <br><code>false</code> otherwise<br>
   *
   * @exception IllegalAccessException thrown if the path is not a sub path
   *                                   of the package root
   * @exception AgentLoaderException thrown if server communication failed
   *
   **************************************************************************/
  public boolean mkdir(String pname) throws IllegalAccessException,
    AgentLoaderException {
    
    if(debug) {
      System.err.println("mkdir: "+pname);
    }
    
    if(isIllegalAccess(pname)) {
      throw new
        IllegalAccessException(pname+" not a subdirectory of package root.");
    }

    boolean retVal;

    synchronized(serverLock) {
      try {
        if(!alive()) {
          connect();
        }

        oos.write(AgentServerProtocol.kREQ_MKDIR);
        oos.writeObject(pname);
        oos.flush();

        byte reply;
        reply = (byte) ois.read();

        if(reply == AgentServerProtocol.kDENIED) {
          throw new IllegalAccessException(
            pname+" not a subdirectory of package root.");
        }
        else if(reply == AgentServerProtocol.kOKAY) {
          retVal = true;
        }
        else { /* reply == AgentServerProtocol.kNO */
          retVal = false;
        }
      }
      catch(IOException ioe) {
        cleanupSockets();
        throw new AgentLoaderException(
          "AgentServer communication stream broken.");
      }
    } //end synchronized
    return retVal;
  }
  
  /***************************************************************************
   *
   * Renames the file or directory specified by src to the
   * one specified by dest. They must both be full paths.
   *
   * @param src the path to rename
   * @param dest the new name
   *
   * @return <code>true</code> if the directory or file was renamed;
   *         <br><code>false</code> otherwise<br>
   *
   * @exception IllegalAccessException thrown if the either path is not
   *                                   a sub path of the package root
   * @exception AgentLoaderException thrown if server communication failed 
   *
   **************************************************************************/
  public boolean rename(String src, String dest) throws IllegalAccessException,
    AgentLoaderException {

    if(isIllegalAccess(src)) {
      throw new
        IllegalAccessException(src + " is not a sub path of package root.");
    }
    if(isIllegalAccess(dest)) {
      throw new
        IllegalAccessException(dest + " is not a sub path of package root.");
    }

    boolean retVal;
    
    synchronized(serverLock) {

      try {
        if(!alive()) {
          connect();
        }

        oos.write(AgentServerProtocol.kREQ_RENAME);
        oos.writeObject(src);
        oos.writeObject(dest);
        oos.flush();

        byte reply;
        reply = (byte) ois.read();

        if(reply == AgentServerProtocol.kDENIED) {
          throw new IllegalAccessException(
            src+" or "+dest+" not a subdirectory of package root.");
        }
        else if(reply == AgentServerProtocol.kOKAY) {
          retVal = true;
        }
        else { /* reply == AgentServerProtocol.kNO */
          retVal = false;
        }
      }
      catch(IOException ioe) {
        cleanupSockets();
        throw new AgentLoaderException(
          "AgentServer communication stream broken.");
      }
    } //end synchronized
    return retVal;
  }
  
  /***************************************************************************
   *
   * Test if the path is a legal AgentServer access.
   * Only paths derived from the package root are legal.
   *
   * @param m the path to check
   *
   * @return <code>true</code> if the path is illegal;
   *         <br><code>false</code> otherwise<br>
   *
   * @exception 
   *
   **************************************************************************/
  private boolean isIllegalAccess(String m) {
    return !m.startsWith(packageRoot);
  }
}
