package adaptive.core;

import java.io.Serializable;
import java.util.*;
/*****************************************************************************
 * <! Copyright 2000, Institute for Complex Engineered Sytems,
 *                    Carnegie Mellon University
 *
 * PROJECT: Adaptable 
 *
 * FILE: MacroDescription.java 
 * >
 *
 * Describes the connections and internal state of an agent. Note: this is
 * only a data storage class. It is the users responsibility to make sure
 * the data stored in this class is consistent for a real agent.
 * 
 * @author yatish?
 * @author Jonathan Jackson <jackson4@andrew.cmu.edu>
 * <!
 * REVISION HISTORY:
 *
 * $Log: MacroDescription.java,v $
 * Revision 1.1.2.6  2000/06/05 05:48:19  jrj
 * more macro fixes.  Wrote a couple more functions to help once we start start storing port types in the AgentDescription and to keep the port information in sync with the PBAgent definition.
 *
 * Revision 1.1.2.5  2000/06/03 19:25:31  jrj
 * fixed an error in mapping inputs on macros.  Added a few diagnostic messages during the system verification step.
 *
 * Revision 1.1.2.4  2000/05/20 19:11:17  jrj
 * fixed a couple more macro bugs.  Took preliminary steps toward making
 * fully recursive macros in MacroDescription.  Have not yet made changes
 * that would damage save file.
 *
 * Revision 1.1.2.3  2000/05/15 22:39:38  jrj
 * Made fixes to macros so that they work.  Changed the executing macro
 * so that it is recursive, can be manipulated as a whole, and its ports
 * can be asked for directly as if it were any other BasicPortAgent.
 * Made several other changes to simplify and assist these changes.
 * Finally purged the last remaining bit of things named clump.
 *
 *
 ****************************************************************************/
public class MacroDescription extends AgentDescription implements Serializable{
  protected Hashtable inputAgentList;
  protected Hashtable outputAgentList;


  /** Constant to signify an input port */
  public static final int INPUT = 0;
  /** Constant to signify an output port */
  public static final int OUTPUT=1;
  
  static final long serialVersionUID = 2687110204160241837L;

  /**
   * Construct a new MacroDescription.
   */
  public MacroDescription() {
    super();
    inputAgentList=new Hashtable();
    outputAgentList=new Hashtable();
    state=new Hashtable();
  }
  
  /** 
    * Construct a new MacroDescription with a specific number of inputs and
    * outputs.
    *
    * @param ni Number of inputs
    * @param no Number of outputs
    */
  public MacroDescription(int ni,int no) {
    this();
    numInputs=ni;
    numOutputs=no;
    inputAgentList=new Hashtable(numInputs);
    outputAgentList=new Hashtable(numOutputs);	
  }
  
  /**
    * Construct a MacroDescription out of a SystemConfiguration.
    * Everything is cloned so the two are independent.
    *
    * Must be careful because the SystemConfiguration contains a copy of one MacroDescription on every host on which any part of that Macro runs
    * @param sc The SystemConfiguration to use.
    */
  public MacroDescription(SystemConfiguration sc) {
    this();
    Vector processedAgents=new Vector();
    System.out.println("Creating macro from system configuration");
    numInputs=sc.getNumInputs();
    numOutputs=sc.getNumOutputs();
    type=sc.getType();
    name=sc.getType();
    //sc.printInputAgentList();
    if (sc.haveExternalPortsBeenPrepared()){
	//set up input ports
	System.out.println("retreiving ports from system configuration");
	for (int x=0;x<sc.getNumInputs();x++) {
	    //System.out.println("looking for element at "+x);
	    if (sc.getInputPortName(x)!=null){
		setInputLocation((PortLocation)sc.getInputAgent(x).clone(),x);
		setInputPortName(sc.getInputPortName(x),x);
	    }
	}
	
	//set up output ports
	for (int x=0;x<sc.getNumOutputs();x++) {
	    if (sc.getOutputPortName(x)!=null){
		setOutputLocation((PortLocation)sc.getOutputAgent(x).clone(),x);
		setOutputPortName(sc.getOutputPortName(x),x);
	    }
	}
    }

    //add internal agents
    Enumeration hostNames=sc.getHostNames();
    Enumeration agents;
    AgentDescription ad;
    String host;
    Hashtable agentList=(Hashtable)state;
    while (hostNames.hasMoreElements()) {
      host=(String)hostNames.nextElement();
      //      System.out.println("On "+host);
      agents=sc.getAgents(host);
      while (agents.hasMoreElements()) {
	ad=(AgentDescription)agents.nextElement();
	if (!processedAgents.contains(ad)){//only want to handle a specific MacroDescription once over all the hosts it sits on
	  agentList.put(ad.getName(),(AgentDescription)ad.clone());
	  //  System.out.println("\t"+ad.getName());
	  processedAgents.addElement(ad);
	}
      }
    }

    if (!sc.haveExternalPortsBeenPrepared()){
	exportUnmappedPorts();//if ports have not been prepared for the macro, prepare them
    }
    else{
	exportUnmappedInputPorts();//always make sure that unmapped inputs are made external
    }

    System.out.println("created macro"+this.toString());
  }
  

     /*********
     * exportUnmappedPorts()
     * set all unmapped ports to connect to outside the macro
     * written as an alternative to a method within ModuleManager and to eventually replace it
     *********/
    public boolean exportUnmappedPorts(){
	inputAgentList=new Hashtable();
	outputAgentList=new Hashtable();
	System.out.println("Exporting all unmapped ports in "+getName());
	AgentDescription ad;
	PortLocation newPort;
	int i,inputNum=0,outputNum=0;
	String source;
	InputLocation il;
	Vector temp;
	Integer sourcePort;
	Hashtable mappedOutputs=new Hashtable();//name-->vector of ints indicating which ports are already mapped
	Enumeration agentDescs=getAgents();
	//export inputs and do setup for exporting outputs
	while (agentDescs.hasMoreElements()){
	    ad=(AgentDescription)agentDescs.nextElement();
	    System.out.println(ad.getName());
	    for (i=0;i<ad.getNumInputs();i++){
		System.out.println("input "+i);
		if (!ad.isInputPortMapped(i)){
		    System.out.println("not mapped");
		    setInputLocation(new PortLocation(ad.getName(),i),inputNum++);
		}
		else{
		    il=ad.getInputMapping(i);
		    //if it is mapped to something external to the macro, we need to export it as well
		    if (il.getAgentID().startsWith(".")){
			setInputLocation(new PortLocation(ad.getName(),i),inputNum++);
		    }
		    System.out.println("mapped to "+il);
		    source=il.getAgentID();
		    sourcePort=new Integer(il.getOutputNum());
		    temp=(Vector)mappedOutputs.get(source);
		    if (temp==null){
			temp=new Vector();
			mappedOutputs.put(source,temp);
		    }
		    if (!temp.contains(sourcePort)) {
			temp.addElement(sourcePort);
		    }
		}
	    }	
	}
	agentDescs=getAgents();
	System.out.println("Checking outputs now");
	while (agentDescs.hasMoreElements()){
	    ad=(AgentDescription)agentDescs.nextElement();
	    System.out.println(ad.getName());
	    temp=(Vector)mappedOutputs.get(ad.getName());
	    if (temp==null){
		for (i=0;i<ad.getNumOutputs();i++){
		    System.out.println(i+" not mapped");
		    setOutputLocation(new PortLocation(ad.getName(),i),outputNum++);
		}
	    }
	    else{
		for (i=0;i<ad.getNumOutputs();i++){
		    System.out.print(i);
		    if (!temp.contains(new Integer(i))){
			System.out.print(" not mapped");
			setOutputLocation(new PortLocation(ad.getName(),i),outputNum++);
		    }
		    System.out.println();
		}
	    }
	}
	this.numInputs=inputNum;
	this.numOutputs=outputNum;
	return true;
    }
 
     /*********
     * exportUnmappedInputPorts()
     * set all unmapped input ports to connect to outside the macro
     * written as an alternative to a method within ModuleManager and to eventually replace it
     *********/
    public boolean exportUnmappedInputPorts(){
	Hashtable oldInputAgentList=this.inputAgentList;
	this.inputAgentList=new Hashtable();
	System.out.println("Exporting all unmapped input ports in "+getName());
	AgentDescription ad;

	int i,inputNum=0;
	InputLocation il;
	
	Enumeration agentDescs=getAgents();
	//export inputs and do setup for exporting outputs
	while (agentDescs.hasMoreElements()){
	    ad=(AgentDescription)agentDescs.nextElement();
	    System.out.println(ad.getName());
	    for (i=0;i<ad.getNumInputs();i++){
		System.out.println("input "+i);
		if (!ad.isInputPortMapped(i)){
		    System.out.println("not mapped");
		    setInputLocation(new PortLocation(ad.getName(),i),inputNum++);
		}
		else{
		    il=ad.getInputMapping(i);
		    //if it is mapped to something external to the macro, we need to export it as well
		    if (il.getAgentID().startsWith(".")){
			setInputLocation(new PortLocation(ad.getName(),i),inputNum++);
		    }
		}
	    }	
	}
	this.numInputs=inputNum;
	return true;
    }
  /**
  * Gets an AgentDescription for the agent with a given name.  Recursively
  * searchs all the macros inside this one until the agent is found.
  * DEPRECATED:jrj 5/19/2000
  * @param name The fully qualified name of the agent to find.
  * @param t The hashtable to look in.
  */
  private AgentDescription getAgentDescriptionFor(String name,Hashtable t) {
    int dot =name.indexOf(".");
    if (dot>0) {
      return getAgentDescriptionFor(name.substring(dot+1),(Hashtable)t.get(name.substring(0,dot)));
    } else {
      return (AgentDescription)(t.get(name));
    }
  }

  /**
    * Helper method for getAgentDescriptionFor.  Recursively searchs for
    * the AgentDescription and returns it along with a fully qualified
    * name of the agent.
    *  * DEPRECATED:jrj 5/19/2000
    * @param pl The PortLocation that contains the agent to look for
    * @param w Whether the agent is on the input or output of this macro.
    * @param t The Hashtable to look in.
    * @param prefix The current prefix of the agent we're looking for
    */
  private PortLocation getAgentDescriptionFor(PortLocation pl,int w,Hashtable t,StringBuffer prefix){
    AgentDescription ad = (AgentDescription)t.get(pl.getAgentName());
    if (ad instanceof MacroDescription) {
      MacroDescription cd=(MacroDescription)ad;
      t=(Hashtable)cd.getState();
      if (w==INPUT) {
	return cd.getAgentDescriptionFor(cd.getInputAgent(pl.getPortNumber()),w,t,prefix.append("."));
      } else {
	return cd.getAgentDescriptionFor(cd.getOutputAgent(pl.getPortNumber()),w,t,prefix.append("."));
      }
    } else {
      return new PortLocation(prefix.toString(),pl.getPortNumber(),ad);
    }
    
  }
  
  /**
    * Recursively searches for and returns the AgentDescription for the
    * agent that PortLocation maps to.  The PortLocation passed in contains
    * a name of an agent or macro, and the portnumber.  If the name is
    * the name of a macro, this will search until it finds an agent.  This
    * is mainly used to get a pointer to the AgentDescription for input
    * mapping.
    *  * DEPRECATED as public:jrj 5/19/2000
    * @param pl The name of the agent and portnumber.
    * @param w INPUT or OUTPUT
    * @return A PortLocation that contains the agentdescription and fully
    * qualified name for the agent passed in.
    */
  public PortLocation getAgentDescriptionFor(PortLocation pl,int w) {
      // System.out.println("table="+((Hashtable)state).toString());
    return getAgentDescriptionFor(pl,w,(Hashtable)state,new StringBuffer(""));
  }
  
  /**
    * Recursively searches for the AgentDescription that name refers too.
    * Name should be a fully qualified name. ie. <code>sys1.potato.agent 
    * </code>
    *  * DEPRECATED as public:jrj 5/19/2000
    * @param name The fully qualified name of the agent to find
    * @return An AgentDescription for name.
    */  
  public AgentDescription getAgentDescriptionFor(String name) {
    //name = name.substring(this.name.length());
    //	 System.out.println(((Hashtable)state).toString());
    return getAgentDescriptionFor(name,(Hashtable)state);
  }
  
  /**
    * Returns the agent and port on that agent that the specified input
    * port maps to.
    *  * DEPRECATED :jrj 5/19/2000 Use getInputMapping or setInputMapping directly.
    * @param port The input port on this macro
    * @return The agent and port on the agent 
    */
  public PortLocation getInputAgent(int port) {
      //    return (PortLocation)inputAgentList.get(new Integer(port));
      return getInputLocation(port);
  }
  
  /**
    * Returns the agent and port on that agent that the specified output
    * port maps to.
    * @param port The output port on this macro
    * @return The agent and port on the agent 
    */
  public PortLocation getOutputAgent(int port) {
      //return (PortLocation)outputAgentList.get(new Integer(port));
      return getOutputLocation(port);
  }
    //* DEPRECATED:jrj 5/19/2000
    //replaced by setInputLocation
  public void setInputAgent(PortLocation pl,int port) {
      //inputAgentList.put(new Integer(port),pl);	 
      setInputLocation(pl,port);
  }
    //* DEPRECATED:jrj 5/19/2000
    //replaced by setOutputLocation
  public void setOutputAgent(PortLocation pl,int port) {
      //outputAgentList.put(new Integer(port),pl);
      setOutputLocation(pl,port);
  }
  
  /***************************************************************************
   *
   * Get the specified input location.
   *
   * @param 
   *
   * @return the sepecified input mapping from the inputs vector.
   *         <br> </br><code>null</code> if the specified input mapping
   *         doesn't exist
   *
   * @exception 
   *
   **************************************************************************/
  public InputLocation getInputMapping(int i) {
      InputLocation results=null;
      if (i<getNumInputs()){
	  int index;
	  //	  System.out.println("Getting input mapping for port "+i+" on "+getName());
	  PortLocation provider=getInputLocation(i);
	  AgentDescription ad=getAgentDescription(provider.getAgentName());
	  int port=provider.getPortNumber();
	  InputLocation il= ad.getInputMapping(port);
	  if (il!=null){
	      String outputAgentName=new String(il.getAgentID());
	      index=outputAgentName.indexOf('.');
	      //System.out.println(il.getAgentID()+" index="+index);
	      outputAgentName=outputAgentName.substring(index+1,outputAgentName.length());
	      results=new InputLocation(outputAgentName,il.getOutputNum(),i);
	  //	  System.out.println(getName()+":"+results);
	  }
      }
      return results;
  }


  /**************************************************************************
  * Set the input mapping at a specific input port. If the vector isn't
  * big enough to support it, increase the size of the vector until it has
  * a spot for the new input mapping.
  *
  * @param il the Input Location to add
  * @param port The port to add it too
  *******************/
  public void setInputMapping(InputLocation il,int i) {
      InputLocation results=null;
      System.out.println("set input mapping port"+i+": "+il);
      PortLocation provider=getInputLocation(i);
      AgentDescription ad=getAgentDescription(provider.getAgentName());
      int port=provider.getPortNumber();
      //      System.out.println("provider="+provider);
      if (il!=null){
	  String outputAgentName=new String("."+il.getAgentID());
	  results=new InputLocation(outputAgentName,il.getOutputNum(),port);
      }
      ad.setInputMapping(results,port);
      if (il!=null){
	  inputs.addElement(il);//this doesn't need to be accurate, because the size of the vector is all that I really care about anymore--eventually I'll rewrite the methods that use it
      }
      else{
	  inputs.removeElementAt(0);
      }
      
  }
    /*****************
    setInputLocation
    set what port on which AgentDescription maps to the ith input port of the macro
    ********************/
    void setInputLocation(PortLocation pl,int i){
	inputAgentList.put(new Integer(i),pl);
    }
    PortLocation getInputLocation(int port){
	return (PortLocation)inputAgentList.get(new Integer(port));
    }
    /*****************
    setOutputLocation
    set what port on which AgentDescription maps to the ith output port of the macro
    ********************/
    void setOutputLocation(PortLocation pl,int i){
	outputAgentList.put(new Integer(i),pl);
    }
    PortLocation getOutputLocation(int port){
	PortLocation retval;
	retval= (PortLocation)outputAgentList.get(new Integer(port));
	return retval;
    }
    public AgentDescription getAgentDescription(String agentName){
	AgentDescription retval=null;
	retval=(AgentDescription)((Hashtable)state).get(agentName);
	return retval;
    }


  public String toString() {
    return toString("");
  }
  
  public String toString(String s) {
    StringBuffer sb=new StringBuffer(s+"MacroDescription:"+name+"\n");
    sb.append(s+" runLocation:"+this.getRunLocation()+"\n");
    sb.append(s+" All runLocations:{");
    Enumeration e;
    e=this.getAllRunLocations();
    while (e.hasMoreElements()){
      sb.append((String)e.nextElement()+(e.hasMoreElements()?", ":""));
    }
    sb.append("}\n");
    for (int x=0;x<numInputs;x++) {
      if (getInputMapping(x)!=null) {
	sb.append(s+" "+getInputMapping(x).toString()+"\n");
      } else {
	sb.append(s+" "+String.valueOf(x)+" NULL\n");
      }
      
    } 
    
    sb.append(s+" InputAgentList:\n");
    e = inputAgentList.elements();
    while (e.hasMoreElements()) {
      sb.append(s+"  "+((PortLocation)e.nextElement()).toString()+"\n");
    }
    sb.append(s+" OutputAgentList:\n");
    e = outputAgentList.elements();
    while (e.hasMoreElements()) {
      sb.append(s+"  "+((PortLocation)e.nextElement()).toString()+"\n");
    }
    
    Hashtable t = (Hashtable)state;
    e = t.elements();
    while (e.hasMoreElements()) {
      sb.append(s+" "+((AgentDescription)e.nextElement()).toString(s+"\t")+"\n");
    }
    
    return sb.toString();
  }
  
  /**
    * Gets all the agents in the macro by recursing through every macro
    * contained within this one.
    * deprecated:  use getAgents() and let any that are Macros trickle changes down themselves recursively
    * @return An Enumeration of all the agents in this macro
    */
  public Enumeration getAllAgents() {
    return new MacroDescriptionEnumerator(this);
  }
  
  /**
    * Gets the AgentDescription for the specified agent name.
    *
    * @param agentName The agent to look for
    * @return The AgentDescription for agentName
    */
  public AgentDescription getAgent(String agentName) {
    return (AgentDescription)((Hashtable)state).get(agentName);
  }
  
  /**
    * Gets the agents and macros in this macro.
    *
    * @return An Enumeration of all the agents and macros in this macro.
    */
  public Enumeration getAgents() {
    return ((Hashtable)state).elements();
  }
  
  /**
    * Add an agent to the macros agent list
    *
    * @param agent AgentDescription to add
    */
  public void addAgent(AgentDescription ad) {
    ((Hashtable)state).put(ad.getName(),ad);
  }
  
  /**
    * Change the run location of all the modules in this macro
    *
    * @param location The new run location
    */
  public void changeHost(String location) { 
    setRunLocation(location);
    Enumeration agents = getAgents();
    while(agents.hasMoreElements()) {
      AgentDescription ad = (AgentDescription)agents.nextElement();
      ad.setRunLocation(location);
    }
  }
  
  /********
  return true if any agent in this macro runs on runLocation
  I should be able to make this more efficient after I rework MacroDescription
  */
  public boolean runsOnLocation(String runLocation){
    if (runLocation==null) return false;
    //System.out.println("Checking runLocation for "+getName());
    boolean retval=false;
    Enumeration e=getAgents();
    AgentDescription ad;
    while (!retval && e.hasMoreElements()){
      ad=(AgentDescription)e.nextElement();
      /*  if (ad instanceof MacroDescription){
	retval |= ((MacroDescription)ad).runsOnLocation(runLocation); 
      } 
      else{*/
	retval |= ad.runsOnLocation(runLocation);
	//      }
      //System.out.println("checking "+ad.getName());
    }
    //    System.out.println("Does any part of "+getName()+" run on "+runLocation+"?"+retval);
    return retval;
  }
  /*************************
  * return all the locations over which this agent runs
  * --will only be one for most agents, but can be more for macros
  * @param none
  * @return Enumeration listing all runLocations
  * @exception none
  **********************/
  public Enumeration getAllRunLocations(){
    //    System.out.println("Getting all runLocations for "+getName());
    Vector temp=new Vector();
    Enumeration agents=getAgents();
    Enumeration locations;
    AgentDescription ad;
    String run;
    //	if (this.location!=null) temp.addElement(this.location);
    
    while (agents.hasMoreElements()){
      ad=(AgentDescription)agents.nextElement();
      if (ad instanceof MacroDescription){
	locations=((MacroDescription)ad).getAllRunLocations();
      }
      else{
	locations=ad.getAllRunLocations();
      }
      
      while (locations.hasMoreElements()){
	run=(String)locations.nextElement();
	if (!temp.contains(run)){
	  temp.addElement(run);
	}
      }
    }
    //    System.out.println(temp);
    return temp.elements();
  }  
  public boolean replaceRunLocation(String oldLocation,String newLocation){
    boolean retval= super.replaceRunLocation(oldLocation,newLocation);
    AgentDescription ad;
    for (Enumeration agents=getAgents();agents.hasMoreElements();){
      ad=(AgentDescription)agents.nextElement();
      retval |=ad.replaceRunLocation(oldLocation,newLocation);
    }
    return retval;
  }
    /***********
     *getInputPortType
     * @return Class indicating the port type 
     *************/
    public Class getInputPortType(int i){
      PortLocation provider=getInputLocation(i);
      if (provider==null) return null;
      String agentName=provider.getAgentName();
      AgentDescription ad=getAgentDescription(agentName);
      int port=provider.getPortNumber();
      return ad.getInputPortType(port);
    }
    /************
     * setInputPortType
     ************/
    public void setInputPortType(Class type, int i){
      PortLocation provider=getInputLocation(i);
      // AgentDescription ad=provider.getAgentDescription();
      String agentName=provider.getAgentName();
      AgentDescription ad=getAgentDescription(agentName);
      int port=provider.getPortNumber();
      ad.setInputPortType(type,port);
    }
    /***********
     * getOutputPortType
     * @return Class indicating the port type 
     *
     *************/
    public Class getOutputPortType(int i){
      PortLocation provider=getOutputLocation(i);
      if (provider==null) return null;
      // System.out.println("getoutputtype provider:"+provider);
      //AgentDescription ad=provider.getAgentDescription();
      String agentName=provider.getAgentName();
      AgentDescription ad=getAgentDescription(agentName);
      int port=provider.getPortNumber();
      return ad.getOutputPortType(port);
    }
    /************
    setOutputPortType
    ************/
    public void setOutputPortType(Class type, int i){
      PortLocation provider=getOutputLocation(i);
      //AgentDescription ad=provider.getAgentDescription();
      String agentName=provider.getAgentName();
      AgentDescription ad=getAgentDescription(agentName);
      int port=provider.getPortNumber();
      ad.setOutputPortType(type,port);
    }

    /***********
     *getInputPortName
     * @return String indicating the port name
     *************/
    /*  public String getInputPortName(int i){
	String retval=null;

	PortLocation provider=getInputLocation(i);
//	AgentDescription ad=provider.getAgentDescription();
      String agentName=provider.getAgentName();
      AgentDescription ad=getAgentDescription(agentName);
	int port=provider.getPortNumber();
	retval= ad.getInputPortName(port);
	
	return retval;
	}*/
    /************
     * setInputPortName
     ************/
    /*    public void setInputPortName(String name, int i){
      PortLocation provider=getInputLocation(i);
      //AgentDescription ad=provider.getAgentDescription();
      String agentName=provider.getAgentName();
      AgentDescription ad=getAgentDescription(agentName);
      int port=provider.getPortNumber();
      ad.setInputPortName(name,port);
      }*/
    /***********
     * getOutputPortName
     * @return String indicating the port name
     *
     *************/
    /*    public String getOutputPortName(int i){
	String retval=null;
	
	PortLocation provider=getOutputLocation(i);
//	AgentDescription ad=provider.getAgentDescription();
      String agentName=provider.getAgentName();
      AgentDescription ad=getAgentDescription(agentName);
	int port=provider.getPortNumber();
	retval= ad.getOutputPortName(port);

	return retval;
	}*/
    /************
    setOutputPortName
    ************/
    /*    public void setOutputPortName(String name, int i){
      PortLocation provider=getOutputLocation(i);
     // AgentDescription ad=provider.getAgentDescription();
      String agentName=provider.getAgentName();
      AgentDescription ad=getAgentDescription(agentName);
      int port=provider.getPortNumber();
      ad.setOutputPortName(name,port);
      }*/
    


  class MacroDescriptionEnumerator implements Enumeration{
    //rewrote so it increments the element pointer correctly
    // jrj 5/9/2000
    Stack enumerationStack;
    Enumeration currentElements;

    MacroDescriptionEnumerator(Hashtable table){
      //deprecated 
      enumerationStack=new Stack();
      currentElements=table.elements();
    }
    MacroDescriptionEnumerator(MacroDescription macro){
      enumerationStack=new Stack();
      currentElements=macro.getAgents();
    }
    public boolean hasMoreElements(){
      return (!enumerationStack.empty() ||currentElements.hasMoreElements());
    }
    public Object nextElement(){
      //if current list of agent descriptions is empty pop off next one
      if (!currentElements.hasMoreElements()){
	if (!enumerationStack.empty()){
	  currentElements=(Enumeration)enumerationStack.pop();
	}
	else{
	  throw new NoSuchElementException("MacroDescriptionEnumerator");
	}
      }
      Object output=currentElements.nextElement();
      if (output instanceof MacroDescription){
	enumerationStack.push(((MacroDescription)output).getAgents());
	output=nextElement();
      }
      return output;
    }
  }
  /*  class MacroDescriptionEnumerator implements Enumeration {
	 boolean keys; 
	 Enumeration elements; 
	 Vector macroDescriptions; 
	 Hashtable table; 
	 AgentDescription ad; 
	 
	 MacroDescriptionEnumerator(Hashtable table) { 
		this.table = table; 
		elements=table.elements(); 
		ad=null; 
		macroDescriptions=new Vector(); //added jrj--2/23/00 
	 } 
	 
	 public boolean hasMoreElements() { 
		if (elements.hasMoreElements()) { 
		  ad=(AgentDescription)elements.nextElement(); 
		  if (ad instanceof MacroDescription) { 
			 macroDescriptions.addElement(ad); 
			 return hasMoreElements(); 
		  } else { 
			 return true; 
		  } 
		} else if (macroDescriptions.size()>0) {
		  table=(Hashtable)macroDescriptions.elementAt(0); 
		  elements=table.keys(); 
		  macroDescriptions.removeElementAt(0); 
		  return hasMoreElements(); 
		} 
		ad=null; 
		return false; 
	 } 
	 
    public Object nextElement() { 
		
		if (ad!=null) { 
		  return ad; 
		} else { 
		  if (hasMoreElements()) return ad; 
		  else			  
			 throw new NoSuchElementException("MacroDescriptionEnumerator"); 
		} 
		
    }
  }  */
} 
