package adaptive.core;

import java.util.Hashtable;
import java.util.Enumeration;
import java.util.zip.*;
import java.io.*;

/*****************************************************************************
 * <! Copyright 1998, Institute for Complex Engineered Sytems,
 *                    Carnegie Mellon University
 *
 * PROJECT: Adaptable 
 *
 * FILE: LocalAgentLoader.java 
 * >
 *
 * Class responsible for loading Agents and support classes from the
 * Local File System. Java automatically calls ClassLoader.loadClass()
 * for an agent's support classes so the NetAgentLoader and
 * LocalAgentLoader must be split up into two seperate class
 * loaders. They will both be contained in AgentLoader.
 * 
 * @author Theodore Q Pham <A HREF="mailto:telamon@CMU.EDU">telamon@CMU.EDU</A>
 *         <br> </br>
 *
 * @version  1.00 12/12/98<br> </br>
 *
 * <!
 * REVISION HISTORY:
 *
 * $Log: LocalAgentLoader.java,v $
 * Revision 1.4.2.2  2000/02/28 00:39:04  telamon
 * less debugging in LocalAgentLoader and NetAgentLoader
 *
 * Revision 1.4.2.1  2000/02/27 22:38:45  telamon
 * safety checking, NetExecutor.java
 *
 * Revision 1.4  1999/04/06 15:11:59  yatish
 * Massive changes to the Save File format along with the addition of Macro
 * support.
 *
 * Revision 1.3  1999/03/07 19:20:43  telamon
 * changed Agent constructors so they don't System.exit() on initialize failure
 *
 * Revision 1.2  1999/03/01 06:15:52  yatish
 * Added port names to the agent container and description.  Added the
 * ability to get an agentcontainer to local/net agent loader. (not fully
 * implemented in net yet)
 *
 * Revision 1.1  1999/02/05 21:15:16  telamon
 * agentloader related
 *
 * >
 ****************************************************************************/
public class LocalAgentLoader extends ClassLoader {
  public static final String kPACKAGEROOT ="./";
  String packageRoot;
  Hashtable agentTable;
  Object lookupLock;
  
  /***************************************************************************
   *
   * Constructor that takes the package root as an argument.
   *
   * @param pr the package root; or null if you want to use the default
   *
   * @return 
   *
   * @exception AgentLoaderException thrown if the package root isn't a real
   *                                 directory
   *
   **************************************************************************/
  public LocalAgentLoader(String pr) throws AgentLoaderException{
    super();
    agentTable = new Hashtable();
    lookupLock = new Object();
    if(pr == null) {
      pr = kPACKAGEROOT;  
    }
    File rootDir = new File(pr);
    if(!rootDir.isDirectory()) {
      throw new AgentLoaderException("Error: Agent Root not a directory.\n");
    }
    try {        
      packageRoot = rootDir.getCanonicalPath();
    }
    catch(IOException ioe) {
      throw new AgentLoaderException(ioe.getMessage());        
    }
  }
  
  /***************************************************************************
   *
   * Returns the package root for the local file system. Browsing should
   * start from here and file system access is restricted to subpaths of
   * the package root.
   *
   * @param 
   *
   * @return the local file system's package root
   *
   * @exception 
   *
   **************************************************************************/
  public String getPackageRoot() {
    return packageRoot;
  }

  protected Class getClass(String cname, boolean agentCheck, boolean resolve)
    throws AgentLoaderException, ClassNotFoundException{
    Class retVal;
    synchronized(lookupLock) {// class lookup needs to be synchronized
      retVal = (Class) agentTable.get(cname);
      if (retVal == null) {
        retVal = Class.forName(cname);
        if(agentCheck) {
          if(!adaptive.core.Agent.class.isAssignableFrom(retVal)) {
            throw
              new AgentLoaderException(retVal.getName()+
                                       " is not an agent type.");
          }
        }
        agentTable.put(retVal.getName(),retVal);
      }
      if(resolve) {
        resolveClass(retVal);        
      }
      
      //System.out.println("The class loader for " +
      //                   cname +
      //                   " " +
      //                   retVal.getClassLoader()); 
      
    } // end class lookup
    return retVal;
  }
  
  /***************************************************************************
   *
   * Method to create a new Agent given it's fully qualified class name
   *
   * @param agentType the Agent's classname
   *
   * @return a new Agent object
   *
   * @exception ClassNotFoundException thrown if the agentType doesn't
   *                                   resolve to a class
   * @exception AgentLoaderException thrown if specified agentType isn't
   *                                 of type Agent
   * @exception InstantiationException thrown if class cannot be instantiated
   * @exception IllegalAccessException thrown if the java securitymanager
   *                                   has been violated
   *
   **************************************************************************/
  public Agent newInstance(String agentType) throws ClassNotFoundException,
    InstantiationException, IllegalAccessException, AgentLoaderException {
    Class c = getClass(agentType,true,false);
    Agent holder;
    try {
      holder = (Agent) c.newInstance();
      if(holder.isBadInit()) {
        throw new AgentLoaderException(
          c.getName()+": setNumInputsOutputs() not called in initialize()");
      }
    }
    catch(ClassCastException cce) {
      throw new AgentLoaderException(c.getName()+" is not an agent type.");
    }
    //if we get here then we definitely have an agent class and
    //we've created a new instance of it
    return holder;
  }
  
  /***************************************************************************
   *
   * Support class loading method. Overrides ClassLoader.loadClass()
   *
   * @see java.lang.ClassLoader#loadClass
   *
   * @param name the support class's name
   * @param resolve whether to resolve the class or not
   *
   * @return the Class for the given name
   *
   * @exception ClassNotFoundException thrown when the name isn't
   *                                   a valid class
   *
   **************************************************************************/
  protected Class loadClass( String cname, boolean resolve ) throws
    ClassNotFoundException {
    System.out.println( "loadClass: " + cname );
    Class c;
    try {
      c = getClass(cname,false, resolve);
    }
    catch (AgentLoaderException ale) {
      throw new ClassNotFoundException(ale.getMessage());
    }
    return c;
  }
  
  /***************************************************************************
   *
   * List the directories available in the specified directory.
   * The Directory must be a sub directory of the package root or
   * package root itself.
   *
   * @param dir the directory to look in
   *
   * @return an array of Strings containing the directory names
   *
   * @exception FileNotFoundException thrown if dir isn't a dir
   * @exception IllegalAccessException thrown if java securitymanager is
   *                                   violated
   *
   ***************************************************************************/
  public String[] listDirectoriesIn(String dir) throws FileNotFoundException, IllegalAccessException {
    if(isIllegalAccess(dir)) {
      throw new
        IllegalAccessException(dir+" not a subdirectory of package root.");
    }
    File directory = new File(dir);
    if(!directory.isDirectory()) {
      throw new FileNotFoundException(dir+" is not a directory.");
    }
    else {
      return directory.list(new DirectoryFilter());
    }
  }
  
  /***************************************************************************
   *
   * Get the files available in the specified directory
   *
   * @param dir the directory to search
   *
   * @return an array of Strings containing the file names
   *
   * @exception FileNotFoundException thrown if the dir isn't a directory
   * @exception IllegalAccessException thrown if java securitymanager is
   *                                   violated
   *
   **************************************************************************/
  public String[] listFilesIn(String dir) throws FileNotFoundException, IllegalAccessException {
    if(isIllegalAccess(dir)) {
      throw new
        IllegalAccessException(dir+" not a subdirectory of package root.");
    }
    
    File directory = new File(dir);
    if(!directory.isDirectory()) {
      throw new FileNotFoundException(dir+" is not a directory.");
    }
    else {
      return directory.list(new FileFilter());
    }
  }


  /***************************************************************************
   *
   * List the Agents available in the specified directory.
   * The names returned are fully qualified java classnames with
   * respect to the package root.
   *
   * @param dir The directory to search.
   *
   * @return an array of strings of the Agent's fully qualified
   *         classnames
   *
   * @exception FileNotFoundException thrown if the specified
   *                                  directory is not a directory
   * @exception IllegalAccessException thrown if the java security manger is
   *                                   violated
   *
   **************************************************************************/
  public String[] listAgentsIn(String dir) throws FileNotFoundException, IllegalAccessException {
    if(isIllegalAccess(dir)) {
      throw new
        IllegalAccessException(dir+" not a subdirectory of package root.");
    }
    File directory = new File(dir);
    if(!directory.isDirectory()) {
      throw new FileNotFoundException(dir+" is not a directory.");
    }
    else {
      String packagePart="";
      try {
        packagePart = directory.getCanonicalPath();
      }
      catch(IOException ioe) {
      }
      
      if(packagePart.equals(packageRoot)) {
        packagePart ="";
      }
      else{
        packagePart = packagePart.substring(packageRoot.length()+1);
        packagePart = packagePart.replace(File.separatorChar,'.');
        packagePart += ".";
      }
      String[] fileNames = directory.list(new AgentNameFilter());
      String[] classes = new String[fileNames.length];

      for(int i = 0; i < fileNames.length;i++) {
        classes[i] = packagePart+
          fileNames[i].substring(0,fileNames[i].indexOf((int) '.'));        
      }
      return classes;      
    }
  }
  
  /***************************************************************************
   *
   * Checks if the specified path is a file and exists.
   *
   * @param fname the specified path
   *
   * @return <code>true</code> if the path is a file and exists;
   *         <br><code>false</code> otherwise<br>
   *
   * @exception IllegalAccessException thrown if pname is not a sub path
   *                                   of the package root
   *
   **************************************************************************/
  public boolean pathIsFile(String pname) throws IllegalAccessException {
    if(isIllegalAccess(pname)) {
      throw new
        IllegalAccessException(pname + " is not a sub path of package root.");
    }
    return (new File(pname)).isFile();    
  }

  /***************************************************************************
   *
   * Checks if the specified path is a directory and exists.
   *
   * @param fname the specified path
   *
   * @return <code>true</code> if the path is a directory and exists;
   *         <br><code>false</code> otherwise<br>
   *
   * @exception IllegalAccessException thrown if pname is not a sub path
   *                                   of the package root
   *
   **************************************************************************/
  public boolean pathIsDir(String pname) throws IllegalAccessException {
    if(isIllegalAccess(pname)) {
      throw new
        IllegalAccessException(pname + " is not a sub path of package root.");
    }
    return (new File(pname)).isDirectory();    
  }

  /***************************************************************************
   *
   * Checks if the specified path exists.
   *
   * @param fname the specified path
   *
   * @return <code>true</code> if the path exists;
   *         <br><code>false</code> otherwise<br>
   *
   * @exception IllegalAccessException thrown if pname is not a sub path
   *                                   of the package root
   *
   **************************************************************************/
  public boolean pathExists(String pname) throws IllegalAccessException {
    if(isIllegalAccess(pname)) {
      throw new
        IllegalAccessException(pname + " is not a sub path of package root.");
    }
    return (new File(pname)).exists();    
  }
  
  /***************************************************************************
   *
   * Get a InputStream for the specified filename.
   *
   * @param fname the input file's name
   *
   * @return the InputStream for the specified file
   *
   * @exception IllegalAccessException thrown if fname is not a subpath
   *                                   of the package root
   * @exception FileNotFoundException thrown if fname does not exist 
   *
   **************************************************************************/
  public InputStream getFileInputStream(String fname) throws IllegalAccessException, FileNotFoundException ,IOException {
    if(isIllegalAccess(fname)) {
      throw new
        IllegalAccessException(fname + " is not a sub path of package root.");
    }
    return new GZIPInputStream(new FileInputStream(fname));    
  }
  
  /***************************************************************************
   *
   * Get an OutputStream for the specified filename.
   *
   * @param fname the output file's name
   * @param append whether to append to the end of an existing file
   *
   * @return the OutputStream for the specified file
   *
   * @exception IllegalAccessException thrown if fname is not a subpath of
   *                                   the package root
   * @exception IOException thrown if fname cannot be opened for writing
   *
   **************************************************************************/
  public OutputStream getFileOutputStream(String fname, boolean append) throws IllegalAccessException, IOException {
    if(isIllegalAccess(fname)) {
      throw new
        IllegalAccessException(fname + " is not a sub path of package root.");
    }
    return new GZIPOutputStream(new FileOutputStream(fname, append));    
  }

  /***************************************************************************
   *
   * Get an OutputStream for the specified filename.
   *
   * @param fname the output file's name
   *
   * @return the SutputStream for the specified file
   *
   * @exception IllegalAccessException thrown if fname is not a
   * subpath of the package root @exception IOException thrown if
   * fname cannot be opened for writing
   *
   **************************************************************************/
  public OutputStream getFileOutputStream(String fname) throws IllegalAccessException, IOException {
    return getFileOutputStream(fname,false);
  }

  /**************************************************************************
	*
	* Get an AgentContainer given the filename
	*
	* @param fname The file name for the agentContainer
	*
	* @return the agentContainer in fname
	*
   * @exception IllegalAccessException thrown if fname is not a subpath of
   *                                   the package root
   * @exception IOException thrown if fname cannot be opened for writing
   *
   **************************************************************************/
  public SystemConfiguration getSystemConfiguration(String fname) throws IllegalAccessException, IOException {
	 ObjectInputStream ois = new ObjectInputStream(getFileInputStream(fname));
	 try {
		SystemConfiguration sc =(SystemConfiguration)ois.readObject();
		return sc;
		
	 } catch (OptionalDataException e) {
		throw new IOException(fname+" does not contain an AgentContainer");
	 } catch (Exception e) {
		throw new IOException("Some random error occured. "+e.toString());
	 }
	 
  }

  public ConfigAndLayout getConfigAndLayout(String fname) throws IllegalAccessException, IOException {
	 ObjectInputStream ois = new ObjectInputStream(getFileInputStream(fname));
	 try {
		SystemConfiguration sc =(SystemConfiguration)ois.readObject();
		Hashtable s = (Hashtable)ois.readObject();
		
		return new ConfigAndLayout(sc,s);
		
	 } catch (OptionalDataException e) {
		throw new IOException(fname+" does not contain an AgentContainer");
	 } catch (Exception e) {
		throw new IOException("Some random error occured. "+e.toString());
	 }
	 
  }
  
  /***************************************************************************
   *
   * Deletes the specified path.
   *
   * @param pname the full path to the file or directory you want to delete
   *              a directory must be empty for the delete to succeed
   *
   * @return <code>true</code> if the file or directory was deleted;
   *         <br><code>false</code> otherwise<br>
   *
   * @exception IllegalAccessException thrown if the path is not a sub path
   *                                   of the package root
   *
   **************************************************************************/
  public boolean delete(String pname) throws IllegalAccessException {
    if(isIllegalAccess(pname)) {
      throw new
        IllegalAccessException(pname + " is not a sub path of package root.");
    }
    return new File(pname).delete();
  }

  /***************************************************************************
   *
   * Creates the specified directory if it doesn't already exist 
   *
   * @param pname the full path to the directory to create
   *
   * @return <code>true</code> if the directory (or directories) could
   *         be created;
   *         <br><code>false</code> otherwise<br>
   *
   * @exception IllegalAccessException thrown if the path is not a sub path
   *                                   of the package root
   *
   **************************************************************************/
  public boolean mkdir(String pname) throws IllegalAccessException {
    if(isIllegalAccess(pname)) {
      throw new
        IllegalAccessException(pname + " is not a sub path of package root.");
    }
    return new File(pname).mkdirs();
  }
  
  /***************************************************************************
   *
   * Renames the file or directory specified by src to the
   * one specified by dest. They must both be full paths.
   *
   * @param src the path to rename
   * @param dest the new name
   *
   * @return <code>true</code> if the directory or file was renamed;
   *         <br><code>false</code> otherwise<br>
   *
   * @exception IllegalAccessException thrown if the either path is not
   *                                   a sub path of the package root
   *
   **************************************************************************/
  public boolean rename(String src, String dest) throws IllegalAccessException{
    if(isIllegalAccess(src)) {
      throw new
        IllegalAccessException(src + " is not a sub path of package root.");
    }
    if(isIllegalAccess(dest)) {
      throw new
        IllegalAccessException(dest + " is not a sub path of package root.");
    }
    return new File(src).renameTo(new File(dest));
  }
  
  /***************************************************************************
   *
   * Test if the path is a legal file system access.
   * Only paths derived from the package root are legal.
   *
   * @param m the path to check
   *
   * @return <code>true</code> if the path is illegal;
   *         <br><code>false</code> otherwise<br>
   *
   * @exception 
   *
   **************************************************************************/
  private boolean isIllegalAccess(String m) {
    return !m.startsWith(packageRoot);
  }  
}
