package adaptive.core;

import java.io.*;

/**
  * Modifications : This class has take CoordinatorProtocol and extended
  * it to perform handshakes between different object that do not 
  * necessarily communicate with the Coordinator. Thus the need was felt
  * to have a separate class (ntolia)
  *
  * The Communications protocol used to talk to the Coordinator.  When the
  * socket is opened create an Object Input/Ouput Stream.  Then send
  * these constants for the protocol.  Initially the person establishing the
  * connection must identify himself.  The otherside will then respond with
  * an OK message and will then identify himself.  The establisher must
  * then respond with an OK.  Then any communications can take place.
  * If a socket is going to be closed, send a GOODBYE message.  It will
  * not be acknowledged.  All other commands will be acknowledged with 
  * either an OK, OKPLUS or ERROR.  OK signifies all is well and to continue
  * with the next message.  OKPLUS signifies that the command was accepted
  * and run, and the next object in the stream contains data that was
  * generated by running the command.  ERROR signifies and error, the next
  * object in the stream will be an Exception of some type.
  *
  * Soon we'll add Event messages. they will not be acknowledged.  This is
  * basically so that if a modulemanager is watching a system, the system
  * can broadcast events and the modulemanager doesn't need to respond to
  * them.
  */
public class HandshakeProtocol {
  /** Signify all is well */
  public static Byte kOK = new Byte((byte)10);
  
  /** Signify OK and that the next object is data to fill a request , when
      the extra object is received properly a kOK should be sent back to
      notify the sender it was received ok. */
  public static Byte kOKPLUS = new Byte((byte)11);
  
  /** Signify an Error occured */
  public static Byte kERROR = new Byte((byte)50);
  
  /** no operation */
  public static Byte kNOOP = new Byte((byte)0);
  
  public static Byte kLISTPROCS = new Byte((byte)102);
  public static Byte kKILLPROCESS = new Byte((byte)103);
  public static Byte kRUNCOMMAND = new Byte((byte)104);
  public static Byte kPROCESSREQUEST = new Byte((byte)105);
  
  public static Byte kSTARTRUN = new Byte((byte)106);
  /* Identify as a Coordinator */
  public static Byte kIAMACOORDINATOR = new Byte((byte)107);

  /** Identify as a ModuleManager */
  public static Byte kIAMAMODULEMANAGER = new Byte((byte)108);
  /** Identify as a NetExecutor */
  public static Byte kIAMANETEXECUTOR = new Byte((byte)109);
  /** Identify as a NetController */
  public static Byte kIAMANETCONTROLLER = new Byte((byte)110);
  /** Identify as a Redirector **/
  public static Byte kIAMAREDIRECTOR = new Byte ((byte)125);
  

  /** send when you are going to close socket */
  public static Byte kGOODBYE = new Byte((byte)111);
  
  /** Tell the modulemanager the system has been started. This message gets
      sent to any modulemanager that's connected to a system that just started.
      Do not use this to acknowledge a kSTARTRUN. That should be acknowledged
      with an OK or ERROR. */
  public static Byte kSYSTEMSTARTED = new Byte((byte)112);
  
  /** tell the other side to receive the configuration for the system, the
      other side responds with an ok, and then the next object it reads in
      is a systemconfiguration. */
  public static Byte kRECEIVESYSTEM = new Byte((byte)113);

  public static Byte kSTOPRUN = new Byte((byte)114);
  public static Byte kNESTARTED = new Byte((byte)115);
  public static Byte kNEERROR = new Byte((byte)116);
  
  public static Byte kPAUSEAG = new Byte((byte)117);
  public static Byte kUNPAUSEAG = new Byte((byte)118);
  public static Byte kSHUTDOWN = new Byte((byte)119);
  public static Byte kGIVEMOVELOCK = new Byte((byte)120);
  public static Byte kREQUESTMOVELOCK = new Byte((byte)121);
  
  /** Tell the other side what agent server to use, the next object
      should be a vector, 1st element a string with the server name, the
      2nd element should be an Integer with the port number */
  public static Byte kUSEAGENTSERVER = new Byte((byte)122);

  public static Byte kREQ_AGENTSERVER = new Byte((byte)123);

  /** request directory server hostname and port number */
  public static Byte kREQ_DIRSERVER = new Byte((byte) 124);

  /** Unexpected object error */
  //public static Byte kUNEXPECTED = new Byte((byte)51);

  /** ClassCastException occured */
  //public static Byte kCANTCAST = new Byte((byte)52);
  

  /** Ask for the SystemConfiguration */
  public static Byte kREQ_SYSCONFIG= new Byte((byte)100);

  /** About to send the SystemConfiguration */
  // public static Byte kSENDING_SYSCONFIG = new Byte(101);
  
  /**
    * Check the object to see if it's an OK signal
    *
    * @param object The Object to check
    *
    * @return true if the object signifies an OK; false otherwise
    */
  public static boolean isOK(Object object)  {
    try {
      return (((Byte)object).equals(kOK));
    } catch (ClassCastException e) {
      return false;
    }	 
  }

  /**
    * Check the object to see if it's an OKPLUS signal
    *
    * @param object The Object to check
    *
    * @return true if the object signifies an OKPLUS; false otherwise
    */
  public static boolean isOKPLUS(Object object)  {
    try {
      return (((Byte)object).equals(kOKPLUS));
    } catch (ClassCastException e) {
      return false;
    }	 
  }

  /**
    * Checks to see if an appropriate sender identification was received and 
    * then acknowledged.  Received the senderID and compares it to senderID, if
    * it's good then an OK is sent.  Then the receiverID is sent and an
    * OK is waited for.
    *
    * @param senderID The sender Identification to check against
    * @param receiverID The Identification to send back to the sender
    * @param inStream the ObjectInputStream to use
    * @param outStream the ObjectOutputStream to use
    * @exception IOException if an IOException was encountered during the
    * process
    */
  public static boolean goodSender(Byte senderID,Byte receiverID,ObjectInputStream inStream,ObjectOutputStream outStream) throws 
    IOException {
    Byte b;
    try {
      System.out.println("Reading sender id");
      b=(Byte)inStream.readObject();
      System.out.println("read in: " + b);
      if (b.equals(senderID)) {
	System.out.println("Writing ok");
	outStream.writeObject(kOK);
	System.out.println("Writing receiver id");
	outStream.writeObject(receiverID);
	b=(Byte)inStream.readObject();
	if (b.equals(kOK)) {
	  System.out.println("all good");
	  return true;
	}
      }
      outStream.writeObject(kERROR);
      outStream.writeObject(new Exception("Unable to Identify"));
    } catch (IOException e) {
      throw e;
    } catch (Exception e) {
      outStream.writeObject(kERROR);
      outStream.writeObject(e);
    }
    return false;
  }
  /**
    * Checks to see if the other side is a good receiver.  First sends the 
    * sender id. Then checks to see if the receiver acknowledges with the 
    * proper id.
    *
    * @param senderID The sender Identification to check against
    * @param receiverID The Identification to send back to the sender
    * @param inStream the ObjectInputStream to use
    * @param outStream the ObjectOutputStream to use
    * @exception IOException if an IOException was encountered during the
    * process
    */
  public static boolean goodReceiver(Byte senderID,Byte receiverID,ObjectInputStream inStream,ObjectOutputStream outStream) throws 
    IOException {
    Byte b;
    try {
      //  System.out.println("Writing sender id");
      outStream.writeObject(senderID);
      //  System.out.println("Reading ok");
      b=(Byte)inStream.readObject();
      if (b.equals(kOK)) {
	//	System.out.println("Reading receiver id");
	b=(Byte)inStream.readObject();
	if (b.equals(receiverID)) {
	  //	  System.out.println("All good.");
	  outStream.writeObject(kOK);
	  return true;
	}
      }
      outStream.writeObject(kERROR);
      outStream.writeObject(new Exception("Unable to Identify"));
    } catch (IOException e) {
      throw e;
    } catch (Exception e) {
      outStream.writeObject(kERROR);
      outStream.writeObject(e);
    }
    return false;
  }
}
