package adaptive.core;

import adaptive.core.util.LinkedList;
import java.util.Hashtable;
import java.io.IOException;


/*****************************************************************************
 * <! Copyright 2000, Institute for Complex Engineered Systems,
 *                    Carnegie Mellon University
 *
 * PROJECT: Adaptable 
 *
 * FILE: EventTriggerThreadPool.java 
 * >
 *
 * The thread pool used to trigger Port based events.
 * 
 * @author Theodore Q Pham <A HREF="mailto:telamon@CMU.EDU">telamon@CMU.EDU</A>
 *         <br> </br>
 *
 * @version  1.00 2/8/2000<br> </br>
 *
 * <!
 * REVISION HISTORY:
 *
 * $Log: EventTriggerThreadPool.java,v $
 * Revision 1.1.2.2  2000/03/16 21:40:30  jrj
 * turned off debug messages
 *
 * Revision 1.1.2.1  2000/02/17 01:17:51  telamon
 * initial demo 2/16 checkin
 *
 * >
 ****************************************************************************/
public class EventTriggerThreadPool extends ThreadGroup {
  protected static final boolean DEBUG=false;
  private static EventTriggerThreadPool poolInstance = null;
  /* Non synchronized LinkedList, must have EventTriggerThreadPool lock to
     access */
  private static LinkedList eventQueue = null;
  /* Must have EventTriggerThreadPool lock to
     access activeTargets */
  private static Hashtable activeTargets = null;
  /* Must have EventTriggerThreadPool lock to
     access numActive

     this is the number of threads that are not currently sleeping,
     this may be different then the number of threads ins the
     activeTargets table
  */
  private static int numActive;

  /*
    true when we want to drain the queue and shutdown the Thread pool
  */
  private static boolean shutdown;
  
  static final int NUM_THREADS = 12;
  static final int TIMEOUT = 900;

  public static void init() {
    if(poolInstance == null) {
      poolInstance = new EventTriggerThreadPool();
    }
  }

  public static void cleanup() {
    if(poolInstance == null) {
      return;  
    }
    poolInstance.drainAndClose();
    poolInstance = null;
  }
  
  private EventTriggerThreadPool() {
    super( "Worker Threads" );
    eventQueue = new LinkedList();
    activeTargets = new Hashtable();
    shutdown = false;
    if (DEBUG) System.out.println( "Starting worker threads - "+ Thread.currentThread() +
                        " - " + System.currentTimeMillis() );
    /* we set all the threads active initially, but they'll all just
       go inactive as soon as they are started
       as each goes inactive, it will decrement the active count */
    numActive = NUM_THREADS;
    for(int i = 0; i < NUM_THREADS; i++) {
      (new EventTriggerThread( "Worker Thread " + i) ).start();
    }
  }
  
  public static EventTriggerThreadPool instance() {
    if(poolInstance == null) {
      init();
    }
    return poolInstance;
  }

  private void drainAndClose() {
      /* join on each of the threads in the thread group */
      Thread[] workers = new Thread[NUM_THREADS];
      int numLeft;
      synchronized(this) {
        shutdown = true;
        numLeft = this.enumerate(workers);
	if (DEBUG)  System.out.println("Number of living workers " + numLeft);
	if (DEBUG)  System.out.println("Waking any sleeping worker threads!");
        notifyAll();
      }
      
      for(int i = 0 ; i < numLeft; i++) {
        boolean finishedJoin = false;
        while(!finishedJoin) {
          try {
            workers[i].join();
            finishedJoin = true;
          }
          catch (InterruptedException ie) {
          }
        }
      }
      if (DEBUG)   System.out.println("All workers shutdown!");
  }
  
  public synchronized void enQueueEvent( TriggerEvent ev ) {
    /* when shutting down, no new events can be put onto the queue otherwise
       draining it could take forever */
    if(shutdown) {
      return;
    } 
    /* put the new event trigger on the event queue.

       if there are inactive threads (threads that are asleep)
       in the pool, wake one up to handle this new event
    */
    eventQueue.addLast( ev );
    if (DEBUG)   System.out.println("Added event to TriggerPool");
    if(numActive < NUM_THREADS ) {
      notify();
      if (DEBUG)   System.out.println("Notifying TriggerPool");
    }
  }
  
  /*        
    return (TriggerEvent) eventQueue.getFirst();
  */
  
  class EventTriggerThread extends Thread {
    /*
      pendingTargetQueue protected by the EventTriggerThread lock,
      must synchronize on EventTriggerThread before you can
      access it
    */
    private LinkedList pendingTriggerQueue;
    
    /*
      the target of this thread is only set when the thread initially does
      a direct dequeue from the ThreadPools event queue and is put into the
      activeTargets hashtable
    */ 
    private BasicPortAgent target;
      
    EventTriggerThread( String name ) {
      super( EventTriggerThreadPool.this, name );
      pendingTriggerQueue = new LinkedList();
      target = null;
      if (DEBUG)   System.out.println("Started "+ name);
    }
    
    public void run() {
      TriggerEvent currTrigger;
      boolean keepGoing = true;
      do {
        /* while the personal queue has stuff on it, process it */
        do {   
          synchronized(this) {
            currTrigger = (TriggerEvent) pendingTriggerQueue.getFirst();
          }
          if(currTrigger != null ) {
	    if (DEBUG)   System.out.println( "Dispatching event - " +
                                this +
                                " - " +
                                System.currentTimeMillis() );
            currTrigger.dispatch();
          }
        } while( currTrigger != null );

        /*
          if we get here, it means our pending queue is clear.
          lock on the ThreadPool,
          check if our pending queue is still clear
        */                     
        synchronized( EventTriggerThreadPool.this ) {
          /* must lock on this thread before can check the pending queue */
          synchronized(this) {
            if(pendingTriggerQueue.size() == 0 ) {
              /*
                the pending queue is really clear and we can now use this
                thread for a new target
                
                remove the target from the activeTarget table,
                note: the target will be null when this thread is started
                for the first time, or after this thread has woken up from
                waiting
              */
              if( target != null ) {
                EventTriggerThreadPool.this.activeTargets.remove(target);
                target = null;
              }

              /*
                now look for new work from the thread pool's event queue,
                note: we still are holding the thread pool's lock and this
                thread's lock
              */
              do {
                currTrigger = (TriggerEvent)
                  EventTriggerThreadPool.this.eventQueue.getFirst();
                
                  
                if( currTrigger == null ) {
                  /*
                    no work in the main eventqueue

                  */
                  if( !EventTriggerThreadPool.this.shutdown ) {
                    /*
                      if we are not shutting down,
                      then make this thread wait and adjust the number
                      of non-sleeping threads
                    */
                    try {
                      EventTriggerThreadPool.this.numActive--;
                      EventTriggerThreadPool.this.wait();
                      EventTriggerThreadPool.this.numActive++;
                    }
                    catch( InterruptedException ie ) {
                    }
                  }
                  else {
                    /*
                      else this thread should quit since we are
                      shutting down
                    */
                    keepGoing = false;                    
                  }
                }
                else {
                  EventTriggerThread ett =
                    (EventTriggerThread) activeTargets.get(currTrigger.dst);
                  
                  if ( ett != null ) {
                    /*
                      this trigger is for an already active target,
                      queue it for that active target's thread
                    */
                    synchronized(ett) {
                      ett.pendingTriggerQueue.addLast(currTrigger);
                    }
                    /* we set this to null so that we'll loop back up
                       and try to process the next triggerevent
                    */
                    currTrigger = null;
                  }
                  else {
                    /*
                      this trigger is not currently active for anyone
                      so we will make it active for the current thread
                      by putting it into the current thread's pending
                      trigger queue and into the activeTarget table with this
                      thread as the owner.
                    */
                    this.target = currTrigger.dst;
                    this.pendingTriggerQueue.addLast(currTrigger);
                    EventTriggerThreadPool.this.activeTargets.put(
                      this.target,
                      this );
                  }
                }
              } while (currTrigger == null && keepGoing );

              /*
                if we get here, either we are shutting down
                or there is more work to do and we have a new target
              */
            }
            else {
              /*
                looks like another thread added more work for us while
                we were waiting to get the thread pool lock,
                better pop out and process more work
              */
	      if (DEBUG)     System.out.println("Work added for target " +
                                 target.getID() +
                                 " while waiting for EventTriggerThreadPool lock, going back to target processing!");
            }
          }
        }
      } while(keepGoing);
    }
  }

  public static void main(String[] argv) {
    EventTriggerThreadPool.init();
    BasicPortAgent bpa1 = new TestBPA();
    BasicPortAgent bpa2 = new TestBPA();
    BasicPortAgent bpa3 = new TestBPA();
    BasicPortAgent bpa4 = new TestBPA();
    BasicPortAgent bpa5 = new TestBPA();
    BasicPortAgent bpa6 = new TestBPA();

    try {
      
      bpa1.setup("bpa1", null, null);
      bpa2.setup("bpa2", null, null);
      bpa3.setup("bpa3", null, null);
      bpa4.setup("bpa4", null, null);
      bpa5.setup("bpa5", null, null);
      bpa6.setup("bpa6", null, null);
    }
    catch(Exception ex) {
    }
    
      
    TriggerEvent tev;
    int[] test = { 1, 2 };
    
    
    System.out.println("Press Q to quit.");
    byte keyPress = 0;
    while(keyPress != (byte) 'q' && keyPress != (byte) 'Q') {
      System.err.print("Quit? ");
      try {
        keyPress = (byte) System.in.read();
        switch(keyPress) {
          case (byte) 'a':
            tev = new TriggerEvent(bpa1, test);
            EventTriggerThreadPool.instance().enQueueEvent(tev);
            break;
          case (byte) 'b':
            tev = new TriggerEvent(bpa2, test);
            EventTriggerThreadPool.instance().enQueueEvent(tev);
            break;
          case (byte) 'c':
            tev = new TriggerEvent(bpa3, test);
            EventTriggerThreadPool.instance().enQueueEvent(tev);
            break;
          case (byte) 'd':
            tev = new TriggerEvent(bpa4, test);
            EventTriggerThreadPool.instance().enQueueEvent(tev);
            break;
          case (byte) 'e':
            tev = new TriggerEvent(bpa5, test);
            EventTriggerThreadPool.instance().enQueueEvent(tev);
            break;
          case (byte) 'f':
            tev = new TriggerEvent(bpa6, test);
            EventTriggerThreadPool.instance().enQueueEvent(tev);
            break;
          default:
            break;
        }
        
      }
      catch (IOException ioe) {
        keyPress = (byte) 'q';
      }
    }
    EventTriggerThreadPool.cleanup();
    //System.exit(0);
  }
}
