package adaptive.core;

import java.net.*;
import comms.core.*;
import java.util.Hashtable;
import java.util.Enumeration;
import java.util.Vector;
import java.io.*;
import adaptive.core.util.Queue;
import adaptive.core.net.DirectoryInfo;


/**
 * Coordinator is the program that keeps track of which hosts are running
 * parts of a system.  ModuleManager sends the SystemConfiguration to
 * a Coordinator.  The Coordinator populates a hashtable for each host
 * with a list of all agents that are running on the host, and all agents
 * that the host needs to communicate with.  It then sends these hashtables
 * to each NetExecutor to run them.
 *
 *
 *  CHANGED! I have to update these comments, they are no longer accurate.
 * Flow of control:
 * 1. Coordinator sits and listens on a given port.
 * 2. ModuleManager contacts coordinator -> CommunicatorThread is created
 * 3. CommunicatorThread talks to modulemanager.  It receives the system
 *    from the modulemanager.
 * 4. CommunicatorThread contacts each NetController and tells them to
 *    start a NetExecutor -> NetExecutorThread is created for each
 *    NetExecutor.
 * 5. The NetExecutorThread communicates with the netexecutor on a specific
 *    host.  It sends the system to it, and waits for any other communications
 *    that have to take place.
 * 6. When all netexecutors have been started and are ready to run, a
 *    SystemThread is created which does all processing for the system.
 * 7. Then a ModuleManagerThread is created. All it does is listen to
 *    the modulemanager and send events to the SystemThread.
 * 8. The Communicator goes away.
 *
 * Now all we have is a SystemThread, ModuleManagerThread and muliple
 * NetExecutorThreads.  If another ModuleManager connects and wants to
 * watch the system, it's added to the SystemThread as a listener.
 *
 * If a messages comes from the modulemanager it goes to the SystemThread.
 * The systemthread then dispatches it to the appropriate NetExecutorThread
 * so it will be sent to the correct NetExecutor.
 *
 * @see NetExecutor,NetController,adaptive.modulemanager.ModuleManager
 *
 * REVISION HISTORY
 * $Log: Coordinator.java,v $
 * Revision 1.1.2.25  2000/06/19 19:26:11  telamon
 * fixes for win32 compatability in module manager, coordinator and net controller
 *
 * Revision 1.1.2.24  2000/05/25 06:56:26  jrj
 * lengthed the socket timeouts a little to help things out on my slow computer at home
 *
 * Revision 1.1.2.23  2000/05/23 00:47:12  jrj
 * fixed exception caught on socket so it works with all JDKs
 *
 * Revision 1.1.2.22  2000/05/22 23:47:15  telamon
 * synched with comms classes reorganization
 *
 * Revision 1.1.2.21  2000/05/22 21:30:03  jrj
 * fixed small exception handling bug
 *
 * Revision 1.1.2.20  2000/05/20 19:11:16  jrj
 * fixed a couple more macro bugs.  Took preliminary steps toward making
 * fully recursive macros in MacroDescription.  Have not yet made changes
 * that would damage save file.
 *
 */
public class Coordinator extends Thread {
  /** The default Directory server IP */
  final static String kDIRIP = "localhost";
  /** The default Directory server port */
  final static int kDIRPORT = 6500;
  final static String kSERVERNAME = "Coordinator"; 
  final static boolean kMLOG = false;
  /** The default NetController port */
  final static int kNCPORT = 7000;
  /** ServerSocket timeout in milliseconds */
  public final static int kTIMEOUT= 10000;
  public final static int kATTEMPTS = 5;
  
  private int ncPort;
  private int dirPort;
  private String dirHost;
  private boolean keepRunning;
  private Vector moduleManagers;
  private Hashtable systems;
  private ServerSocket serverSocket;
  private String name;
  
  /**
    * Constructs a new Coordinator that will connect to the directory server
    * on host:port.  
    */
  public Coordinator(String host,int port,boolean mLog,int ncPort) {
    this.ncPort=ncPort;
    this.dirHost=host;
    this.dirPort=port;
    systems=new Hashtable();
    moduleManagers=new Vector();
  }

  /**
    * @return the hostname of the directory server
    */
  public String getDirHost() {
    return this.dirHost;
  }

  /**
    * @return the port of of the host at which the directory server resides
    */
  public int getDirPort() {
    return this.dirPort;
  }

  /**
    * @return hostname and port of directory server in a vector
    */
  public Vector getDirInfo() {
    Vector v = new Vector(2);
    v.insertElementAt(this.dirHost, 0);
    v.insertElementAt(new Integer(this.dirPort), 1);
    return v;
  }
  /**
    * Remove a SystemThread from the list of active threads.
    *
    * @param nct The SystemThread to remove
    */
  public void removeSystem(String s) {
    systems.remove(s);
  }
  
  public void addSystem(SystemThread systemThread) {
    systems.put(systemThread.getSystemName(),systemThread);
  }
  
  public void removeModuleManagerThread(ModuleManagerThread mmt) {
    moduleManagers.removeElement(mmt);
  }
  
  public String getCoordinatorName() {
    return name;
  }
  
  public void run() {
    System.out.println("started running");
    DirectoryInfo dirInfo=null;
    try {
      Socket connection;
      serverSocket = new ServerSocket(0);
      int localPort = serverSocket.getLocalPort();
      // Connect to the directory server and tell it what port we're
      // listening on
      ModuleManagerThread nct;
		
      //Communications comms = new Communications(kSERVERNAME, dirHost, dirPort,
      //					kMLOG,true);
      // register with the directory server
     
      int rand = (int) (Math.random() * 1000);
      name="Coordinator_"+String.valueOf(rand);  
      dirInfo = new DirectoryInfo(dirHost,dirPort,name);
      System.out.println("port"+localPort);
      dirInfo.registerWithDirectory
	(localPort,comms.core.EntryTypes.ADAPTABLE_COORDINATOR );

      serverSocket.setSoTimeout(kTIMEOUT);
      keepRunning=true;
      System.out.println("Should be up and ready");
      while (keepRunning) {		  
	// listen for a connection
	try {
	  connection = serverSocket.accept();
	  nct=new ModuleManagerThread(connection,this,ncPort);
	  System.out.println("\nStarting new modulemanager thread..");
	  nct.start();
	  moduleManagers.addElement(nct);
	} catch (java.io.IOException ex) {
	  // if there was a time out, do nothing, and try listening
	  // again.  This is only to prevent indefinite blocking on
	  // the accept call.
	}
		  
	yield();
      }

      //dirInfo.unregisterWithDirectory();
      // we need to close all connects to systems and modulemanagers

      /*
	Enumeration c = communicators.elements();
		
	while (c.hasMoreElements()) {
	nct=(ModuleManagerThread)c.nextElement();
	nct.close();
	}
      */

      // serverSocket.close();
		
    } catch (Exception e) {
      System.err.println("Coordinator main thread error:"+e);
      e.printStackTrace();
    }
    try {
      dirInfo.unregisterWithDirectory();	 
    } catch (IOException ioe) {
    }
    
  }

  /**
    * Set the daemon to keep running or not.  At the next socket timeout
    * it will check and stop running if keepRunning is false.
    *
    * @param keepRunning Whether or not to keep running
    */
  public void setKeepRunning(boolean keepRunning) {
    this.keepRunning=keepRunning;
    try {
      serverSocket.close();
    } catch (Exception e) {
    }
    
  }
  

  public static void commandLineError(char ch) throws Exception {
    throw new Exception("Bad Usage of -"+ch+". Use -h or -? for help");
  }

  public static String usage() {
    return "java adaptive.core.Coordinator [options]\n"+
      "Defaults: DirIP = "+kDIRIP + "\n"+
      "          DirPort = "+kDIRPORT + "\n"+
      "          NetController Port = "+kNCPORT+"\n"+
      "          MessageLogging = "+kMLOG+"\n"+
      "Options:\n"+
      "Use -N <DirIP> <DirPort> to override defaults\n"+ 
      "Note: -N overrides -h and -p as well\n"+
      "use -H <DirIP> to override default DirIP\n"+ 
      "Use -p <DirPort> to override default DirPort\n"+
      "use -c <port> to override default NetController Port\n"+
      "Use -m to turn on message logging\n"+
      "Use -? or -h for this help file\n";
  }

  public static void main(String args[]) {
    boolean bigNet = false;
    boolean littleNet = false;
    String hostname = kDIRIP;
    int port = kDIRPORT;
    String rootPath = null;
    boolean mLog = kMLOG;
    int currArg = 0;
    int ncPort=kNCPORT;
	 
    try {
      while(currArg < args.length) {
	if(args[currArg].length() >= 2 && args[currArg].charAt(0) == '-') {
	  switch(args[currArg].charAt(1)) {
	  case 'm':
	    mLog = true;
	    currArg+=1;
	    break;
	  case 'N':
	    if((args.length - currArg-1) >= 2) {
	      hostname = args[currArg+1];
	      port = Integer.parseInt(args[currArg+2]);
	      bigNet = true;
	      currArg+=3;
	      if(littleNet) {
		System.err.println("-N overriding -h and -p");
	      }
	    }
	    else {
	      commandLineError('N');
	    }
	    break;
	  case 'H':
	    if((args.length - currArg-1) >= 1) {
	      if(!bigNet) {
		hostname = args[currArg+1];
		littleNet = true;
	      }
	      else {
		System.err.println("-N already overriding -H");
	      }
	      currArg+=2;
	    }
	    else {
	      commandLineError('H');
	    }
	    break;
	  case 'p':
	    if((args.length - currArg-1) >= 1) {
	      if(!bigNet) {
		port = Integer.parseInt(args[currArg+1]);
		littleNet = true;  
	      }
	      else {
		System.err.println("-N already overriding -p");
	      }
	      currArg+=2;
	    }
	    else {
	      commandLineError('p');
	    }
	    break;
	  case 'c':
	    if((args.length - currArg-1) >= 1) {
	      ncPort = Integer.parseInt(args[currArg+1]);
	      currArg+=2;
	    }
	    else {
	      commandLineError('c');
	    }
	    break;
	  case '?':
	  case 'h':
	    System.err.println(usage());
	    System.exit(0);
	    break;
	  default:
	    throw new Exception("Bad Switch: "+args[currArg]+"\n"+usage());
	  }
	}
	else {
	  throw new Exception("Bad argument: "+args[currArg]+"\n"+usage());
	}
      }
	
      /*
       * resolve the hostname and see if it's valid.
       * if it's localhost, look up the actual name
       */

      boolean goodHost = true;
      
      if(hostname.toLowerCase().equals("localhost")) {
	try {
	  hostname = adaptive.core.net.LocalHost.getFullLocalHostName();
	} 
	catch (UnknownHostException uhe) {
	  goodHost = false;
	  System.err.println("Could not get the real hostname for localhost");
	}
      }
      else {
	hostname = hostname.toLowerCase();
	try {
	  InetAddress.getByName(hostname);
	}
	catch(UnknownHostException uhe) {
	  goodHost = false;
	  System.err.println("Could not resolve the directory server hostname");
	}
      }

      if(goodHost) {
	Coordinator coordinator = new Coordinator(hostname,port,mLog,ncPort);
	coordinator.setDaemon(true);
	coordinator.start();
	
	
	System.out.println("Press Q to quit.");
	byte keyPress = 0;
	while(keyPress != (byte) 'q' && keyPress != (byte) 'Q') {
	  System.err.print("Quit? ");
	  keyPress = (byte) System.in.read();
	}
	coordinator.setKeepRunning(false);
	while(coordinator.isAlive()) {
	}
      }
      System.exit(0);
    }
    catch(Exception e) {
      e.printStackTrace();
      System.err.println("Error in Main");
      System.err.println(e.getMessage());
    }
  }  /* main() */

  /**
    * Returns a list of all the systems known to this coordinator and their 
    * status.
    *
    * @return a list of systems with there status.
    */
  public Vector getSystemList() {
    Vector v = new Vector(systems.size());
    Enumeration e = systems.keys();
    Object temp;
    // loop through all the system names, and put them in
    // a vector and return that
    SystemThread st;
    while (e.hasMoreElements()) {
      //
      // Changes made by ntolia - Two debugging lines which output code
      // Plus.. Enumeration was taking place through the keys and thus
      // it was assumed that what was actually wanted was SystemThreads 
      // (the elements) and thus the change was made. 
      //
      // Then what was inserted  into the vector was not strings but 
      // the object  SystemInfo and thus there was a change made in the 
      // method recieveSystem() where we do a SystemInfo.getName()
      // This should remove all ClassCast exceptions from the code.
      // 
      // However it is to be noted that all these exceptions were 
      // found only when trying to start something after a stop in
      // the NetController
      //

      temp = e.nextElement();
      System.out.println ("temp is of class "+ temp.getClass()+ "and is " + temp );
      st = (SystemThread) systems.get(temp);
      //      st=(SystemThread)e.nextElement();
       System.out.println("St is " + st.getSystemInfo() );
      v.addElement(st.getSystemInfo());
    }
    return v;
  }
  

  public boolean startSystem(String name,ModuleManagerThread starter) {
    SystemThread st = (SystemThread)systems.get(name);
    System.out.print("coor.startSystem..");
    if (st!=null) {
      System.out.println("..dispatching message..");
      // set the modulemanager as a lister to the systemthread
      starter.setSystemThread(st);
      st.addModuleManager(starter);
      st.dispatchMessage(CoordinatorProtocol.kSTARTRUN,starter,null);
    } else {
      System.out.println("..no system.");
      // the system doesn't exist
      return false;
    }
    return true;
  }
 
  
}

/**
  * The thread that handles communicating with a specific system.  Muliple
  * modulemanagers can connect to a system and manipulate it
  */
class SystemThread extends Thread {
  Vector netExecutors;
  Vector moduleManagers;
  Queue messages;
  Thread starter;
  SystemInfo systemInfo;
  int numberOfHosts;
  Coordinator coordinator;
  int ncPort;
  Vector agentServer;
  Hashtable startedPieces;
  
  SystemThread(SystemInfo systemInfo,Coordinator coordinator,int ncPort) {
    this.ncPort=ncPort;
    netExecutors=new Vector();
    moduleManagers=new Vector(5);
    messages=new Queue();
    starter=null;
    this.systemInfo=systemInfo;
    this.coordinator=coordinator;
    agentServer=null;
  }

  /*
   *   
   */
  public Coordinator getCoordinator() {
    return this.coordinator;
  }

  public String getSystemName() {
    return systemInfo.getName();
  }

  public SystemInfo getSystemInfo() {
    return systemInfo;
  }
  

  public void addModuleManager(ModuleManagerThread mmt) {
    if (!moduleManagers.contains(mmt)) 
      moduleManagers.addElement(mmt);
  }
  
  /**
    * Removes a modulemanager from this system.  This does not tell
    * the modulemanager it's being disconnected. Nor does it actually
    * disconnect the modulemanager.  All it does is stop the modulemanager
    * from receiving messages from this sytem.  Technically the modulemanager
    * can still send messages to this system if it's still alive.
    *
    * @param mmt The modulemanager to remove
    */
  public void removeModuleManager(ModuleManagerThread mmt) {
    moduleManagers.removeElement(mmt);
  }
  
  
  /**
    * Inserts a message into the message queue.  The message is
    * then sent to the system to run.
    *
    * @param b The message to send
    * @param mmt The sender
    */
  public void dispatchMessage(Byte b,Thread mmt,Object data) {
    System.out.println("Dispatching a message");
    messages.queueIn(new Message(b,mmt,data));
  }
  
  private void stopSystem(Thread t) {
    System.out.print("stopping all ne's....");
    ModuleManagerThread mmt = (ModuleManagerThread)t;
    if (systemInfo.isRunning()) {
      starter=mmt;
      Enumeration nets=netExecutors.elements();
      NetExecutorThread net;
      while (nets.hasMoreElements()) {
	net=(NetExecutorThread)nets.nextElement();
	try {
	  net.send(CoordinatorProtocol.kSTOPRUN);
	} catch (IOException ioe) {
	  // let the modulemanager know one of the netexecutors bombed out
	  dispatchMessage(CoordinatorProtocol.kERROR,mmt,
			  new Exception
			  ("IOException while contacting "+net.getHost()));
	}
	
      }
      // FIX THIS! verify the things actually stopped!
      System.out.print("...sending ok...");
      mmt.send(CoordinatorProtocol.kOK);
    } else {
      System.out.println("...system not running");
      mmt.send(CoordinatorProtocol.kERROR);
      mmt.send(new Exception("System is not running."));
    }
    System.out.println("...finish stop");
  }

  /**
    * Shut down all the NetExecutors.
    */
  private void shutDown(Thread t) {
    System.out.println("shutting down netexecutors...");
    ModuleManagerThread mmt = (ModuleManagerThread)t;
  
  
    Enumeration nets=netExecutors.elements();
    NetExecutorThread net;
    while (nets.hasMoreElements()) {
      net=(NetExecutorThread)nets.nextElement();
      try {
	net.send(CoordinatorProtocol.kSHUTDOWN);
      } catch (IOException e) {
	// let the modulemanager know one of the netexecutors bombed out
	dispatchMessage(CoordinatorProtocol.kERROR,mmt,
			new Exception
			("IOException while contacting "+net.getHost()));
      }
      
      }
    // FIX THIS! verify the things actually stopped!
    System.out.print("...sending ok...");
    mmt.send(CoordinatorProtocol.kOK);
    System.out.println("...finish shutdown");
  }
  /**
    * Starts up the system by sending a STARTRUN to each NetExecutor.
    *
    * @param mmt The ModuleManager that sent the command
    */
  private void startSystem(Thread mmt) {
    if (agentServer==null) {
      System.out.println("no agentserver");
      // we can't run if we don't have an agent server to use
      ((ModuleManagerThread)mmt).send(CoordinatorProtocol.kERROR);
      ((ModuleManagerThread)mmt).send(new 
				      Exception("No Agent Server specified."));
      return;
    }
    

    if (systemInfo.isIdle()) {
      System.out.println("telling netexecutors..");
      starter=mmt;
      Enumeration nets=netExecutors.elements();
      NetExecutorThread net;
      while (nets.hasMoreElements()) {
	net=(NetExecutorThread)nets.nextElement();
	try {
	  net.send(CoordinatorProtocol.kSTARTRUN);
	} catch (IOException ioe) {
	  // let the modulemanager know one of the netexecutors bombed out
	  dispatchMessage(CoordinatorProtocol.kERROR,mmt,
			  new UnableToStartException
			  ("IOException while contacting "+net.getHost()));
	}
	
      }
      systemInfo.setStatus(SystemInfo.kRUNNING);
    } else {
      ((ModuleManagerThread)mmt).send
	(CoordinatorProtocol.kERROR);
      if (systemInfo.isRunning()) {
	((ModuleManagerThread)mmt).send
	  (new Exception("System is already running."));
      } else {
	((ModuleManagerThread)mmt).send
	  (new Exception("System is initializing."));	  
      }
    }
  }
  
  /**
    * Notifys all modulemanager that are registered listeners of a
    * specific message.
    *
    * @param b the message to send.
    */
  private void notifyModuleManagers(Byte b) {
    Enumeration mmts = moduleManagers.elements();
    ModuleManagerThread mmt;
    while (mmts.hasMoreElements()) {
      System.out.println("notifying a modulemanager.."+b);
      mmt=(ModuleManagerThread)mmts.nextElement();
      mmt.send(b);
    }
  }

  public void closeDown() {
	 
  }
  
  private void processInput() throws InterruptedException {
    // right now i'm not dealing with permissions, or command conflicts,
    // etc.  We'll have to do something about that later.  Personally
    // I wouldn't want one of my running systems to be stopped by
    // some random grad student just because he wants to decrease
    // the load on a computer. :-) 

    Enumeration mmcs;
    NetExecutorThread net;
    //Hashtable startedPieces=null;
    Message message;
    message=(Message)messages.queueOutBlocking();
    if (message.getMessage().equals(CoordinatorProtocol.kSTARTRUN)) {
      // we can only start if the system is idle, if not, let
      // the modulemanager know
      System.out.println("STARTRUN");
      if (systemInfo.isIdle()) startedPieces=new Hashtable();
      startSystem(message.getSender());
    } else if(message.getMessage().equals(CoordinatorProtocol.kSTOPRUN)) {
      stopSystem(message.getSender());
    } else if(message.getMessage().equals(CoordinatorProtocol.
					  kSYSTEMSTARTED)) {
      startedPieces.put(message.getSender(),new Boolean(true));
      // if the number of started equals the number of netexecutors,
      // we're all good and the system is running
      if (startedPieces.size()==netExecutors.size()) {
	// this is short sighted, i should implement an interface
	// so that both netexecutor's and modulemanagers can be
	// a starter.  who knows, maybe a netexecutor will start
	// additional systems someday..
		  
	// reply to the original sender 
	((ModuleManagerThread)starter).send(CoordinatorProtocol.kOK);
	// all the netexecutors have started up the system
	notifyModuleManagers(CoordinatorProtocol.kSYSTEMSTARTED);
      }
		
    } else if(message.getMessage().equals(CoordinatorProtocol.kNEERROR)) {
      Exception e = (Exception)message.getData();
      if(e instanceof UncooperativeNetExecutorException) {
	// this is fatal! 
	((ModuleManagerThread)starter).send(CoordinatorProtocol.kERROR);
	((ModuleManagerThread)starter).send(e);
	throw (new InterruptedException("This system can't run"));
      } else {
	// non fatal exceptions.. just tell the modulemanager and resume
	// communications..
	((ModuleManagerThread)starter).send(CoordinatorProtocol.kERROR);
	((ModuleManagerThread)starter).send(e);
      } 
    } else if (message.getMessage().equals
	       (CoordinatorProtocol.kUSEAGENTSERVER)) {
      // first element is a String of the agentserver host
      // second element is an Integer of the agentserver port
      System.out.println("using agentserver..");
      agentServer = (Vector)message.getData();
      if (message.getSender()!=null) {
	((ModuleManagerThread)message.getSender()).
	  send(CoordinatorProtocol.kOK);
      }
    } else if(message.getMessage().equals
	       (CoordinatorProtocol.kRECEIVESYSTEM)) {
      System.out.println("Receiving system");
      startedPieces=new Hashtable();
      // set the system status to initializtion
      systemInfo.setStatus(SystemInfo.kINIT);
      receiveSystem(message);
    } else if(message.getMessage().equals(CoordinatorProtocol.kSHUTDOWN)) {
      // i should notify other modulemanages that this system is gone
      shutDown(message.getSender());
      throw new InterruptedException("kSHUTDOWN");
      // the run method will catch that and shutdown the system
    } else if (message.getMessage().equals(CoordinatorProtocol.kNESTARTED)) {
      System.out.print("A NetExecutor started. that makes ");
      startedPieces.put(message.getSender(),new Boolean(true));
      System.out.println(startedPieces.size());
      if (startedPieces.size()==numberOfHosts) {
	System.out.println("All the netexecutors are started");
	// all the netexecutors have started
	systemInfo.setStatus(SystemInfo.kIDLE);
	((ModuleManagerThread)starter).send(CoordinatorProtocol.kOK);	
      }
    }
  }
  
  public void run() {
    try {
      while (true) {
	processInput();						
      }
    } catch (InterruptedException ie) {
      // need to close things down here..
    } 
    // close all open netexecutors
    closeNetExecutors(netExecutors);
	 
    // this systemthread is dying, so all modulemanagers that
    // reference it should be told not to send messages to it.
    Enumeration mmts = moduleManagers.elements();
    ModuleManagerThread mmt;
    while (mmts.hasMoreElements()) {
      mmt=(ModuleManagerThread)mmts.nextElement();
      mmt.setSystemThread(null);
    }	 
    // tell the coordinator this sytem no longer exists.
    coordinator.removeSystem(getSystemName());
  }
  
	 

  /**
    * Adds a clump and all the agents/clumps inside of it to the
    * main agentList.  Converts all names to a flat namespace.
    *
    * @param cd MacroDescription to add to the list
    * @param prefix Prefix to append to each name
    * @param agentList the main hashtable with all the hosts->agents
    * @param hostNames a list of all hostnames encountered so far
    */
  //jrj--I think this method is obsolete
  private void addToAgentList(MacroDescription cd,String prefix,
			      Hashtable agentList,Hashtable hostNames) {
    Enumeration agents=cd.getAgents();
    AgentDescription agentDes;
    InputLocation il;
    int x;
    String p;

    prefix="";

    //System.out.println("Processing "+cd.getName()+"   prefix="+prefix);
    while (agents.hasMoreElements()) {
      agentDes=(AgentDescription)agents.nextElement();
      //System.out.println("  Processing "+agentDes.getName());
      // We need to clone the description.  If we try and change
      // the agents name without cloning, the original description will
      // also get changed
		
      // it the whole clump is cloned beforehand, don't need to do it here
      // agentDes=(AgentDescription)agentDes.clone();
		
		
      // modify the InputLocation to reflect full names
      for (x=0;x<agentDes.getNumInputs();x++) {
        il = agentDes.getInputMapping(x);		  
        if (il!=null) {
          int dot;
          p = prefix+cd.getName();
          while (il.getAgentID().charAt(0)=='.') {
            il.setAgentID(il.getAgentID().substring(1));
            dot=p.lastIndexOf(".");
            if (dot>0)
              p=p.substring(0,dot);
            else p="";
          }
          if (!p.equals("")) p=p+".";
          il.setAgentID(p+il.getAgentID());				
        }
      }
		
      /*      if (agentDes instanceof MacroDescription) {
        // add all the agents inside the clump.  
        // we have to put this clump name in front of the agents inside
        // of it so all agents will have unique names
        addToAgentList((MacroDescription)agentDes,prefix+cd.getName()+".", 
		       agentList,hostNames); 
		       } */
      agentDes.setName(prefix+cd.getName()+"."+agentDes.getName());
      System.out.println("Adding "+agentDes.getName()+". Runs on "+agentDes.getRunLocation());
      agentList.put(agentDes.getName(),agentDes);
      Hashtable h =(Hashtable)hostNames.get(agentDes.getRunLocation());
      if (h==null) {
	h = new Hashtable();
      }
      h.put(agentDes.getName(),agentDes);
      hostNames.put(agentDes.getRunLocation(),h);
      
    }
  }
  

  /**
    * Process the SystemConfiguration and create a hashtable with
    * the information about what is running on what host.  Each
    * host will have a hashtable with the AgentDescriptions of all
    * the modules running on the system, along with an AgentDescription
    * for any module that provides input to this system.
    *
    * @param sysConfig the SystemConfiguration to process
    *
    * @return A hashtable indexed by hostname whose elements are
    * hashtables of all the AgentDescriptions on that host
    */
  private Hashtable processConfig(SystemConfiguration sysConfig) {
    AgentDescription agentDes;
    Enumeration agents;
    Enumeration hosts;
    String host;
    Hashtable hostInfo = new Hashtable(); 
    Hashtable agentList = new Hashtable();

    InputLocation il;
    AgentDescription ad;
    AgentDescription realProvider;
    MacroDescription cd;
    PortLocation pl;
    int x;	 
	 
    
    hosts = sysConfig.getHostNames();
    String agentName;
    System.out.println("Parsing system configuration...");
    System.out.println(sysConfig.toString());
    while (hosts.hasMoreElements()) {
      host = (String)hosts.nextElement();
      System.out.println("Looking at "+host);
      agentList=new Hashtable();

      hostInfo.put(host,agentList);
      // agents=sysConfig.getAgentNames(host);
      agents=sysConfig.getAllAgents();//inefficient hack here to make multi-host macros work

      while (agents.hasMoreElements()) {
	//	agentName=(String)agents.nextElement();
	//	agentDes=(AgentDescription)sysConfig.getAgent(host,agentName);
	agentDes=(AgentDescription)agents.nextElement();
	agentName=agentDes.getName();
	//if any part of this agent runs on this host, add it 
	if (agentDes.runsOnLocation(host)){
	  System.out.println("Processing "+agentDes.getName());
	  agentList.put(agentName,agentDes);
	}
      }
      //must add input providers as well
      agents=agentList.elements();
      while (agents.hasMoreElements()){
	agentDes=(AgentDescription)agents.nextElement();
	if (agentDes.runsOnLocation(host)){
	  System.out.println("adding input providers for "+agentDes.getName());
	  for (x=0;x<agentDes.getNumInputs();x++) {
	    il = agentDes.getInputMapping(x);
	    if (il!=null) {
	      System.out.println(" Input "+x+": "+il.toString());
	      // we have to add all the agents that this agent gets it's
	      // inputs from.  Undoubtedly, an agent can be added multiple times fairly easily, but it will just overwrite the older copy of itself.  It is not a problem.
	      ad=(AgentDescription)agentList.get(il.getAgentID());	
	      System.out.println(ad);
	      System.out.println(ad.getName());
	      agentList.put(ad.getName(),ad);
	    }
	  } 
	}
      }
    }
    System.out.println("Got all the hostnames needed");
	 

    System.out.println("Before return hostInfo");
    return hostInfo;
  }
  
  /**
    * Checks to see if we can contact and send system information to
    * all the hosts that are needed to run this system
    *
    * @param hostInfo The hashtable indexed by host of all the agentdesc's
    * 
    * @return true if all the hosts are available, false if not
    */
  private Hashtable checkHosts(Hashtable hostInfo) throws UnknownHostException, NoNetControllerException {
    Enumeration hosts = hostInfo.keys();
    Socket socket;
    String host;
    RemoteConnection remoteConnection;
    ObjectInputStream ois;
    ObjectOutputStream oos;
    Hashtable connectionList=new Hashtable();
	 
    while ((hosts.hasMoreElements()) && (!interrupted())) {
      host = (String)hosts.nextElement();  
      try {
	System.out.println("opening netcontroller on "+host);
	socket = new Socket(InetAddress.getByName(host),ncPort);
	oos = new ObjectOutputStream(socket.getOutputStream());
	oos.flush();
	ois = new ObjectInputStream(socket.getInputStream());
	// check to see if it's really a netcontroller
	if (CoordinatorProtocol.goodReceiver
	    (CoordinatorProtocol.kIAMACOORDINATOR,
	     CoordinatorProtocol.kIAMANETCONTROLLER,ois,oos)) {
	  // the otherside is indeed a netcontroller
	  remoteConnection=new RemoteConnection(socket,oos,ois);
	  System.out.println("Adding "+host+" to rc list");
	  connectionList.put(host,remoteConnection);
	} else {
	  closeNetControllers(connectionList);
	  throw(new NoNetControllerException(host));
	}
	
      } catch (UnknownHostException uhe) {
	closeNetControllers(connectionList);
	throw(uhe);
      } catch (IOException ioe) {
	closeNetControllers(connectionList);
	throw(new NoNetControllerException(host));		  
      } 
    }
    if (interrupted()) {
      closeNetControllers(connectionList);
    }
    return connectionList;
  }
  
  /**
    * Closes all the open socket connections	  
    */
  private void closeNetControllers(Hashtable connectionList) {
    Enumeration hosts = connectionList.keys();
    RemoteConnection s=null;
    while (hosts.hasMoreElements()) {
      s =(RemoteConnection)connectionList.get(hosts.nextElement());
      try {
	s.send(CoordinatorProtocol.kGOODBYE);
      } catch (IOException ioe) {
	// error outputting the goodbye, doesn't really matter..
      }
      s.close();
    }
  }
  
  private void closeNetExecutors(Vector list) {
    Enumeration connections = list.elements();
    NetExecutorThread net;
    while (connections.hasMoreElements()) {
      net = (NetExecutorThread)connections.nextElement();
      // abort - we don't want to send the configuration
      try {
	net.send(CoordinatorProtocol.kGOODBYE);
      } catch (IOException ioe) {
      } catch (NullPointerException npe){
      }
      net.close();
    }
  }
  

  /**
    * Sends the host information to each host.  Each NetController is
    * told to start a NetExecutor.
    *
    * @param hostInfo the hashtable with all the host information in it
    */
  public void startNetExecutors(Hashtable hostInfo,Hashtable connectionList) throws UncooperativeNetControllerException {
    Enumeration hosts = hostInfo.keys();
    Hashtable h;
    String host;
    RemoteConnection rc;
    netExecutors= new Vector(hostInfo.size());
    Enumeration connections;
    NetExecutorThread cs;
    Vector goodConnections=new Vector();
	 
    while ((hosts.hasMoreElements()) && (!interrupted())) {
      host = (String)hosts.nextElement();
      h = (Hashtable)hostInfo.get(host);
      try {
	cs = new NetExecutorThread(h,this,host,agentServer);
      } catch (IOException ioe) {
	System.out.println("startnet exception");
	ioe.printStackTrace();
	closeNetExecutors(goodConnections);
	closeNetControllers(connectionList);
	throw (new UncooperativeNetControllerException(ioe,host));
      }
      // we have to start the listener before we tell NetController
      // to start NetExecutor otherwise NetExecutor won't have anything
      // to connect to
      cs.start();
      goodConnections.addElement(cs);
      rc=(RemoteConnection)connectionList.get(host);
      try {
	// Tell the NetController to start a NetExecutor
	rc.send(CoordinatorProtocol.kPROCESSREQUEST);
	rc.flush();
	// send the information about the listener
	Vector v = new Vector(4);
	System.out.println("created new Vector");
	v.setSize(4);
	v.setElementAt("NE_"+coordinator.getCoordinatorName(),0);
	v.setElementAt(coordinator.getCoordinatorName(),1);
	v.setElementAt(InetAddress.getLocalHost().getHostAddress(),2);
	v.setElementAt(new Integer(cs.getPort()),3);
	rc.send(v);
	rc.flush();
	Byte b = (Byte)rc.receive();
	if (!b.equals(CoordinatorProtocol.kOK)) {
	  // something is wrong.. the netexecutor didn't say
	  // ok to the process request
	  if (b.equals(CoordinatorProtocol.kERROR)) {
	    Exception e = (Exception)rc.receive();
	    // Crap! there was an error.. now we have to close
	    // all the listeners, and tell the user something happened
	    closeNetExecutors(goodConnections);
	    closeNetControllers(connectionList);
	    throw (new UncooperativeNetControllerException(e,host));
	  }
			 
	} else {
	  // no error..
	  netExecutors.addElement(cs);
	}
      } catch (Exception e) {
	System.out.println("startnetexet exception");
	e.printStackTrace();
	closeNetExecutors(goodConnections);
	closeNetControllers(connectionList);
	throw (new UncooperativeNetControllerException(e,host));
      } 
		
    }
    // all the netcontrollers were contacted and they all successfully
    // started netexecutors. now we close all the netcontroller connections
    // because we no longer need them
    closeNetControllers(connectionList);
    if (interrupted()) {
      closeNetExecutors(goodConnections);
    } 
  }
  
  /**
    * The Modulemanager asked the coordinator to receive a system
    * configuration.
    */
  public void receiveSystem(Message message) throws InterruptedException {
    try {
      // Process the configuration file. 
      Hashtable hostInfo = 
	processConfig((SystemConfiguration)message.getData());
      starter=message.getSender();
      // Check to see if all the hosts needed are available
      numberOfHosts=hostInfo.size();
      Hashtable netControllers=checkHosts(hostInfo);
      System.out.println("Starting net executors");
      startNetExecutors(hostInfo,netControllers);

    } catch (Exception e) {
      // let the modulemanager know what happaned, and tell it the
      // command wasn't processed
      System.out.println("receive system exception!");
      e.printStackTrace();
      ((ModuleManagerThread)message.getSender()).send
	(CoordinatorProtocol.kERROR);
      ((ModuleManagerThread)message.getSender()).send(e);
      systemInfo.setStatus(SystemInfo.kLIMBO);
      throw (new InterruptedException("This system can't run."));
    }
  } // receive
  
  class Message {
    Byte message;
    Thread thread;
    Object data;
	 
    Message(Byte m,Thread mmt,Object d) {
      message=m;
      thread=mmt;
      data=d;
    }
	 
    public Object getData() {
      return data;
    }
	 
	 
    public Thread getSender() {
      return thread;
    }
	 
    public Byte getMessage() {
      return message;
    }
	 
  }  
}
  
/**
  * A class to store a connection to a modulemanager.
  */
class ModuleManagerThread extends Thread {
  ObjectInputStream inStream;
  ObjectOutputStream outStream;
  Socket socket;
  boolean closeDown;
  SystemThread systemThread;
  Coordinator coordinator;
  SystemConfiguration sysConfig;
  int ncPort;
  Hashtable connectionList;
  Vector messageQueue;
  Vector netExecutors; 
  
  public ModuleManagerThread(Socket s,Coordinator c,int ncPort) {
    socket=s;
    closeDown=false;
    systemThread=null;
    coordinator=c;
    connectionList = new Hashtable(); 
    this.ncPort = ncPort;
  }
  
  public void send(Object b) {
    try {
      outStream.writeObject(b);
    } catch (IOException ioe) {
      // close the sockets, etc..
      close();
    }
  }
  
  public void setSystemThread(SystemThread st) {
    systemThread=st;
  }
  

  /**
    * Get a list of systems from the coordinator and pass the info back
    * to the modulemanager
    */
  private void listSystems() throws Exception {
    Vector list = coordinator.getSystemList();
    outStream.writeObject(CoordinatorProtocol.kOKPLUS);
    outStream.writeObject(list);
    Byte b = (Byte)inStream.readObject();
    if (b.equals(CoordinatorProtocol.kOK)) {
      // all good
    } else if (b.equals(CoordinatorProtocol.kERROR)) {
      Exception e = (Exception)inStream.readObject();
      throw(new Exception("Error sending System list: "+e.toString()));
    } else {
      throw(new Exception("Unknown response to system list request."));
    }
  }
  
  
  public void processInput() throws IOException {
    Byte b;
    boolean processed=false;
    try {
      b = (Byte)inStream.readObject();
      if (b.equals(CoordinatorProtocol.kSTARTRUN)) {
	// the next object sent should be a string with the system name
	System.out.println("Module manager received start");
	String systemName = (String)inStream.readObject();
	if (!coordinator.startSystem(systemName,this)) {
	  // if we get a false back, the system doesn't exist
	  outStream.writeObject(CoordinatorProtocol.kERROR);
	  outStream.writeObject(new Exception("Unknown System"));
	}
	processed=true;
      } else if(b.equals(CoordinatorProtocol.kLISTPROCS)) {
	// pass back a list of what's running here
	listSystems();
	processed=true;
      } else if(b.equals(CoordinatorProtocol.kSTOPRUN)) {
	systemThread.dispatchMessage(CoordinatorProtocol.kSTOPRUN,this,null);
	processed=true;
      } else if(b.equals(CoordinatorProtocol.kRECEIVESYSTEM)) {
	receiveSystem();			
	processed=true;
      } else if (b.equals(CoordinatorProtocol.kUSEAGENTSERVER)) {
	System.out.println("USEAGENTSERVER");
	Vector v = (Vector)inStream.readObject();
	outStream.writeObject(CoordinatorProtocol.kOK);
	System.out.println("systhread=>"+systemThread);
	systemThread.dispatchMessage(CoordinatorProtocol.kUSEAGENTSERVER,this,
				     v);
	processed=true;
      } else if(b.equals(CoordinatorProtocol.kSHUTDOWN)) {
	processed=true;
	systemThread.dispatchMessage(CoordinatorProtocol.kSHUTDOWN,this,null);
      } else if(b.equals(CoordinatorProtocol.kGOODBYE)) {
	// modulemanager is hanging up
	// tell the system we aren't connected any more
	systemThread.removeModuleManager(this);
	// call interrupt to tell the thread to die off
	interrupt();
	processed=true;
      }
      if (!processed) {
	outStream.writeObject(CoordinatorProtocol.kERROR);
	outStream.writeObject(new Exception("I don't understand "+b));
      }
      
    } catch (IOException ioe) {
      throw ioe;
    } catch (Exception e){
      outStream.writeObject(CoordinatorProtocol.kERROR);
      outStream.writeObject(e);
      e.printStackTrace();
    }
	 
	 
  }
  
  private void receiveSystem() throws Exception {
    outStream.writeObject(CoordinatorProtocol.kOK);
    Vector v=null;
    try {
      v = (Vector)inStream.readObject();
    } catch (ClassCastException e) {
      throw new ClassCastException("Please send a vector, index 0 the "+
				   "systemconfig, index 1 the vector for"+
				   " the agentserver");
    }
    
    SystemConfiguration sysConfig=
      (SystemConfiguration)v.elementAt(0);
    // check to see if a system with that name already is running
    // on this coordinator
    Vector systemList = coordinator.getSystemList();
    Enumeration e = systemList.elements();
    String name;
    Object temp1;

    while (e.hasMoreElements()) {
      // Debugging code by ntolia. Having problems with the entries in the hashtable
      // and then casting them (exceptions being thrown)
      // Check also the changes made in getSystemList.... Assumption is that
      // we wanted to get the name (as cast is made into a string)


      name = (String)( ( (SystemInfo) e.nextElement() ).getName()  );
      if (name.equals(sysConfig.getType())) {
	// the system is already on this coordinator
	throw new Exception("A system of type "+sysConfig.getType()+" already"+
			    " exists on this coordinator.");
      }
    }
    
    SystemInfo si = new SystemInfo(sysConfig.getType());
    SystemThread st = new SystemThread(si,coordinator,ncPort);
    st.addModuleManager(this);
    coordinator.addSystem(st);
    st.start();
    st.dispatchMessage(CoordinatorProtocol.kUSEAGENTSERVER,null,
		       v.elementAt(1));
    st.dispatchMessage(CoordinatorProtocol.kRECEIVESYSTEM,this,sysConfig);
    systemThread=st;
  }
  

  /**
    * Close down the socket and remove this modulemanager's reference
    * from the SystemThread (if we've connected to one).
    *
    */
  public void close() {
    try {
      // remove my reference in the system thread
      if (systemThread!=null) systemThread.removeModuleManager(this);
      outStream.close();
      inStream.close();
      socket.close();
    } catch (Exception e) {
      // facilitate garbage collection
      outStream=null;
      inStream=null;
      socket=null;
    }
  }
  
  public void run() {
    Object o;
    try {
      // open the streams
      inStream = new 
	ObjectInputStream(new BufferedInputStream(socket.getInputStream()));
      outStream = new
	ObjectOutputStream(socket.getOutputStream());
		
      //Byte b = (Byte)inStream.readObject();
      // should first receive a message saying who the connector is
      // we acknowledge it,and identify who we are.
      if (CoordinatorProtocol.goodSender
	  (CoordinatorProtocol.kIAMAMODULEMANAGER,
	   CoordinatorProtocol.kIAMACOORDINATOR,inStream,outStream)) 
	{
	  System.out.println("Good id");
	  while (!isInterrupted()) {
	    processInput();
	  } 
	} else {
	  // if the identifier is wrong, just disconnect the socket.
	  // and exit the thread. close gets called at the end of run
	  System.out.println("Bad id");
	}
    } catch (IOException ioe) {
      // io error, just close down and exit the thread
    } catch (Exception e) {
    }
	 
    close();
  }
  

  public boolean hasData() {
    try {
      return inStream.available()>0;
    } catch (IOException ioe) {
      // Should something happen here? 
    }
    return false;
  }

  /* call interrupt
  public void setCloseDown() {
    closeDown=true;
  }*/
  

}	  

/**
 * The thread that does all the communication from the coordinator to
 * the NetExecutor
 *
 */
class NetExecutorThread extends Thread {
  Hashtable config;
  SystemThread systemThread;
  boolean configurationSent,remainConnected;
  ObjectOutputStream outStream;
  ObjectInputStream inStream;
  Socket socket;
  ServerSocket serverSocket;
  int port;
  String host;
  Vector agentServer;
  
  /**
    * Constructs a new thread to communicate with the NetExecutor.
    *
    * @param config The Configuration to send to the remote host
    * @param messageQueue The queue to insert status messages
    * @param host The host that will be contacting us
    */
  NetExecutorThread(Hashtable config,SystemThread systemThread,String host,
		    Vector agentServer) throws IOException {
    this.config=config;
    this.systemThread=systemThread;
    this.host=host;
    remainConnected=true;
    this.agentServer=agentServer;
    this.serverSocket = new ServerSocket(0);
    this.port = serverSocket.getLocalPort();
    System.out.println("setting port to "+port);
  }

  /**
    * Closes all the streams and sockets
    */
  public void close()  {
    try {
      outStream.close();
      inStream.close();
    } catch (IOException ioe) {
    } catch (NullPointerException e) {
    }
    

    try {
      socket.close();
      serverSocket.close();
    } catch (IOException ioe) {
    } catch (NullPointerException e) {
    }
    

    outStream=null;
    inStream=null;
    socket=null;
    serverSocket=null;	 
  }
  
  /**
    * Get the host name for this NetExecutor.
    *
    * @return The host name for this net executor.
    */
  public String getHost() {
    return host;
  }
  
  /**
    * Send an object to this net executor.
    *
    * @param o The Object to send.
    * @exception IOException if there was an IOException durning the send
    *            NullPointerException if outStream was never created because of an earlier exception
    */
  public void send(Object o) throws IOException,NullPointerException {
    outStream.writeObject(o); 
  }

  /**
    * Get the port number this listener is listening on.
    *
    * @return the port number the lisetener is listening on.
    */  
  public int getPort() {
    //  System.out.println("sending port of "+port);
    return port;
  }
  
  private SystemThread getSysThread() {
    return this.systemThread;
  }

  /**
    * Notify the netexecutor that it has to use a specific agent server
    * to retrieve agents.
    *
    * @param agentServer the agentserver vector that the netexecutor should
    * use.  The first element is a String of the hostname, the second element
    * is an Integer with the port number.
    *
    * @exception Exception If an error was encountered while sending.
    */
  private void sendAgentServer(Vector agentServer) throws Exception {
    outStream.writeObject(CoordinatorProtocol.kUSEAGENTSERVER);
    outStream.writeObject(agentServer);
    Byte b=(Byte)inStream.readObject();
    if (b.equals(CoordinatorProtocol.kOK)) {
      // success
    } else if (b.equals(CoordinatorProtocol.kERROR)) {
      Exception ex = (Exception)inStream.readObject();	
      throw (new UncooperativeNetExecutorException(ex,host));
    } else {
      Exception ex = 
	new Exception("Unknown response to agentserver command.");
      throw (new UncooperativeNetExecutorException(ex,host));
    }
  }
  
  /**
    * Respond to a kREQ_SYSCONFIG comand.  This sends the system to be run
    * on the netexecutor.
    *
    * @exception Exception if there was an error while sending the system.
    */
  private void sendConfiguration() throws Exception {
    outStream.writeObject(CoordinatorProtocol.kOKPLUS);
    outStream.writeObject(config);
    Enumeration e=config.keys();
    while (e.hasMoreElements()) {
      System.out.println("Agent: "+e.nextElement());
    }
    
    Byte b = (Byte)inStream.readObject();
    if (b.equals(CoordinatorProtocol.kOK)) {
      configurationSent=true;
      System.out.println("Dispatching NESTARTED message.");
      systemThread.dispatchMessage(CoordinatorProtocol.kNESTARTED,this,null);
    } else if (b.equals(CoordinatorProtocol.kERROR)) {
      //noError=false;
      UncooperativeNetExecutorException unee = new 
	UncooperativeNetExecutorException
	((Exception)inStream.readObject(),host);
      throw unee;
      //  messageQueue.addElement(unee);
    } else {
      //noError=false;
      outStream.writeObject(CoordinatorProtocol.kERROR);
      Exception ex=new Exception("Unknown Command");
      outStream.writeObject(ex);
      throw (new UncooperativeNetExecutorException(ex,host));
    }
  }
  
  /**
    * Processes the commands received from the input stream.
    *
    * @exception IOException if there was an io exception while processing
    * @exception UncooperativeNetExecutorException if the netexecutor didn't
    * cooperate with a command that was sent to it
    */
  private void mainMessageLoop() throws IOException,UncooperativeNetExecutorException {
    System.out.println("Connected to netexecutor..");
    while (true) {
      try {
	//if (inStream.available()>0) {
	Byte b = (Byte)inStream.readObject();
	  if (b.equals(CoordinatorProtocol.kREQ_SYSCONFIG)) {
	    System.out.println("system requested");
	    sendConfiguration();
	  } else if (b.equals(CoordinatorProtocol.kREQ_AGENTSERVER)) {
	    System.out.print("agentserver requested...");
	    System.out.print(agentServer.elementAt(1));
	    outStream.writeObject(CoordinatorProtocol.kOKPLUS);
	    outStream.writeObject(agentServer);
	    b = (Byte)inStream.readObject();
	    if (b.equals(CoordinatorProtocol.kOK)) {
	      // all good.
	      System.out.println("..sent");
	    } else if (b.equals(CoordinatorProtocol.kERROR)) {
	      Exception e = (Exception)inStream.readObject();
	      systemThread.dispatchMessage(CoordinatorProtocol.kNEERROR,this,
					   e);
	    } else {
	      Exception e = new Exception("Unknown response to agentserver" +
					  "request.");
	      systemThread.dispatchMessage(CoordinatorProtocol.kNEERROR,this,
					   e);
	    }
	    
	  } else if (b.equals(CoordinatorProtocol.kREQ_DIRSERVER)) {
	    System.out.println("dir server requested");
	    /* send info about dir server, e.g. IP address & port no. */
	    this.send(CoordinatorProtocol.kOKPLUS);
	    System.out.println("sent OKPLUS");
	    this.send(this.getSysThread().getCoordinator().getDirInfo());
	    System.out.println("sent dir info");
	  } else if(b.equals(CoordinatorProtocol.kOK)) {
	    // well this is nice. netexecutor successfully did something,
	    // but we don't know what it did.. oh-well..
	  } else if(b.equals(CoordinatorProtocol.kSYSTEMSTARTED)) {
	    System.out.println("Netexecutor started");
	    systemThread.dispatchMessage(b,this,null);
	  } else if(b.equals(CoordinatorProtocol.kGOODBYE)) {
	    // netexecutor is hanging up, but it may still be running
	    interrupt();
	  }
	  //}
      //yield();
      } catch (UncooperativeNetExecutorException e) {
	// some sort of error from the netexecutor. 
	throw e;
      } catch (IOException ioe) {
	// We can't recover from an IOException
	throw ioe;
      } catch (Exception e) {
	// The error isn't fatal, we'll let the other side know
	// what happened over here, and see if we should continue
	outStream.writeObject(CoordinatorProtocol.kERROR);
	outStream.writeObject(e);
      }
    }
  }
  
  /**
    * main thread run method.  Opens a server socket and listens for 
    * incoming connectinos.  Only Coordinator.kATTEMPTS attempts are
    * made to establish a connection.  After that the thread closes.
    * Once a valid netexecutor connects, no more connections are allowed.
    */
  public void run() {
    configurationSent=false;
    boolean noError=true;
    Byte b;
    int attempts=0;
	 
    try {
      // We only wait for a connection for a specified amount of time.
      // If the NetExecutor doesn't contact us we assume something is
      // wrong with it
      serverSocket.setSoTimeout(Coordinator.kTIMEOUT);
      while ((attempts++ < Coordinator.kATTEMPTS) && !isInterrupted()) {
	try {
	  System.out.print("Waiting for netexecutor connection..");
	  socket=serverSocket.accept();
	  System.out.println("...connected");
	  outStream = new ObjectOutputStream(socket.getOutputStream());
	  inStream = new ObjectInputStream(socket.getInputStream());
	  if (CoordinatorProtocol.goodSender(CoordinatorProtocol.kIAMANETEXECUTOR,CoordinatorProtocol.kIAMACOORDINATOR,inStream,outStream)) {
	    attempts=Coordinator.kATTEMPTS;
	    serverSocket.close();
	    mainMessageLoop();
	  }
	} catch (IOException ioe) {
	  System.out.println("...ioexception "+ioe );
	} 
      }

      if (attempts>=Coordinator.kATTEMPTS) {
	Exception e;
	e = new Exception("No Response");
	e = new UncooperativeNetExecutorException(e,host);
	systemThread.dispatchMessage(CoordinatorProtocol.kNEERROR,this,e);
      }
    } catch (SocketException se) {
      // if this exception is caught then we couldn't open a server socket.
      systemThread.dispatchMessage(CoordinatorProtocol.kNEERROR,this,se);
    } catch (UncooperativeNetExecutorException utse) {
      systemThread.dispatchMessage(CoordinatorProtocol.kNEERROR,this,utse);
    }

    close();
  }

}





/**
  * Stores the information about an open Remote Connection
  *
  */
class RemoteConnection {
  //SystemConfiguration config;
  //Hashtable agentList;
  private Socket socket;
  private ObjectOutputStream oos;
  private ObjectInputStream ois;

  /**
    * Construct a new RemoteConnection
    *
    * @param socket The open socket to the host
    * @param oos The ObjectOutputStream
    * @param ois The ObjectInputStream
    */
  RemoteConnection(Socket socket,ObjectOutputStream oos, 
		   ObjectInputStream ois) {
    //agentList=new Hashtable();
    this.socket = socket;
    this.oos = oos;
    this.ois = ois;
  }

  void close() {
    try {
      oos.close();
      ois.close();
    } catch (IOException ioe1) {
      oos=null;
      ois=null;
    }
    try {
      socket.close();
    } catch (IOException ioe2){
      socket=null;
    }
    
  }
  

  ObjectOutputStream getOOS() {
    return oos;
  }
  
  ObjectInputStream getOIS() {
    return ois;
  }
  

  void setOOS(ObjectOutputStream objectOStream) {
    oos = objectOStream;
  }
  
  void setOIS(ObjectInputStream objectIStream) {
    ois = objectIStream;
  }

  void send(Object o) throws IOException {
    oos.writeObject(o);
  }

  Object receive() throws Exception {
    return ois.readObject();
  }
  
  void flush() throws IOException {
    oos.flush();
  }  
}
