package adaptive.core;

import adaptive.modulemanager.*;
import adaptive.core.net.*;
import java.awt.*;
import java.awt.event.*;
import java.util.*;
import java.io.*;

/*****************************************************************************
 *
 * Copyright 2000, Institute for Complex Engineered Systems,
 *                 Carnegie Mellon University
 * PROJECT: Self-adaptive Software
 * DESCRIPTION: Class to execute a single SystemConfiguration.  It brings up a list of available Coordinators.  After one of those is selected, it allows you to Start, Stop, or Shutdown that SystemConfiguration.
 *
 * @author Jonathan R Jackson <A HREF="mailto:jackson4@andrew.cmu.edu">jackson4@andrew.cmu.edu>/A>
 *
 * REVISION HISTORY:
 *
 * $Log: ConfigurationExecutor.java,v $
 * Revision 1.1.2.6  2000/05/25 03:34:01  jrj
 * changed List to full reference of java.awt.List to resolve conflict when using java 1.2 or 1.3
 *
 * Revision 1.1.2.5  2000/05/20 22:16:16  jrj
 * made into Observable so I can send error messages and other
 * information to ModuleManager
 *
 * Revision 1.1.2.4  2000/05/09 19:12:05  jrj
 * added check whether the SystemConfiguration is valid before we try to execute
 *
 * Revision 1.1.2.3  2000/03/07 05:31:06  jrj
 * *** empty log message ***
 *
 * Revision 1.1.2.2  2000/02/20 21:38:03  jrj
 * added main so we can run config files from  command line
 *
 * Revision 1.1.2.1  2000/02/17 07:52:49  jrj
 * created as substitute for equivalent methods buried within
 * ModuleManager.  This way someone can launch a configuration without
 * starting modulemanager.
 *
 *
 * 
 ****************************************************************************/
public class ConfigurationExecutor extends Observable implements ItemListener,ActionListener,Observer{
    public static final String FINISHED="finished execution";
    public static final String STARTED="started execution";

  public static String kDIRIP="localhost";
  public static int kDIRPORT=6500;
  public static boolean kMLOG=false;//is this still used by anything????

  protected DirectoryInfo dirInfo;
  protected SystemConfiguration sc;
  protected CoordinatorInfo coord;

  protected String dirHost;
  protected int dirPort;
  protected String myName;

  Frame listWindow;

  public ConfigurationExecutor(String dirHost, int dirPort, String myName,SystemConfiguration sc){
    this.dirHost=dirHost;
    this.dirPort=dirPort;
    this.myName=myName;
    System.out.println(dirHost+" "+dirPort+" "+myName);

    this.listWindow=new Frame();

    this.listWindow.addWindowListener( new WindowAdapter() {
      public void windowClosing( WindowEvent ev ) {  
	System.out.println("Closing ConfigurationExecutor now"); 
	listWindow.dispose();  
      }  
    } 
		       ); 
    this.sc=sc;

  }

  public void executeConfiguration() throws java.io.IOException,InvalidSystemConfigurationException{
    java.awt.List coordinatorOptions;
    if (!this.sc.getHostNames().hasMoreElements()){
      throw new InvalidSystemConfigurationException("Empty SystemConfiguration");
    }
    this.dirInfo=new DirectoryInfo(dirHost,dirPort,myName);
    this.dirInfo.retrieveDirInfo();
    System.out.println(dirInfo);
    coordinatorOptions=new java.awt.List(10);

    fillList(coordinatorOptions,this.dirInfo.getCoordinators());
    coordinatorOptions.addActionListener(this);
    coordinatorOptions.addItemListener(this);

    this.listWindow.add(coordinatorOptions);

    this.listWindow.pack();
    this.listWindow.show();
  }

  public void shutdown(){
    System.out.println("Shutting down Configuration Executor");
    if (this.coord!=null){
      if (this.coord.isAlive()) this.coord.shutDown();
    }
    this.listWindow.dispose();
    signalFinished();
  }
  
  public void itemStateChanged(ItemEvent e){
    // System.out.println("entered itemStateChanged");
    if (e.getSource() instanceof java.awt.List){
      PhoneEntry pe=dirInfo.getAgentServerEntry();
      Vector v=new Vector();
      v.addElement(pe.getHost());
      v.addElement(new Integer(pe.getPort()));
      Object[] o = e.getItemSelectable().getSelectedObjects();
      pe = dirInfo.getCoordinatorEntry((String)o[0]);

      CoordinatorInfo ci=new CoordinatorInfo(pe.getHost(),pe.getPort(),this.sc,v);
      //System.out.println("About to start...");
      this.coord=ci;
      this.coord.addObserver(this);
      this.coord.start();

      this.listWindow.dispose();
    }

  }

  private void fillList(java.awt.List l,Enumeration e){
    while(e.hasMoreElements()){
      String s = (String)e.nextElement();
      //      System.out.println(s);
      l.addItem(s);
    }
  }
  public void actionPerformed(ActionEvent evt){
    System.out.println(evt.toString());
  }

    public void signalError(Exception e){
	setChanged();
	notifyObservers(e);
    }
    public void signalFinished(){
	setChanged();
	notifyObservers(FINISHED);
    }
    public void signalStarted(){
	setChanged();
	notifyObservers(STARTED);
    }

    public void update(Observable o,Object arg){
	//	System.out.println("Received message:"+arg+" from "+o);
	//used to receive messages from CoordinatorInfo
	//and act appropriately
	Exception e;
	String message;
	if (arg instanceof Exception){
	    e=(Exception)arg;
	    signalError(e);
	}
	else if (arg instanceof String){
	    message=(String)arg;
	    if (ConfigurationExecutor.FINISHED.equals(message)){
		this.shutdown();
	    }
	}
	else{
	    System.out.println(this+" received unknown message format "+arg);
	}

    }
    //static methods for command line execution

  public static String usage() {
    return
      "java adaptive.core.ConfigurationExecutor filename [options]\n"+
      "Defaults: DirIP = "+kDIRIP + "\n"+
      "          DirPort = "+kDIRPORT + "\n"+
      "          RootPath = "+adaptive.core.LocalAgentLoader.kPACKAGEROOT+"\n"+
      "          MessageLogging = "+kMLOG+"\n"+
      "Options:\n"+
      "Use -N <DirIP> <DirPort> to override defaults\n"+ 
      "Note: -N overrides -h and -p as well\n"+
      "use -H <DirIP> to override default DirIP\n"+ 
      "Use -p <DirPort> to override default DirPort\n"+ 
      "Use -r <RootPath> to set the root path\n"+ 
      "Use -m to turn on message logging\n"+
      "Use -h for this help file\n";
  } 
  public static void commandLineError(char ch) throws Exception {
    throw new Exception("Bad Usage of -"+ch+". Use -h for help");
  }
  private static ConfigurationExecutor processArgs(String[] argv){
    ConfigurationExecutor exec=null;
    boolean netOn = true;
    boolean bigNet = false;
    boolean littleNet = false;
    String hostname = kDIRIP;
    int port = kDIRPORT;
    String rootPath = adaptive.core.LocalAgentLoader.kPACKAGEROOT;
    boolean mLog = kMLOG;
    int currArg = 0;

    String filename;

    try {
      if(argv.length < 1) {
        System.err.println(usage());
        throw new Exception("");
      }
      filename=argv[0];
      currArg = 1;
      while(currArg < argv.length) {
	if(argv[currArg].length() >= 2 && argv[currArg].charAt(0) == '-') {
	  switch(argv[currArg].charAt(1)) {

            case 'r': 
              if((argv.length - currArg -1) >= 1) {
                rootPath = argv[currArg +1];
                currArg+=2;
              }
              else {
                commandLineError('r');
              }
              break;
            case 'm':
              mLog = true;
              currArg+=1;
              break;
            case 'N':
              if((argv.length - currArg-1) >= 2) {
                netOn = true;
                hostname = argv[currArg+1];
                port = Integer.parseInt(argv[currArg+2]);
                bigNet = true;
                currArg+=3;
                if(littleNet) {
                  System.err.println("-N overriding -h and -p");
                }
              }
              else {
                commandLineError('N');
              }
              break;
            case 'H':
              if((argv.length - currArg-1) >= 1) {
                if(!bigNet) {
                  netOn = true;
                  hostname = argv[currArg+1];
                  littleNet = true;
                }
                else {
                  System.err.println("-N already overriding -H");
                }
                currArg+=2;
              }
              else {
                commandLineError('H');
              }
              break;
            case 'p':
              if((argv.length - currArg-1) >= 1) {
                if(!bigNet) {
                  netOn = true;
                  port = Integer.parseInt(argv[currArg+1]);
                  littleNet = true;  
                }
                else {
                  System.err.println("-N already overriding -p");
                }
                currArg+=2;
              }
              else {
                commandLineError('p');
              }
              break;
            case 'h':
              System.err.println(usage());
              System.exit(0);
              break;
            default:
              throw new Exception("Bad Switch: "+argv[currArg]+"\n"+usage());
	  }
	}
	else {
	  throw new Exception("Bad argument: "+argv[currArg]+"\n"+usage());
	}
      }

      SystemConfiguration sc;
      Hashtable layout;

      ObjectInputStream ois =
	new ObjectInputStream(new java.util.zip.GZIPInputStream(new FileInputStream(filename)));
      sc = (SystemConfiguration)ois.readObject();
      layout=(Hashtable)ois.readObject();
      ois.close();
      exec=new ConfigurationExecutor(hostname,port,null,sc);


    }catch(Exception e) {
      e.printStackTrace();
      System.err.println("Error in processing arguments");
      System.err.println(e.getMessage());
    }


    return exec;
  }
  public static void main(String[] args){
    ConfigurationExecutor exec=processArgs(args);
    exec.addObserver(new Observer(){
	public void update(Observable o, Object arg){
	    System.out.println("Observed "+arg);
	    if (arg instanceof Exception){
		System.err.println("The following exception occurred during execution");
		((Exception)arg).printStackTrace();
	    }
	    if (arg instanceof String){
		String message=(String)arg;
		if (message.equals(ConfigurationExecutor.FINISHED)){
		    System.exit(1);
		}
	    }
	}
    }
		     );

    try{
      if (exec!=null){
	exec.executeConfiguration();
      }
    }catch(Exception e){
      e.printStackTrace();
    }
  }


}
