package adaptive.core;

import java.util.*;

/*****************************************************************************
 * <! Copyright 1998, Institute for Complex Engineered Sytems,
 *                    Carnegie Mellon University
 *
 * PROJECT: Adaptable 
 *
 * FILE: BasicPortAgent.java 
 * >
 *
 * Class <code>BasicPortAgent</code> is the package internal base
 * class for all real Port Based Agents. It implements the
 * <code>Agent</code> skeleton. None of its methods are finalized.  To
 * create a new core Agent type (PBAgent, PBSAgent) that will be
 * extendible outside this package, extend this class and finalize the
 * appropriate methods.
 *
 * @see Agent
 * @see PBAgent
 * @see PBSAgent
 * 
 * @author Theodore Q Pham <A HREF="mailto:telamon@CMU.EDU">telamon@CMU.EDU</A>
 *         <br> </br>
 *
 * @version  1.7 2/09/99<br> </br>
 *
 * <!
 * REVISION HISTORY:
 *
 * $Log: BasicPortAgent.java,v $
 * Revision 1.14.2.9  2000/05/20 19:11:16  jrj
 * fixed a couple more macro bugs.  Took preliminary steps toward making
 * fully recursive macros in MacroDescription.  Have not yet made changes
 * that would damage save file.
 *
 * Revision 1.14.2.8  2000/05/15 22:39:36  jrj
 * Made fixes to macros so that they work.  Changed the executing macro
 * so that it is recursive, can be manipulated as a whole, and its ports
 * can be asked for directly as if it were any other BasicPortAgent.
 * Made several other changes to simplify and assist these changes.
 * Finally purged the last remaining bit of things named clump.
 *
 * Revision 1.14.2.7  2000/04/28 04:22:27  jrj
 * minor changes in interface regarding getStateEditor;  also an attempt
 * on AgentDescription to have it catch deserialization exceptions in the
 * state that still fails the same as the default method, but maybe
 * someone else can see what I'm doing wrong there.  I think I also
 * increased the max text box size in Netcontroller
 *
 * Revision 1.14.2.6  2000/03/16 21:39:30  jrj
 * turned off some debug messages for events
 *
 * Revision 1.14.2.5  2000/02/27 22:26:12  telamon
 * BasicPortAgent: less printing debug statements
 *
 * Revision 1.14.2.4  2000/02/17 01:17:51  telamon
 * initial demo 2/16 checkin
 *
 * Revision 1.14.2.3  2000/01/31 22:52:02  telamon
 * fixed example cvs conflict
 *
 * Revision 1.14.2.2  2000/01/31 22:46:22  jeffreyl
 * *** empty log message ***
 *
 * Revision 1.14.2.1  1999/12/14 21:45:30  heller
 * added event classes
 * added code to create event connections in system executor
 * updated state viewers
 *
 * Revision 1.14  1999/04/10 21:27:14  yatish
 * Fixed SystemExecutor and Launcher so they run macros properly.
 *
 * Revision 1.13  1999/04/03 03:04:09  telamon
 * new BasicPortAgent.java for StateEditor
 *
 * Revision 1.12  1999/03/11 19:40:40  telamon
 * changes for kevin's neural net
 *
 * Revision 1.11  1999/03/07 19:20:43  telamon
 * changed Agent constructors so they don't System.exit() on initialize failure
 *
 * Revision 1.10  1999/03/03 22:54:08  telamon
 * SystemExecutor debug info added
 *
 * Revision 1.9  1999/02/22 01:49:43  telamon
 * fixes for NetExecutor disconnection
 *
 * Revision 1.8  1999/02/20 02:29:04  nzc
 * Add port labels
 *
# Revision 1.7  99/02/09  20:08:15  telamon
#  new Adaptive core for 2/10 demo
# 
 * Revision 1.6  1999/02/05 21:13:04  telamon
 * new agentloader
 *
 * >
 ****************************************************************************/
abstract class BasicPortAgent implements Agent, Runnable, TriggerListener {
  /** package visible so that only Classes from within adaptive.core
      can extend and manipulate the internals. */
  Thread mainLoop;
  Object runLock;
  boolean go;
  boolean stateFreeze;
  boolean stopped;
  boolean usesThread;
  
  String myAgentID;
  
  int numOutputs=0;;
  int numInputs=0;
  
  Vector outputPorts;  
  Vector inputPorts; 

  long perSleepTime;
  boolean started;
  boolean badInit;

   /******************************************************************
   *
   * No argument constructor required for dynamic loading.
   * Calls initialize() to set input and output numbers.
   *
   * @param 
   *
   * @return 
   *
   * @exception 
   ******************************************************************/
  public BasicPortAgent() {
    //by default, uses a thread. inside initialize,
    //call setNoThread() if you don't need one
    usesThread = true;
    badInit = false;
    perSleepTime = Agent.DEFAULT_SLEEP_TIME;
    initialize();
    if(outputPorts == null || inputPorts == null) {
      badInit = true;
      //System.err.println(this.getClass().getName()+
      //": setNumInputsOutputs() not called in initialize()");
    }
    //call this to make sure all ports have been created.
    //makes Float type ports for any not already created.
    defaultPorts();
  }
  /******************************************************************
   * <! Description:>
   *
   * Implement this method to set the number of inputs/outputs and 
   * configure port types.
   * Use <code> setNumInputsOutputs() </code>,
   *     <code> setOutputPortType()</code>,
   *     <code> setInputPortType()</code>.
   *
   * @see #setNumInputsOutputs
   * @see #setOutputPortType
   * @see #setInputPortType
   *
   * @param 
   *
   * @return 
   *
   * @exception 
   ******************************************************************/
  protected abstract void initialize();

  /******************************************************************
   * <!
   * Method: allocateInternals()
   *
   * Description:
   * >
   * Implement this method to allocate the Agent's internal data 
   * structures.
   *
   * @param 
   *
   * @return 
   *
   * @exception 
   ******************************************************************/
  protected void allocateInternals() {    
  }

  /******************************************************************
   * <!
   * Method: processInternalState()
   *
   * Description: 
   * > 
   *
   * Implement this method to restore the Agent's
   * internal state. <b>Note:</b> the state argument may be null if no
   * state was saved previously.
   *
   *
   * @param state the Agent's deserialized internal state
   *
   * @return <code>true</code> if the state was restored correctly;
   * <br><code>false</code> if a non-recoverable error occured during 
   * restoration
   * </br>
   *
   * @exception 
   ******************************************************************/
  protected boolean processInternalState(Object state) {
    return true;
  }

  /******************************************************************
   * <!
   * Method: initOutputVals()
   *
   * Description:
   * >
   * Implement this method to initialize the Agent's outputs.
   * Use setOutput(). DO NOT LOOK AT YOUR INPUTS WHEN INITIALIZING
   * OUTPUTS. THERE IS NO GUARANTEE THAT THE THOSE INPUTS HAVE
   * BEEN INITIALIZED YET. Anything you don't explicitly initialize will
   * be initialized using the port types default no argument constructor.
   *
   * @see #setOutput
   *
   * @param 
   *
   * @return <code>true</code> if the initial output values were set correctly;
   *         <br> false if an error occured </br>
   *
   * @exception 
   ******************************************************************/
  protected abstract boolean initOutputVals();
    
  /******************************************************************
   * <!
   * Method: runLoop()
   *
   * Description:
   * >
   * Called repeatedly inside a while loop in the Agent's thread
   * All processing should be done in this method.
   *
   * @param 
   *
   * @return 
   *
   * @exception 
   ******************************************************************/
  protected void runLoop() {
  }
  
  /**
   * Called when the module recieves a trigger event
   * @param ev The TriggerEvent listing what ports have changed
   */
  protected void runEvent( TriggerEvent ev ) {
  }
  
  /**
   * Internal method to take care of trigger events on the output ports.
   * Sends events on the output ports appropriately by aggregating all
   * events from this agent only.
   * @param event The trigger event to give to runEvent
   * @param loopModule true if runLoop should be used, false if
   *                   runEvent should be used
   */
  void runModule( TriggerEvent event, boolean loopModule ) {
    // assume all ports are clean
    //System.out.println( "runModule: " + this );
    if( loopModule ) {
      runLoop();
    } else {
      synchronized (runLock) {
        runEvent( event );
      }
    }
    
    triggerOutputEvent();
  }
  
  /**
   * sends appropriate trigger events on all changed output ports.
   */
  protected final void triggerOutputEvent() {
    //System.out.println( "triggering output " + this );
    Enumeration en = outputPorts.elements();
    Hashtable eventTable = new Hashtable();
    while( en.hasMoreElements() ) {
      Object e = en.nextElement();
      if( e instanceof OutputPort ) {
	OutputPort op = (OutputPort) e;
	//	System.out.println(op+" from "+getID()+" clean?"+op.isClean());
	if( ! op.isClean() ) {
	  Hashtable listenerTable = op.getListenerTable();
	  Enumeration listeners = listenerTable.keys();
	  while( listeners.hasMoreElements() ) {
	    Object listener = listeners.nextElement();
	    // System.out.println(this+" has listener"+listener);
	    if( eventTable.containsKey( listener ) ) {
	      Vector portList = (Vector) eventTable.get( listener );
	      portList.addElement( listenerTable.get( listener ) );
	    } else {
	      Vector portList = new Vector();
	      portList.addElement( listenerTable.get( listener ) );
	      eventTable.put( listener, portList );
	    }
	  }
	}
      }
    }
    
    en = eventTable.keys();
    while( en.hasMoreElements() ) {
      Object dstObj = en.nextElement();
      if( dstObj instanceof BasicPortAgent) {
	BasicPortAgent dst = (BasicPortAgent) dstObj;
	Vector portVector = (Vector) eventTable.get( dst );
	int[] portNums = new int[portVector.size()];
	for( int i = 0; i < portNums.length; i++ ) {
	  portNums[i] = ( (Integer) portVector.elementAt( i ) ).intValue();
	}
	//	System.out.println( "enQing event for " + dst );
	TriggerEvent ev = new TriggerEvent( dst, portNums );
	EventTriggerThreadPool.instance().enQueueEvent( ev );
      } else {
	System.out.println( "non TriggerListener in list: " + dstObj );
      }
    }

    en = outputPorts.elements();
    while( en.hasMoreElements() ) {
      Object e = en.nextElement();
      if( e instanceof OutputPort ) {
	( (OutputPort) e ).setClean( true );
      }
    }
  }

  /******************************************************************
   * <!
   * Method: dumpState()
   *
   * Description: > 
   * 
   * Implement this method to save the Agent's state. The return value
   * should be a Serializable object. This method will be called
   * inside <code>saveState()</code>. <b>Note:</b> State may not have
   * been explicitly frozen and Agent may not have been stopped prior
   * to this call. However, the <code>saveState()</code> method is
   * internally synchronized such that it will only occur between
   * successive calls to <code>runLoop()</code>.
   *
   * @param 
   *
   * @return An Serializable Object representing the state of the Agent.
   *
   * @exception 
   ******************************************************************/
  protected Object dumpState() {
    return null;
  }

  /******************************************************************
   * <!
   * Method: deAllocateInternals()
   *
   * Description:
   * >
   * Implement this method to deallocate the Agent's internal data 
   * structures. The Agent will have been stopped before this method 
   * is called.
   *
   * @param 
   *
   * @return 
   *
   * @exception 
   ******************************************************************/
  protected void deAllocateInternals() {
    
  }

  /***************************************************************************
   *
   * Returns the GUI dialog internal state editor. This should return
   * a new instance of the StateEditor. Or null if there is no state
   * editor.
   *
   * @param parent frame, since the state editor should extend java.awt.Dialog, but nothing forces it too --added jrj 4/27/00
   *
   * @return the gui state editor
   *
   * @exception 
   *
   **************************************************************************/
  public adaptive.modulemanager.StateEditor getStateEditor(java.awt.Frame parent) {
    return null;
  }
  
  /******************************************************************
   * <!
   * Method: setup()
   *
   * Description:
   * >
   *
   * Psuedo-constructor to prepare agent for runtime.
   * Allocate support data. Call allocateInternals() to allow
   * derived classes to setup. Call processInternalState() to restore
   * saved state.
   *
   * @param agentID the agent's name
   * @param myLoader for use by composed agents
   * @param internalState the agent's saved internal state
   *
   * @return  <code>true</code> if setup completed successfully; <br>
   *          <code>false</code> otherwise </br>
   *
   * @exception 
   ******************************************************************/
  public boolean setup(String agentID,
		       AgentLoader myLoader, Object internalState ) {
    go = false;
    stateFreeze = false;
    stopped = false;
    runLock = new Object();
    setID(agentID);
    if (usesThread) {
      mainLoop = new Thread(this);
    }
    else {
      mainLoop = null;
    }
    
    //allow derived classes to allocate data sturctures needed for
    //operation
    allocateInternals();
    
    if(!processInternalState(internalState)) {
      return false;
    }
    if(!initOutputVals()) {
      return false;
    }

    return checkOutputVals();
  }

  /******************************************************************
   * <!
   * Method: getID()
   *
   * Description:
   * >
   *
   * Returns the Agent's name.
   *
   * @param 
   *
   * @return the Agent's name
   *
   * @exception 
   ******************************************************************/
  public String getID() {
    return( myAgentID );
  }

  /******************************************************************
   * <!
   * Method: setID()
   *
   * Description:
   * >
   *
   * Set the Agent's name.
   *
   * @param agentID The Agent's name.
   *
   * @return 
   *
   * @exception 
   ******************************************************************/
  void setID( String agentID ) {
    myAgentID = agentID;
  }

  /******************************************************************
   * <!
   * Method: getNumInputs()
   *
   * Description:
   * >
   *
   * Get the number of inputs this agent needs.
   *
   * @param 
   *
   * @return the number of inputs this agent needs. 
   *
   * @exception 
   ******************************************************************/

  public int getNumInputs() {
    return numInputs;
  }

  /******************************************************************
   * <!
   * Method: getNumOutputs()
   *
   * Description:
   * >
   *
   * Get the number of outputs this agent provides.
   *
   * @param 
   *
   * @return the number of outputs this agent provides. 
   *
   * @exception 
   ******************************************************************/
  public int getNumOutputs() {
    return numOutputs;
  }

  /******************************************************************
   * <!
   * Method: setNumInputsOutputs()
   *
   * Description:
   * >
   *
   * Call to set the agent's number of inputs and outputs. Should be
   * called inside initialize(). Otherwise the agent will have 0 inputs
   * and 0 outputs.
   *
   * @param numin Number of inputs
   * @param numout Number of outputs
   *
   * @return 
   *
   * @exception 
   ******************************************************************/
  protected void setNumInputsOutputs( int numin, int numout ) {
    numInputs = numin;
    numOutputs = numout;
    outputPorts = new Vector(getNumOutputs());
    inputPorts = new Vector(getNumInputs());
    outputPorts.setSize(getNumOutputs());
    inputPorts.setSize(getNumInputs());
  }

  /******************************************************************
   * <!
   * Method: isInputIndex()
   *
   * Description:
   * >
   *
   * Checks if <code>which</code> is a valid input port index.
   *
   * @param which the input port index
   *
   * @return <code>true</code> if valid;
   *         <br><code>false</code> otherwise</br>
   *
   * @exception 
   ******************************************************************/
  public boolean isInputIndex(int which) {
    if(which < 0 || which >= numInputs) {
      return false;
    }
    return true;
  }
  
  /******************************************************************
   * <!
   * Method: getInputPort()
   *
   * Description:
   * >
   *
   * Get the specified InputPort.
   *
   * @param which the input port index
   *
   * @return the InputPort Object at the specified port number;
   *         <code>null</code> if the index is invalid
   *
   * @exception 
   ******************************************************************/
  public InputPort getInputPort(int which) {
    if(!isInputIndex(which)) {
      return null;
    }
    return (InputPort) inputPorts.elementAt(which);
  }

  /******************************************************************
   * <!
   * Method: setInputPort()
   *
   * Description:
   * >
   *
   * Set the input port at index to be the given InputPort.
   *
   * @param inp the InputPort object
   * @param which the input port index
   *
   * @return <code>true</code> if the input port was set;
   *         <br><code>false</code> if the index is invalid<br>
   *
   * @exception 
   ******************************************************************/
  public boolean setInputPort(InputPort inp, int which) {
    if(!isInputIndex(which)) {
      return false;
    }
    inputPorts.setElementAt(inp,which);
    return true;
  }
  
  /******************************************************************
   * <!
   * Method: getInputPortType()
   *
   * Description:
   * >
   *
   * Get the type of the InputPort at the specified index.
   *
   * @param which the input port index
   *
   * @return the Class Object for the input port's type;
   *         <br><code>null</code> if the index is invalid</br>
   *
   * @exception 
   *
   ******************************************************************/
  public Class getInputPortType( int which ) {
    if(!isInputIndex(which)) {
      return null;
    }
    return( getInputPort( which ).getType() );
  }

  /******************************************************************
   * <!
   * Method: setInputPortType()
   *
   * Description:
   * >
   *
   * Set the InputPort type. If the input port doesn't yet exist,
   * create it.
   *
   * @param stp the fully qualified class name of the input type 
   * @param which the input port index
   *
   * @return <code>true</code> if the input port type was set;
   *         <br><code>false</code> if the index is invalid</br>
   *
   * @exception ClassNotFoundException thrown when stp does not
   *            indicate a valid java class
   ******************************************************************/
  public boolean setInputPortType( String stp, int which ) throws ClassNotFoundException{
    return setInputPortType( Class.forName( stp ), which );
  }
  
  /******************************************************************
   * <!
   * Method: setInputPortType()
   *
   * Description:
   * >
   *
   * Set the InputPort type. If the input port doesn't yet exist,
   * create it.
   *
   * @param otp the port should be set to this Object's type 
   * @param which the input port index
   *
   * @return <code>true</code> if the input port type was set;
   *         <br><code>false</code> if the index is invalid</br>
   *
   * @exception 
   ******************************************************************/
  public boolean setInputPortType( Object otp, int which ) {
    return setInputPortType( otp.getClass(), which );
  }

  /******************************************************************
   * <!
   * Method: setInputPortType()
   *
   * Description:
   * >
   *
   * Set the InputPort type. If the input port doesn't yet exist,
   * create it.
   *
   * @param tp the Class object for the type the InputPort should
   *           be set to
   * @param which the input port index
   *
   * @return <code>true</code> if the input port type was set;
   *         <br><code>false</code> if the index is invalid</br>
   *
   * @exception 
   ******************************************************************/
  public boolean setInputPortType( Class tp, int which ) {
    if(!isInputIndex(which)) {
      return false;
    }

    InputPort cur = getInputPort( which );
    
    if(cur == null) {
      //this InputPort is in the legal range, but has not been created yet.
      //create it with the type setting constructor
      cur = new InputPort(tp);
      setInputPort(cur,which);
    }
    else {
      //already exists, so just change it's type
      cur.setType( tp );
    }
    return true;
  }
  
  /******************************************************************
   * <!
   * Method: getInputPortName()
   *
   * Description:
   * >
   *
   * Get the type of the InputPort at the specified index.
   *
   * @param which the input port index
   *
   * @return the Class Object for the input port's name;
   *         <br><code>null</code> if the index is invalid</br>
   *
   * @exception 
   *
   ******************************************************************/
  public String getInputPortName( int which ) {
    if(!isInputIndex(which)) {
      return null;
    }
    return( getInputPort(which).getName() );
  }

  /******************************************************************
   * <!
   * Method: setInputPortName()
   *
   * Description:
   * >
   *
   * Set the InputPort name. If the input port doesn't yet exist,
   * create it.
   *
   * @param nm the String for the name the InputPort should
   *           be set to
   * @param which the input port index
   *
   * @return <code>true</code> if the input port name was set;
   *         <br><code>false</code> if the index is invalid</br>
   *
   * @exception 
   ******************************************************************/
  public boolean setInputPortName( String nm, int which ) {
    if(!isInputIndex(which)) {
      return false;
    }

    InputPort cur = (InputPort) inputPorts.elementAt( which );
    
    if(cur == null) {
      //this InputPort is in the legal range, but has not been created yet.
      //create it with the name setting constructor
      cur = new InputPort(nm);
      setInputPort(cur,which);
    }
    else {
      //already exists, so just change it's name
      cur.setName( nm );
    }
    return true;
  }
  
  /******************************************************************
   * <!
   * Method: getInputMapping()
   *
   * Description:
   * >
   *
   * Get the input mapping for the specified input port.
   *
   * @param which the input port index
   *
   * @return a space seperated string containing the providing agent's
   *         name and output number; <br><code>null</code> if no
   *         provider is mapped to the specified input port or the if
   *         specified input port doesn't exist
   *
   * @exception 
   ******************************************************************/
  public String getInputMapping(int which) {
      return ((InputPort)(getInputPort(which))).getMapping();
  }

  /******************************************************************
   * <!
   * Method: setInputMapping()
   *
   * Description:
   * >
   *
   * Set the output port that will provide data for the specified
   * input port.
   *
   * @param which the index of the input port
   * @param provider the
   *
   * @return <code>true</code> if the mapping was made;
   * <br><code>false</code> otherwise. Failure results when the port
   * index are invalid or the provider argument is <code>null</code>
   * </br>
   *
   * @exception 
   ******************************************************************/
  public boolean setInputMapping(int which, Agent provider,
				 int whichOut) {
    //check for bad input index,
    //no provider
    //and bad output index on provider
    if( !isInputIndex(which)
	|| (provider == null )
	|| !provider.isOutputIndex(whichOut)) {
      return false;
    }
    InputPort curInput = getInputPort( which );
    if( ( getModuleMode() & Agent.EVENT_ONLY ) != 0 ) {
      System.out.println( "adding " + this + " as a listener to "
	  + provider.getOutputPort( whichOut ) );
      provider.getOutputPort( whichOut ).addTriggerListener( this, which );
    }

    return curInput.setMapping(provider.getOutputPort(whichOut));
  }

  /******************************************************************
   * <!
   * Method: getInput()
   *
   * Description:
   * >
   *
   * Get the current value of the Agent's specified input.
   *
   * @param which the desired input number
   *
   * @return the current input value;
   *
   * @exception 
   ******************************************************************/
  public Object getInput( int which ) {
    return( ((InputPort)(getInputPort( which ))).getInput() );
  }
  
  /******************************************************************
   * <!
   * Method: isOutputIndex()
   *
   * Description:
   * >
   *
   * Checks if <code>which</code> is a valid output port index.
   *
   * @param which the output port index
   *
   * @return <code>true</code> if valid;
   *         <br><code>false</code> otherwise</br>
   *
   * @exception 
   ******************************************************************/
  public boolean isOutputIndex(int which) {
    if(which < 0 || which >= getNumOutputs()) {
      return false;
    }
    return true;
  }

  /******************************************************************
   * <!
   * Method: getOutputPort()
   *
   * Description:
   * >
   *
   * Gets the specified OutputPort.
   *
   * @param which the output port index
   *
   * @return the OutputPort Object;
   *         <br><code>null</code> if index is invalid</br>
   *
   * @exception 
   ******************************************************************/
  public OutputPort getOutputPort(int which) {
    if(!isOutputIndex(which)) {
      return null;
    }
    return (OutputPort) outputPorts.elementAt(which);
  }

  /******************************************************************
   * <!
   * Method: setOutputPort()
   *
   * Description:
   * >
   *
   * Set Agent's specified OutputPort.
   *
   * @param outp the OutputPort object
   * @param which index of the output port to set
   *
   * @return <code>true</code> if the OutputPort was set;
   *         <br><code>false</code> if the specified output port
   *         number is invalid</br>
   *
   * @exception 
   ******************************************************************/
  public boolean setOutputPort(OutputPort outp, int which) {
    if(!isOutputIndex(which)) {
      return false;
    }
    outputPorts.setElementAt(outp,which);
    return true;
  }

  /******************************************************************
   * <!
   * Method: getOutputPortType()
   *
   * Description:
   * >
   *
   * Get the specified output port's type.
   *
   * @param which index of the output port
   *
   * @return the Class object for the output port's type;
   *         <br><code>null</code> if the output port index is
   *         invalid</br>
   *
   * @exception 
   ******************************************************************/
  public Class getOutputPortType( int which ) {
    if(!isOutputIndex(which)) {
      return null;
    }
    return( getOutputPort( which ).getType() );
  }

  /******************************************************************
   * <!
   * Method: setOutputPortType()
   *
   * Description:
   * >
   *
   * Set the specified output port's type. If the OutputPort doesn't
   * yet exist, create it.
   *
   * @param stp fully qualified name representing of the type
   * @param which index of the output port to set
   *
   * @return <code>true</code> if the port type was set;
   *         <br><code>false</code> if the output port index is
   *         invalid</br>
   *
   * @exception ClassNotFoundException thrown if stp doesn't indicate a valid
   *            type
   ******************************************************************/
  public boolean setOutputPortType( String stp, int which ) 
    throws ClassNotFoundException {
    return setOutputPortType( Class.forName( stp ), which );
  }
  
  /******************************************************************
   * <!
   * Method: setOutputPortType()
   *
   * Description:
   * >
   *
   * Set the specified output port's type. If the OutputPort doesn't
   * yet exist, create it.
   *
   * @param otp port's type should be set to this Object's type
   * @param which index of the output port to set
   *
   * @return <code>true</code> if the port type was set;
   *         <br><code>false</code> if the output port index is
   *         invalid</br>
   *
   * @exception 
   ******************************************************************/
  public boolean setOutputPortType( Object otp, int which ) {
    return setOutputPortType( otp.getClass(), which );
  }

  /******************************************************************
   * <!
   * Method: setOutputPortType()
   *
   * Description:
   * >
   *
   * Set the specified output port's type. If the OutputPort doesn't
   * yet exist, create it.
   *
   * @param tp the type to set to
   * @param which index of the output port to set
   *
   * @return <code>true</code> if the port type was set;
   *         <br><code>false</code> if the output port index is
   *         invalid</br>
   *
   * @exception 
   ******************************************************************/
  public boolean setOutputPortType( Class tp, int which ) {
    if(!isOutputIndex(which)) {
      return false;
    }

    OutputPort cur = getOutputPort( which );
    
    if(cur == null) {
      //this OutputPort is in the legal range, but has not been created yet.
      //create it with the type setting constructor
      cur = new OutputPort(which, myAgentID, tp);
      setOutputPort(cur,which);
    }
    else {
      //already exists, so just change it's type
      cur.setType( tp );
    }
    return true;
  }

  /******************************************************************
   * <!
   * Method: getOutputPortName()
   *
   * Description:
   * >
   *
   * Get the specified output port's name.
   *
   * @param which index of the output port
   *
   * @return the String for the output port's name;
   *         <br><code>null</code> if the output port index is
   *         invalid</br>
   *
   * @exception 
   ******************************************************************/
  public String getOutputPortName( int which ) {
    if(!isOutputIndex(which)) {
      return null;
    }
    return( getOutputPort( which ).getName() );
  }

  /******************************************************************
   * <!
   * Method: setOutputPortName()
   *
   * Description:
   * >
   *
   * Set the specified output port's name. If the OutputPort doesn't
   * yet exist, create it.
   *
   * @param nm the name to set to
   * @param which index of the output port to set
   *
   * @return <code>true</code> if the port name was set;
   *         <br><code>false</code> if the output port index is
   *         invalid</br>
   *
   * @exception 
   ******************************************************************/
  public boolean setOutputPortName( String nm, int which ) {
    if(!isOutputIndex(which)) {
      return false;
    }

    OutputPort cur = (OutputPort) getOutputPort( which );
    
    if(cur == null) {
      //this OutputPort is in the legal range, but has not been created yet.
      //create it with the name setting constructor
      cur = new OutputPort(which, myAgentID, nm);
      setOutputPort(cur,which);
    }
    else {
      //already exists, so just change it's name
      cur.setName( nm );
    }
    return true;
  }

  /******************************************************************
   * <!
   * Method: getOutput()
   *
   * Description:
   * >
   *
   * Get the current value of the Agent's specified output.
   *
   * @param which the desired output number
   *
   * @return the current output value; <br> <code>null</code> if the
   *         specified output does not exist </br>
   *
   * @exception 
   ******************************************************************/
  public Object getOutput(int which) {
    return ( ((OutputPort)(getOutputPort( which ))).getOutput() );
  }
  
  /******************************************************************
   * <!
   * Method: setOutput()
   *
   * Description:
   * >
   *
   * Set the value of the Agent's specified output.
   *
   * @param value the new output value
   * @param which the desired output port number
   *
   * @return <code>true</code> if the value was writen to the port;
   *         <br><code>false</code> if the specified port is invalid
   *         or there is a type mismatch</br>
   *
   * @exception 
   ******************************************************************/
  public boolean setOutput( Object value, int which ) {
    return ((OutputPort)(getOutputPort( which ))).setOutput( value );
  }

  /******************************************************************
   * <!Description: >
   *
   * Put's the Agent to sleep for the specified amount of time. This
   * method automatically releases the internal runLock monitor. DO
   * NOT USE ANY OTHER METHODS THAT WOULD BLOCK YOUR RUNLOOP.
   *
   * @param sleepTime how long to sleep for   
   *
   * @return 
   *
   * @exception InterruptedException when the thread is interrupted
   *
   ******************************************************************/
  public void sleep(long sleepTime) throws InterruptedException {
    //System.out.println("Sleeping "+sleepTime);
    long startTime = System.currentTimeMillis();
    long currentTime;    
    synchronized(runLock) {
      while(((currentTime = System.currentTimeMillis()) - startTime) < sleepTime){
	sleepTime -= (currentTime - startTime);
	startTime = currentTime;
	runLock.wait(sleepTime);
      }
      while(!go) {	
	try {
	  runLock.wait();
	}
	catch (Exception e) {
	}
      }
    }
  }
  
  /****************************************************************************
   * <!
   * Method: getSleepTime()
   *
   * Description:
   * >
   *
   * Get the current per loop sleep time value.
   *
   * @param 
   *
   * @return the current per loop sleep time value.
   *
   * @exception 
   ***************************************************************************/
  public long getSleepTime() {
    return perSleepTime;
  }
  
  /****************************************************************************
   * <!
   * Method: setSleepTime()
   *
   * Description:
   * >
   *
   * Set's the current per loop sleep value.
   *
   * @param sleepTime the new current sleep value
   *
   * @return 
   *
   * @exception 
   ***************************************************************************/
  protected void setSleepTime(long sleepTime) {
    perSleepTime = sleepTime;
  }
  
  /******************************************************************
   * <!
   * Method: start()
   *
   * Description:
   * >
   *
   * Called to start execution.
   *
   * @param 
   *
   * @return 
   *
   * @exception 
   ******************************************************************/
  public void start() {
    if (!started) {
      go = true;
      synchronized(runLock) {
        runLock.notifyAll();
      }
      if(usesThread) {
        System.out.print( "with THREAD " );
        mainLoop.start();
        started=true;
      }
      System.out.println(getID() + " started.");
    }
  }

  /******************************************************************
   * <!
   * Method: pause()
   *
   * Description:
   * >
   *
   * Called to pause execution. <code>unpause()</code> may be called
   * later to resume execution.
   *
   * @param 
   *
   * @return 
   *
   * @exception 
   ******************************************************************/
  public void pause() {
    synchronized(runLock) {
      go = false;
    }
    System.out.println(getID() + " paused.");
  }

  /******************************************************************
   * <!
   * Method: unpause()
   *
   * Description:
   * >
   *
   * Called to unpause execution. <code>pause()</code> should have
   * been called before. If <code>pause()</code> wasn't called before,
   * then do nothing.
   *
   * @param 
   *
   * @return 
   *
   * @exception 
   ******************************************************************/
  public void unpause() {
     synchronized(runLock) {
       if(!go) {
	 go = true;
	 runLock.notifyAll();
       }
     }
     System.out.println(getID() + " unpaused.");
  }

  /******************************************************************
   * <!
   * Method: stop()
   *
   * Description:
   * >
   *
   * Called to end execution. <b>Note:</b> Do not deallocate resources
   * or clear state in this method. Do so in <code>destroy()</code>.
   * <code>dumpState</code> may be called after this method has been
   * called.
   *
   * @param 
   *
   * @return 
   *
   * @exception 
   ******************************************************************/
  public void stop() {
    System.out.println("Stopping "+getID());
    synchronized(runLock) {
      if(!stopped) {
	if(usesThread) {
	  mainLoop.stop();
	}
	stopped = true;
      }
    }
    System.out.println(getID() + " stopped.");
  }

  /******************************************************************
   * <!
   * Method: saveState()
   *
   * Description:
   * >
   *
   * Returns the Agent's internal state. This will be serialized and
   * saved for later restoration. <b>Note:</b> State may not have been
   * explicitly frozen and Agent may not have been stopped prior to
   * this call. However, this method is internally synchronized such
   * that it will only occur between successive calls to
   * <code>runLoop()</code>.
   *
   * @param 
   *
   * @return a Serializable Object representing the Agent's saved state
   *
   * @exception 
   ******************************************************************/
  public Object saveState() {
      // if (runLock==null) return dumpState(); //if agent has not been initialized, it can't run so there is no reason to synchronize on runLock
    synchronized(runLock) {
      return dumpState();
    }
  }
  
  /******************************************************************
   * <!
   * Method: isStateFrozen()
   *
   * Description:
   * >
   *
   * Whether the state should currently be frozen. This method,
   * <code>freezeState()</code>, and <code>unfreezeState()</code> are
   * internally synchronized so that they will never freeze or
   * unfreeze state in the middle of a call to
   * <code>runLoop()</code>. The Agent implementer should check to see
   * if state is frozen at the start of <code>runLoop()</code> and trust
   * that it will not change
   *
   * @param 
   *
   * @return an Object representing the Agent's saved state
   *
   * @exception 
   ******************************************************************/
  public boolean isStateFrozen() {
    synchronized(runLock) {
      return stateFreeze;
    }
  }

  /******************************************************************
   * <!
   * Method: freezeState()
   *
   * Description:
   * >
   *
   * Called to prevent the Agent from changing it's internal state.
   * The Agent should continue to update it's outputs, but internal
   * data structures such as neural net weights or adaptive filter
   * coefficients should not be changed while state is frozen. See
   * <code>isStateFrozen()</code> for notes on synchronization.
   *
   * @see #isStateFrozen
   *
   * @param 
   *
   * @return 
   *
   * @exception 
   ******************************************************************/
  public void freezeState() {
    synchronized(runLock) {
      stateFreeze = true;
    }
  }

  /******************************************************************
   * <!
   * Method: unfreezeState()
   *
   * Description:
   * >
   *
   * Called to undo a state freeze. The Agent is then free to change
   * it's internal state again. See <code>isStateFrozen()</code> for
   * notes on synchronization.
   *
   * @see #isStateFrozen
   *
   * @param 
   *
   * @return 
   *
   * @exception 
   ******************************************************************/
  public void unfreezeState() {
    synchronized(runLock) {
      stateFreeze = false;
    }
  }

  /******************************************************************
   * <!
   * Method: destroy()
   *
   * Description:
   * >
   *
   * Called at end of Agent's run cycle. Agent should release all
   * resources it's holding and free up memory where necessary. If
   * stop hasn't been called before destroy, destroy will call stop
   * before proceeding with deallocation.
   *
   * @param 
   *
   * @return 
   *
   * @exception 
   ******************************************************************/
  public void destroy() {
    System.out.println("DESTROY: Before lock");
    synchronized(runLock) {
      System.out.println("DESTROY: After lock");
      if(!stopped) {
        System.out.println("Before stopping.");
	this.stop();
        System.out.println("Done stopping.");
      }
      deAllocateInternals();
    }
    inputPorts = null;
    outputPorts = null;
    mainLoop = null;
    runLock = null;
  }

  /******************************************************************
   * <!
   * Method: verify()
   *
   * Description:
   * >
   *
   * Called to check if the Agent's inputs have all been satisfied
   * correctly. That is they have mapped sources and matching
   * input/output types.
   *
   * @param 
   *
   * @return <code>true</code> if all inputs satisfied correctly;
   *         <br><code>false</code></br> otherwise
   *
   * @exception 
   ******************************************************************/
  public boolean verify() {
    for(int j=0; j < getNumInputs();j++) {
      if(getInputPort(j) == null
	 || !( ((InputPort)(getInputPort(j))).verify()) ) {
	return false;
      }
    }
    return true;
  }

  /******************************************************************
   * <!
   * Method: isBadInit()
   *
   * Description:
   * >
   *
   * Called to check if initialization completed correctly
   *
   * @param 
   *
   * @return <code>true</code> if initialize failed;
   *         <br><code>false</code></br> otherwise
   *
   * @exception 
   ******************************************************************/
  public boolean isBadInit() {
    return badInit;
  }

  /******************************************************************
   * <!
   * Method: getStateDescription()
   *
   * Description:
   * >
   *
   * Called to get a description of the state object. For use with
   * StateEditors. To use the DefaultStateEditor, you should return
   * an array of Strings where each string is a description
   * of the corresponding element in the State array of Objects or
   * primitives. DefaultStateEditor can edit primitives
   * (int, short, long, double, float, boolean, byte, char) or Objects
   * of type (Integer, Short, Long, Double, Float, Boolean, Byte, Character).
   *
   * @param 
   *
   * @return the state description
   *
   * @exception 
   ******************************************************************/
  public Object getStateDescription() {
    return (Object) null;
  }
  
  /******************************************************************
   * <!
   * Method: getDefaultState()
   *
   * Description:
   * >
   *
   * Returns a new object representing the default state for this
   * agent. This object will be editted by the StateEditor, so
   * make sure it is non-static and non-final.
   *
   * @param 
   *
   * @return the default state object
   *
   * @exception 
   ******************************************************************/
  public Object getDefaultState() {
    return (Object) null;
  }

  /******************************************************************
   * <!
   * Method: verifyState()
   *
   * Description:
   * >
   *
   * See if the state object is of valid format for this agent.
   *
   * @param 
   *
   * @return null if the obj is a valid state object;
   *         an error string representing what was wrong otherwise
   *
   * @exception 
   ******************************************************************/
  public String verifyState(Object obj) {
    return (String) null;
  }
  
  /**
     Internal method called to block the mainLoop thread 
     when Agent is paused.
   */
  /*  private void progress() {
      while(!go) {
      synchronized(runLock) {
      try {
      runLock.wait();
      }
      catch (Exception e) {
      }
      }
    } 
  }*/

  /******************************************************************
   * <!
   * Method: run()
   *
   * Description:
   * >
   * Satisfies Runnable interface. The main work loop. Derived
   * classes should not override this method. 
   * Override runLoop instead. This is public because java demands
   * that it be public.
   *
   * @param 
   *
   * @return 
   *
   * @exception 
   ******************************************************************/
  public void run() {
    while( true ) {
      synchronized(runLock) {
	while(!go) {	
	  try {
	    runLock.wait();
	  }
	  catch (Exception e) {
	  }	  
	}
	runModule( null, true );
        try {
          sleep(perSleepTime);
        }
        catch(InterruptedException ie) {  
        }
      }
    }
  }

  /******************************************************************
   * 
   * <! Description: >
   * after setup() calls initOutputVals, it will call this function
   * to verify that all output vals have been initialized
   *
   * @param 
   *
   * @return 
   *
   * @exception 
   ******************************************************************/
  boolean checkOutputVals() {
    OutputPort curPort;
    Class ptype;    
    for(int i = 0; i < getNumOutputs();i++) {
      curPort = (OutputPort) getOutputPort(i);
      if(curPort.getOutput() == null) {
	
	System.err.println("Not all output vals were initialized: " +
			   this.getClass().getName());
	return false;
	
      }
    }
    return true;    
  }

  /******************************************************************
   * 
   * <! Description: >
   * after the constructor calls configurePorts, it will call this
   * method to make sure all ports have been instantiated.
   * any port not already instantiated will get instantiated and set to Float
   *
   * @param 
   *
   * @return 
   *
   * @exception 
   ******************************************************************/
  void defaultPorts() {
    try {
      for(int i = 0; i < getNumOutputs();i++) {
	if(getOutputPort(i) == null) {
	  setOutputPortType("java.lang.Float",i);
	}
      }
      for(int i =0; i < getNumInputs();i++) {
	if(getInputPort(i) == null) {
	  setInputPortType("java.lang.Float",i);
	}
      }
    }
    catch(ClassNotFoundException cnfe) {
      System.err.println("Invalid port type in default ports.");
    }
  }

  /******************************************************************
   * 
   * <! Description: >
   * tells us to not use a seperate thread for this agent.
   *
   * @param 
   *
   * @return 
   *
   * @exception 
   ******************************************************************/
  protected void setNoThread(){
    System.out.println("Setting no thread for "+ this);
    usesThread = false;
  }

  /**
   * says what modes this module is capable of running it.
   * should return constant.  Cannot be a static method since
   * we don't know the class name.  Returns LOOP_ONLY by default.
   * @return a coded int which should be one of the following:
   *           BasicPortAgent.EVENT_ONLY, 
   *           BasicPortAgent.LOOP_ONLY, or
   *           BasicPortAgent.EVENT_OR_LOOP 
   */
  public int getModuleModeCapability() {
    return LOOP_ONLY;
  }

  private int moduleModeSetting = LOOP_ONLY;

  /**
   * returns the mode that this module is set to
   * @return a coded int which should be one of the following:
   *           BasicPortAgent.EVENT_ONLY, 
   *           BasicPortAgent.LOOP_ONLY, or
   *           BasicPortAgent.EVENT_OR_LOOP 
   */
  public int getModuleMode() {
    return moduleModeSetting;
  }

  /** set the mode for this module.  modules are initially set to
   * be in mode LOOP_ONLY
   * @param newMode a coded int which should be one of the following:
   *           BasicPortAgent.EVENT_ONLY, 
   *           BasicPortAgent.LOOP_ONLY, or
   *           BasicPortAgent.EVENT_OR_LOOP 
   */
  void setModuleMode( int newMode ) {
    System.out.println( this + " . setModuleMode ( " + newMode + " )" );
    moduleModeSetting = newMode;
  }
}
