package adaptive.core;

//import java.util.Hashtable;
//import java.util.Enumeration;
import java.io.*;
//import java.net.*;
//import java.util.zip.*;
//import java.lang.reflect.*;

import comms.core.*;

/*****************************************************************************
 * <! Copyright 1998, Institute for Complex Engineered Sytems,
 *                    Carnegie Mellon University
 *
 * PROJECT: Adaptable 
 *
 * FILE: AgentLoader.java 
 * >
 *
 * This class acts as an all purpose remote and local Agent class loader.
 * The ModuleManager uses AgentLoader to browse both the local file 
 * system and AgentServer for usable Agents. Executors use AgentLoader
 * to instantiate the Agents in their configurations and dynamically 
 * add/swap agent classes.
 *
 * @see Agent 
 * 
 * @author Theodore Q Pham <A HREF="mailto:telamon@CMU.EDU">telamon@CMU.EDU</A>
 *         <br> </br>
 *
 * @version  1.00 11/22/98<br> </br>
 *
 * <!
 * REVISION HISTORY:
 *
 * $Log: AgentLoader.java,v $
 * Revision 1.12.2.2  2000/02/27 22:38:44  telamon
 * safety checking, NetExecutor.java
 *
# Revision 1.12.2.1  99/10/08  13:58:00  telamon
# NetAgentLoader updated for new communications
# 
 * Revision 1.12  1999/04/06 19:40:06  telamon
 * moving comms out of cyberscout
 *
 * Revision 1.11  1999/04/06 15:11:57  yatish
 * Massive changes to the Save File format along with the addition of Macro
 * support.
 *
 * Revision 1.10  1999/03/01 06:15:51  yatish
 * Added port names to the agent container and description.  Added the
 * ability to get an agentcontainer to local/net agent loader. (not fully
 * implemented in net yet)
 *
 * Revision 1.9  1999/02/05 21:13:03  telamon
 * new agentloader
 *
 * >
 ****************************************************************************/
public class AgentLoader {
  LocalAgentLoader lal;
  NetAgentLoader nal;
  
  /***************************************************************************
   *
   * Constructor for AgentLoader. Takes the local package root and the
   * Communications object as arguments. If the local package root is
   * null, then the default local package root will be used. If host
   * is null, then networking will be disabled and port is ignored.
   *
   * @param lpr  the local package root
   * @param host the AgentServer hostname
   * @param port the AgentServer port number
   *
   * @return 
   *
   * @exception AgentLoaderException if the local root package isn't
   *                                 a valid path or no AgentServer
   *                                 connection could be made
   *
   **************************************************************************/
  public AgentLoader(String lpr,
                     String host,
                     int port) throws AgentLoaderException{
    super();
    lal = new LocalAgentLoader(lpr);
    
    if(host != null) {
      nal = new NetAgentLoader(host, port);
    }
    else {
      nal = null;
    }
  }
  
  /***************************************************************************
   *
   * Method to throw an exception when trying to use disabled networking.
   *
   * @param 
   *
   * @return 
   *
   * @exception 
   *
   **************************************************************************/
  private void checkNet() throws IllegalAccessException {
    if(nal == null ) {
      throw new IllegalAccessException("AgentServer support disabled!");
    }
  }
  
  /***************************************************************************
   *
   * Returns the package root for the local file system if localSrc is true.
   * Otherwise returns the package root for the remote file system.
   * Browsing should start from these paths and file system access is
   * restricted to subpaths of the package roots.
   *
   * @param localSrc true for the local file system
   *                 false for the AgentServer
   *
   * @return the desired file system's package root
   *
   * @exception IllegalAccessException thrown if network services
   *                                   requested when networking disabled
   *
   **************************************************************************/
  public String getPackageRoot(boolean localSrc) throws IllegalAccessException{
    if(localSrc) {
      return lal.getPackageRoot();
    }
    else {
      checkNet(); //throw an exception if cannot use net
      return nal.getPackageRoot();
    }
  }
  
  /***************************************************************************
   *
   * Method to create a new Agent given it's fully qualified class name and
   * and an indicator of where to load from (net or local)
   *
   * @param agentType the Agent's fully qualified classname
   * @param localSrc true to load from local file system;
   *                 false to load from AgentServer
   *
   * @return a new Agent object
   *
   * @exception ClassNotFoundException thrown if the agentType doesn't
   *                                   resolve to a class
   * @exception AgentLoaderException thrown if specified agentType isn't
   *                                 of type Agent
   * @exception InstantiationException thrown if class cannot be instantiated
   * @exception IllegalAccessException thrown if the java securitymanager
   *                                   has been violated or network services
   *                                   requested when networking disabled
   *
   **************************************************************************/
  public Agent newInstance(String agentType, boolean localSrc) throws
    ClassNotFoundException, InstantiationException, IllegalAccessException,
    AgentLoaderException {
    if(localSrc) {
      return lal.newInstance(agentType);
    }
    else {
      checkNet(); //throw an exception if cannot use net
      return nal.newInstance(agentType);
    }
  }
  
  /***************************************************************************
   *
   * List the directories available in the specified directory.
   * The Directory must be a sub directory of the package root or
   * package root itself.
   *
   * @param dir the directory to look in
   * @param localSrc true to load from local file system;
   *                 false to load from AgentServer
   *
   * @return an array of Strings containing the directory names
   *
   * @exception FileNotFoundException thrown if dir isn't a dir
   * @exception IllegalAccessException thrown if java securitymanager is
   *                                   violated or networking requested
   *                                   when networking disabled
   * @exception AgentLoaderException thrown when a networking error occurs
   *
   ***************************************************************************/
  public String[] listDirectoriesIn(String dir,boolean localSrc) throws FileNotFoundException, AgentLoaderException, IllegalAccessException {
    if(localSrc) {
      return lal.listDirectoriesIn(dir);
    }
    else {
      checkNet(); //throw an exception if cannot use net
      return nal.listDirectoriesIn(dir);
    }
  }

  /***************************************************************************
   *
   * Get the files available in the specified directory
   *
   * @param dir the directory to search
   * @param localSrc true to load from local file system;
   *                 false to load from AgentServer
   *
   * @return an array of Strings containing the file names
   *
   * @exception FileNotFoundException thrown if the dir isn't a directory
   * @exception IllegalAccessException thrown if java securitymanager is
   *                                   violated or networking requested
   *                                   when networking disabled
   * @exception AgentLoaderException thrown when a networking error occurs
   *
   **************************************************************************/
  public String[] listFilesIn(String dir, boolean localSrc) throws FileNotFoundException,IllegalAccessException, AgentLoaderException {
    if(localSrc) {
      return lal.listFilesIn(dir);
    }
    else {
      checkNet(); //throws an exception if cannot use net
      return nal.listFilesIn(dir);
    }
  }

  /***************************************************************************
   *
   * List the Agents available in the specified directory.
   * The names returned are fully qualified java classnames with
   * repect to the package root.
   *
   * @param dir The directory to search.
   * @param localSrc true to load from local file system;
   *                 false to load from AgentServer
   *
   * @return an array of strings of the Agent's fully qualified
   *         classnames
   *
   * @exception FileNotFoundException thrown if the specified
   *                                  directory is not a directory
   * @exception IllegalAccessException thrown if the java security manger is
   *                                   violated or network services requested
   *                                   when networking disabled
   * @exception AgentLoaderException thrown when a networking error occurs
   *
   **************************************************************************/
  public String[] listAgentsIn(String dir, boolean localSrc) throws FileNotFoundException, IllegalAccessException, AgentLoaderException {
    if(localSrc){
      return lal.listAgentsIn(dir);
    }
    else{
      checkNet(); //throws an exception if cannot use net
      return nal.listAgentsIn(dir);
    }
  }

  /***************************************************************************
   *
   * Checks if the specified path is a file and exists.
   *
   * @param fname the specified path
   * @param localSrc true to load from local file system;
   *                 false to load from AgentServer
   *
   * @return <code>true</code> if the path is a file and exists;
   *         <br><code>false</code> otherwise<br>
   *
   * @exception IllegalAccessException thrown if pname is not a sub path
   *                                   of the package root or
   *                                   network services requested when
   *                                   networking disabled
   * @exception AgentLoaderException thrown when a networking error occurs
   *
   **************************************************************************/
  public boolean pathIsFile(String pname, boolean localSrc) throws IllegalAccessException, AgentLoaderException {
    if(localSrc) {
      return lal.pathIsFile(pname);
    }
    else {
      checkNet(); //throws an exception if network error occurrs
      return nal.pathIsFile(pname);
    }
  }
  
  /***************************************************************************
   *
   * Checks if the specified path is a directory and exists.
   *
   * @param fname the specified path
   * @param localSrc true to load from local file system;
   *                 false to load from AgentServer
   *
   * @return <code>true</code> if the path is a file and exists;
   *         <br><code>false</code> otherwise<br>
   *
   * @exception IllegalAccessException thrown if pname is not a sub path
   *                                   of the package root or
   *                                   network services requested when
   *                                   networking disabled
   * @exception AgentLoaderException thrown when a networking error occurs
   *
   **************************************************************************/
  public boolean pathIsDir(String pname, boolean localSrc) throws IllegalAccessException, AgentLoaderException {
    if(localSrc) {
      return lal.pathIsDir(pname);
    }
    else {
      checkNet(); //throws an exception if network error occurrs
      return nal.pathIsDir(pname);
    }
  }

  /***************************************************************************
   *
   * Checks if the specified path is a file and exists.
   *
   * @param fname the specified path
   * @param localSrc true to load from local file system;
   *                 false to load from AgentServer
   *
   * @return <code>true</code> if the path is a file and exists;
   *         <br><code>false</code> otherwise<br>
   *
   * @exception IllegalAccessException thrown if pname is not a sub path
   *                                   of the package root or
   *                                   network services requested when
   *                                   networking disabled
   * @exception AgentLoaderException thrown when a networking error occurs
   *
   **************************************************************************/
  public boolean pathExists(String pname, boolean localSrc) throws IllegalAccessException, AgentLoaderException {
    if(localSrc) {
      return lal.pathExists(pname);
    }
    else {
      checkNet(); //throws an exception if network error occurrs
      return nal.pathExists(pname);
    }
  }
  
  /***************************************************************************
   *
   * Get a InputStream for the specified filename.
   *
   * @param fname the input file's name
   * @param localSrc true to load from local file system;
   *                 false to load from AgentServer
   *
   * @return the InputStream for the specified file
   *
   * @exception IllegalAccessException thrown if fname is not a subpath
   *                                   of the package root or
   *                                   network services requested when
   *                                   networking disabled
   * @exception FileNotFoundException thrown if fname does not exist
   * @exception AgentLoaderException thrown when a networking error occurs
   *
   **************************************************************************/
  public InputStream getFileInputStream(String fname, boolean localSrc) throws IllegalAccessException, FileNotFoundException, AgentLoaderException,IOException {
    if(localSrc) {
      return lal.getFileInputStream(fname);
    }
    else {
      checkNet(); //throws an exception if network error occurrs
      return nal.getFileInputStream(fname);
    }
  }

  /***************************************************************************
   *
   * Get an OutputStream for the specified filename.
   *
   * @param fname the output file's name
   * @param append whether to append to the end of an existing file
   * @param localSrc true to load from local file system;
   *                 false to load from AgentServer
   *
   * @return the OutputStream for the specified file
   *
   * @exception IllegalAccessException thrown if fname is not a subpath of
   *                                   the package root or
   *                                   network services requested when
   *                                   networking disabled
   * @exception IOException thrown when fname cannot be opened for writing
   * @exception AgentLoaderException thrown when a networking error occurs
   *
   **************************************************************************/
  public OutputStream getFileOutputStream(String fname, boolean append, boolean localSrc) throws IllegalAccessException, IOException, AgentLoaderException {
    if(localSrc) {
      return lal.getFileOutputStream(fname,append);
    }
    else {
      checkNet(); //throws an exception if network error occurrs
      return nal.getFileOutputStream(fname,append);
    }
  }

  /***************************************************************************
   *
   * Get an OutputStream for the specified filename.
   *
   * @param fname the output file's name
   * @param localSrc true to load from local file system;
   *                 false to load from AgentServer
   *
   * @return the OutputStream for the specified file
   *
   * @exception IllegalAccessException thrown if fname is not a subpath of
   *                                   the package root or
   *                                   network services requested when
   *                                   networking disabled
   * @exception FileNotFoundException thrown if fname cannot be opened for
   *                                  writing
   * @exception IOException thrown when fname cannot be opened for writing
   * @exception AgentLoaderException thrown when a networking error occurs
   *
   **************************************************************************/
  public OutputStream getFileOutputStream(String fname, boolean localSrc) throws IllegalAccessException, IOException, AgentLoaderException {
    if(localSrc) {
      return lal.getFileOutputStream(fname);
    }
    else {
      checkNet(); //throws an exception if network error occurrs
      return nal.getFileOutputStream(fname);
    }
  }

  /**************************************************************************
   *
   * Get an AgentContainer given the filename
   *
   * @param fname The file name for the agentContainer
   * @param localSrc true to load from local file system;
   *                 false to load from AgentServer
   *
   * @return the agentContainer in fname
   *
   * @exception IllegalAccessException thrown if fname is not a subpath of
   *                                   the package root
   * @exception IOException thrown if fname cannot be opened for writing
   *
   **************************************************************************/
  public SystemConfiguration getSystemConfiguration(String fname,boolean localSrc) throws IllegalAccessException,IOException,AgentLoaderException {
    if (localSrc) {
      return lal.getSystemConfiguration(fname);
    } else {
      checkNet();
      return nal.getSystemConfiguration(fname);
    }
  }
  
  public ConfigAndLayout getConfigAndLayout(String fname,boolean localSrc) throws IllegalAccessException, IOException {
    if (localSrc) {
      return lal.getConfigAndLayout(fname);
    } else {
      checkNet();
      return nal.getConfigAndLayout(fname);
    }
  }

  /***************************************************************************
   *
   * Deletes the specified path.
   *
   * @param pname the full path to the file or directory you want to delete
   *              a directory must be empty for the delete to succeed
   * @param localSrc true to load from local file system;
   *                 false to load from AgentServer   
   *
   * @return <code>true</code> if the file or directory was deleted;
   *         <br><code>false</code> otherwise<br>
   *
   * @exception IllegalAccessException thrown if the path is not a sub path
   *                                   of the package root or
   *                                   network services requested when
   *                                   networking disabled
   * @exception AgentLoaderException thrown when a networking error occurs
   *
   **************************************************************************/
  public boolean delete(String pname, boolean localSrc) throws IllegalAccessException, AgentLoaderException {
    if(localSrc) {
      return lal.delete(pname);
    }
    else {
      checkNet(); //throws an exception if network error occurrs
      return nal.delete(pname);
    }
  }

  /***************************************************************************
   *
   * Creates the specified directory if it doesn't already exist 
   *
   * @param the full path to the directory to create
   * @param localSrc true to load from local file system;
   *                 false to load from AgentServer    
   *
   * @return <code>true</code> if the directory (or directories) could
   *         be created;
   *         <br><code>false</code> otherwise<br>
   *
   * @exception IllegalAccessException thrown if the path is not a sub path
   *                                   of the package root or
   *                                   network services requested when
   *                                   networking disabled
   * @exception AgentLoaderException thrown when a networking error occurs
   *
   **************************************************************************/
  public boolean mkdir(String pname, boolean localSrc) throws IllegalAccessException,
    AgentLoaderException{
    if(localSrc) {
      return lal.mkdir(pname);
    }
    else {
      checkNet(); //throws an exception if network error occurrs
      return nal.mkdir(pname);
    }
  }

  /***************************************************************************
   *
   * Renames the file or directory specified by src to the
   * one specified by dest. They must both be full paths.
   *
   * @param src the path to rename
   * @param dest the new name
   * @param localSrc true to load from local file system;
   *                 false to load from AgentServer
   *
   * @return <code>true</code> if the directory or file was renamed;
   *         <br><code>false</code> otherwise<br>
   *
   * @exception IllegalAccessException thrown if the either path is not
   *                                   a sub path of the package root or
   *                                   network services requested when
   *                                   networking disabled
   * @exception AgentLoaderException thrown when a networking error occurs
   *
   **************************************************************************/
  public boolean rename(String src, String dest, boolean localSrc) throws IllegalAccessException,
    AgentLoaderException{
    if(localSrc) {
      return lal.rename(src,dest);
    }
    else {
      checkNet(); //throws an exception if network error occurrs
      return nal.rename(src,dest);
    }
  }
  
  /***************************************************************************
   *
   * Call this to manually close a connection to the AgentServer.
   *
   * @param 
   *
   * @return 
   *
   * @exception 
   *
   **************************************************************************/
  public void disconnectAgentServer() {
    if(nal != null) {
      nal.disconnect();
    }
  }
  
  /* 
     public static void main(String[] args) {
     
     AgentLoader aLoader;
     
     if(args.length > 0) {
     System.out.println("Agent Root Dir: "+args[0]+"\n");
     try {
     aLoader = new AgentLoader(args[0]);
     
     Agent myFirstAgent;
     
     myFirstAgent = aLoader.newInstance("adaptive.agents.PBATestAgent");
     myFirstAgent.initialize("Test1",aLoader);
     System.out.println("ID: "+ myFirstAgent.getID());
     System.out.println("Inputs: "+ myFirstAgent.queryNumInputs());
     System.out.println("Outputs: "+myFirstAgent.queryNumOutputs());
     
     Agent runningAgent;
     
     runningAgent = 
     aLoader.newInstance("adaptive.agents.counting.PBACountAgent");
     
     runningAgent.initialize("Run1",aLoader);
     runningAgent.start();
     
     
     System.out.println("Agent: "+ runningAgent.getID());
     for(int j = 0; j < 25;j++) {
     System.out.println("Output 0: "+runningAgent.readOutput(0));
     System.out.println("Output 1: "+runningAgent.readOutput(1));
     Thread.sleep(10);
     }
     
     System.exit(1);
     }
     catch (AgentLoaderException alex) {
     System.err.println(alex.getMessage());
     System.exit(1);
     }
     catch (Exception e) {
     if (e instanceof InterruptedException) {
     
     }
     else if (e instanceof ClassNotFoundException) {
     System.err.println("Class "+ 
     e.getMessage()+ " Not Found");
     }
     else if (e instanceof InstantiationException ) {
     System.err.println("Can't Instantiate Class "+ 
     e.getMessage());
     }
     else if (e instanceof IllegalAccessException ) {
     System.err.println("Illegal Access: "+
     e.getMessage());
     }
     }
     }
     else { 
     System.err.println("Usage: java AgentLoader <Agent Root Path>\n");
     }
     }
  */

}

