package adaptive.core;

import java.util.*;
import java.io.*;

/*****************************************************************************
 * <! Copyright 1998, Institute for Complex Engineered Sytems,
 *                    Carnegie Mellon University
 *
 * PROJECT: Adaptable 
 *
 * FILE: AgentDescription.java 
 * >
 *
 * Describes the connections and internal state of an agent. Note: this is
 * only a data storage class. It is the users responsibility to make sure
 * the data stored in this class is consistent for a real agent.
 * 
 * @author Charles Tennent <A HREF="mailto:tennent@andrew.cmu.edu">
 *         tennent@andrew.cmu.edu</A>
 *         <br> </br>
 *
 * @version  1.00 11/18/98<br> </br>
 *
 * <!
 * REVISION HISTORY:
 *
 * $Log: AgentDescription.java,v $
 * Revision 1.15.2.9  2000/06/06 00:21:21  jrj
 * wrote function to synchronize the ports with the Agent defining this
 *
 * Revision 1.15.2.8  2000/06/05 05:48:02  jrj
 * more macro fixes.  Wrote a couple more functions to help once we start start storing port types in the AgentDescription and to keep the port information in sync with the PBAgent definition.
 *
 * Revision 1.15.2.7  2000/06/03 19:25:07  jrj
 * fixed an error in mapping inputs on macros.  Added a few diagnostic messages during the system verification step.
 *
 * Revision 1.15.2.6  2000/05/20 19:11:14  jrj
 * fixed a couple more macro bugs.  Took preliminary steps toward making
 * fully recursive macros in MacroDescription.  Have not yet made changes
 * that would damage save file.
 *
 * Revision 1.15.2.5  2000/05/15 22:39:36  jrj
 * Made fixes to macros so that they work.  Changed the executing macro
 * so that it is recursive, can be manipulated as a whole, and its ports
 * can be asked for directly as if it were any other BasicPortAgent.
 * Made several other changes to simplify and assist these changes.
 * Finally purged the last remaining bit of things named clump.
 *
 * Revision 1.15.2.4  2000/04/28 04:22:27  jrj
 * minor changes in interface regarding getStateEditor;  also an attempt
 * on AgentDescription to have it catch deserialization exceptions in the
 * state that still fails the same as the default method, but maybe
 * someone else can see what I'm doing wrong there.  I think I also
 * increased the max text box size in Netcontroller
 *
 * Revision 1.15.2.3  2000/02/20 23:57:26  arb
 * no more clumps
 *
 * Revision 1.15.2.2  1999/12/14 21:45:30  heller
 * added event classes
 * added code to create event connections in system executor
 * updated state viewers
 *
# Revision 1.15.2.1  99/09/24  20:32:07  yatish
# Fixed Macros.
# 
 * Revision 1.15  1999/04/18 22:55:04  yatish
 * Made some modifications to the way port names are stored.
 *
 * Revision 1.14  1999/04/10 03:39:53  yatish
 * Macros work.  Minor changes to modulemanager.  Fixed delete mapping bug.
 *
 * Bugs:  Deleting modules no longer works.  I have a slight problem with
 * the keylistener.
 *
 * Added some methods to the description/configuration files.
 *
 * Revision 1.13  1999/04/06 15:11:56  yatish
 * Massive changes to the Save File format along with the addition of Macro
 * support.
 *
 * Revision 1.12  1999/03/04 17:11:40  telamon
 * fixed null pointer clone bug
 *
 * Revision 1.11  1999/03/01 06:15:51  yatish
 * Added port names to the agent container and description.  Added the
 * ability to get an agentcontainer to local/net agent loader. (not fully
 * implemented in net yet)
 *
 * Revision 1.10  1999/02/18 01:41:56  yatish
 * Added groupId to the agentDescription.
 *
 * Added version,author,date,description,lastrevision to agentcontainer
 *
 * Revision 1.9  1999/02/12 22:38:56  yatish
 * Added a group id field and accessor methods to agentDescription.
 *
 * Added a pegged field to agentcontainer
 *
 *
 * Rewriting the README. (not done yet)
 *
 * Fixed the error messages from the browser.  Especially when the mm is
 * run in local mode and it needs to locate network agents.
 *
# Revision 1.8  99/02/09  20:08:14  telamon
#  new Adaptive core for 2/10 demo
# 
# Revision 1.7  98/12/07  13:17:32  telamon
# fixed the SystemExecutor
# 
 * >
 ****************************************************************************/
public class AgentDescription implements Serializable, Cloneable {

  static final long serialVersionUID = 5010156495454942566L;

  public static final int LOCAL_FILE = 0;
  public static final int AGENT_SERVER = 1;
  public static final String RUN_LOCAL = "local";

  protected int numInputs;
  protected int numOutputs;
  /** class name of this agent */
  protected String type;
  /** current instance name */
  protected String name;
  /** should contain InputLocation classes. has the source of all the inputs */
  protected Vector inputs;
  /** where to find class  ie LOCAL_FILE or AGENT_SERVER*/
  protected int source;
  /** ip address to run agent */
  protected String location;


  /** if this is a macro, then the state is a hashtable of agent names.
	   the hashtable contains all the agents in this macro */
  /** user defined internal state */
  protected Object state;
  
  protected Hashtable inputPortNames;
  protected Hashtable outputPortNames;

    private int moduleModeSetting = Agent.LOOP_ONLY;

  
  /********************************************************************
   *  
   * Description  Creates a new Vector for inputs  
   *
   * @param none 
   * @return 
   * @exception none 
   ********************************************************************/
  public AgentDescription() {
    inputs = new Vector();
    inputPortNames= new Hashtable();
    outputPortNames=new Hashtable();
    source=LOCAL_FILE;
    location=RUN_LOCAL;
    numInputs=0;
    numOutputs=0;
  }

  public AgentDescription(String n,String t) {
    this();
    name=n;
    type=t;
  }
    /**************
     *  added new constructor--jrj 5/19/2000
     * extract from an Agent the information necessary to use and save it in a configuration
     ************************/
    public AgentDescription(Agent agent){ 
	this(agent.getID(),agent.getClass().getName());
	this.numInputs=agent.getNumInputs();
	this.numOutputs=agent.getNumOutputs();
	int i;
	//read in and store port types--not implemented yet because it requires save file changes I'm not ready to make yet
	for (i=0;i<numInputs;i++){
	    setInputPortType(agent.getInputPortType(i),i);
	}
	for (i=0;i<numOutputs;i++){
	    setOutputPortType(agent.getOutputPortType(i),i);
	}
	
	//read in and store port names
	for (i=0;i<numInputs;i++){
	    setInputPortName(agent.getInputPortName(i),i);
	}
	for (i=0;i<numOutputs;i++){
	    setOutputPortName(agent.getOutputPortName(i),i);
	}

	//set up state--might need to switch to BasicPortAgent to do this
	//this.setState(agent.saveState());	

    }
  
  /***************************************************************************
   *
   * Get the number of outputs.
   *
   * @param 
   *
   * @return 
   *
   * @exception 
   *
   **************************************************************************/
  public int getNumOutputs() {
    return numOutputs;    
  }

  public int getNumInputs() {
	 return numInputs;
  }
  /***************************************************************************
   *
   * Set the number of outputs.
   *
   * @param i the number of inputs 
   *
   * @return 
   *
   * @exception 
   *
   **************************************************************************/
  public void setNumOutputs(int i) {
    numOutputs = i;    
  }
  public void setNumInputs(int d) {
	 numInputs=d;
  }
    /***********
     * getInputPortType
     * @return Class indicating the port type 
     *        or null indicating that the port type is not stored
     *	     the null output is a temporary measure to ease the transition to storing port types in the AgentDescription
     *
     *************/
    public Class getInputPortType(int portNum){
	return null;
    }
    /************
    setInputPortType
    ************/
    public void setInputPortType(Class type, int portNum){
    }
    /***********
    getOutputPortType
    * @return Class indicating the port type 
             or null indicating that the port type is not stored
	     the null output is a temporary measure to ease the transition to storing port types in the AgentDescription
    *
    *************/
    public Class getOutputPortType(int portNum){
	return null;
    }
    /************
    setOutputPortType
    ************/
    public void setOutputPortType(Class type, int portNum){
    }
    /*************************************************************************
     * syncPorts
     * get information about the ports from an agent of this type
     * this method is used to update AgentDescriptions that are out of date
     * from an Agent that has been changed since first created
     *
     * any existing input mappings are not changed--this may cause a verification error if the types have changed, but ModuleManager will report this and allow the user to remap the inputs
     * @return true=synchronization successful
     *         false=error occurred
     ***********************************************************************/
    public boolean syncPorts(Agent agent){
	boolean retval=false;
	int i;
	System.out.println("Synchronizing ports for "+getName());
	if (agent!=null){
	    //if the agent's type is the same as this AgentDescription
	    try{
		if (this.getType().equals(agent.getClass().getName())){
		  System.out.println("inside");
		    setNumInputs(agent.getNumInputs());
		    setNumOutputs(agent.getNumOutputs());
		    for (i=0;i<getNumInputs();i++){
			setInputPortName(agent.getInputPortName(i),i);
			setInputPortType(agent.getInputPortType(i),i);
			System.out.println("input"+i);
		    }
		    for (i=0;i<getNumOutputs();i++){
			setOutputPortName(agent.getInputPortName(i),i);
			setOutputPortType(agent.getInputPortType(i),i);
			System.out.println("output"+i);
		    }
		}
		retval=true;
	    }catch(NullPointerException npe){
		System.out.println("Exception in syncPorts:"+npe.toString());
		retval=false;
	    }
	}
	return retval;
    }
  public boolean syncPortTypes(Agent agent){
	boolean retval=false;
	int i;
	System.out.println("Synchronizing port types for "+getName());
	if (agent!=null){
	    //if the agent's type is the same as this AgentDescription
	    try{
		if (this.getType().equals(agent.getClass().getName())){
		  System.out.println("inside");
		    setNumInputs(agent.getNumInputs());
		    setNumOutputs(agent.getNumOutputs());
		    for (i=0;i<getNumInputs();i++){
		      setInputPortType(agent.getInputPortType(i),i);
		      System.out.println("input"+i);
		    }
		    for (i=0;i<getNumOutputs();i++){
			setOutputPortType(agent.getInputPortType(i),i);
			System.out.println("output"+i);
		    }
		}
		retval=true;
	    }catch(NullPointerException npe){
		System.out.println("Exception in syncPorts:"+npe.toString());
		retval=false;
	    }
	}
	return retval;
  }
  /***************************************************************************
   *
   * Get the agent type.
   *
   * @param 
   * @return the agent type
   * @exception 
   **************************************************************************/
  public String getType(){
	 return type;
  }

  /***************************************************************************
   *
   * Set the agent type.
   *
   * @param type the new agent type
   * @return 
   * @exception 
   **************************************************************************/
  public void setType(String t){
    type = t;    
  }
  /***************************************************************************
   *
   * Get the agentID.
   *
   * @param 
   * @return the agentID
   * @exception 
   **************************************************************************/
  public String getName() {
	 return name;
  }
  /************************************************************************
   *
   * Set the agentID.
   *
   * @param 
   * @return the agentID
   * @exception 
   **************************************************************************/
  public void setName(String n) {
    name = n;  
  }
  
  /***************************************************************************
   *
   * Get num inputs mapped.
   *
   * @param 
   * @return the number of mapped inputs.
   * @exception 
   **************************************************************************/
  public int getNumMappedInputs() {
    return inputs.size();
  }
  
  /***************************************************************************
   *
   * Get the specified input location from the inputs vector.
   *
   * @param 
   * @return the sepecified input mapping from the inputs vector.
   *         <br> </br><code>null</code> if the specified input mapping
   *         doesn't exist
   * @exception 
   **************************************************************************/
  public InputLocation getInputMapping(int i) {
      InputLocation result=null;
    if(i < inputs.size()) {
      result= (InputLocation) inputs.elementAt(i);
    }
    //    System.out.println("input mapping for port "+i+" on "+getName()+" is "+result);
    return result;
  }
    public boolean isInputPortMapped(int i){
	return (getInputMapping(i)!=null);
    }
  /***************************************************************************
   *
   * Add an input location to the inputs vector.
   *
   * @param il the input location to add
   * @return 
   * @exception 
   **************************************************************************/
  public void addInputMapping(InputLocation il) {
    inputs.addElement(il);      
  }

  /**
   * Set the input mapping at a specific input port. If the vector isn't
   * big enough to support it, increase the size of the vector until it has
   * a spot for the new input mapping.
   *
   * @param il the Input Location to add
   * @param port The port to add it too
   */
  public void setInputMapping(InputLocation il,int port) {
      //      System.out.println("set input mapping port"+port+": "+il);
      if (port >= inputs.size()) {
	  inputs.setSize(port+1); 
      }
      inputs.setElementAt(il,port);
  }
  
  /***************************************************************************
   *
   * Get where to run the class from.
   *
   * @param 
   *
   * @return an integer flag equal to LOCAL_FILE or AGENT_SERVER
   *
   * @exception 
   *
   **************************************************************************/
  public int getSource() {
    return source;    
  }

  /***************************************************************************
   *
   * Set where to run the class from.
   *
   * @param where a flag indicating where to load the class from
   *
   * @return 
   *
   * @exception 
   *
   **************************************************************************/
  public void setSource(int where) {
    source = where;    
  }
  
  /***************************************************************************
   *
   * Get the hostname of where the agent is to run.
   *
   * @param 
   *
   * @return the hostname of where the agent is to run
   *
   * @exception 
   *
   **************************************************************************/
  public String getRunLocation() {
    return location;    
  }
  /*************************
  * return all the locations over which this agent runs
  * --will only be one for most agents, but can be more for macros
  * @param none
  * @return Enumeration listing all runLocations
  * @exception none
  **********************/
  public Enumeration getAllRunLocations(){
    Vector temp=new Vector();
    if (this.location!=null) temp.addElement(this.location);
    return temp.elements();
  }  
  /******
   * runsOnLocation
   * return true if any part of this agent runs on runLocation
   *******/
  public boolean runsOnLocation(String runLocation){
    boolean retval=(runLocation==null?false:runLocation.equalsIgnoreCase(this.location));
    return retval;
  }
  /***************************************************************************
   * setRunLocation
   * Set the hostname of where the agent is to run.
   * @param runWhere the hostname of where the agent is to run
   * @return <code>true</code> if the runLocation was set;
   *         <br> </br><code>false</code>
   * @exception 
   **************************************************************************/
  public boolean setRunLocation(String runWhere) {
    if(runWhere == null) {
      return false;
    }
    else {      
      location = runWhere;
      return true;
    }
  }
  /*********
  * change any runLocation on this agent that was oldLocation to be newLocation
  * @param String oldLocation
  *        String newLocation
  * @output true=replacements made
  *         false=no replacements made
  * @exception none
  **********/
  public boolean replaceRunLocation(String oldLocation,String newLocation){
    boolean retval=false;
    if (this.getRunLocation().equals(oldLocation)){
      this.setRunLocation(newLocation);
      retval=true;
    }
    return retval;
  }
  /***************************************************************************
   *
   * Get the object graph representing the Agent's state
   *
   * @param 
   * @return the object graph of the agent's state
   * @exception 
   **************************************************************************/
  public Object getState() {
    return state;    
  }
  
  /***************************************************************************
   *
   * Set the object graph representing the Agent's state
   *
   * @param theState the object graph representing the Agent's state
   *                 may be null.
   * @return 
   * @exception 
   **************************************************************************/
  public void setState(Object theState) {
    state = theState;    
  }
    /*************
    setStateField
    attempts to set a particular field of theState, even though it has no real idea of what the structure of the state is.  This is not implemented yet, and I'm not sure if I will implement it here
    ********/
    public void setStateField(Object descriptor,Object theState){
	setState(theState);
    }
  
  /*********************************************************************
   *
   * <! Description: >
   *
   * Returns whether the Agent represented by this description
   * requires networking. An Agent requires networking when it runs
   * remotely (a stub) or it loads it's class over the network.
   *
   * @param 
   *
   * @return <code>true</code> if the agent requires networking; 
   *         <br><code>false</code> if the agent doesn't need networking;
   *
   * @exception 
   *
   *********************************************************************/
  public final boolean requiresNetwork() {
    return (!this.location.equals(RUN_LOCAL) 
	    || this.source == AGENT_SERVER);
  }
  
  /**********************************************************************
   * Description Returns a new Agent Description object
   * @param 
   * @return the new agent description
   * @exception 
   *********************************************************************/
  public Object clone() {
    // cheat a little bit 
    return cloneSerializableObject(this);
  }

  /**
    * Clones a serializeable object.  If the object isn't serializable then
    * return null.
    *
    * @param o The object to Clone
    * @return A clone of the object.
    */
  protected static Object cloneSerializableObject(Object o) {
    if (o==null) return null;
    try {
      ByteArrayOutputStream bout = new ByteArrayOutputStream();
      ObjectOutputStream out = new ObjectOutputStream(bout);
      out.writeObject(o);
      out.close();
      ByteArrayInputStream bin = new ByteArrayInputStream(bout.toByteArray());
      ObjectInputStream in = new ObjectInputStream(bin);
      Object ret = in.readObject();
      in.close();
      return ret;
    } catch (Exception e)
      {
	e.printStackTrace();
        return null; 
      }
  }

  /**
   * Return the group name for this agent
   *
   * @return the group name for this agent
   */
    
  public String getInputPortName(int port) {
    //if (inputPortNames!=null)
    return (String)inputPortNames.get(new Integer(port));
    //else return null;
  }
  
  public String getOutputPortName(int port) {
    //if (outputPortNames!=null)
    return (String)outputPortNames.get(new Integer(port));
    //else return null;
  }
  
  public void setInputPortName(String n,int port) {
    //if (inputPortNames==null) inputPortNames=new Hashtable();
    if (n!=null)
      inputPortNames.put(new Integer(port),n);
  }
  
  public void setOutputPortName(String n,int port) {
    //if (outputPortNames==null) outputPortNames=new Hashtable();
    if (n!=null)
      outputPortNames.put(new Integer(port),n);
  }
  
  public String toString() {
    return toString("");
  }
  
    public String toString(String s) {
    StringBuffer sb=new StringBuffer(s+"AgentDescription: "+name+" on "+
				     location+"\n");
    int x;
    for (x=0;x<getNumInputs();x++) {
	sb.append(s+"input "+x+" type:"+getInputPortType(x)+" mapped to ");
      if (getInputMapping(x)!=null) {
	sb.append(getInputMapping(x).toString()+"\n");
      } else {
	sb.append(String.valueOf(x)+" NULL\n");
      }      
    } 
    for (x=0;x<getNumOutputs();x++){
	sb.append(s+"output "+x+" type:"+getOutputPortType(x));
    }
    return sb.toString();
  }
  /**
    * @see BasicPortAgent.getModuleMode()
    */
  public int getModuleMode() {
    return moduleModeSetting;
  }
  
  /**
    * @see BasicPortAgent.setModuleMode()
    */ 
  public void setModuleMode( int newMode ) {
    moduleModeSetting = newMode;
  }

  //overwrote the default deserialization method, so an exception on the state wouldn't prevent the reading of the rest of the AgentDescription--jrj 4/7/00
  //THIS DOESN'T WORK because for some reason it still echos the caught exceptions ?????WTF????
  private void readObject(ObjectInputStream in) throws IOException, ClassNotFoundException{
    try{
      in.defaultReadObject();
    }catch(java.io.InvalidClassException e){
      System.out.println("Caught exception in AgentDescription deserialization");
      System.out.println(e.toString());
      state=null;
    }catch(ClassNotFoundException cnfe){
      System.out.println("Caught exception in AgentDescription deserialization");
      System.out.println(cnfe.toString());
      state=null;
    }
  }

}
