package adaptive.core;

import java.io.*;
import java.util.*;

/*****************************************************************************
 * <! Copyright 1998, Institute for Complex Engineered Sytems,
 *                    Carnegie Mellon University
 *
 * PROJECT: Adaptable 
 *
 * FILE: AgentContainer.java 
 * >
 *
 * Agent configuration file container.
 * 
 * @author  Charles Tennent <A HREF="mailto:cht@cs.cmu.edu">cht@cs.cmu.edu</A>
 *
 *
 * @version  1.10 12/05/98<br> </br>
 *
 * <!
 * REVISION HISTORY:
 *
 * $Log: AgentContainer.java,v $
 * Revision 1.13.2.1  2000/02/20 23:56:54  arb
 * no more clumps
 *
# Revision 1.13  99/04/06  15:11:56  yatish
# Massive changes to the Save File format along with the addition of Macro
# support.
# 
 * Revision 1.12  1999/03/01 06:15:50  yatish
 * Added port names to the agent container and description.  Added the
 * ability to get an agentcontainer to local/net agent loader. (not fully
 * implemented in net yet)
 *
 * Revision 1.11  1999/02/18 01:41:56  yatish
 * Added groupId to the agentDescription.
 *
 * Added version,author,date,description,lastrevision to agentcontainer
 *
 * Revision 1.10  1999/02/12 22:38:55  yatish
 * Added a group id field and accessor methods to agentDescription.
 *
 * Added a pegged field to agentcontainer
 *
 *
 * Rewriting the README. (not done yet)
 *
 * Fixed the error messages from the browser.  Especially when the mm is
 * run in local mode and it needs to locate network agents.
 *
 * Revision 1.9  1999/02/07 14:38:30  telamon
 * new accessors
 *
 * Revision 1.8  98/12/07  13:17:31  telamon
 * fixed the SystemExecutor
 * 
 * >
 ****************************************************************************/
public class AgentContainer implements Serializable, Cloneable {

  static final long serialVersionUID = 1549224596619833426L;
  
  public String systemType; /** the NAME of the system */
  public Hashtable agentlist; /** list of AgentDescriptions */
  public String[] gui;  /** gui save info */

  /** Does this configuration have to be run on one computer, or can it
	be broken up and run on multiple machines*/
  private boolean pegged; 

  
  /** author of the system */
  private String author;
  /** system version */
  private String version;
  /** description of system */
  private String description;
  /** Date of creation */
  private String date;
  /** Things that were changed since last revision */
  private String lastRevision;

  private Hashtable inputPortNames;
  private Hashtable outputPortNames;

  /** list of all the macros (agentcontainers) in this system. */
  private Hashtable macroList;

  private Hashtable inputAgentList;
  private Hashtable outputAgentList;
  
  /**************************************************************************
   * <!
   * Method:AgentContainer()
   *
   * Description:
   * >
   *  Initialize agentlist with a new Hashtable, set rest to null.
   * 
   * @param 
   *
   * @return 
   *
   * @exception 
   *************************************************************************/  
  public AgentContainer( ) {
    agentlist = new Hashtable();
    gui = null;
    systemType = null;
	 inputPortNames=new Hashtable();
	 outputPortNames=new Hashtable();
	 macroList=null;
  }
  
  /**************************************************************************
   * <!
   * Method: AgentContainer()
   *
   * Description:
   * >
   * Creates a new Hashtable obect for agentlist
   * sets systemType.
   * 
   * @param sysType sets systemType 
   *
   * @return 
   *
   * @exception 
   *************************************************************************/
  public AgentContainer(String sysType) {
    this();
    systemType = sysType;
    agentlist = new Hashtable();
    gui = null;
  }

  

  /**************************************************************************
   * <!
   * Method: addAgent()
   * Description:
   * >
   * Adds an agent to agentlist.
   * 
   * @param AgentDescription ad agent to add 
   *
   * @return 
   *
   * @exception 
   *************************************************************************/
  public void addAgent(AgentDescription ad) {
    agentlist.put(ad.getName(),ad);
  }

  /**************************************************************************
   * <!Description:>
   *
   * Gets the system type.
   * 
   * @param 
   *
   * @return the system type 
   *
   * @exception 
   *************************************************************************/
  public String getSystemType() {
    return systemType;
  }
  
  /**************************************************************************
   * <!Description:>
   *
   * Set the system type.
   * 
   * @param 
   *
   * @return 
   *
   * @exception 
   *************************************************************************/
  public void setSystemType(String sysType) {
    systemType = sysType;
  }
  
  /**************************************************************************
   * <!
   * Method: getAgent()
   *
   * Description:
   * >
   * Find and AgentDescription.
   * 
   * @param agentID the name of the AgentDescription
   *
   * @return AgentDescription or null if not found  
   *
   * @exception 
   *************************************************************************/
  public AgentDescription getAgent(String agentID) {
    return (AgentDescription)agentlist.get(agentID);    
  }

  
  /***************************************************************************
   *
   * Return an enumeration of the agent names.
   *
   * @param 
   *
   * @return an enumerations of the agent names.
   *
   * @exception 
   *
   **************************************************************************/
  public Enumeration getAgentNames() {    
    return agentlist.keys();    
  }

  
  /***************************************************************************
   *
   * Get the reference to the gui string array.
   *
   * @param 
   *
   * @return the gui string array
   *
   * @exception 
   *
   **************************************************************************/
  public String[] getGuiStrings() {
    return gui;    
  }

  
  /***************************************************************************
   *
   * Set the gui string array
   *
   * @param 
   *
   * @return 
   *
   * @exception 
   *
   **************************************************************************/
  public void setGuiStrings(String[] guiStrs) {
    gui = guiStrs;    
  }

  
  /***************************************************************************
   *
   * Returns whether this agent container is empty.
   *
   * @param 
   *
   * @return <code>true</code> if the agent container is empty;
   *         <br><code>false</code> otherwise</br>
   *
   * @exception 
   *
   **************************************************************************/
  public boolean isEmpty() {
    return agentlist.isEmpty();
  }
  
  /*************************************************************************
   * <!
   * Method: clone()
   *
   * Description:
   * >
   * Produces a new AgentContainer that has it's own Hashtable
   *             and copies of each Agent Description object, 
   *             but it does not clone the Agent's internalParameters fields
   *             nor the Agent's inputLocations
   *   
   * 
   * @param 
   *
   * @return new copy of it's self
   *
   * @exception 
   ************************************************************************/
  public Object clone() {

    AgentContainer retVal = new AgentContainer();
    Enumeration keys;
    String id;
    retVal.setSystemType(this.getSystemType());    
	 retVal.setPegged(isPegged());
	 retVal.setAuthor(getAuthor());
	 retVal.setVersion(getVersion());
	 retVal.setDescription(getDescription());
	 retVal.setDate(getDate());
	 retVal.setLastRevision(getLastRevision());
	 retVal.setInputPortNames((Hashtable)getInputPortNames().clone());
	 retVal.setOutputPortNames((Hashtable)getOutputPortNames().clone());
	 
	 if (macroList!=null) {
		retVal.macroList = new Hashtable(macroList.size());
		keys = getMacroNames();
		while (keys.hasMoreElements()) {
		  id = (String)keys.nextElement();
		  retVal.addMacro(getMacro(id));
		}
	 }
    //copy the agentlist
    keys = this.getAgentNames();
    while(keys.hasMoreElements()) {
      id = (String)keys.nextElement();        
      retVal.addAgent((AgentDescription) this.getAgent(id).clone());   
    }
      
    //copy gui fields; not really copying, just making our own array
    //pointing back to the same constant strings
    String[] gStrs;    
    if((gStrs = this.getGuiStrings()) != null) {
      String[] newStrs = new String[gStrs.length];
      for(int i=0; i < gStrs.length;i++) {
	newStrs[i] = gStrs[i];
      }
      retVal.setGuiStrings(newStrs);      
    }
    else {
      retVal.setGuiStrings(null);
    }

    return retVal;
  }

  private Hashtable getInputPortNames() {
	 return inputPortNames;
  }
  
  private Hashtable getOutputPortNames() {
	 return outputPortNames;
  }

  private void setInputPortNames(Hashtable in) {
	 inputPortNames=in;
  }
  
  private void setOutputPortNames(Hashtable on) {
	 outputPortNames=on;
  }

  /**
	* Returns whether or not the configuration is pegged.
	*
	* @return whether or not the configuratino is pegged.
	*/
  public boolean isPegged() {
	 return pegged;
  }

  /**
	* Set whether the configuration is pegged.
	*
	* @param value Is the configuration pegged
	*/
  public void setPegged(boolean value) {
	 pegged=value;
  }

  public String getAuthor() {
	 return author;
  }
  
  public String getDescription() {
	 return description;
  }
  
  public String getVersion() {
	 return version;
  }
  
  public String getDate() {
	 return date;
  }
  
  public String getLastRevision() {
	 return lastRevision;
  }

  public void setAuthor(String a) {
	 author=a;
  }
  
  public void setDate(String d) {
	 date=d;
  }
  
  public void setVersion(String v) {
	 version=v;
  }
  
  public void setDescription(String d) {
	 description=d;
  }
  
  public void setLastRevision(String lr) {
	 lastRevision=lr;
  }  

  public String getInputPortName(int port) {
	 if (inputPortNames!=null)
		return (String)inputPortNames.get(new Integer(port));
	 else return null;
  }

  public String getOutputPortName(int port) {
	 if (outputPortNames!=null)
		return (String)outputPortNames.get(new Integer(port));
	 else return null;
  }
  
  public void setInputPortName(String n,int port) {
	 // for compatibility with old containers...
	 if (inputPortNames==null) inputPortNames=new Hashtable();

	 inputPortNames.put(new Integer(port),n);
  }
  
  public void setOutputPortName(String n,int port) {
	 if (outputPortNames==null) outputPortNames=new Hashtable();
	 outputPortNames.put(new Integer(port),n);
  }

  public Enumeration getMacroNames() {
	 if (macroList!=null) 
		return macroList.keys();
	 else return null;
  }
  
  public AgentContainer getMacro(String s) {
	 if (macroList!=null) 
		return (AgentContainer)macroList.get(s);
	 else
		return null;
  }

  public void addMacro(AgentContainer ac) {
	 if (macroList==null) macroList=new Hashtable();
	 macroList.put(ac.getSystemType(),ac);
  }
  
  
}
