package adaptive.core;

/*****************************************************************
 * <!
 * PROJECT: Adaptable 
 *
 * FILE: Agent.java
 * 
 * ABSTRACT: > <p> This interface defines the basic skeleton of a Port
 * Based Agent. Specific agent types should extend Class
 * <code>PBAgent</code>.
 * <p> 
 * NOTE: For dynamic loading, all classes that implement this
 * interface must have a no argument constructor. This cannot be
 * enforced at compile time. A NoSuchMethodError exception will be
 * thrown during dynamic instantiation, if the no argument constructor
 * is missing.
 *
 * @see PBAgent
 *
 * @author Ted Pham <A HREF="mailto:telamon@CMU.EDU">telamon@CMU.EDU</A>
 *         <br> </br>
 *
 * @version  1.10 11/20/98<br> </br>
 *
 * <!
 * REVISION HISTORY:
 * Version  Date       Who        What
 * -------  -------    -------    --------------------  
 * 1.00     9/24/98    telamon    Created.
 * 1.10     11/20/98   telamon    Fixed input/output ports.
 * >
 *****************************************************************/

public interface Agent {

  /******************************************************************
   * <!
   * Method: setup()
   *
   * Description:
   * >
   *
   * Psuedo-constructor to prepare agent for runtime.
   *
   * @param agentID the agent's name
   * @param myLoader for use by composed agents (coming soon)
   * @param internalState the agent's saved internal state
   *
   * @return true if setup completed successfully; <br>
   *         false otherwise </br>
   *
   * @exception 
   ******************************************************************/
  public abstract boolean setup(String agentID,AgentLoader myLoader,
				Object internalState);
  
  /******************************************************************
   * <!
   * Method: getID()
   *
   * Description:
   * >
   *
   * Returns the Agent's name.
   *
   * @param 
   *
   * @return the Agent's name
   *
   * @exception 
   ******************************************************************/
  public abstract String getID();
  /******************************************************************
   * <!
   * Method: setID()
   *
   * Description:
   * >
   *
   * Set the Agent's name.
   *
   * @param agentID The Agent's name.
   *
   * @return 
   *
   * @exception 
   ******************************************************************/
  /*  public abstract void setID( String agentID );*/

  /******************************************************************
   * <!
   * Method: getNumInputs()
   *
   * Description:
   * >
   *
   * Get the number of inputs this agent needs.
   *
   * @param 
   *
   * @return the number of inputs this agent needs. 
   *
   * @exception 
   ******************************************************************/
  public abstract int getNumInputs();

  /******************************************************************
   * <!
   * Method: getNumOutputs()
   *
   * Description:
   * >
   *
   * Get the number of outputs this agent provides.
   *
   * @param 
   *
   * @return the number of outputs this agent provides. 
   *
   * @exception 
   ******************************************************************/
  public abstract int getNumOutputs();

  /******************************************************************
   * <!
   * Method: setNumInputsOutputs()
   *
   * Description:
   * >
   *
   * Call to set the agent's number of inputs and outputs.
   *
   * @param numin Number of inputs
   * @param numout Number of outputs
   *
   * @return 
   *
   * @exception 
   ******************************************************************/
  /*  protected abstract void setNumInputsOutputs( int numin, int numout ); */

  /******************************************************************
   * <!
   * Method: isInputIndex()
   *
   * Description:
   * >
   *
   * Checks if <code>which</code> is a valid input port index.
   *
   * @param which the input port index
   *
   * @return <code>true</code> if valid;
   *         <code>false</code> otherwise
   *
   * @exception 
   ******************************************************************/
  public boolean isInputIndex(int which);

  /******************************************************************
   * <!
   * Method: getInputPort()
   *
   * Description:
   * >
   *
   * Get the specified InputPort.
   *
   * @param which the input port index
   *
   * @return the InputPort Object at the specified port number;
   *         <code>null</code> if the index is invalid
   *
   * @exception 
   ******************************************************************/
  public InputPort getInputPort(int which);

  /******************************************************************
   * <!
   * Method: setInputPort()
   *
   * Description:
   * >
   *
   * Set the input port at index to be the given InputPort.
   *
   * @param inp the InputPort object
   * @param which the input port index
   *
   * @return true if the input port was set;
   *         <code>false</code> if the index is invalid
   *
   * @exception 
   ******************************************************************/
  public boolean setInputPort(InputPort inp, int which);
    
  /******************************************************************
   * <!
   * Method: getInputPortType()
   *
   * Description:
   * >
   *
   * Get the type of the InputPort at the specified index.
   *
   * @param which the input port index
   *
   * @return the Class Object for the input port's type;
   *         <br><code>null</code> if the index is invalid</br>
   *
   * @exception 
   *
   ******************************************************************/
  public Class getInputPortType( int which );

  /******************************************************************
   * <!
   * Method: setInputPortType()
   *
   * Description:
   * >
   *
   * Set the InputPort type. If the input port doesn't yet exist,
   * create it.
   *
   * @param stp the fully qualified class name of the input type 
   * @param which the input port index
   *
   * @return <code>true</code> if the input port type was set;
   *         <br><code>false</code> if the index is invalid</br>
   *
   * @exception ClassNotFoundException thrown when stp does not
   *            indicate a valid java class
   ******************************************************************/
  public boolean setInputPortType( String stp, int which ) throws ClassNotFoundException;
  
  /******************************************************************
   * <!
   * Method: setInputPortType()
   *
   * Description:
   * >
   *
   * Set the InputPort type. If the input port doesn't yet exist,
   * create it.
   *
   * @param otp the port should be set to this Object's type 
   * @param which the input port index
   *
   * @return <code>true</code> if the input port type was set;
   *         <br><code>false</code> if the index is invalid</br>
   *
   * @exception 
   ******************************************************************/
  public boolean setInputPortType( Object otp, int which );

  /******************************************************************
   * <!
   * Method: setInputPortType()
   *
   * Description:
   * >
   *
   * Set the InputPort type. If the input port doesn't yet exist,
   * create it.
   *
   * @param tp the Class object for the type the InputPort should
   *           be set to
   * @param which the input port index
   *
   * @return <code>true</code> if the input port type was set;
   *         <br><code>false</code> if the index is invalid</br>
   *
   * @exception 
   ******************************************************************/
  public boolean setInputPortType( Class tp, int which );

  /******************************************************************
   * <!
   * Method: getInputPortName()
   *
   * Description:
   * >
   *
   * Get the type of the InputPort at the specified index.
   *
   * @param which the input port index
   *
   * @return the String for the input port's name;
   *         <br><code>null</code> if the index is invalid</br>
   *
   * @exception 
   *
   ******************************************************************/
  public String getInputPortName( int which );

  /******************************************************************
   * <!
   * Method: setInputPortName()
   *
   * Description:
   * >
   *
   * Set the InputPort name. If the input port doesn't yet exist,
   * create it.
   *
   * @param nm the String name the InputPort should be set to
   * @param which the input port index
   *
   * @return <code>true</code> if the input port name was set;
   *         <br><code>false</code> if the index is invalid</br>
   *
   * @exception 
   ******************************************************************/
  public boolean setInputPortName( String nm, int which );

   /******************************************************************
   * <!
   * Method: getInputMapping()
   *
   * Description:
   * >
   *
   * Get the input mapping for the specified input port.
   *
   * @param which the input port index
   *
   * @return a space seperated string containing the providing agent's
   *         name and output number; <br><code>null</code> if no
   *         provider is mapped to the specified input port or the if
   *         specified input port doesn't exist
   *
   * @exception 
   ******************************************************************/
  public abstract String getInputMapping(int which);

  /******************************************************************
   * <!
   * Method: setInputMapping()
   *
   * Description:
   * >
   *
   * Set the output port that will provide data for the specified
   * input port.
   *
   * @param which the index of the input port
   * @param provider the
   *
   * @return <code>true</code> if the mapping was made;
   *         <br><code>false</code> otherwise. Failure results when
   *         the port * index are invalid or the provider argument is
   *         <code>null</code>
   *
   * @exception 
   ******************************************************************/
  public abstract boolean setInputMapping(int which, Agent provider,
					  int whichOut);

  /******************************************************************
   * <!
   * Method: getInput()
   *
   * Description:
   * >
   *
   * Get the current value of the Agent's specified input.
   *
   * @param which the desired input number
   *
   * @return the current input value;
   *
   * @exception 
   ******************************************************************/
  public Object getInput( int which );

  /******************************************************************
   * <!
   * Method: isOutputIndex()
   *
   * Description:
   * >
   *
   * Checks if <code>which</code> is a valid output port index.
   *
   * @param which the output port index
   *
   * @return <code>true</code> if valid;
   *         <br><code>false</code> otherwise</br>
   *
   * @exception 
   ******************************************************************/
  public boolean isOutputIndex(int which);
  
  /******************************************************************
   * <!
   * Method: getOutputPort()
   *
   * Description:
   * >
   *
   * Gets the specified OutputPort.
   *
   * @param which the output port index
   *
   * @return the OutputPort Object;
   *         <br><code>null</code> if index is invalid</br>
   *
   * @exception 
   ******************************************************************/
  public OutputPort getOutputPort(int which);

  /******************************************************************
   * <!
   * Method: setOutputPort()
   *
   * Description:
   * >
   *
   * Set Agent's specified OutputPort.
   *
   * @param outp the OutputPort object
   * @param which index of the output port to set
   *
   * @return <code>true</code> if the OutputPort was set;
   *         <br><code>false</code> if the specified output port
   *         number is invalid</br>
   *
   * @exception 
   ******************************************************************/
  public boolean setOutputPort(OutputPort outp, int which);

  /******************************************************************
   * <!
   * Method: getOutputPortType()
   *
   * Description:
   * >
   *
   * Get the specified output port's type.
   *
   * @param which index of the output port
   *
   * @return the Class object for the output port's type;
   *         <br><code>null</code> if the output port index is
   *         invalid</br>
   *
   * @exception 
   ******************************************************************/
  public Class getOutputPortType( int which );

  /******************************************************************
   * <!
   * Method: setOutputPortType()
   *
   * Description:
   * >
   *
   * Set the specified output port's type. If the OutputPort doesn't
   * yet exist, create it.
   *
   * @param stp fully qualified name representing of the type
   * @param which index of the output port to set
   *
   * @return <code>true</code> if the port type was set;
   *         <br><code>false</code> if the output port index is
   *         invalid</br>
   *
   * @exception ClassNotFoundException thrown when stp does not
   *            indicate a valid java class
   ******************************************************************/
  public boolean setOutputPortType( String stp, int which ) throws ClassNotFoundException;

  /******************************************************************
   * <!
   * Method: setOutputPortType()
   *
   * Description:
   * >
   *
   * Set the specified output port's type. If the OutputPort doesn't
   * yet exist, create it.
   *
   * @param otp port's type should be set to this Object's type
   * @param which index of the output port to set
   *
   * @return <code>true</code> if the port type was set;
   *         <br><code>false</code> if the output port index is
   *         invalid</br>
   *
   * @exception 
   ******************************************************************/
  public boolean setOutputPortType( Object otp, int which );

  /******************************************************************
   * <!
   * Method: setOutputPortType()
   *
   * Description:
   * >
   *
   * Set the specified output port's type. If the OutputPort doesn't
   * yet exist, create it.
   *
   * @param tp the type to set to
   * @param which index of the output port to set
   *
   * @return <code>true</code> if the port type was set;
   *         <br><code>false</code> if the output port index is
   *         invalid</br>
   *
   * @exception 
   ******************************************************************/
  public boolean setOutputPortType( Class tp, int which );

  /******************************************************************
   * <!
   * Method: getOutputPortName()
   *
   * Description:
   * >
   *
   * Get the specified output port's type.
   *
   * @param which index of the output port
   *
   * @return the String for the output port's name;
   *         <br><code>null</code> if the output port index is
   *         invalid</br>
   *
   * @exception 
   ******************************************************************/
  public String getOutputPortName( int which );

  /******************************************************************
   * <!
   * Method: setOutputPortName()
   *
   * Description:
   * >
   *
   * Set the specified output port's name. If the OutputPort doesn't
   * yet exist, create it.
   *
   * @param nm the name to set to
   * @param which index of the output port to set
   *
   * @return <code>true</code> if the port type was set;
   *         <br><code>false</code> if the output port index is
   *         invalid</br>
   *
   * @exception 
   ******************************************************************/
  public boolean setOutputPortName( String nm, int which );

  /******************************************************************
   * <!
   * Method: getOutput()
   *
   * Description:
   * >
   *
   * Get the current value of the Agent's specified output.
   *
   * @param which the desired output number
   *
   * @return the current output value; <br><code>null</code> if the specified 
   *         output does not exist </br>
   *
   * @exception 
   ******************************************************************/
  public abstract Object getOutput(int which);

  /******************************************************************
   * <!
   * Method: setOutput()
   *
   * Description:
   * >
   *
   * Set the value of the Agent's specified output.
   *
   * @param value the new output value
   * @param which the desired output port number
   *
   * @return <code>true</code> if the value was writen to the port;
   *         <br><code>false</code> if the specified port is invalid
   *         or there is a type mismatch</br>
   *
   * @exception 
   ******************************************************************/
  public boolean setOutput( Object value, int which );

  /******************************************************************
   * <!
   * Method: start()
   *
   * Description:
   * >
   *
   * Called to start execution.
   *
   * @param 
   *
   * @return 
   *
   * @exception 
   ******************************************************************/
  public abstract void start();

  /******************************************************************
   * <!
   * Method: pause()
   *
   * Description:
   * >
   *
   * Called to pause execution. <code>unpause()</code> may be called
   * later to resume execution.
   *
   * @param 
   *
   * @return 
   *
   * @exception 
   ******************************************************************/
  public abstract void pause();

  /******************************************************************
   * <!
   * Method: unpause()
   *
   * Description:
   * >
   *
   * Called to unpause execution. <code>pause()</code> should have
   * been called before. If <code>pause()</code> wasn't called before,
   * then do nothing.
   *
   * @param 
   *
   * @return 
   *
   * @exception 
   ******************************************************************/
  public abstract void unpause();

  /******************************************************************
   * <!
   * Method: stop()
   *
   * Description:
   * >
   *
   * Called to end execution. <b>Note:</b> Do not deallocate resources
   * or clear state in this method. Do so in <code>destroy()</code>.
   * <code>dumpState</code> may be called after this method has been
   * called.
   *
   * @param 
   *
   * @return 
   *
   * @exception 
   ******************************************************************/
  public abstract void stop();

  /******************************************************************
   * <!
   * Method: saveState()
   *
   * Description:
   * >
   *
   * Returns the Agent's internal state. This will be serialized and
   * saved for later restoration. <b>Note:</b> State may not have been
   * explicitly frozen and Agent may not have been stopped prior to
   * this call. However, this method should guarantee that the Agent's
   * saved state is consistent. (Use internal synchronization.)
   *
   * @param 
   *
   * @return an Serializable Object representing the Agent's saved state
   *
   * @exception 
   ******************************************************************/
  public abstract Object saveState();

  /******************************************************************
   * <!
   * Method: isStateFrozen()
   *
   * Description:
   * >
   *
   * Whether the state should currently be frozen. This method,
   * <code>freezeState()</code>, and <code>unfreezeState()</code>
   * should be internally synchronized so that they will not cause
   * inconsistent state data to be frozen.
   *
   * @param 
   *
   * @return an Object representing the Agent's saved state
   *
   * @exception 
   ******************************************************************/
  public boolean isStateFrozen();


  /******************************************************************
   * <!
   * Method: freezeState()
   *
   * Description:
   * >
   *
   * Called to prevent the Agent from changing it's internal state.
   * The Agent should continue to update it's outputs, but internal
   * data structures such as neural net weights or adaptive filter
   * coefficients should not be changed while state is frozen. See
   * <code>isStateFrozen()</code> for notes on synchronization.
   *
   * @see #isStateFrozen
   *
   * @param 
   *
   * @return 
   *
   * @exception 
   ******************************************************************/
  public abstract void freezeState();

  /******************************************************************
   * <!
   * Method: unfreezeState()
   *
   * Description:
   * >
   *
   * Called to undo a state freeze. The Agent is then free to change
   * it's internal state again. See <code>isStateFrozen()</code> for
   * notes on synchronization.
   *
   * @see #isStateFrozen
   *
   * @param 
   *
   * @return 
   *
   * @exception 
   ******************************************************************/
  public abstract void unfreezeState();

  /******************************************************************
   * <!
   * Method: destroy()
   *
   * Description:
   * >
   *
   * Called at end of Agent's run cycle. Agent should release all
   * resources it's holding and free up memory where necessary. If
   * stop hasn't been called before destroy, destroy will call stop
   * before proceeding with deallocation.
   *
   * @param 
   *
   * @return 
   *
   * @exception 
   ******************************************************************/
  public abstract void destroy();

  /******************************************************************
   * <!
   * Method: verify()
   *
   * Description:
   * >
   *
   * Called to check if the Agent's inputs have all been satisfied
   * correctly. That is they have mapped sources and matching
   * input/output types.
   *
   * @param 
   *
   * @return <code>true</code> if all inputs satisfied correctly;
   *         <br><code>false</code></br> otherwise
   *
   * @exception 
   ******************************************************************/
  public abstract boolean verify();

  /******************************************************************
   * <!
   * Method: isBadInit()
   *
   * Description:
   * >
   *
   * Called to check if initialization completed correctly
   *
   * @param 
   *
   * @return <code>true</code> if initialize failed;
   *         <br><code>false</code></br> otherwise
   *
   * @exception 
   ******************************************************************/
  public abstract boolean isBadInit();


  /***************************************************************************
   * <!
   * Method : getStateEditor()
   * Description:
   * Returns the GUI dialog internal state editor. This should return
   * a new instance of the StateEditor. Or null if there is no state
   * editor.
   * >
   * @param parent frame, since the state editor should extend java.awt.Dialog, but nothing forces it too --added jrj 4/27/00
   *
   * @return the gui state editor
   *
   * @exception 
   *
   **************************************************************************/
    public adaptive.modulemanager.StateEditor getStateEditor(java.awt.Frame parent);
  /******************************************************************
   * <!
   * Method: getStateDescription()
   *
   * Description:
   * >
   *
   * Called to get a description of the state object. For use with
   * StateEditors. To use the DefaultStateEditor, you should return
   * an array of Strings where each string is a description
   * of the corresponding element in the State array of Objects or
   * primitives. DefaultStateEditor can edit primitives
   * (int, short, long, double, float, boolean, byte, char) or Objects
   * of type (Integer, Short, Long, Double, Float, Boolean, Byte, Character).
   *
   * @param 
   *
   * @return the state description object.
   *         
   * @exception 
   ******************************************************************/
  public abstract Object getStateDescription();
  
  /******************************************************************
   * <!
   * Method: getDefaultState()
   *
   * Description:
   * >
   *
   * Returns a new object representing the default state for this
   * agent. This object will be editted by the StateEditor, so
   * make sure it is non-static and non-final.
   *
   * @param 
   *
   * @return the default state object
   *
   * @exception 
   ******************************************************************/
  public abstract Object getDefaultState();

  /******************************************************************
   * <!
   * Method: verifyState()
   *
   * Description:
   * >
   *
   * See if the state object is of valid format for this agent.
   * Note, this really should be a static function, but there isn't
   * a way to do static interface methods.
   *
   * @param 
   *
   * @return null if the obj is a valid state object;
   *         an error string representing what was wrong otherwise
   *
   * @exception 
   ******************************************************************/
  public abstract String verifyState(Object obj);

  /**
   * says what modes this module is capable of running it.
   * should return constant.  should return LOOP_ONLY by default.
   * @return a coded int which should be one of the following:
   *           Agent.EVENT_ONLY, 
   *           Agent.LOOP_ONLY, or
   *           Agent.EVENT_OR_LOOP 
   */
  public int getModuleModeCapability();
  int getModuleMode();//returns what mode the Agent is actually running in

  // these constants can be used as bit masks!
  /** return code for getModuleModeCapability for modules that should only
   * be event based
   */
  public static final int EVENT_ONLY = 1;

  /** return code for getModuleModeCapability for modules that should only
   * be loop based
   */
  public static final int LOOP_ONLY = 2;

  /** return code for getModuleModeCapability for modules that can be
   * either event or loop based
   */
  public static final int EVENT_OR_LOOP = 3;

  public static final long DEFAULT_SLEEP_TIME = 250;
}
