package adaptive.agentserver;

import java.net.Socket;
import java.net.ServerSocket;
import java.io.IOException;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.FileNotFoundException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.BufferedOutputStream;
import java.io.BufferedInputStream;
import java.io.InvalidClassException;
import java.io.StreamCorruptedException;



import adaptive.core.net.LocalHost;
import adaptive.core.AgentLoaderException;
import adaptive.core.AgentServerProtocol;



//import comms.core.*;
//import adaptive.core.*;

public class AgentServerClientThread extends Thread {  
  private final static boolean debug = true;
  private final static String clientClosed = "Client closing connection.";
  String clientName;
  Socket socket;
  BufferedInputStream bis;
  ObjectInputStream ois;
  BufferedOutputStream bos;
  ObjectOutputStream oos;
  AgentServer agentServer;
  
  public AgentServerClientThread( Socket socket, AgentServer server ) {
    super();
    this.socket = socket;
    agentServer = server;
  }

  private void cleanupSockets() {
    try {
     if(oos != null) {
        oos.close();
      }
    }
    catch (IOException ioe) {      
    }

    try {
      if(ois != null) {
        ois.close();
      }
    }
    catch (IOException ioe) {  
    }
    
    try {
      if(socket != null) {
        socket.close();
      }
    }
    catch (IOException ioe) {      
    }
    
    oos = null;
    bos = null;
    ois = null;
    bis = null;
    socket = null;
  }
  
  public void run() {
    try {
      /* open streams and prep for handsake */      
      bos = new BufferedOutputStream(socket.getOutputStream());
      oos = new ObjectOutputStream(bos);
      oos.flush();
      
      bis = new BufferedInputStream(socket.getInputStream());
      ois = new ObjectInputStream(bis);

      boolean error = false;
      try {
        /* handshaking */
        String incoming;
        incoming = (String) ois.readObject();
      
        if(!AgentServerProtocol.kHELLOCLIENT.equals(incoming)) {
          try {
            oos.writeObject(AgentServerProtocol.kREJECTCONNECT);
            oos.flush();
          }
          catch(IOException ioe2) {
            if(debug) {
              ioe2.printStackTrace();
            }
          }
          cleanupSockets();
          System.err.println(
            "Error: Rejecting handshake. Client thread dying...");
          return;
        }
        else {
          oos.writeObject(AgentServerProtocol.kHELLOSERVER);
          oos.flush();
        }
      }
      catch(ClassNotFoundException cnfe) {
        error = true;
      }
      catch(InvalidClassException ice) {
        error = true;
      }
      catch(StreamCorruptedException sce) { 
        error = true;       
      }
      catch(IOException ioe3) {
        error = true;        
      }

      if(error == true) {
        cleanupSockets();
        System.err.println(
          "Error: Corrupted handshake. Client thread dying...");
        return;
      } 
      
      /*Begin request handling phase */
      byte request;
      while( true ) {
        try {
          request = (byte) ois.read();
        }
        catch(Exception excep) {
          request = AgentServerProtocol.kBYE;
        }
        switch(AgentServerProtocol.identifyRequest(request)) {
          case AgentServerProtocol.kREQ_PACKAGEROOT:
            //send the OKAY
            //then send the package root over
            requestPackageRoot();
            break;
          case AgentServerProtocol.kREQ_CLASS:
            //if class is found,
            //send the OKAY
            //then send the class over
            //else send NO
            requestClass();
            break;
          case AgentServerProtocol.kREQ_LISTDIR:
            //if valid path,
            //send OKAY
            //send the listing over
            //else send DENIED when path not sub of package root
            //else send NO
            requestListDir();
            break;
          case AgentServerProtocol.kREQ_LISTFILES:
            //if valid path,
            //send OKAY
            //send the listing over
            //else send DENIED when path not sub of package root
            //else send NO
            requestListFiles();
            break;
          case AgentServerProtocol.kREQ_LISTAGENTS:
            //if valid path,
            //send OKAY
            //send the listing
            //else send DENIED when path not sub of package root
            //else send NO
            requestListAgents();
            break;
          case AgentServerProtocol.kREQ_PATHISFILE:
            //if valid path,
            //send OKAY if true
            //else send NO if false
            //else send DENIED when path not sub of package root
            requestPathIsFile();
            break;
          case AgentServerProtocol.kREQ_PATHISDIR:
            //if valid path,
            //send OKAY if true
            //else send NO if false
            //else send DENIED when path not sub of package root
            requestPathIsDir();
            break;
          case AgentServerProtocol.kREQ_PATHEXISTS:
            //if valid path,
            //send OKAY if true
            //else send NO if false
            //else send DENIED when path not sub of package root
            requestPathExists();
            break;
          case AgentServerProtocol.kREQ_GETFILEINSTREAM:
            //NOT IMPLEMENTED
            requestGetFileInStream();
            break;
          case AgentServerProtocol.kREQ_GETFILEOUTSTREAM:
            //NOT IMPLEMENTED
            requestGetFileOutStream();
            break;
          case AgentServerProtocol.kREQ_DELETE:
            //if valid path,
            //send OKAY if delete successful
            //else send NO if delete failed
            //else send DENIED when path not sub of package root
            requestDelete();
            break;
          case AgentServerProtocol.kREQ_MKDIR:
            //if valid path,
            //send OKAY if dir created
            //else send NO if dir not created
            //else send DENIED when path not sub of package root
            requestMkdir();
            break;
          case AgentServerProtocol.kREQ_RENAME:
            //if valid path,
            //send OKAY if renamed
            //else send NO if not renamed
            //else send DENIED when paths not sub of package root
            requestRename();
            break;
          case AgentServerProtocol.kUNKNOWN:
            //send the DENIED message since we have no idea what this
            //was supposed to be
            oos.write(AgentServerProtocol.kDENIED);
            oos.flush();
            break;
          case AgentServerProtocol.kBYE:
            //the other side died so we can exit.
            throw new IOException(clientClosed);
        }
      }
      /* End request handling phase */
    }
    catch(IOException ioe) {
      if( (ioe.getMessage() != null) &&
          (ioe.getMessage().equals(clientClosed))) {
        System.err.println(ioe.getMessage());
      }
      else {
        ioe.printStackTrace();
      }
      cleanupSockets();
      /* the thread will die now */
    }
  }

  void requestRename() throws IOException {
    String src;
    String dest;
    try {
      src  = (String) ois.readObject();
      dest = (String) ois.readObject();
    }
    catch(Exception ex) {
      throw new IOException(ex.getMessage());
    }
    
    boolean retVal = false;
    try {
      retVal = agentServer.rename(src,dest);
    }
    catch(IllegalAccessException iae) {
      oos.write(AgentServerProtocol.kDENIED);
      oos.flush();
    }
    
    if(retVal) {
      oos.write(AgentServerProtocol.kOKAY);
      oos.flush();
    }
    else {
      oos.write(AgentServerProtocol.kNO);
      oos.flush();
    }
  }

  void requestMkdir() throws IOException {
    String dest;
    try {
      dest = (String) ois.readObject();
    }
    catch(Exception ex) {
      throw new IOException(ex.getMessage());
    }
    
    boolean retVal = false;
    try {
      retVal = agentServer.mkdir(dest);
    }
    catch(IllegalAccessException iae) {
      oos.write(AgentServerProtocol.kDENIED);
      oos.flush();
    }
    
    if(retVal) {
      oos.write(AgentServerProtocol.kOKAY);
      oos.flush();
    }
    else {
      oos.write(AgentServerProtocol.kNO);
      oos.flush();
    }    
  }

  void requestDelete() throws IOException {
    String dest;
    try {
      dest = (String) ois.readObject();
    }
    catch(Exception ex) {
      throw new IOException(ex.getMessage());
    }
    
    boolean retVal = false;
    try {
      retVal = agentServer.delete(dest);
    }
    catch(IllegalAccessException iae) {
      oos.write(AgentServerProtocol.kDENIED);
      oos.flush();
    }
    
    if(retVal) {
      oos.write(AgentServerProtocol.kOKAY);
      oos.flush();
    }
    else {
      oos.write(AgentServerProtocol.kNO);
      oos.flush();
    }    
  }

  void requestGetFileOutStream() throws IOException {
    String dest;
    try {
      dest = (String) ois.readObject();
    }
    catch(Exception ex) {
      throw new IOException(ex.getMessage());
    }
    
    int append;
    append = (byte) ois.read();
    
    try {
      throw new IllegalAccessException("Not implemented.");
    }
    catch(IllegalAccessException iae) {
      oos.write(AgentServerProtocol.kDENIED);
      oos.flush();
    }
  }

  void requestGetFileInStream() throws IOException {
    String dest;
    try {
      dest = (String) ois.readObject();
    }
    catch(Exception ex) {
      throw new IOException(ex.getMessage());
    }
    
    
    try {
      throw new IllegalAccessException("Not implemented.");
    }
    catch(IllegalAccessException iae) {
      oos.write(AgentServerProtocol.kDENIED);
      oos.flush();
    }
  }
  
  void requestPathExists() throws IOException {
    String dest;
    try {
      dest = (String) ois.readObject();
    }
    catch(Exception ex) {
      throw new IOException(ex.getMessage());
    }   

    boolean retVal = false;    
    try {
      retVal = agentServer.pathExists(dest);
    }
    catch(IllegalAccessException iae) {
      oos.write(AgentServerProtocol.kDENIED);
      oos.flush();
      return;
    }
    
    if(retVal) {
      oos.write(AgentServerProtocol.kOKAY);
      oos.flush();
    }
    else {
      oos.write(AgentServerProtocol.kNO);
      oos.flush();
    }
  }

  void requestPathIsDir() throws IOException {
    String dest;
    try {
      dest = (String) ois.readObject();
    }
    catch(Exception ex) {
      throw new IOException(ex.getMessage());
    }
    
    boolean retVal;    
    try {
      retVal = agentServer.pathIsDir(dest);
    }
    catch(IllegalAccessException iae) {
      oos.write(AgentServerProtocol.kDENIED);
      oos.flush();
      return;
    }
    
    if(retVal) {
      oos.write(AgentServerProtocol.kOKAY);
      oos.flush();
    }
    else {
      oos.write(AgentServerProtocol.kNO);
      oos.flush();
    }
  }

  void requestPathIsFile() throws IOException {
    String dest;
    try {
      dest = (String) ois.readObject();
    }
    catch(Exception ex) {
      throw new IOException(ex.getMessage());
    }

    boolean retVal;    
    try {
      retVal = agentServer.pathIsFile(dest);
    }
    catch(IllegalAccessException iae) {
      oos.write(AgentServerProtocol.kDENIED);
      oos.flush();
      return;
    }
    
    if(retVal) {
      oos.write(AgentServerProtocol.kOKAY);
      oos.flush();
    }
    else {
      oos.write(AgentServerProtocol.kNO);
      oos.flush();
    }  
  }

  void requestListAgents() throws IOException {
    String dest;
    try {
      dest = (String) ois.readObject();
    }
    catch(Exception ex) {
      throw new IOException(ex.getMessage());
    }
    
    String[] retVal;
    try {
      retVal = agentServer.listAgentsIn(dest);
    }
    catch(IllegalAccessException iae) {
      oos.write(AgentServerProtocol.kDENIED);
      oos.flush();
      return;
    }
    catch(FileNotFoundException fnfe) {
      oos.write(AgentServerProtocol.kNO);
      oos.flush();
      return;
    }

    oos.write(AgentServerProtocol.kOKAY);
    oos.writeObject(retVal);
    oos.flush();    
  }

  void requestListFiles() throws IOException {
    String dest;
    try {
      dest = (String) ois.readObject();
    }
    catch(Exception ex) {
      throw new IOException(ex.getMessage());
    }    
    
    String[] retVal;
    try {
      retVal = agentServer.listFilesIn(dest);
    }
    catch(IllegalAccessException iae) {
      oos.write(AgentServerProtocol.kDENIED);
      oos.flush();
      return;
    }
    catch(FileNotFoundException fnfe) {
      oos.write(AgentServerProtocol.kNO);
      oos.flush();
      return;
    }
    
    oos.write(AgentServerProtocol.kOKAY);
    oos.writeObject(retVal);
    oos.flush();
  }

  void requestListDir() throws IOException {
    String dest;
    try {
      dest = (String) ois.readObject();
    }
    catch(Exception ex) {
      throw new IOException(ex.getMessage());
    }
    
    String[] retVal;
    try {
      retVal = agentServer.listDirectoriesIn(dest);
    }   
    catch(IllegalAccessException iae) {
      oos.write(AgentServerProtocol.kDENIED);
      oos.flush();
      return;
    }
    catch(FileNotFoundException fnfe) {
      oos.write(AgentServerProtocol.kNO);
      oos.flush();      
      return;
    }

    oos.write(AgentServerProtocol.kOKAY);
    oos.writeObject(retVal);
    oos.flush();
  }
  
  void requestClass() throws IOException {
    String dest;
    try {
      dest = (String) ois.readObject();
    }
    catch(Exception ex) {
      throw new IOException(ex.getMessage());
    }
    
    byte[] retVal;
    if(debug) {
      System.err.println("Retrieving class:"+dest);
    }
    
    try {
      retVal = agentServer.getClassBytes(dest);
    }
    catch(Exception except) {
      /* Class not found or some other error retrieving the byte
         definition for the class */
      retVal = null;
    }
    
    if(retVal != null) {
      oos.write(AgentServerProtocol.kOKAY);
      oos.writeObject(retVal);
      oos.flush();
    }
    else {
      if(debug) {
        System.err.println("ClassNotFound");
      }
      oos.write(AgentServerProtocol.kNO);
      oos.flush();
    }
  }
  
  void requestPackageRoot() throws IOException {
    String retVal;
    retVal = agentServer.getPackageRoot();
    oos.write(AgentServerProtocol.kOKAY);
    oos.writeObject(retVal);
    oos.flush();
  }

  protected void finalize() {
    cleanupSockets();
  }
}
