(* moc-applet: control the MOC audio player
   Copyright (C) 2006  Eric C. Cooper <ecc@cmu.edu>
   Released under the GNU General Public License *)

type state = Play | Pause | Stop

let player_state () =
  try
    match Info.lookup "state" with
    | "PLAY" -> Play
    | "PAUSE" -> Pause
    | _ -> Stop
  with
    Not_found -> Stop

let _ = GMain.init ()

let tips = GData.tooltips ()

let update_title applet =
  let text =
    if not (Moc.connected ()) then "[Not connected]"
    else if player_state () = Stop then "[Stopped]"
    else Glib.Convert.locale_to_utf8 (Info.lookup "title")
  in
  tips#set_tip ~text applet#coerce

let button_press ev =
  if GdkEvent.Button.button ev <> 1 then GtkSignal.stop_emit ();
  false

let open_moc _ = Moc.open_player ()

let button_ids =
  [ `MEDIA_PLAY; `MEDIA_PAUSE; `MEDIA_STOP; `MEDIA_PREVIOUS; `MEDIA_NEXT ]

let images =
  List.map (fun stock -> stock, GMisc.image ~stock ()) button_ids

let play_image = (List.assoc `MEDIA_PLAY images)#coerce
let pause_image = (List.assoc `MEDIA_PAUSE images)#coerce

let toggle_play () =
  match player_state () with
  | Play -> Moc.pause ()
  | Pause -> Moc.unpause ()
  | Stop -> Moc.play ()

let actions =
  [ `MEDIA_PREVIOUS, Moc.prev;
    `MEDIA_PLAY, toggle_play;
    `MEDIA_STOP, Moc.stop;
    `MEDIA_NEXT, Moc.next ]

let make_button (id, callback) =
  let b = GButton.button () in
  b#add (List.assoc id images)#coerce;
  b#connect#clicked ~callback;
  b#event#connect#button_press ~callback: button_press;
  id, b

let buttons = List.map make_button actions

let play_button = List.assoc `MEDIA_PLAY buttons

let update_controls applet =
  let image =
    match player_state () with
    | Play -> pause_image
    | Pause | Stop -> play_image
  in
  play_button#remove (List.hd play_button#children);
  play_button#add image

let rec update applet () =
  if Moc.connected () then
    begin
      Info.update ();
      update_title applet;
      update_controls applet
    end
  else
    begin
      Moc.start_server ();
      Moc.connect (update applet)
    end

let make_widgets ?packing () =
  let box = GPack.hbox ?packing () in
  List.iter (fun (_, b) -> box#pack b#coerce) buttons;
  box

let xml =
  "<popup name=\"button3\">
     <menuitem _label=\"_Open MOC\"
	       name=\"open\" verb=\"open\"
	       pixtype=\"stock\" pixname=\"gnome-stock-volume\" />
   </popup>"

let initialize applet =
  make_widgets ~packing: applet#add ();
  applet#setup_menu ~xml ["open", open_moc ];
  update applet ();
  applet#misc#show ()

let applet_iid = "OAFIID:MOC_Applet"
let factory_iid = "OAFIID:MOC_Applet_Factory"

let factory applet ~iid =
  if iid = applet_iid then
    try initialize applet; true
    with _ -> false
  else
    false

let _ = Panel.factory_main ~iid: factory_iid factory
