(* moc-applet: control the MOC audio player
   Copyright (C) 2006  Eric C. Cooper <ecc@cmu.edu>
   Released under the GNU General Public License *)

let debug = false
let trace = prerr_endline

let moc_dir = Sys.getenv "HOME" ^ "/.moc/"
let socket_file = moc_dir ^ "socket2"
let pid_file = moc_dir ^ "pid"

let start_server () =
  ignore (Unix.system "/usr/bin/mocp --server >& /dev/null")

let server_running () = Sys.file_exists pid_file

open Unix

let sleep secs = ignore (select [] [] [] secs)

let server_conn = ref None

let connected () = !server_conn <> None

let from_server () =
  match !server_conn with
  | Some (chan, _) -> chan
  | None -> failwith "from_server"

let to_server () =
  match !server_conn with
  | Some (_, chan) -> chan
  | None -> failwith "to_server"

let (<<) = (lsl)
let (>>) = (lsr)
let (||) = (lor)
let (&&) = (land)
let (~~) = lnot

(* Some input operations on Glib.Io.channel objects *)

let try_read chan ~buf ~pos ~len =
  try Some (Glib.Io.read chan ~buf ~pos ~len)
  with Glib.GError "g_io_channel_read: G_IO_ERROR_AGAIN" -> None

let really_input chan ~buf ~pos ~len =
  let rec loop pos len =
    if len > 0 then
      match try_read chan ~buf ~pos ~len with
      | Some n ->
	  if n > 0 then loop (pos + n) (len - n)
	  else failwith "really_input"
      | None ->
	  sleep 0.25;
	  loop pos len
  in
  loop pos len

let char_buf = String.create 1

let input_byte chan =
  really_input chan ~buf: char_buf ~pos: 0 ~len: 1;
  int_of_char char_buf.[0]

(* WARNING: Little-Endian only *)

let read_int () =
  let chan = from_server () in
  let b0 = input_byte chan in
  let b1 = input_byte chan in
  let b2 = input_byte chan in
  let b3 = input_byte chan in
  (b3 << 24) || (b2 << 16) || (b1 << 8) || (b0 << 0)

let read_string () =
  let n = read_int () in
  if n > 0 then
    let str = String.create n in
    really_input (from_server ()) ~buf: str ~pos: 0 ~len: n;
    str
  else
    ""

let skip n =
  let chan = from_server () in
  for i = 1 to n do input_byte chan done

let skip_int () = skip 4
let skip_time () = skip 4
let skip_string () = skip (read_int ())

let skip_tags () =
  skip_string ();
  skip_string ();
  skip_string ();
  skip_int ()

let skip_plist () =
  if read_string () <> "" then
    begin
      skip_string ();
      skip_tags ();
      skip_time ()
    end

let write_int n =
  let chan = to_server () in
  (* output_byte performs mod 256 operation *)
  output_byte chan (n >>  0);
  output_byte chan (n >>  8);
  output_byte chan (n >> 16);
  output_byte chan (n >> 24)

let flush_int n =
  write_int n;
  flush (to_server ())

let flush_string str =
  write_int (String.length str);
  let chan = to_server () in
  output_string chan str;
  flush chan

let play () = if connected () then write_int 0x00; flush_string ""
let pause () = if connected () then flush_int 0x05
let unpause () = if connected () then flush_int 0x06
let stop () = if connected () then flush_int 0x04
let prev () = if connected () then flush_int 0x20
let next () = if connected () then flush_int 0x10

let ping () =
  if debug then trace "ping";
  flush_int 0x19 (* CMD_PING *);
  if read_int () <> 0x0B (* EV_PONG *) then failwith "ping"
  else if debug then trace "pong"

let handle_event hook =
  match read_int () with
  | 0x01 (* EV_STATE *) ->
      if debug then trace "state change";
      hook ()
  | 0x0A (* EV_EXIT *) ->
      if debug then trace "server exit";
      raise End_of_file
  | 0x0F (* EV_STATUS_MSG *) | 0x5F (* EV_PLIST_DEL *) ->
      skip_string ()
  | 0x5E (* EV_PLIST_ADD *) ->
      skip_plist ()
  | _ -> ()

let open_player () = ignore (system "x-terminal-emulator -e mocp")

let watch = ref None

let disconnect hook =
  if debug then trace "disconnecting";
  match !server_conn with
  | Some (_, chan) ->
      (match !watch with
       | Some id -> Glib.Io.remove id; watch := None
       | None -> ());
      close (descr_of_out_channel chan);
      server_conn := None;
      if debug then trace "disconnected -- calling hook";
      hook ()
  | None ->
      failwith "already disconnected"

let handle_input hook conditions =
  try
    if List.mem `HUP conditions then raise End_of_file;
    handle_event hook;
    true
  with e ->
    disconnect hook;
    if e <> End_of_file then prerr_endline (Printexc.to_string e);
    false

let connect hook =
  assert (!server_conn = None);
  if debug then trace "connecting";
  try
    let fd = socket PF_UNIX SOCK_STREAM 0 in
    set_nonblock fd;
    let (in_chan, _) as conn =
      Glib.Io.channel_of_descr fd, out_channel_of_descr fd
    in
    let callback = handle_input hook in
    server_conn := Some conn;
    try
      Unix.connect fd (ADDR_UNIX socket_file);
      ping ();
      watch := Some (Glib.Io.add_watch ~cond: [`IN; `HUP] ~callback in_chan);
      flush_int 0x1D (* CMD_SEND_EVENTS *);
      if debug then trace "connected -- calling hook";
      hook ()
    with e ->
      disconnect hook;
      raise e
  with e ->
    prerr_endline (Printexc.to_string e)
