/*
 * gkrellm-moc: MOC plugin
 * Copyright (C) 2006  Eric Cooper <ecc@cmu.edu>
 * Released under the GNU General Public License
 */

#include <gkrellm2/gkrellm.h>
#include "moc.h"
#include "moc/protocol.h"

static char *style_name = "moc";
static gint style_id;
static GkrellmMonitor *monitor;
static GkrellmStyle *style;
static GkrellmTextstyle *textstyle;

static GkrellmPanel *title_panel;
static GtkTooltips *tooltip;
static GkrellmDecal *title_decal;
static int title_width;
static char *title_string = NULL;
static gint title_delta, title_offset, title_pause;

static GkrellmPanel *control_panel;

#define TITLE_PAUSE_TICKS	5
#define TITLE_SCROLL_UNIT	10

static void
pause_title_start(void)
{
    title_pause = TITLE_PAUSE_TICKS;
    title_offset = 0;
}

static void
pause_title_end(void)
{
    title_pause = TITLE_PAUSE_TICKS;
    title_offset = title_delta;
}

static void
update_title(void)
{
    char *str = song_info[INFO_SONGTITLE];
    if (!streq(str, title_string)) {  // title has changed
	title_string = str;
	if (str != NULL) {
	    gint w = gkrellm_gdk_string_width(textstyle->font, str);
	    title_delta = title_width - w;
	    gkrellm_decal_scroll_text_set_text(title_panel, title_decal, str);
	    gtk_tooltips_set_tip(tooltip, title_panel->drawing_area,
				 song_info[INFO_TITLE], "");
	    gtk_tooltips_enable(tooltip);
	} else {
	    title_delta = title_width;
	    gkrellm_decal_scroll_text_set_text(title_panel, title_decal, "");
	    gtk_tooltips_disable(tooltip);
	}
	pause_title_start();
    }
    if (title_delta < 0) {
	if (title_pause == 0) {
	    title_offset -= TITLE_SCROLL_UNIT;
	    if (title_offset <= title_delta)
		pause_title_end();
	} else {
	    title_pause -= 1;
	    if (title_pause == 0 && title_offset <= title_delta)
		pause_title_start();
	}
    }
    gkrellm_decal_text_set_offset(title_decal, title_offset, 0);
}

static gboolean
button_event(GtkWidget *widget, GdkEventButton *ev)
{
    switch (ev->button) {
      case 2:
      case 3:
	open_player();
	break;
    }
    return FALSE;
}

static gint
expose_event(GtkWidget *widget, GdkEventExpose *ev, GkrellmPanel *p)
{
    gdk_draw_pixmap(widget->window,
		    widget->style->fg_gc[GTK_WIDGET_STATE(widget)],
		    p->pixmap,
		    ev->area.x, ev->area.y, ev->area.x, ev->area.y,
		    ev->area.width, ev->area.height);
    return FALSE;
}

struct button {
    char *stock_id;
    void (*callback)(void);
};

static GkrellmDecalbutton *
create_decal_button(GtkWidget *widget, struct button *b, gint x, gint y)
{
    char *id = b->stock_id;
    GdkPixbuf *pixbuf = gtk_widget_render_icon(widget, id, -1, NULL);
    GdkPixmap *pixmap = NULL;
    GdkBitmap *mask = NULL;
    gkrellm_scale_pixbuf_to_pixmap(pixbuf, &pixmap, &mask, 0, 0);
    gint w = 0;
    gdk_drawable_get_size(pixmap, &w, NULL);
    GkrellmDecal *d = gkrellm_create_decal_pixmap(control_panel,
						  pixmap, mask, 1,
						  style, x - w/2, y);
    return gkrellm_make_decal_button(control_panel, d, b->callback, NULL,
				     0, 0);
}

struct button button_info[] = {
    { GTK_STOCK_MEDIA_PREVIOUS, moc_prev },
    { GTK_STOCK_MEDIA_PLAY, moc_play },
    { GTK_STOCK_MEDIA_STOP, moc_stop },
    { GTK_STOCK_MEDIA_NEXT, moc_next }
};

#define NBUTTONS	(sizeof(button_info) / sizeof(struct button))

struct button pause_info = { GTK_STOCK_MEDIA_PAUSE, moc_play };

GkrellmDecalbutton *play_button, *pause_button;

static void
make_buttons(GtkWidget *widget)
{
    gint width = gkrellm_chart_width();
    gint delta = width / (NBUTTONS + 1);
    gint x, y;
    gkrellm_panel_label_get_position(style, &x, &y);
    int i;
    struct button *bp;
    for (i = 0, bp = button_info, x = delta;
	 i < NBUTTONS;
	 i++, bp++, x += delta) {
	GkrellmDecalbutton *b = create_decal_button(widget, bp, x, y);
	if (streq(bp->stock_id, GTK_STOCK_MEDIA_PLAY)) {
	    play_button = b;
	    pause_button = create_decal_button(widget, &pause_info, x, y);
	    gkrellm_hide_button(pause_button);
	}
    }
}

static void
update_controls(void)
{
    int s = player_state();
    switch (s) {
      case STATE_PLAY:
	gkrellm_hide_button(play_button);
	gkrellm_show_button(pause_button);
	break;
      case STATE_PAUSE:
      case STATE_STOP:
	gkrellm_hide_button(pause_button);
	gkrellm_show_button(play_button);
	break;
    }
}

static void
update_plugin(void)
{
    if (!server_connected()) {
	GkrellmTicks *gt = gkrellm_ticks();
	if (gt->second_tick)
	    server_init();
    }
    update_title();
    update_controls();
    gkrellm_draw_panel_layers(title_panel);
    gkrellm_draw_panel_layers(control_panel);
}

static void
create_plugin(GtkWidget *widget, gint first_create)
{
    if (first_create) {
	title_panel = gkrellm_panel_new0();
	control_panel = gkrellm_panel_new0();
	tooltip = gtk_tooltips_new();
    }
    textstyle = gkrellm_panel_textstyle(style_id);
    style = gkrellm_panel_style(style_id);

    title_decal = gkrellm_create_decal_text(title_panel, "Ay",
					    textstyle, style,
					    -1, -1, -1);
    gkrellm_decal_get_size(title_decal, &title_width, NULL);
    gkrellm_panel_configure(title_panel, NULL, style);
    gkrellm_panel_create(widget, monitor, title_panel);

    make_buttons(widget);
    gkrellm_panel_configure(control_panel, NULL, style);
    gkrellm_panel_create(widget, monitor, control_panel);

    if (first_create) {
	g_signal_connect(title_panel->drawing_area, "expose-event",
			 G_CALLBACK(expose_event), title_panel);
	g_signal_connect(title_panel->drawing_area, "button-press-event",
			 G_CALLBACK(button_event), NULL);
	g_signal_connect(control_panel->drawing_area, "expose-event",
			 G_CALLBACK(expose_event), control_panel);
    }
}

static GkrellmMonitor plugin_mon = {
    .name = "MOC",
    .id = 0,
    .create_monitor = create_plugin,
    .update_monitor = update_plugin,
    .insert_before_id = MON_PLUGIN,
};

GkrellmMonitor *gkrellm_init_plugin(void) {
    style_id = gkrellm_add_meter_style(&plugin_mon, style_name);
    monitor = &plugin_mon;
    return monitor;
}
