/*
 * gkrellm-moc: MOC plugin
 * Copyright (C) 2006  Eric Cooper <ecc@cmu.edu>
 * Released under the GNU General Public License
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <libgen.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <glib.h>

#include "moc.h"
#include "moc/protocol.h"
#include "moc/playlist.h"

/*
 * See also moc/protocol.c and moc/interface.c
 */

static gboolean connected = FALSE;
static int server_fd;
static gboolean updating = FALSE;

gboolean
server_connected(void)
{
    return connected;
}

/*
 * These must agree with the enumeration in moc.h
 */
static char *info_tag[] = {
    "State: ",
    "File: ",
    "Title: ",
    "Artist: ",
    "SongTitle: ",
    "Album: ",
    "TotalTime: ",
    "CurrentTime: ",
    "TimeLeft: ",
    "TotalSec: ",
    "CurrentSec: ",
    "Bitrate: ",
    "Rate: ",
};

#define NTAGS	(sizeof(info_tag) / sizeof(char *))

char *song_info[NTAGS];

static void
reset_info(void)
{
    int i;
    for (i = 0; i < NTAGS; i++) {
	g_free(song_info[i]);
	song_info[i] = NULL;
    }
}

static int state = STATE_STOP;

int
player_state(void)
{
    return state;
}

void
open_player(void)
{
    g_spawn_command_line_async("x-terminal-emulator -e mocp", NULL);
}

static int
read_int(void)
{
    int result = -1;
    read(server_fd, &result, sizeof(int));
    return result;
}

static void
write_int(int n)
{
    write(server_fd, &n, sizeof(int));
}

static time_t
read_time(void)
{
    time_t result = 0;
    read(server_fd, &result, sizeof(time_t));
    return result;
}

static char *
read_string(void)
{
    int len = read_int();
    if (len <= 0 || len > 500)
	return NULL;
    char *str = g_malloc(len + 1);
    int i = 0;
    while (i < len) {
	int n = read(server_fd, &str[i], len - i);
	if (n <= 0) {
	    g_free(str);
	    return NULL;
	}
	i += n;
    }
    str[len] = '\0';
    return str;
}

static void
write_string(char *str)
{
    int len = strlen(str);
    write_int(len);
    write(server_fd, str, len);
}

static char *
read_nonempty_string(void)
{
    char *str = read_string();
    if (str == NULL || str[0])
	return str;
    g_free(str);
    return NULL;
}

static struct file_tags *
read_tags(void)
{
    struct file_tags *tags = g_new0(struct file_tags, 1);
    tags->title = read_nonempty_string();
    tags->artist = read_nonempty_string();
    tags->album = read_nonempty_string();
    tags->track = read_int();
    return tags;
}

static void
free_tags(struct file_tags *tags)
{
    g_free(tags->title);
    g_free(tags->artist);
    g_free(tags->album);
    g_free(tags);
}

static struct plist_item *
read_plist_item(void)
{
    struct plist_item *item = g_new0(struct plist_item, 1);
    item->file = read_string();
    if (item->file[0]) {
	item->title_tags = read_string();
	if (!item->title_tags[0]) {
	    g_free(item->title_tags);
	    item->title_tags = NULL;
	}
	item->tags = read_tags();
	item->mtime = read_time();
    }
    return item;
}

static void
free_plist_item(struct plist_item *item)
{
    g_free(item->file);
    g_free(item->title_tags);
    free_tags(item->tags);
    g_free(item);
}

static gboolean
ping(void)
{
    write_int(CMD_PING);
    return read_int() == EV_PONG;
}

static gboolean
prefix(char *str, char *prefix, char **rest)
{
    int i = 0;
    while (prefix[i] && prefix[i] == str[i])
	i += 1;
    if (prefix[i])
	return FALSE;
    *rest = &str[i];
    return TRUE;
}

static void
extract_info(char *line)
{
    static int last_index = 0;
    char *value = NULL;
    int i = last_index;
    for (;;) {
	if (prefix(line, info_tag[i], &value))
	    break;
	i = (i + 1) % NTAGS;
	if (i == last_index) {
	    g_free(line);
	    return;
	}
    }
    last_index = i;
    song_info[i] = g_strdup(value);
    g_free(line);
    if (i == INFO_STATE) {
	if (streq(value, "PLAY"))
	    state = STATE_PLAY;
	else if (streq(value, "PAUSE"))
	    state = STATE_PAUSE;
	else
	    state = STATE_STOP;
    }
}

static gboolean
handle_info(GIOChannel *chan, GIOCondition condition, gpointer data)
{
    if (condition & G_IO_IN) {
	for (;;) {
	    char *line = NULL;
	    gsize eol = 0;
	    switch (g_io_channel_read_line(chan, &line, NULL, &eol, NULL)) {
	      case G_IO_STATUS_NORMAL:
		line[eol] = '\0';  /* chomp */
		extract_info(line);
		continue;
	      case G_IO_STATUS_AGAIN:
		break;
	      case G_IO_STATUS_EOF:
	      case G_IO_STATUS_ERROR:
		condition |= G_IO_HUP;
		break;
	    }
	    break;
	}
    }
    if (condition & G_IO_HUP) {
	updating = FALSE;
	return FALSE;
    } else {
	return TRUE;
    }
}

static void
initiate_update(void)
{
    if (updating)
	return;
    reset_info();
    static char *info_cmd_argv[] = { "/usr/bin/mocp", "--info", NULL };
    int fd = -1;
    if (!g_spawn_async_with_pipes(NULL, info_cmd_argv, NULL, 0,
				  NULL, NULL, NULL,
				  NULL, &fd, NULL, NULL))
	return;
    updating = TRUE;
    fcntl(fd, F_SETFL, O_NONBLOCK);
    GIOChannel *chan = g_io_channel_unix_new(fd);
    g_io_channel_set_encoding(chan, "ISO-8859-1", NULL);
    g_io_channel_set_close_on_unref(chan, TRUE);
    g_io_add_watch(chan, G_IO_IN | G_IO_HUP, handle_info, NULL);
}

static int
handle_event()
{
    int ev = read_int();
    switch (ev) {
      case EV_STATE:
	initiate_update();
	break;
      case EV_STATUS_MSG:
      case EV_PLIST_DEL:
	g_free(read_string());
	break;
      case EV_PLIST_ADD:
	free_plist_item(read_plist_item());
	break;
      default:
	break;
    }
    return ev;
}

static void
close_connection(GIOChannel *chan)
{
    g_io_channel_unref(chan);
    connected = FALSE;
    reset_info();
}

static gboolean
handle_input(GIOChannel *chan, GIOCondition condition, gpointer data)
{
    if (condition & G_IO_IN) {
	if (handle_event() == EV_EXIT)
	    condition |= G_IO_HUP;
    }
    if (condition & G_IO_HUP) {
	close_connection(chan);
	return FALSE;
    } else {
	return TRUE;
    }
}

void
moc_play(void)
{
    switch (state) {
      case STATE_PLAY:
	write_int(CMD_PAUSE);
	break;
      case STATE_PAUSE:
	write_int(CMD_UNPAUSE);
	break;
      case STATE_STOP:
	write_int(CMD_PLAY);
	write_string("");
	break;
    }
}

void
moc_stop(void)
{
    write_int(CMD_STOP);
}

void
moc_prev(void)
{
    write_int(CMD_PREV);
}

void
moc_next(void)
{
    write_int(CMD_NEXT);
}

void
server_init(void)
{
    server_fd = socket(PF_LOCAL, SOCK_STREAM, 0);
    if (server_fd == -1)
	return;
    struct sockaddr_un sock = { .sun_family = AF_LOCAL };
    snprintf(sock.sun_path, sizeof(sock.sun_path), "%s/.moc/socket2",
	     getenv("HOME"));
    if (connect(server_fd, (struct sockaddr *)&sock, SUN_LEN(&sock)) == -1) {
	close(server_fd);
	return;
    }
    fcntl(server_fd, F_SETFL, O_NONBLOCK);
    if (!ping()) {
	close(server_fd);
	return;
    }
    connected = TRUE;
    updating = FALSE;
    GIOChannel *chan = g_io_channel_unix_new(server_fd);
    g_io_channel_set_close_on_unref(chan, TRUE);
    g_io_add_watch(chan, G_IO_IN | G_IO_HUP, handle_input, NULL);
    write_int(CMD_SEND_EVENTS);
    initiate_update();
}
