(* BabelBuster by Jonathan Baccash *)
structure BabelBuster :> CTT =
struct
  fun errState () = Error.mkErrState TextIO.stdErr

  (* CKit wrappers. *)
  structure CK = ParseToAst
  fun cToAst x = (Tid.reset 0; Pid.reset 0; Aid.reset 0; CK.fileToAst x)
  fun astToC strm ({ast, tidtab, ...} : CK.astBundle) =
        (PPAst.printConst := true;
         PPLib.ppToStrm (PPAst.ppAst () tidtab) strm ast)

  (* Wrappers for my functions. *)
  val eToAst = EnglishToAst.englishToAst
  fun astToE strm ast = AstToEnglish.astBundleToEnglish (strm, ast)

  fun cte' (f, strm) = (astToE strm (cToAst f); TextIO.flushOut strm)
  fun etc' (f, strm) = (astToC strm (eToAst f); TextIO.flushOut strm)
  fun ctc' (f, strm) = (astToC strm (cToAst f); TextIO.flushOut strm)
  fun ete' (f, strm) = (astToE strm (eToAst f); TextIO.flushOut strm)

  fun cte f = cte' (f, TextIO.stdOut)
  fun etc f = etc' (f, TextIO.stdOut)
  fun ctc f = ctc' (f, TextIO.stdOut)
  fun ete f = ete' (f, TextIO.stdOut)

  fun cte2 (f1, f2) = let val strm = TextIO.openOut f2
                      in (cte' (f1, strm); TextIO.closeOut strm) end
  fun etc2 (f1, f2) = let val strm = TextIO.openOut f2
                      in (etc' (f1, strm); TextIO.closeOut strm) end
  fun ctc2 (f1, f2) = let val strm = TextIO.openOut f2
                      in (ctc' (f1, strm); TextIO.closeOut strm) end
  fun ete2 (f1, f2) = let val strm = TextIO.openOut f2
                      in (ete' (f1, strm); TextIO.closeOut strm) end

  val success = OS.Process.success
  val failure = OS.Process.failure

  (* Print English translation of C file, for export. *)
  fun e2c' (s, nil) = (print (s ^ " usage: e2c source.e [out.c]\n"); failure)
    | e2c' (s, x::nil) = (etc x; success)
    | e2c' (s, x1::x2::nil) = (etc2 (x1, x2); success)
    | e2c' (s, x::xs) = (print (s ^ " usage: e2c source.e [out.c]\n"); failure)

  (* Print C translation of English file, for export. *)
  fun c2e' (s, nil) = (print (s ^ " usage: c2e source.c [out.e]\n"); failure)
    | c2e' (s, x::nil) = (cte x; success)
    | c2e' (s, x1::x2::nil) = (cte2 (x1, x2); success)
    | c2e' (s, x::xs) = (print (s ^ " usage: c2e source.c [out.e]\n"); failure)

  (* Pretty print C file, for export. *)
  fun c2c' (s, nil) = (print (s ^ " usage: c2c source.c [out.e]\n"); failure)
    | c2c' (s, x::nil) = (ctc x; success)
    | c2c' (s, x1::x2::nil) = (ctc2 (x1, x2); success)
    | c2c' (s, x::xs) = (print (s ^ " usage: c2c source.c [out.e]\n"); failure)

  fun afterSentinel nil = nil
    | afterSentinel (x::xs) = if x="__sentinel__" then xs
                              else afterSentinel xs

  fun e2c (s, L) = e2c' (s, afterSentinel L)
  fun c2e (s, L) = c2e' (s, afterSentinel L)
  fun c2c (s, L) = c2c' (s, afterSentinel L)

end
