/**
***************************************************************************
* @file X11Interface.hh
*
* Header file declaring X11Interface class.
*
* Copyright (C) 2000-2006 David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
***************************************************************************
**/

#ifndef _XCOMPLETE_X11INTERFACE_H_
#define _XCOMPLETE_X11INTERFACE_H_

#include <string>
#include <X11/X.h>
#include <X11/Xlib.h>
#include <dlrCommon/exception.h>
#include "ModifierTag.hh"


namespace xComplete {

  // Forward declaration.
  class KeyDescription;

  // Forward declaration.
  class WindowDescription;
  

  /**
   ** The X11Interface class connects to the X server and takes care
   ** of monitoring & sending keystrokes.  If we were to port
   ** xComplete2 to run under different windowing systems, we'd do it
   ** by making alternate classes which have this interface, and then
   ** making CompleterApp subclasses similar to CompleterAppX11 which
   ** use the new interface classes.  See CompleterAppX11.hh for more
   ** thoughts on this.
   **/
  class X11Interface {
  public:

    /* ============== Public types ============== */

    /**
     ** This public type is a small, inexpensive scalar for uniquely
     ** identifying keys on the keyboard.  We simply borrow the
     ** corresponding X11 typedef.
     **/
    typedef KeyCode KeyID;


    /**
     ** This public type is a small, inexpensive scalar for uniquely
     ** identifying windows on the display.  We simply borrow the
     ** corresponding X11 typedef.
     **/
    typedef Window WindowID;

    
    /**
     ** This public type is used to represent whether/which modifier
     ** keys have been pressed at the time of the keystroke.  Modifier
     ** keys are keys like Shift, Ctrl, etc.  This is an opaque type,
     ** in that the user should not try to directly recover modifier
     ** information from it.  Instead use the checkModifierState() and
     ** getModifierState() functions to translate to a more portable
     ** state representation.
     **/
    typedef long int ModifierState;

    
    /* ============== Public member functions ============== */

    /** 
     * The constructor creates an X11Interface instance and connects
     * to the X server.
     * 
     * @param displayName This argument specifies which X server to
     * use.  By default we connect to display ":0".  Other values,
     * such as ":1" are currently untested, but might work.
     */
    X11Interface(std::string displayName="");


    /** 
     * The destructor destroys the class instance and cleans up any
     * system resources.
     */
    ~X11Interface();


    /** 
     * This member function checks to see if a keystroke is available,
     * copies the next keystroke (if one is available), and returns
     * immediately.
     * 
     * @param keyDescription If a keystroke is available, this
     * reference argument will be populated with the details of the
     * keystroke.  If a keystroke is not available, this argument is
     * not used.
     * 
     * @param windowDescription If a keystroke is available, this
     * reference argument will be populated with the details of the
     * window which had input focus when the keystroke was issued.  If
     * a keystroke is not available, this argument is not used.
     * 
     * @return The return value is a bool indicating whether or not a
     * keystroke was available.
     */
    bool
    checkForKeystroke(KeyDescription& keyDescription,
                      WindowDescription& windowDescription);


    /** 
     * This member function returns information about the window which
     * currently has input focus.  It returns WindowID, rather than
     * WindowDescription, for efficiency.
     * 
     * @return The return value is the ID of the window which
     * currently has input focus, i.e. the window to which keystrokes
     * are currently directed.
     */
    WindowID
    getCurrentWindowID();


    /** 
     * This member function synthesizes keypress events and sends them
     * to the specified window.
     *
     * NOTE(xxx): We currently ignore the windowDescription argument
     * and simply send keystrokes to the window currently having input
     * focus.  This will be fixed soon.
     * 
     * @param keyDescription This argument describes the keystroke to
     * be simulated.
     * 
     * @param windowDescription This argument specifies which window
     * should receive the synthesized keypress events.
     */
    void
    sendKeystroke(const KeyDescription& keyDescription,
                  const WindowDescription& windowDescription);


    /** 
     * This member function is just like checkForKeystroke(), except
     * that if no keystroke is in the queue, it blocks until one
     * becomes available.
     * 
     * @param keyDescription This reference argument will be populated
     * with the details of the keystroke.
     * 
     * @param windowDescription This reference argument will be
     * populated with the details of the window which had input focus
     * when the keystroke was issued.
     */
    void
    waitForKeystroke(KeyDescription& keyDescription,
                     WindowDescription& windowDescription);
    
  private:

    // An error handler to deal with the (many) times when we fall
    // slightly afoul of the X server.
    static int
    xErrorHandler(Display *displayPtr, XErrorEvent *errorEvent);
    

    // Convenience function for populating XEvent structures.
    XEvent
    buildEventTemplate(Display* displayPtr);


    // Returns the window currently having input focus.
    Window
    getOutputWindow(int &rootX, int &rootY,
                    int &windowX, int &windowY,
                    unsigned int &buttonMask);


    // Returns the name of the specified Window.  Note that "Window"
    // is an X11 type.
    std::string
    getWindowName(Window window);


    // Returns true if the specified X11 keysym refers to a modifier
    // key.
    bool
    isModifierKeySym(KeySym keySym);

    
    // Returns true for printable characters.  Currently uses a very
    // simple ascii-based heuristic.  We'll have to rework this
    // eventually.
    bool
    isValidChar(char rep);
    

    // Registers the X11Interface instance to receive keyboard events,
    // window creation events, etc. for all windows in the display.
    void
    selectInput(Display* displayPtr, Window window, long int inputMask);


    // Takes an X11 XKeyEvent structure pointer and uses it to
    // populate the (portable) KeyDescription and WindowDescription
    // classes.
    void
    unpackKeyPressEvent(XKeyEvent *eventPtr,
                        KeyDescription& keyDescription,
                        WindowDescription& windowDescription);
    

    
    Display* m_displayPtr;
    XEvent m_event;
    long int m_inputMask;
  };

  
  /* ============== Non-member function declarations ============== */
  

  /** 
   * This function translates ModifierTag instances into
   * X11Interface::ModifierState instances.  It provides, along with
   * checkModifierState() the interface for working with the otherwise
   * completely opaque ModifierState type.  Currently the only
   * supported ModifierTag value is MODIFIER_NONE.
   * 
   * @param modifierTag This argument specifies the tag to translate,
   * and currenly may only be set to MODIFIER_NONE.
   * 
   * @return The return value is the ModifierState value corresponding
   * to the specified ModifierTag.
   */
  inline X11Interface::ModifierState
  getModifierState(ModifierTag modifierTag) {
    switch(modifierTag) {
    case MODIFIER_NONE:
      return static_cast<X11Interface::ModifierState>(0);
      break;
    default:
      DLR_THROW(dlr::NotImplementedException, "getModifierState()",
                "Sorry, can only decode a few modifiers at present.");
      break;
    }
  }


  /** 
   * This function checks whether the specified modifier is active in
   * a particular ModifierState instance.  For example, if a
   * particular ModifierState value, m, corresponds to both the left
   * control and left shift keys being down, then checkModifier(m,
   * MODIFIER_SHIFT), checkModifier(m, MODIFIER_LEFT_SHIFT),
   * checkModifier(m, MODIFIER_CONTROL), and checkModifier(m,
   * MODIFIER_LEFT_CONTROL) will all return true.  This function
   * provides, along with checkModifierState() the interface for
   * working with the otherwise completely opaque ModifierState type.
   * 
   * @param modifierState This argument is the ModifierState value to
   * query.
   * 
   * @param modifierTag This argument specifies which modifier key we
   * care about.
   * 
   * @return The return value is true if modifierState corresponds to
   * the specified modifier key being down, false otherwise.
   */
  inline bool
  checkModifier(X11Interface::ModifierState modifierState,
                ModifierTag modifierTag) {

    // Note: operator!=() has precedence over operator&()
    switch(modifierTag) {
    case MODIFIER_NONE:
      return modifierState == static_cast<X11Interface::ModifierState>(0);
      break;
    case MODIFIER_LEFT_SHIFT:
    case MODIFIER_RIGHT_SHIFT:
    case MODIFIER_SHIFT:
      return ((modifierState
               & static_cast<X11Interface::ModifierState>(ShiftMask))
              != 0);
      break;
    case MODIFIER_LEFT_CONTROL:
    case MODIFIER_RIGHT_CONTROL:
    case MODIFIER_CONTROL:
      return ((modifierState
               & static_cast<X11Interface::ModifierState>(ControlMask))
              != 0);
      break;
    case MODIFIER_LEFT_MODIFIER1:
    case MODIFIER_RIGHT_MODIFIER1:
    case MODIFIER_MODIFIER1:
      return ((modifierState
               & static_cast<X11Interface::ModifierState>(Mod1Mask))
              != 0);
      break;
    case MODIFIER_LEFT_MODIFIER2:
    case MODIFIER_RIGHT_MODIFIER2:
    case MODIFIER_MODIFIER2:
      return ((modifierState
               & static_cast<X11Interface::ModifierState>(Mod2Mask))
              != 0);
      break;
    case MODIFIER_LEFT_MODIFIER3:
    case MODIFIER_RIGHT_MODIFIER3:
    case MODIFIER_MODIFIER3:
      return ((modifierState
               & static_cast<X11Interface::ModifierState>(Mod3Mask))
              != 0);
      break;
    case MODIFIER_LEFT_MODIFIER4:
    case MODIFIER_RIGHT_MODIFIER4:
    case MODIFIER_MODIFIER4:
      return ((modifierState
               & static_cast<X11Interface::ModifierState>(Mod4Mask))
              != 0);
      break;
    case MODIFIER_LEFT_MODIFIER5:
    case MODIFIER_RIGHT_MODIFIER5:
    case MODIFIER_MODIFIER5:
      return ((modifierState
               & static_cast<X11Interface::ModifierState>(Mod5Mask))
              != 0);
      break;
    case MODIFIER_CAPS_LOCK:
      return ((modifierState
               & static_cast<X11Interface::ModifierState>(LockMask))
              != 0);
      break;
    default:
      DLR_THROW(dlr::NotImplementedException, "checkModifier()",
                "Unrecognized Modifier.  X11Interface.hh is out of sync with "
                "ModifierTag.hh.");
      break;
    }
    return false;
  }
      

} // namespace xComplete

/* ============ Definitions of inline & template functions ============ */

namespace xComplete {

  // Empty.
  
} // namespace xComplete

#endif /* #ifndef _XCOMPLETE_X11INTERFACE_H_ */
