/**
***************************************************************************
* @file X11Interface.cc
* Source file defining X11Interface class.
*
* Copyright (C) 2000-2006 David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: $
* $Date: $
* 
* $Log: $
***************************************************************************
**/

#include <iostream>
#include <sstream>
#include <X11/X.h>
#include <X11/Xlib.h>
#include <X11/Intrinsic.h>
#include <X11/keysym.h>
// #include <X11/StringDefs.h>
#include <X11/Xatom.h>		// for XA_WM_NAME
// #include <X11/Xutil.h>
// #include <X11/Shell.h>
#include <dlrCommon/exception.h>
#include "X11Interface.hh"
#include "KeyDescription.hh"
#include "WindowDescription.hh"

namespace {

  const bool l_printX11Errors = false;

}


namespace xComplete {

  X11Interface::
  X11Interface(std::string displayName)
    : m_displayPtr(0),
      m_event(),
      m_inputMask(KeyPressMask | SubstructureNotifyMask)
  {
    if(displayName == "") {
      displayName = ":0";
    }
    
    this->m_displayPtr = XOpenDisplay(displayName.c_str());
    if (this->m_displayPtr == 0) {
      std::ostringstream message;
      message << "Can't open display " << displayName;
      DLR_THROW(dlr::RunTimeException, "X11Interface::X11Interface()",
                message.str().c_str());
    }

    XSetErrorHandler(X11Interface::xErrorHandler);
    this->m_event = this->buildEventTemplate(m_displayPtr);
    this->selectInput(
      m_displayPtr, DefaultRootWindow(this->m_displayPtr), m_inputMask);
  }

  
  X11Interface::
  ~X11Interface()
  {
    // Empty.
  }

  
  bool
  X11Interface::
  checkForKeystroke(KeyDescription& keyDescription,
                    WindowDescription& windowDescription)
  {
    static XEvent event;
    XEvent* eventPtr = &event;
    bool foundKeyPressEvent = false;
    while(foundKeyPressEvent == false) {
      if(!XCheckMaskEvent(this->m_displayPtr, this->m_inputMask, eventPtr)) {
        break;
      }
      switch(eventPtr->type) {
      case CreateNotify:
        this->selectInput(
          m_displayPtr, ((XCreateWindowEvent *)eventPtr)->window, m_inputMask);
        break;
      case KeyPress:
        foundKeyPressEvent = true;
        this->unpackKeyPressEvent(
          (XKeyEvent*)eventPtr, keyDescription, windowDescription);
        break;
      default:
        break;
      }
    }
    return foundKeyPressEvent;
  }
  

  X11Interface::WindowID
  X11Interface::
  getCurrentWindowID()
  {
    int rootX, rootY, winX, winY;
    unsigned int buttonMask;
    Window cw = this->getOutputWindow(rootX, rootY, winX, winY, buttonMask);
    return WindowID(cw);
  }

    
//   bool
//   X11Interface::
//   isModifierKey(KeyDescription::KeyID keyID)
//   {
//     KeySym keySym = XKeycodeToKeysym(this->m_displayPtr, keyID, 0);
//     return this->isModifierKeySym(keySym);
//   }

    
  void
  X11Interface::
  sendKeystroke(const KeyDescription& keyDescription,
                const WindowDescription& windowDescription)
  {
    // Note(xxx): We ought to use windowDescription here...
    int rootX, rootY, winX, winY;
    unsigned int buttonMask;
    Window outputWindow =
      this->getOutputWindow(rootX, rootY, winX, winY, buttonMask);
    this->m_event.xkey.x = winX;
    this->m_event.xkey.y = winY;
    this->m_event.xkey.x_root = rootX;
    this->m_event.xkey.y_root = rootY;
    this->m_event.xkey.window = outputWindow;
    if(this->isValidChar(keyDescription.getRepresentation())) {
      this->m_event.xkey.state = keyDescription.getState();
//       this->m_event.xkey.keycode =
//         XKeysymToKeycode(this->m_displayPtr, keyDescription.getKeyID());
      this->m_event.xkey.keycode = keyDescription.getKeyID();
      this->m_event.xkey.type = KeyPress;
      XSendEvent(this->m_displayPtr, outputWindow, True, KeyPressMask,
                 &(this->m_event));
      this->m_event.xkey.type = KeyRelease;
      XSendEvent(this->m_displayPtr, outputWindow, True, KeyReleaseMask,
                 &(this->m_event));
      XFlush(this->m_displayPtr);
    }
  }

  
  void
  X11Interface::
  waitForKeystroke(KeyDescription& keyDescription,
                   WindowDescription& windowDescription)
  {
    static XEvent event;
    XEvent* eventPtr = &event;
    bool foundKeyPressEvent = false;
    while(foundKeyPressEvent == false) {
      XNextEvent(this->m_displayPtr, eventPtr);
      switch(eventPtr->type) {
      case CreateNotify:
        this->selectInput(
          m_displayPtr, ((XCreateWindowEvent *)eventPtr)->window, m_inputMask);
        break;
      case KeyPress:
        foundKeyPressEvent = true;
        this->unpackKeyPressEvent(
          (XKeyEvent*)eventPtr, keyDescription, windowDescription);
        break;
      default:
        break;
      }
    }
  }


  // An error handler to deal with the (many) times when we fall
  // slightly afoul of the X server.
  int
  X11Interface::
  xErrorHandler(Display *displayPtr,
                XErrorEvent *errorEvent)
  {
    const int bufferLength = 255;
    char buffer[bufferLength];
    XGetErrorText(displayPtr, errorEvent->error_code, buffer,
                  bufferLength);
    if(l_printX11Errors) {
      std::cerr << "\n\nWARNING: Detected the following X11 Error:\n\n";
      std::cerr << buffer << std::endl;
    }
    return 0; // ??
  }

  
  // Convenience function for populating XEvent structures.
  XEvent
  X11Interface::
  buildEventTemplate(Display* displayPtr)
  {
    XEvent event;
    event.xkey.send_event = True;
    event.xkey.display = displayPtr;
    event.xkey.root = DefaultRootWindow(displayPtr);
    event.xkey.subwindow = 0;
    event.xkey.time = 0;
    return event;
  }


  // Returns the window currently having input focus.
  Window
  X11Interface::
  getOutputWindow(int &rootX, int &rootY,
                  int &windowX, int &windowY,
                  unsigned int &buttonMask)
  {
    Window window = DefaultRootWindow(this->m_displayPtr);
    Window newChild = window;
    Window newRoot;
    while(newChild != static_cast<Window>(0)) {    
      XQueryPointer(this->m_displayPtr, window, &newRoot, &newChild,
                    &rootX, &rootY, &windowX, &windowY, &buttonMask);
      if(newChild != static_cast<Window>(0)) {
        window = newChild;
      }
    }
    return window;
  }


  // Returns the name of the specified Window.  Note that "Window"
  // is an X11 type.
  std::string
  X11Interface::
  getWindowName(Window window)
  {
    std::string windowName;
    char *borrowedWindowName;
    if(XFetchName(this->m_displayPtr, window, &borrowedWindowName) != 0) {
      windowName = borrowedWindowName;
      XFree(borrowedWindowName);
    } else {
      Window root;
      Window parent;
      Window *childrenPtr;
      unsigned int numberOfChildren;
      if(XQueryTree(this->m_displayPtr, window, &root, &parent, &childrenPtr,
                    &numberOfChildren) != 0
         && window != root) {
        if(childrenPtr != NULL) {
          XFree(childrenPtr);
        }
        windowName = this->getWindowName(parent);
      } else {
        windowName = "";
      }
    }
    return windowName;
  }

  
  // Returns true if the specified X11 keysym refers to a modifier
  // key.
  bool
  X11Interface::
  isModifierKeySym(KeySym keySym)
  {
    if((keySym == XK_Shift_L)
       || (keySym == XK_Shift_R)
       || (keySym == XK_Control_L)
       || (keySym == XK_Control_R)
       || (keySym == XK_Alt_L)
       || (keySym == XK_Alt_R)
       || (keySym == XK_Meta_L)
       || (keySym == XK_Meta_R)
       || (keySym == XK_Caps_Lock)
       || (keySym == XK_Shift_Lock)) {
      return true;
    }
    return false;
  }


  // Returns true for printable characters.
  bool
  X11Interface::
  isValidChar(char rep)
  {
    if(((rep >= ' ') && (rep <= '~')) //alphaNumeric + many symbols
       || (rep == '\b')) {            //consult an ascii table for
      return 1;                       //details.
    }
    return 0;
  }
  
  
  // Registers the X11Interface instance to receive keyboard events,
  // window creation events, etc. for all windows in the display.
  void
  X11Interface::
  selectInput(Display* displayPtr, Window window, long int inputMask)
  {
    Window newRoot;
    Window newParent;
    Window* newChildrenPtr;
    unsigned int numberOfChildren;

    // Does window still exist?
    XEvent dummyEvent;
    if(XCheckTypedWindowEvent(displayPtr, window, DestroyNotify, &dummyEvent)
       != False) {
      // Window doesn't still exist.
      return;
    }

    // Check for children of this window.
    if(XQueryTree(displayPtr, window, &newRoot, &newParent,
                  &newChildrenPtr, &numberOfChildren) == 0) {
      if(l_printX11Errors) {
        std::cerr << "X11Interface::selectInput() -- XQueryTree() failure."
                  << std::endl;
      }
      return;
    }

    // Any children?
    if(numberOfChildren == 0) {
      return;
    }

    // Select input on the new root window.
    XSelectInput(displayPtr, newRoot, inputMask);

    // And recurse.
    for(size_t childIndex = 0; childIndex < numberOfChildren; childIndex++) {
      XSelectInput(displayPtr, newChildrenPtr[childIndex], inputMask);
      this->selectInput(
        displayPtr, newChildrenPtr[childIndex], inputMask);
    }     
    XFree((void *)newChildrenPtr);
  }


  // Takes an X11 XKeyEvent structure pointer and uses it to
  // populate the (portable) KeyDescription and WindowDescription
  // classes.
  void
  X11Interface::
  unpackKeyPressEvent(XKeyEvent *eventPtr,
                      KeyDescription& keyDescription,
                      WindowDescription& windowDescription)
  {
    const int bufferLength = 255;
    static char buffer[bufferLength];

    Window window = eventPtr->window;
    unsigned int state = eventPtr->state;
    KeyCode keyCode = eventPtr->keycode;
    KeySym keySym;
    int length = XLookupString(eventPtr, buffer, bufferLength,
                               &keySym, NULL);
    char rep;
    if(length == 1) {
      rep = *buffer;
    } else {
      rep = '\0';
    }    
//     char *staticKeyString = XKeysymToString(keySym);
//     if(staticKeyString != NULL) {
//       keyDescription.setString(staticKeyString);
//     }

    keyDescription.setKeyID(keyCode);
    keyDescription.setRepresentation(rep);
    keyDescription.setState(state);

    windowDescription.setWindowID(window);
    windowDescription.setWindowName(this->getWindowName(window));
  }
  
} // namespace xComplete
