/**
***************************************************************************
* @file WindowRule.hh
* Header file declaring WindowRule class.
*
* Copyright (C) 2006 David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
***************************************************************************
**/

#ifndef _XCOMPLETE_WINDOWRULE_H_
#define _XCOMPLETE_WINDOWRULE_H_

#include <boost/regex.hpp>
#include "WindowDescription.hh"

namespace xComplete {

  /**
   ** This is an abstract base class which defines the interface for
   ** WindowRule subclasses.  The purpose of a WindowRule (subclass)
   ** instance is to test WindowDescription instances and indicate
   ** whether or not word completion should be performed in the
   ** corresponding window.  Of course, you could also use WindowRule
   ** subclass instances for other task which involves associating
   ** boolean values with Windows.
   **/
  class WindowRule {
  public:

    /* ============== Public Types ============== */

    /** 
     * This enum lists all of the available rule types, and is used,
     * in conjunction with the getRuleType() virtual function, to
     * simulate Run-Time Type Information.
     */
    enum RuleType {ID_MATCH, NAME_MATCH};

    
    /* ============== Public member functions ============== */

    /** 
     * Default constructor does nothing.
     */
    WindowRule() {}
    

    /** 
     * The destructor deletes the class instance and cleans up any
     * system resources.
     */
    virtual
    ~WindowRule() {}


    /** 
     * This pure virtual function should be overridden by subclasses
     * to indicate what type of rule they implement.
     * 
     * @return The return value should be hardcoded by the subclass to
     * be the appropriate value of the RuleType enum.
     */
    virtual RuleType
    getRuleType() const = 0;
    

    /** 
     * This member function returns true if its argument implements
     * the same rule as *this, false otherwise.  In other words, this
     * member function returns true if, for all possible
     * WindowDescription instances, this->() isMatch() ==
     * otherPtr->isMatch().
     * 
     * @param otherPtr This argument is a pointer to the WindowRule
     * instance with which to compare *this.
     * 
     * @return The return value a bool indicating whether the two
     * rules are equivalent.
     */
    virtual bool
    isEquivalent(const WindowRule* otherPtr) const {
      return this->getIDString() == otherPtr->getIDString();
    }
    

    /** 
     * This pure virtual function should be overridden by subclasses
     * to indicate whether or not completion should be performed in
     * the window described by its argument.
     * 
     * @param windowDescription This argument indicates the window in
     * which completion might be performed.
     * 
     * @return The return value should be true if it's OK to do
     * completion (monitor keystrokes, send keystrokes, etc.) and
     * false if not.
     */
    virtual bool
    isMatch(const WindowDescription& windowDescription) = 0;

  protected:

    /** 
     * This pure virtual protected member function should be
     * overridden to return a string which uniquely describes the
     * decision rule.  getIDString() is used by isEquivalent() to
     * decide whether or not two WindowRule instances implement the
     * same rule.
     * 
     * @return The return value is a string describing the implemented
     * rule.
     */
    virtual std::string
    getIDString() const = 0;
    
  }; // class WindowRule
  

  /**
   ** This WindowRule subclass matches a single window.  It works by
   ** comparing the WindowID value of WindowDescription instances with
   ** a stored WindowID.
   **/
  class WindowRuleIDMatch : public WindowRule {
  public:

    /* ============== Public member functions ============== */

    /** 
     * This constructor extracts the WindowID value from its argument
     * and saves that value internally.
     * 
     * @param windowDescription This argument describes the window to
     * be matched.
     */
    WindowRuleIDMatch(const WindowDescription& windowDescription)
      : WindowRule(),
        m_windowID(windowDescription.getWindowID())
      {}
    
    
    /** 
     * The destructor deletes the class instance and cleans up any
     * system resources.
     */
    virtual
    ~WindowRuleIDMatch() {}


    /** 
     * This member function always returns ID_MATCH, indicating that
     * this WindowRule subclass implements a comparison of WindowIDs.
     * 
     * @return The return value is always ID_MATCH.
     */
    virtual RuleType
    getRuleType() const {return ID_MATCH;}


    /** 
     * This member function returns true if its argument has the same
     * WindowID as the WindowDescription that was passed as the
     * constructor argument.
     * 
     * @param windowDescription This argument specifies the
     * WindowDescription instance to test.
     * 
     * @return The return value is true if the argument's WindowID
     * value matches the stored WindowID.
     */
    virtual bool
    isMatch(const WindowDescription& windowDescription) {
      return windowDescription.getWindowID() == m_windowID;
    }

  protected:

    
    /** 
     * This protected member function returns a string which uniquely
     * describes the decision rule implemented by *this.
     * 
     * @return The return value is a string of the form "ID: [#]",
     * where "[#]" is a numeric value describing the stored WindowID.
     */
    virtual std::string
    getIDString() const {
      std::ostringstream outputStream;
      outputStream << "ID: " << m_windowID;
      return outputStream.str();
    }


    WindowDescription::WindowID m_windowID;
    
  }; // class WindowRule
  


  /**
   ** This WindowRule subclass matches windows based on their window
   ** name.  It works by comparing the name of the window to a stored
   ** regular expression.
   **/
  class WindowRuleNameMatch : public WindowRule {
  public:

    /* ============== Public member functions ============== */

    /** 
     * This constructor builds a class instance around the regular
     * expression described by its argument.
     * 
     * @param windowDescription This argument describes a roughly
     * perl-compatible regular expression.  All windows whose names
     * match this regular expression will be considered to match the
     * WindowRuleNameMatch instance.
     */
    WindowRuleNameMatch(const std::string& regularExpressionString)
      : WindowRule(),
        m_regex(regularExpressionString),
        m_regexString(regularExpressionString),
        m_what()
      {}
    

    /** 
     * The destructor deletes the class instance and cleans up any
     * system resources.
     */
    virtual
    ~WindowRuleNameMatch() {}


    /** 
     * This member function always returns NAME_MATCH, indicating that
     * this WindowRule subclass implements a comparison of window
     * names.
     * 
     * @return The return value is always NAME_MATCH.
     */
    virtual RuleType
    getRuleType() const {return NAME_MATCH;}


    /** 
     * This member function returns true if the name of the window
     * described by its argument matches the regular expression that
     * was passed as the constructor argument.
     * 
     * @param windowDescription This argument specifies the
     * WindowDescription instance to test.
     * 
     * @return The return value is true if the argument's name
     * value matches the stored regular expression.
     */
    virtual bool
    isMatch(const WindowDescription& windowDescription) {
      return boost::regex_match(
        windowDescription.getWindowName(), m_what, m_regex);
    }

  protected:

    /** 
     * This protected member function returns a string which uniquely
     * describes the decision rule implemented by *this.
     * 
     * @return The return value is a string of the form "Name: [s]",
     * where "[s]" is a string describing the stored regular expression.
     */
    virtual std::string
    getIDString() const {
      return "Name: " + m_regexString;
    }
    

    boost::regex m_regex;
    std::string m_regexString;
    boost::smatch m_what;
    
  }; // class WindowRule
  
  
  
  /* ============== Non-member function declarations ============== */
  

} // namespace xComplete

/* ============ Definitions of inline & template functions ============ */

namespace xComplete {

  // Empty.
  
} // namespace xComplete

#endif /* #ifndef _XCOMPLETE_WINDOWRULE_H_ */
