/**
***************************************************************************
* @file WindowDescription.hh
* Header file declaring WindowDescription class.
*
* Copyright (C) 2006 David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
***************************************************************************
**/

#ifndef _XCOMPLETE_WINDOWDESCRIPTION_H_
#define _XCOMPLETE_WINDOWDESCRIPTION_H_

#include <string>
#include "X11Interface.hh"

namespace xComplete {

  /**
   ** The WindowDescription class encapsulates the things we need to
   ** know about which window a keypress was directed to.  These
   ** include a unique identifier for the window, the name of the
   ** window, and anything else important for the functioning of the
   ** application.
   **/
  class WindowDescription {
  public:

    /* ============== Public types ============== */

    /**
     ** This public type is a small, inexpensive scalar for uniquely
     ** identifying windows on the display.  We simply borrow the
     ** corresponding X11Interface typedef.
     **/
    typedef X11Interface::WindowID WindowID;

    
    /* ============== Public member functions ============== */

    /** 
     * This constructor creates a WindowDescription instance and
     * initializes everything to safe defaults.
     */
    WindowDescription()
      : m_windowName(),
        m_windowID(0)
      {
        // Empty.
      }
    

    /** 
     * This constructor creates a WindowDescription instance and
     * initializes everything as specified by its arguments.
     * 
     * @param windowID This argument indicates the WindowID of the
     * referent window.
     * 
     * @param windowName This argument indicates the name of the
     * referent window.
     */
    WindowDescription(WindowID windowID,
                      const std::string& windowName)
      : m_windowName(windowName),
        m_windowID(windowID)
      {
        // Empty.
      }

    
    /** 
     * The destructor destroys the class instance and cleans up any
     * system resources.
     */
    ~WindowDescription() {}


    /** 
     * This member function returns the WindowID of the window
     * described by *this.
     * 
     * @return The return value is a unique WindowID corresponding to
     * the window described by *this.
     */
    WindowID
    getWindowID() const {return m_windowID;}


    /** 
     * This member function returns the name of the window described
     * by *this.
     * 
     * @return The return value is a string containing the window
     * name.
     */
    std::string
    getWindowName() const {return m_windowName;}


    /** 
     * This member function can be used to set the stored WindowID.
     * It is useful for initializing WindowDescription instances that
     * were created using the default constructor.
     * 
     * @param windowID This argument specifies the WindowID to be
     * stored.
     */
    void
    setWindowID(WindowID windowID) {m_windowID = windowID;}


    /** 
     * This member function can be used to set the stored window name.
     * It is useful for initializing WindowDescription instances that
     * were created using the default constructor.
     * 
     * @param windowID This argument specifies the name to be stored.
     */
    void
    setWindowName(const std::string& windowName) {m_windowName = windowName;}
    
  private:

    std::string m_windowName;
    WindowID m_windowID;
    
  }; // class WindowDescription
  
  /* ============== Non-member function declarations ============== */
  

} // namespace xComplete

/* ============ Definitions of inline & template functions ============ */

namespace xComplete {

  // Empty.
  
} // namespace xComplete

#endif /* #ifndef _XCOMPLETE_WINDOWDESCRIPTION_H_ */
