/**
***************************************************************************
* @file KeyHistory.hh
* Header file declaring KeyHistory class.
*
* Copyright (C) 2006 David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
***************************************************************************
**/

#ifndef _XCOMPLETE_KEYHISTORY_H_
#define _XCOMPLETE_KEYHISTORY_H_

#include <deque>
#include <map>
#include <vector>
#include "KeyDescription.hh"
#include "WindowDescription.hh"


namespace xComplete {

  /**
   ** The KeyHistory class keeps track of what's been typed in each
   ** window, and makes the most recent word or word fragment
   ** available for inspection.
   **/
  class KeyHistory {
  public:

    /* ============== Public types ============== */

    /**
     ** This public type is a small, inexpensive scalar for uniquely
     ** identifying windows on the display.  We'd use the
     ** WindowDescription public type anyway, but making a local
     ** typedef saves typing below.
     **/
    typedef WindowDescription::WindowID WindowID;

    
    /* ================ Public member functions ================ */

    /** 
     * The default constructor creates a KeyHistory instance.
     * 
     * @param historySize This argument specifies the number of
     * keystrokes to remember for each window.  It therefore also sets
     * the length of the longest word that can be accurately recorded.
     */
    KeyHistory(size_t historySize=256);


    /** 
     * The copy constructor deep-copies its argument.
     * 
     * @param source This argument 
     */
    KeyHistory(const KeyHistory& source);


    /** 
     * The destructor destroys the class instance and cleans up any
     * system resources.
     */
    ~KeyHistory();


    /** 
     * The assignment operator deep copies its argument.
     * 
     * @param source This argument is a reference to the KeyHistory
     * instance to be copied.
     * 
     * @return The return value is a reference to *this.
     */
    KeyHistory&
    operator=(const KeyHistory& source);


    /** 
     * This member function adds a keystroke to the history of a
     * specific window.  If the window is one that hasn't been seen
     * before, a new (empty) history is created for that window, and
     * the keystroke is added to the new history.  If the history for
     * for the selected window currently contains a number of
     * keystrokes equal to historyLength (see the documentation for
     * the default constructor) then the oldest keystroke will be
     * discarded from the history.
     * 
     * @param keyDescription This argument describes the keystroke to
     * add.  It will be copied into the appropriate history.
     * 
     * @param windowID This argument specifies which window's history
     * should be updated.
     * 
     * @param isNonWordCharacter This argument indicates whether the
     * keystroke sould be considered to separate words or not.  If
     * isNonWordCharacter is true, then this keystroke ends any word
     * currently being typed, and a new word will be started with the
     * next keystroke for which isNonWordCharacter is false.  If
     * isNonWordCharacter is false, this keystroke will simple add to
     * the currently growing word.
     */
    void
    addKeystroke(const KeyDescription& keyDescription,
                 WindowID windowID,
                 bool isNonWordCharacter);


    /** 
     * This member function deletes the most recently added keystroke,
     * returning the history to nearly the same state as before that
     * keystroke was added.  We say "nearly the same state" because
     * the addition of the most recent keystroke may have caused an
     * old keystroke to be discarded from the history, and that old
     * keystroke will not be recovered.  Multiple calls to
     * deleteKeystroke() will cause multiple keystrokes to be deleted.
     * If the window is one that hasn't been seen before, a new
     * (empty) history is created for that window.
     * 
     * @param windowID This argument specifies the window from whose
     * history the most recent keystroke should be deleted.
     */
    void
    deleteKeystroke(const WindowID& windowID);


    /** 
     * This member function returns the most recently added word or
     * word-fragment for the specified window.  If the most recently
     * addeed keystroke was a non-word character, or if the history
     * for the specified window is empty, then the returned vector
     * will be empty.
     * 
     * @param windowID This argument specifies the window whose
     * history should be consulted to construct the word.
     * 
     * @return The return value is a vector of KeyDescription
     * instances describing the keystrokes which were used to type the
     * word.
     */
    std::vector<KeyDescription>
    getFinalWordFragment(const WindowID& windowID);


    /** 
     * This member function resets the history for the specified
     * window, erasing all of the recorded keystrokes for that window.
     * 
     * @param windowID This argument specifies the window whose
     * history is to be erased.
     */
    void
    reset(WindowID windowID);

  private:

    // This private member function returns a deque of KeyDescription
    // instances representing the keystroke history of the specified
    // window.
    std::deque<KeyDescription>&
    lookupKeystrokeDeque(WindowID windowID);


    // This private member function returns the length of the word
    // which is currently being typed in the specified window.  The
    // return value is zero if the most recently typed keystroke is a
    // non-word character (see addKeystroke).
    size_t&
    lookupWordLength(WindowID windowID);


    // This private member function returns a deque of bools which
    // correspond to the KeyDescription instances in the deque
    // returned by lookupKeySrokeDeque().  Each bool indicates whether
    // the corresponding KeyDescription instance is a non-word
    // character.
    std::deque<bool>&
    lookupWordSeparatorIndicatorDeque(WindowID windowID);

    
    /* ================ Private data members ================ */

    // This member indicates the maximum capacity of the KeyHistory
    // instance.
    size_t m_historySize;


    // This member holds the actual key press history.  It maps
    // WindowID values to deques which contain the keypress history
    // for the associated window.  Each time member function
    // addKeystroke() is called, the keystroke is appended to one of
    // the deques in m_keyStrokeDequeMap.
    std::map<WindowID, std::deque<KeyDescription> >
    m_keyStrokeDequeMap;


    // This member maps from WindowID instance to the length of the
    // word or word fragment most recently typed in the corresponding
    // window.
    std::map<WindowID, size_t>
    m_wordLengthMap;


    // The deques in m_wordSeparatorIndicatorMap correspond to those
    // in m_keyStrokeDequeMap, with the exception that instead of
    // containing KeyDescription instances, these deques contain bools
    // indicating whether or not the corresponding KeyDescription
    // instances represent whitespace.
    std::map<WindowID, std::deque<bool> >
    m_wordSeparatorIndicatorDequeMap;

  }; // class KeyHistory


  /* ============== Non-member function declarations ============== */
  

} // namespace xComplete

/* ============ Definitions of inline & template functions ============ */

namespace xComplete {

  // Empty.
  
} // namespace xComplete

#endif /* #ifndef _XCOMPLETE_KEYHISTORY_H_ */
