/**
***************************************************************************
* @file KeyHistory.cc
* Source file defining KeyHistory class.
*
* Copyright (C) 2006 David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: $
* $Date: $
* 
* $Log: $
***************************************************************************
**/

#include <KeyHistory.hh>

namespace xComplete {


  // The default constructor creates a KeyHistory instance.
  KeyHistory::
  KeyHistory(size_t historySize)
    : m_historySize(historySize),
      m_keyStrokeDequeMap(),
      m_wordLengthMap(),
      m_wordSeparatorIndicatorDequeMap()
  {
    // Empty.
  }


  // The copy constructor deep-copies its argument.
  KeyHistory::
  KeyHistory(const KeyHistory& source)
    : m_historySize(source.m_historySize),
      m_keyStrokeDequeMap(source.m_keyStrokeDequeMap),
      m_wordLengthMap(source.m_wordLengthMap),
      m_wordSeparatorIndicatorDequeMap(source.m_wordSeparatorIndicatorDequeMap)
  {
    // Empty.
  }


  // The destructor destroys the class instance and cleans up any
  // system resources.
  KeyHistory::
  ~KeyHistory()
  {
    // Empty.
  }


  // The assignment operator deep copies its argument.
  KeyHistory&
  KeyHistory::
  operator=(const KeyHistory& source)
  {
    m_historySize = source.m_historySize;
    m_keyStrokeDequeMap = source.m_keyStrokeDequeMap;
    m_wordLengthMap = source.m_wordLengthMap;
    m_wordSeparatorIndicatorDequeMap = source.m_wordSeparatorIndicatorDequeMap;
    return *this;
  }


  // This member function adds a keystroke to the history of a
  // specific window.
  void
  KeyHistory::
  addKeystroke(const KeyDescription& keyDescription,
               WindowID windowID,
               bool isNonWordCharacter)
  {
    // Do all the windowID lookups in advance.
    std::deque<KeyDescription>& keyStrokeDeque =
      this->lookupKeystrokeDeque(windowID);
    size_t& wordLength = 
      this->lookupWordLength(windowID);
    std::deque<bool>& wordSeparatorIndicatorDeque =
      this->lookupWordSeparatorIndicatorDeque(windowID);

    // Remember the new keystroke.
    keyStrokeDeque.push_back(keyDescription);
    wordSeparatorIndicatorDeque.push_back(isNonWordCharacter);
    if(keyStrokeDeque.size() > m_historySize) {
      keyStrokeDeque.pop_front();
      wordSeparatorIndicatorDeque.pop_front();
    }

    if(isNonWordCharacter) {
      // If this character couldn't be part of a word, zero wordlength
      // to indicate that we'll be starting a new word soon.
      wordLength = 0;
    } else {
      // Else assume this character is part of the previous word.
      if(wordLength < m_historySize) {
        ++wordLength;
      }
    }
  }
  

  // This member function deletes the most recently added keystroke,
  // returning the history to nearly the same state as before that
  // keystroke was added.
  void
  KeyHistory::
  deleteKeystroke(const WindowID& windowID)
  {
    // Do all the windowID lookups in advance.
    std::deque<KeyDescription>& keyStrokeDeque =
      this->lookupKeystrokeDeque(windowID);
    size_t& wordLength = 
      this->lookupWordLength(windowID);
    std::deque<bool>& wordSeparatorIndicatorDeque =
      this->lookupWordSeparatorIndicatorDeque(windowID);

    // Normally, we just delete the last character and decrease the
    // word length.  However, if the word length is already zero, then
    // we need to search backwards to find the beginning of the
    // previous word.
    if(!keyStrokeDeque.empty()) {
      keyStrokeDeque.pop_back();
      wordSeparatorIndicatorDeque.pop_back();
      if(wordLength != 0) {
        --wordLength;
      } else {
        std::deque<bool>::const_reverse_iterator rbeginIterator =
          wordSeparatorIndicatorDeque.rbegin();
        std::deque<bool>::const_reverse_iterator indicatorIterator =
          std::find(wordSeparatorIndicatorDeque.rbegin(),
                    wordSeparatorIndicatorDeque.rend(),
                    true);
        wordLength = indicatorIterator - rbeginIterator;
        // while(indicatorIterator != rbeginIterator) {
        //   ++indicatorIterator;
        //   ++wordLength;
        // }
      }
    }
  }
  

  // This member function returns the most recently added word or
  // word-fragment for the specified window.
  std::vector<KeyDescription>
  KeyHistory::
  getFinalWordFragment(const WindowID& windowID)
  {
    // Do all the windowID lookups in advance.
    std::deque<KeyDescription>& keyStrokeDeque =
      this->lookupKeystrokeDeque(windowID);
    size_t& wordLength = 
      this->lookupWordLength(windowID);

    std::vector<KeyDescription> wordFragment(wordLength);
    std::copy(keyStrokeDeque.end() - wordLength, keyStrokeDeque.end(),
              wordFragment.begin());

    return wordFragment;
  }


  // This member function resets the history for the specified
  // window, erasing all of the recorded keystrokes for that window.
  void
  KeyHistory::
  reset(WindowID windowID)
  {
    // Do all the windowID lookups in advance.
    std::deque<KeyDescription>& keyStrokeDeque =
      this->lookupKeystrokeDeque(windowID);
    size_t& wordLength = 
      this->lookupWordLength(windowID);
    std::deque<bool>& wordSeparatorIndicatorDeque =
      this->lookupWordSeparatorIndicatorDeque(windowID);

    // Now do the reset.
    keyStrokeDeque.clear();
    wordLength = 0;
    wordSeparatorIndicatorDeque.clear();
  }

  
  // This private member function returns a deque of KeyDescription
  // instances representing the keystroke history of the specified
  // window.
  std::deque<KeyDescription>&
  KeyHistory::
  lookupKeystrokeDeque(WindowID windowID)
  {
    std::deque<KeyDescription>& keyStrokeDeque =
      this->m_keyStrokeDequeMap[windowID];
    return keyStrokeDeque;
  }

  
  // This private member function returns the length of the word
  // which is currently being typed in the specified window.
  size_t&
  KeyHistory::
  lookupWordLength(WindowID windowID)
  {
    std::map<WindowID, size_t>::iterator mapIterator =
      m_wordLengthMap.find(windowID);

    // If this window is new to us, we initialize wordLength to zero.
    if(mapIterator == m_wordLengthMap.end()) {
      size_t& wordLength = m_wordLengthMap[windowID];
      wordLength = 0;
      return wordLength;
    }

    return mapIterator->second;
  }


  // This private member function returns a deque of bools which
  // correspond to the KeyDescription instances in the deque
  // returned by lookupKeySrokeDeque().  Each bool indicates whether
  // the corresponding KeyDescription instance is a non-word
  // character.
  std::deque<bool>&
  KeyHistory::
  lookupWordSeparatorIndicatorDeque(WindowID windowID)
  {
    std::deque<bool>& wordSeparatorIndicatorDeque =
      this->m_wordSeparatorIndicatorDequeMap[windowID];
    return wordSeparatorIndicatorDeque;
  }
  
} // namespace xComplete
