/**
***************************************************************************
* @file KeyDescription.hh
* Header file declaring KeyDescription class.
*
* Copyright (C) 2006 David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
***************************************************************************
**/

#ifndef _XCOMPLETE_KEYDESCRIPTION_H_
#define _XCOMPLETE_KEYDESCRIPTION_H_

#include <iostream>
#include <X11/X.h>
#include "X11Interface.hh"

namespace xComplete {

  /**
   ** The KeyDescription class encapsulates the things we need to know
   ** about a keypress.  That is, it specifies which key was pressed,
   ** and which modifier keys were active.  If necessary, this class
   ** can be expanded to include other things, such as a string
   ** representing any characters which we expect to be generated by
   ** the keypress.
   **/
  class KeyDescription {
  public:

    /* ============== Public types ============== */

    /**
     ** This public type is a small, inexpensive scalar for uniquely
     ** identifying keys on the keyboard.  We simply borrow the
     ** corresponding X11Interface typedef.
     **/
    typedef X11Interface::KeyID KeyID;

    
    /**
     ** This public type is a small, inexpensive scalar for uniquely
     ** identifying windows on the display.  We simply borrow the
     ** corresponding X11Interface typedef.
     **/
    typedef X11Interface::ModifierState ModifierState;


    /** 
     * This public type is a small inexpensive scalar for uniquely
     * identifying individual characters.  For now, we assume ascii
     * representation is sufficient and use the built in char type.
     */
    typedef char CharacterType;

    
    /* ============== Public member functions ============== */

    /** 
     * The default constructor simply initializes to a null character.
     */
    KeyDescription()
      : m_keyID(0),
	m_modifierState(getModifierState(MODIFIER_NONE)),
        m_representation(0)
      {
        // Empty.
      }
    

    /** 
     * This constructor allows the caller to explicitly state the
     * details of the keypress.
     * 
     * @param keyID This argument specifies which key on the keyboard
     * was pressed.
     * 
     * @param modifierState This argument specifies which modifiers
     * were active at the time of the keypress.
     * 
     * @param representation This argument specifies the character
     * representation of the keypress.
     */
    KeyDescription(KeyID keyID,
                   ModifierState modifierState,
                   CharacterType representation)
      : m_keyID(keyID),
	m_modifierState(modifierState),
        m_representation(representation)
      {
        // Empty.
      }

    
    /** 
     * The destructor destroys the class instance and cleans up any
     * system resources.
     */
    ~KeyDescription() {}


    /** 
     * This member function returns the KeyID associated with the
     * KeyDescription instance.
     * 
     * @return The return value is the KeyID corresponding to the key
     * which was pressed (or would be pressed) to generate this
     * KeyDescription instance.
     */
    KeyID
    getKeyID() const {
      return m_keyID;
    }


    /** 
     * This member function returns the character representation
     * associated with the KeyDescription instance.
     * 
     * @return The return value is the character which corresponds to
     * the key and modifiers which were pressed (or would be pressed)
     * to generate this KeyDescription instance.
     */
    CharacterType
    getRepresentation() const {
      return m_representation;
    }


    /** 
     * This member function returns a scalar corresponding to the set
     * of modifier keys associated with the KeyDescription instance.
     * 
     * @return The return value is the ModifierState value
     * corresponding to the set of modifier keys which were pressed
     * (or would be pressed) to generate this KeyDescription instance.
     */
    ModifierState
    getState() const {
      return m_modifierState;
    }


    /** 
     * This member function allows the user to set the KeyID of the
     * keystroke represented by *this.  It is useful for initializing
     * KeyDescription instances which were constructed with the
     * default constructor.
     * 
     * @param keyID This argument specifies the KeyID value to use.
     */
    void
    setKeyID(KeyID keyID) {
      m_keyID = keyID;
    }


    /** 
     * This member function allows the user to set the printable
     * representation of the keystroke represented by *this.  It is
     * useful for initializing KeyDescription instances which were
     * constructed with the default constructor.
     * 
     * @param representation This argument specifies character value
     * to use.
     */
    void
    setRepresentation(CharacterType representation) {
      m_representation = representation;
    }


    /** 
     * This member function allows the user to set the modifier state
     * of the keystroke represented by *this.  It is useful for
     * initializing KeyDescription instances which were constructed
     * with the default constructor.
     * 
     * @param modifierState This argument specifies the ModifierState
     * value to use.
     */
    void
    setState(ModifierState modifierState) {
      m_modifierState = modifierState;
    }

  private:

    KeyID m_keyID;
    ModifierState m_modifierState;
    CharacterType m_representation;
    
  }; // class KeyDescription

  
  /* ============== Non-member function declarations ============== */

  /** 
   * This comparison function is useful for sorting sequences of
   * KeyDescription instances.  It imposes an arbitrary but repeatable
   * ordering on KeyDescription instance based on their contents.  The
   * ordering is "symmetric" only in the sense that left and right
   * modifier keys are considered equivalent.
   * 
   * @param key0 This argument is the first KeyDescription instance to
   * be compared.
   * 
   * @param key1 This argument is the second KeyDescription instance to
   * be compared.
   * 
   * @return The return value is a bool indicating whether the first
   * KeyDescription instance is "greater" than the second
   * KeyDescription instance.
   */
  inline bool
  isSymmetricGreater(const KeyDescription& key0, const KeyDescription& key1);

  
  /** 
   * This comparison function indicates whether two KeyDescription
   * instances represent the same keystroke.  It is "symmetric" only
   * in the sense that left and right modifier keys are considered
   * equivalent.
   * 
   * @param key0 This argument is the first KeyDescription instance to
   * be compared.
   * 
   * @param key1 This argument is the second KeyDescription instance to
   * be compared.
   * 
   * @return The return value is a bool indicating whether the two
   * KeyDescription instances represent the same keystroke.
   */
  inline bool
  isSymmetricMatch(const KeyDescription&, const KeyDescription&);


  /** 
   * This operator sends a printable description if a KeyDescription
   * instance to an output stream.
   * 
   * @param outputStream This argument is the output stream to which
   * the description should be written.
   * 
   * @param keyDescription This argument is the KeyDescription
   * instance to write to the output stream.
   * 
   * @return The return value is a reference to the output stream
   * after writing.
   */
  inline std::ostream&
  operator<<(std::ostream& outputStream, const KeyDescription& keyDescription);


  /** 
   * This operator reads an KeyDescription instance from an input
   * stream.  It matches operator<<(std::ostream&, const
   * KeyDescription&) in that the description written by
   * operator<<(...) can be converted back into a KeyDescription
   * instance by operator>>(...).
   * 
   * @param inputStream This argument is the input stream from which
   * to read.
   * 
   * @param keyDescription This reference argument is modified to
   * match the data from the input stream.
   * 
   * @return The return value is a reference to the input stream after
   * reading.
   */
  inline std::istream&
  operator>>(std::istream& inputStream, KeyDescription& keyDescription);
  
} // namespace xComplete

/* ============ Definitions of inline & template functions ============ */

#include <dlrCommon/inputStream.h>

namespace xComplete {

  // This comparison function is useful for sorting sequences of
  // KeyDescription instances.
  inline bool
  isSymmetricGreater(const KeyDescription& key0, const KeyDescription& key1)
  {
    if(key0.getKeyID() > key1.getKeyID()) {
      return true;
    }

    if(key1.getKeyID() > key0.getKeyID()) {
      return false;
    }

    KeyDescription::ModifierState state0 = key0.getState();
    KeyDescription::ModifierState state1 = key1.getState();

    if(checkModifier(state0, MODIFIER_SHIFT)
       && !checkModifier(state1, MODIFIER_SHIFT)) {
      return true;
    }
    
    if(checkModifier(state1, MODIFIER_SHIFT)
       && !checkModifier(state0, MODIFIER_SHIFT)) {
      return false;
    }
    
    if(checkModifier(state0, MODIFIER_CONTROL)
       && !checkModifier(state1, MODIFIER_CONTROL)) {
      return true;
    }
    
    if(checkModifier(state1, MODIFIER_CONTROL)
       && !checkModifier(state0, MODIFIER_CONTROL)) {
      return false;
    }
    
    if(checkModifier(state0, MODIFIER_MODIFIER1)
       && !checkModifier(state1, MODIFIER_MODIFIER1)) {
      return true;
    }
    
    if(checkModifier(state1, MODIFIER_MODIFIER1)
       && !checkModifier(state0, MODIFIER_MODIFIER1)) {
      return false;
    }

    if(checkModifier(state0, MODIFIER_MODIFIER2)
       && !checkModifier(state1, MODIFIER_MODIFIER2)) {
      return true;
    }
    
    if(checkModifier(state1, MODIFIER_MODIFIER2)
       && !checkModifier(state0, MODIFIER_MODIFIER2)) {
      return false;
    }

    if(checkModifier(state0, MODIFIER_MODIFIER3)
       && !checkModifier(state1, MODIFIER_MODIFIER3)) {
      return true;
    }
    
    if(checkModifier(state1, MODIFIER_MODIFIER3)
       && !checkModifier(state0, MODIFIER_MODIFIER3)) {
      return false;
    }

    if(checkModifier(state0, MODIFIER_MODIFIER4)
       && !checkModifier(state1, MODIFIER_MODIFIER4)) {
      return true;
    }
    
    if(checkModifier(state1, MODIFIER_MODIFIER4)
       && !checkModifier(state0, MODIFIER_MODIFIER4)) {
      return false;
    }

    if(checkModifier(state0, MODIFIER_MODIFIER5)
       && !checkModifier(state1, MODIFIER_MODIFIER5)) {
      return true;
    }
    
    if(checkModifier(state1, MODIFIER_MODIFIER5)
       && !checkModifier(state0, MODIFIER_MODIFIER5)) {
      return false;
    }

    return false;
  }    


  // This comparison function indicates whether two KeyDescription
  // instances represent the same keystroke.
  inline bool
  isSymmetricMatch(const KeyDescription& key0, const KeyDescription& key1)
  {
    if(key1.getKeyID()
       != key0.getKeyID()) {
      return false;
    }

    KeyDescription::ModifierState state0 = key0.getState();
    KeyDescription::ModifierState state1 = key1.getState();

    if(checkModifier(state1, MODIFIER_SHIFT)
       != checkModifier(state0, MODIFIER_SHIFT)) {
      return false;
    }
    if(checkModifier(state1, MODIFIER_CONTROL)
       != checkModifier(state0, MODIFIER_CONTROL)) {
      return false;
    }
    if(checkModifier(state1, MODIFIER_MODIFIER1)
       != checkModifier(state0, MODIFIER_MODIFIER1)) {
      return false;
    }
    if(checkModifier(state1, MODIFIER_MODIFIER2)
       != checkModifier(state0, MODIFIER_MODIFIER2)) {
      return false;
    }
    if(checkModifier(state1, MODIFIER_MODIFIER3)
       != checkModifier(state0, MODIFIER_MODIFIER3)) {
      return false;
    }
    if(checkModifier(state1, MODIFIER_MODIFIER4)
       != checkModifier(state0, MODIFIER_MODIFIER4)) {
      return false;
    }
    if(checkModifier(state1, MODIFIER_MODIFIER5)
       != checkModifier(state0, MODIFIER_MODIFIER4)) {
      return false;
    }
    return true;
  }    


  // This operator sends a printable description if a KeyDescription
  // instance to an output stream.
  inline std::ostream&
  operator<<(std::ostream& outputStream, const KeyDescription& keyDescription)
  {
    outputStream << "KeyDescription {"
                 << "keyID = "
                 << static_cast<int>(keyDescription.getKeyID()) << "; "
                 << "modifierState = "
                 << static_cast<int>(keyDescription.getState()) << "; "
                 << "representation = "
                 << static_cast<int>(keyDescription.getRepresentation())
                 << ";}";
    return outputStream;
  }


  // This operator reads an KeyDescription instance from an input
  // stream.  It matches operator<<(std::ostream&, const
  inline std::istream&
  operator>>(std::istream& inputStream, KeyDescription& keyDescription)
  {
    // It's a lot easier to use a try block than to be constantly
    // testing whether the IO has succeeded, so we tell inputStream to
    // complain if anything goes wrong.
    std::ios_base::iostate oldExceptionState = inputStream.exceptions();
    inputStream.exceptions(
      std::ios_base::badbit | std::ios_base::failbit | std::ios_base::eofbit);

    // Now on with the show.
    try{
      // We'll use this as an intermediary.
      int tempInt;
      
      // Construct an InputStream instance so we can use our
      // convenience functions.
      dlr::InputStream stream(inputStream, dlr::InputStream::SKIP_WHITESPACE);

      // The preamble is short and sweet, but must be there.
      stream.expect("KeyDescription");
      stream.expect("{");

      // Read the individual data members.
      stream.expect("keyID");
      stream.expect("=");
      inputStream >> tempInt;
      keyDescription.setKeyID(
        static_cast<KeyDescription::KeyID>(tempInt));
      stream.expect(";");

      stream.expect("modifierState");
      stream.expect("=");
      inputStream >> tempInt;
      keyDescription.setState(
        static_cast<KeyDescription::ModifierState>(tempInt));
      stream.expect(";");

      stream.expect("representation");
      stream.expect("=");
      inputStream >> tempInt;
      keyDescription.setRepresentation(
        static_cast<KeyDescription::CharacterType>(tempInt));
      stream.expect(";");

      // Finally, the closing brace.
      stream.expect("}");

      // Warning(xxx): instead of buffering info above and copying it
      // here, we simply copy above, which could leave us in a
      // half-read state if the config file is hosed.
      // 
      // And finally, copy the data.
      // Code here...
    } catch(std::ios_base::failure) {
      // Empty
    }

    inputStream.exceptions(oldExceptionState);
    return inputStream;
  }
  
} // namespace xComplete

#endif /* #ifndef _XCOMPLETE_KEYDESCRIPTION_H_ */
