/**
***************************************************************************
* @file CompletionGenerator.hh
* Header file declaring CompletionGenerator class.
*
* Copyright (C) 2006 David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
***************************************************************************
**/

#ifndef _XCOMPLETE_COMPLETIONGENERATOR_H_
#define _XCOMPLETE_COMPLETIONGENERATOR_H_

#include <map>
#include <vector>
#include "KeyDescription.hh"

namespace xComplete {

  /**
   ** The CompletionGenerator class maintains a database of words, and
   ** compares word fragments with the database, returning complete
   ** words which match the fragments.
   **/
  class CompletionGenerator {
  public:

    /** 
     * The default constructor creates an empty database.
     */
    CompletionGenerator();

    
    /** 
     * The destructor destroys the class instance and cleans up any
     * system resources.
     */
    ~CompletionGenerator();


    /** 
     * This member function adds a word to the database.
     * 
     * @param word This argument is a vector of KeyDescription
     * instances describing the word to be added.
     */
    void
    addWord(const std::vector<KeyDescription>& word);


    /** 
     * This member function accepts a partially-typed word as its
     * argument, and returns a complete word which matches the input
     * word fragment.
     * 
     * @param wordBeginning This argument specifies the word fragment
     * to be completed.
     * 
     * @param completionIndex This argument affects the return value
     * only if more than one word from the database matches the input
     * word fragment.  If more than one word matches, and
     * completionIndex is set to 0, then the most recently added word
     * will be returned.  Similarly, if completionIndex is set to 1,
     * then the second-most recently added word will be returned, and
     * so on until the least recently added word is added.  Increasing
     * completionIndex beyond this point will cause the return value
     * to loop back to the most recently added word again, etc.
     * 
     * @return The return value is a vector of KeyDescription
     * instances representing a word which completes wordBeginning.
     * If no completion is found, the returned vector will be empty.
     */
    const std::vector<KeyDescription>&
    getCompletion(const std::vector<KeyDescription>& wordBeginning,
                  size_t completionIndex) const;


    /** 
     * This member function loads the database to a file.  In case of
     * errors while trying to read the file, an exception will be
     * thrown.
     * 
     * @param fileName This argument specifies the file to be read.
     */
    void
    load(const std::string& fileName);


    /** 
     * This member function writes the database to a file.  In case of
     * errors while trying to write the file, an exception will be
     * thrown.
     * 
     * @param fileName This argument specifies the file to be written.
     * If it already exists, it will be overwritten.
     */
    void
    save(const std::string& fileName);
    

  private:

    /**
     ** This functor imposes an arbetrary but consistent ordering on
     ** vectors of KeyDescription instances.
     **/
    struct
    IsLessThanFunctor {
      bool
      operator()(const std::vector<KeyDescription>& word0,
                 const std::vector<KeyDescription>& word1) const;
    };


    /**
     ** This typedef is provided for convenience in internal code
     ** which uses iterators.
     **/
    typedef std::map<std::vector<KeyDescription>, size_t, IsLessThanFunctor>
    ::iterator WordMapIterator;

    
    /**
     ** This typedef is provided for convenience in internal code
     ** which uses iterators.
     **/
    typedef std::map<std::vector<KeyDescription>, size_t, IsLessThanFunctor>
    ::const_iterator WordMapConstIterator;


    /**
     ** This function is used in reading the database from disk.
     **/
    std::map<std::vector<KeyDescription>, size_t, IsLessThanFunctor>
    deserializeWordMap(std::istream& inputStream);


    
    /**
     ** This function is used in writing the database to disk.
     **/
    std::string
    serializeWordMap(
      const std::map<std::vector<KeyDescription>, size_t, IsLessThanFunctor>&
      wordMap);


    /**
     ** this function returns true if the specified word is a
     ** plausible completion for the specified word stub.
     **/
    bool
    isMatch(const std::vector<KeyDescription>& stub,
            const std::vector<KeyDescription>& word) const;
    

    /**
     ** This function throws away old words, keeping only half the
     ** number of words specified by m_maximumNumberOfWords, and
     ** resetting timestamps so that the retained words have
     ** timestamps from 0 to (m_maximumNumberOfWords / 2).
     **/
    void
    purge();


    std::vector<KeyDescription> m_emptyCompletion;
    size_t m_maximumNumberOfWords;
    size_t m_timestamp;
    std::map<std::vector<KeyDescription>, size_t, IsLessThanFunctor> m_wordMap;
        
  }; // class CompletionGenerator


  /* ============== Non-member function declarations ============== */
  

} // namespace xComplete

/* ============ Definitions of inline & template functions ============ */

namespace xComplete {

  // Empty.
  
} // namespace xComplete

#endif /* #ifndef _XCOMPLETE_COMPLETIONGENERATOR_H_ */
