/**
***************************************************************************
* @file CompletionGenerator.cc
* Source file defining CompletionGenerator class.
*
* Copyright (C) 2006 David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: $
* $Date: $
* 
* $Log: $
***************************************************************************
**/

#include <fstream>
#include <sstream>
#include <dlrCommon/exception.h>
#include "CompletionGenerator.hh"

namespace xComplete {

  // The default constructor creates an empty database.
  CompletionGenerator::
  CompletionGenerator()
    : m_emptyCompletion(),
      m_maximumNumberOfWords(5000),
      m_timestamp(0),
      m_wordMap(IsLessThanFunctor())
  {
    // Empty.
  }

    
  // The destructor destroys the class instance and cleans up any
  // system resources.
  CompletionGenerator::
  ~CompletionGenerator()
  {
    // Empty.
  }


  // This member function adds a word to the database.
  void
  CompletionGenerator::
  addWord(const std::vector<KeyDescription>& word)
  {
    m_wordMap[word] = m_timestamp;
    ++m_timestamp;
    if(m_timestamp > m_maximumNumberOfWords) {
      this->purge();
    }
  }

    
  // This member function accepts a partially-typed word as its
  // argument, and returns a complete word which matches the input
  // word fragment.
  const std::vector<KeyDescription>&
  CompletionGenerator::
  getCompletion(const std::vector<KeyDescription>& wordBeginning,
                size_t completionIndex) const
  {
    // Build a sorted container for potential completions.
    std::map<size_t, const std::vector<KeyDescription>*> matchMap;

    // // Debug(xxx):
    // WordMapConstIterator iter0 = m_wordMap.begin();
    // while(iter0 != m_wordMap.end()) {
    //   for(size_t wordIndex = 0; wordIndex < (iter0->first).size();
    //       ++wordIndex) {
    //     std::cout << (iter0->first)[wordIndex].getRepresentation();
    //   }
    //   std::cout << "\t\t" << iter0->second << std::endl;
    //   ++iter0;
    // }
    
    // Add all of the potential completions to the sorted container.
    WordMapConstIterator iter = m_wordMap.lower_bound(wordBeginning);
    while(iter != m_wordMap.end()) {
      if(!this->isMatch(wordBeginning, iter->first)) {
        break;
      }
      matchMap.insert(std::make_pair(iter->second, &(iter->first)));
      ++iter;
    }

    // If there are no matches, we can't continue.
    if(matchMap.empty()) {
      return m_emptyCompletion;
    }
    
    // Choose the Nth-from-end match, with wraparound.
    completionIndex = completionIndex % matchMap.size();
    std::map<size_t, const std::vector<KeyDescription>*>::const_iterator
      matchIter = --(matchMap.end());
    while(completionIndex != 0) {
      --matchIter;
      --completionIndex;
    }

    // And return the associated word.
    return *(matchIter->second);
  }


  // This member function loads the database to a file.
  void
  CompletionGenerator::
  load(const std::string& fileName)
  {
    std::ifstream inputStream(fileName.c_str());
    if(!inputStream) {
      std::ostringstream message;
      message << "Couldn't open input file: " << fileName;
      DLR_THROW(dlr::IOException, "CompletionGenerator::load()",
                message.str().c_str());
    }

    // It's a lot easier to use a try block than to be constantly
    // testing whether the IO has succeeded, so we tell inputStream to
    // complain if anything goes wrong.
    // std::ios_base::iostate oldExceptionState = inputStream.exceptions();
    inputStream.exceptions(
      std::ios_base::badbit | std::ios_base::failbit | std::ios_base::eofbit);

    // Now on with the show.
    try{
      // Construct an InputStream instance so we can use our
      // convenience functions.
      dlr::InputStream stream(inputStream, dlr::InputStream::SKIP_WHITESPACE);

      // The preamble is short and sweet, but must be there.
      stream.expect("CompletionGenerator");
      stream.expect("{");

      // Read the individual data members.
      stream.expect("maximumNumberOfWords");
      stream.expect("=");
      inputStream >> m_maximumNumberOfWords;

      stream.expect("timestamp");
      stream.expect("=");
      inputStream >> m_timestamp;

      stream.expect("wordMap");
      stream.expect("=");
      m_wordMap = this->deserializeWordMap(inputStream);

      // Warning(xxx): instead of buffering info above and copying it
      // here, we simply copy above, which could leave us in a
      // half-read state if the config file is hosed.
      // 
      // And finally, copy the data.
      // Code here...
    } catch(std::ios_base::failure) {
      inputStream.close();
      std::ostringstream message;
      message << "Failed to read complete state from open input file: "
              << fileName;
      DLR_THROW(dlr::IOException, "CompletionGenerator::load()",
                message.str().c_str());
    }
    // inputStream.exceptions(oldExceptionState);
    inputStream.close();
  }
  

  // This member function writes the database to a file.
  void
  CompletionGenerator::
  save(const std::string& fileName)
  {
    std::ofstream outputStream(fileName.c_str());
    if(!outputStream) {
      DLR_THROW(dlr::IOException, "CompletionGenerator::save()",
                "Couldn't open output file.");
    }
    outputStream
      << "CompletionGenerator {\n\n"
      << "  maximumNumberOfWords = "
      << m_maximumNumberOfWords << "\n\n"
      << "  timestamp = "
      << m_timestamp << "\n\n"
      << "  wordMap = "
      << this->serializeWordMap(m_wordMap) << "\n\n"
      << "}" << std::endl;
    if(!outputStream) {
      outputStream.close();
      DLR_THROW(dlr::IOException, "CompletionGenerator::save()",
                "Error writing to open output file.");
    }
    outputStream.close();
  }

  
  // This function is used in reading the database from disk.
  std::map<std::vector<KeyDescription>, size_t,
           CompletionGenerator::IsLessThanFunctor>
  CompletionGenerator::
  deserializeWordMap(std::istream& inputStream)
  {
    // Construct an InputStream instance so we can use our
    // convenience functions.
    dlr::InputStream stream(inputStream, dlr::InputStream::SKIP_WHITESPACE);

    // Construct a map to hold the result.
    std::map<std::vector<KeyDescription>, size_t, IsLessThanFunctor> wordMap;

    // We need an opening brace to start the whole thing off.
    stream.expect("{");

    while(1) {
      // KeyDescription keyDescription;
      stream.skipWhiteSpace();
      if(stream.peek() == '}') {
        break;
      }

      // Read a key, value pair.
      stream.expect("{");

      // Key is a vector of KeyDescription.
      // Construct a vector to hold the result.
      std::vector<KeyDescription> word;
      stream.expect("{");
      while(1) {
        KeyDescription keyDescription;
        stream.skipWhiteSpace();
        if(stream.peek() == '}') {
          break;
        }
        inputStream >> keyDescription;
        word.push_back(keyDescription);
      }
      stream.expect("}");

      // Value is a size_t.
      size_t timestamp;
      inputStream >> timestamp;

      // Done with pair.
      stream.expect("}");

      // Record the newly read pair.
      wordMap.insert(std::make_pair(word, timestamp));
    }
    stream.expect("}");
    return wordMap;
  }


  // This function is used in writing the database to disk.
  std::string
  CompletionGenerator::
  serializeWordMap(
    const std::map<std::vector<KeyDescription>, size_t, IsLessThanFunctor>&
    wordMap)
  {
    typedef std::map<std::vector<KeyDescription>, size_t, IsLessThanFunctor>
      WordMapType;
    typedef WordMapType::const_iterator ConstIteratorType;
    
    std::ostringstream outputStream;
    outputStream << "{\n";
    ConstIteratorType iter = wordMap.begin();
    while(iter != wordMap.end()) {
      outputStream << "    {\n";
      outputStream << "      {\n";
      for(size_t vectorIndex = 0;
          vectorIndex < (iter->first).size();
          ++vectorIndex) {
        outputStream << "        " << (iter->first)[vectorIndex] << "\n";      
      }
      outputStream << "      }\n";
      outputStream << "      " << iter->second << "\n";
      outputStream << "    }\n";
      ++iter;
    }
    outputStream << "  }";
    return outputStream.str();
  }

    
  // this function returns true if the specified word is a
  // plausible completion for the specified word stub.
  bool
  CompletionGenerator::
  isMatch(const std::vector<KeyDescription>& stub,
          const std::vector<KeyDescription>& word) const
  {
    if(stub.size() > word.size()) {
      return false;
    }
    for(size_t index0 = 0; index0 < stub.size(); ++index0) {
      if(!isSymmetricMatch(stub[index0], word[index0])) {
        return false;
      }
    }
    return true;
  }
  
    
  // This function throws away old words, keeping only half the
  // number of words specified by m_maximumNumberOfWords, and
  // resetting timestamps so that the retained words have
  // timestamps from 0 to (m_maximumNumberOfWords / 2).
  void
  CompletionGenerator::
  purge()
  {
    size_t threshold = m_maximumNumberOfWords / 2;

    WordMapIterator mapIter = m_wordMap.begin();
    WordMapIterator mapEndIter = m_wordMap.end();

    // We'd like to operate directly on m_wordMap, but we're not sure
    // whether deleting elements of a std::map invalidates the
    // iterators, so instead we'll create a new map and (ugh) copy it.
    std::map<std::vector<KeyDescription>, size_t, IsLessThanFunctor> newMap;

    while(mapIter != mapEndIter) {
      if(mapIter->second >= threshold) {
        newMap.insert(
          std::make_pair(mapIter->first, mapIter->second - threshold));
      }
      ++mapIter;
    }
    m_wordMap = newMap;
    m_timestamp -= threshold;
  }


  bool
  CompletionGenerator::
  IsLessThanFunctor::
  operator()(const std::vector<KeyDescription>& word0,
             const std::vector<KeyDescription>& word1) const
  {
    typedef std::vector<KeyDescription>::const_iterator WordIterator;
    WordIterator iter0 = word0.begin();
    WordIterator iter1 = word1.begin();
    while((iter0 != word0.end()) && (iter1 != word1.end())) {
      if(isSymmetricGreater(*iter0, *iter1)) {
        return false;        
      } else if(isSymmetricGreater(*iter1, *iter0)) {
        return true;
      }
      ++iter0;
      ++iter1;
    }
    if(iter1 != word1.end()) {
      return true;
    }
    return false;
  }
  
    
} // namespace xComplete
