/**
***************************************************************************
* @file CompleterConfig.hh
* Header file declaring CompleterConfig class.
*
* Copyright (C) 2006 David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: $
* $Date: $
* 
* $Log: $
***************************************************************************
**/

#ifndef _XCOMPLETE_COMPLETERCONFIG_H_
#define _XCOMPLETE_COMPLETERCONFIG_H_

#include <string>
#include <vector>
#include "KeyDescription.hh"

namespace xComplete {

  /**
   ** The CompleterConfig class encapsulates all of the configuration
   ** variables for the Completer application.  Its only purpose is to
   ** keep these variables nicely grouped so that we can conveniently
   ** write routines to read and write them from a config file.
   **/
  class CompleterConfig {
  public:

    /* ============== Public types ============== */

    
    /* ============== Public member functions ============== */

    /** 
     * The constructor creates a new CompleterConfig instance.  If a
     * file name is specified, the configuration will be read from
     * file.  If no file name is specified, then the configuration
     * will be initialized to reasonable defaults.
     * 
     * @param configFileName This argument, if provided, specifies the
     * name of the file from which to read the configuration.
     */
    CompleterConfig(std::string configFileName="");


    /** 
     * The destructor destroys the CompleterConfig instance.
     */
    ~CompleterConfig();


    /** 
     * This function returns true of a backspace character should be
     * sent to undo the effect of pressing the completion key.
     * 
     * @return The return value is true of a backspace should be sent.
     */
    bool
    isCompletionKeyBackspaceRequired();


    /** 
     * This member function adds a regular expression to the list of
     * "valid" window names.  A "valid" window is one in which
     * completion and keystoke snooping is allowed.  I window will be
     * considered valid if its name matches any of the regular
     * expressions in the list.
     * 
     * @param regularExpression This argument 
     */
    void
    addWindowNamePattern(const std::string& regularExpression);


    /** 
     * This member function removes all regular expressions from the
     * list of valid window names.
     */
    void
    clearWindowNamePatterns() {m_windowNamePatterns.clear();}
    

    /** 
     * This member function returns a vector of all KeyDescription
     * instances which count as backspaces.
     * 
     * @return The return value is a vector of all the keystrokes
     * configured to be backspaces.
     */
    const std::vector<KeyDescription>&
    getBackspaceKeys() const {return m_backspaceKeys;}
    

    /** 
     * This member function returns a vector of all KeyDescription
     * instances which count as completion characters.  Whenever a
     * keystroke is received which matches one of the elements of this
     * vector, it means the user requested us to send a completion.
     * 
     * @return The return value is a vector of all the keystrokes
     * configured to be completion keys.
     */
    const std::vector<KeyDescription>&
    getCompletionKeys() const {return m_completionKeys;}


    /** 
     * This member function returns a vector of all KeyDescription
     * instances which count as modifier (shift, control, etc.) keys.
     * In general, keystrokes which matche one of the elements of this
     * vector should be ignored.
     * 
     * @return The return value is a vector of all the keystrokes
     * configured to be modifier keys.
     */
    const std::vector<KeyDescription>&
    getModifierKeys() const {return m_modifierKeys;}


    /** 
     * This member function returns the length at which a partially
     * typed word is long enough to merit searching for completions.
     * 
     * @return The return value is the minimum length at which to
     * start prompting with completions and accepting completion key
     * presses.
     */
    size_t
    getMinimumStubSize() const {return m_minimumStubSize;}


    /** 
     * This member function returns the shortest word length which
     * qualifies for inclusion in the database of learned words.
     * 
     * @return The return value is the minimum length at which a word
     * should be remembered and used for generating completions.
     */
    size_t
    getMinimumWordSize() const {return m_minimumWordSize;}
    

    /** 
     * This member function returns a vector of characters which
     * should be considered to break a word, but which do not cause
     * the cursor to move in an unusual way.  Examples are punctuation
     * whitespace, etc.  This function is very similar to
     * getSeparatorKeys(), but it is significantly easier to manually
     * configure.
     * 
     * @return The return value is a vector of the actual character
     * values.
     */
    const std::vector<KeyDescription::CharacterType>&
    getSeparatorCharacters() const {return m_separatorCharacters;}


    /** 
     * This member function returns a vector of KeyStroke instances
     * which should be considered to break a word, but which do not
     * cause the cursor to move in an unusual way.  Examples are
     * punctuation whitespace, etc.  This function is very similar to
     * getSeparatorCharacters().  It is included because there are
     * some some non-word keystrokes which are not representable as
     * chars.
     * 
     * @return The return value is a vector of KeyStroke instances.
     */
    const std::vector<KeyDescription>&
    getSeparatorKeys() const {return m_separatorKeys;}


    /** 
     * This member function returns a vector of perl-style regular
     * expressions.  The name of each window should be compared
     * against this list to see if keystrokes should be monitored (and
     * completions sent) when the user is typing in that window.
     * 
     * @return The return value a list of perl-style regular
     * expressions.
     */
    const std::vector<std::string>&
    getWindowNamePatterns() const {return m_windowNamePatterns;}
    

    /** 
     * This member function returns a vector of characters which
     * should be considered not to break a word.  Examples are all of
     * the alphanumeric charcters, and perhaps '_'.  This function is
     * very similar to getWordKeys(), but it is significantly easier
     * to manually configure.
     * 
     * @return The return value is a vector of the actual character
     * values.
     */
    const std::vector<KeyDescription::CharacterType>&
    getWordCharacters() const {return m_wordCharacters;}


    /** 
     * This member function returns a vector of KeyStroke instances
     * which should be considered not to break a word. This function
     * is very similar to getWordCharacters().  It is included to
     * handle any characters which should not break a word, but are
     * not easy to represent using a character.
     * 
     * @return The return value is a vector of KeyStroke instances.
     */
    const std::vector<KeyDescription>&
    getWordKeys() const {return m_wordKeys;}


    /** 
     * This member function causes the CompleterConfig instance to
     * load its state from a file.
     * 
     * @param fileName This argument is the name of the file to read.
     */
    void
    load(const std::string& fileName);


    /** 
     * This member function deletes the specified regular expression
     * from the the list of window name regular expressions (see
     * getWindowNamePatterns()).
     * 
     * @param regularExpression This argument is the regular
     * expression to delete.
     */
    void
    removeWindowNamePattern(const std::string& regularExpression);

    
    /** 
     * This member function causes the CompleterConfig instance to
     * save its state to a file.
     * 
     * @param fileName This argument is the name of the file to write.
     */
    void
    save(const std::string& fileName);


    
    /** 
     * This member function specifies which keys should count as
     * backspaces, so that the completer will know when the user has
     * backspaced over a a character.
     * 
     * @param backspaceKeys A vector of all the keystrokes which
     * should count as backspaces.
     */
    void
    setBackspaceKeys(const std::vector<KeyDescription>& backspaceKeys);


    /** 
     * This function sets whether or not a backspace character should
     * be sent to undo the effect of pressing the completion key.
     * 
     * @param isRequired A bool indicating whether or not the
     * backspace should be sent.
     */
    void
    setCompletionKeyBackspaceRequired(bool isRequired);

    
    /** 
     * This member function sets which keystrokes count as completion
     * characters.
     *
     * @param completionKeys A vector of keystrokes which should count
     * as completion keys.
     */
    void
    setCompletionKeys(
      const std::vector<KeyDescription>& completionKeys);
    

    /** 
     * This sets which keystrokes count as modifier (shift, control,
     * etc.) keys.  Keystrokes which matche one of the elements of
     * this vector will not be added to words when they are typed.
     * 
     * @param modifierKeys A vector of keystrokes corresponding to
     * shift, control, etc.
     */
    void
    setModifierKeys(
      const std::vector<KeyDescription>& modifierKeys);


    /** 
     * This member function sets which keystrokes should be considered
     * to break a word, but do not cause the cursor to move in an
     * unusual way.  Examples are punctuation whitespace, etc.
     * 
     * @param separatorKeys A vector of keystrokes which should break
     * words without making us lose our place.
     */
    void
    setSeparatorKeys(
      const std::vector<KeyDescription>& separatorKeys);

  private:

    std::istream&
    deserialize(std::istream& inputStream);


    std::vector<KeyDescription::CharacterType>
    deserializeCharacterVector(std::istream& inputStream);

    
    std::vector<std::string>
    deserializeStringVector(std::istream& inputStream);
    
    
    std::vector<KeyDescription>
    deserializeVector(std::istream& inputStream);
    

    std::string
    serialize();
    
    
    std::string
    serializeCharacterVector(
      const std::vector<KeyDescription::CharacterType>& inputVector);

      
    std::string
    serializeStringVector(const std::vector<std::string>& inputVector);

    
    std::string
    serializeVector(const std::vector<KeyDescription>& inputVector);


    std::vector<KeyDescription> m_backspaceKeys;
    bool m_completionKeyNeedsBackspace;
    std::vector<KeyDescription> m_completionKeys;
    size_t m_minimumStubSize;
    size_t m_minimumWordSize;
    std::vector<KeyDescription> m_modifierKeys;
    std::vector<KeyDescription::CharacterType> m_separatorCharacters;
    std::vector<KeyDescription> m_separatorKeys;
    std::vector<std::string> m_windowNamePatterns;
    std::vector<KeyDescription::CharacterType> m_wordCharacters;
    std::vector<KeyDescription> m_wordKeys;
    
  }; // class CompleterConfig
  
  /* ============== Non-member function declarations ============== */
  

} // namespace xComplete

/* ============ Definitions of inline & template functions ============ */

namespace xComplete {

  // Empty.
  
} // namespace xComplete

#endif /* #ifndef _XCOMPLETE_COMPLETERCONFIG_H_ */
